/*
*
*  A2DPD - Bluetooth A2DP daemon for Linux
*
*  Copyright (C) 2006-2007  Frédéric DALLEAU <frederic.dalleau@palmsource.com>
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <unistd.h>
#include <string.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <getopt.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/poll.h>

#include "a2dpd_output_a2dp.h"
#include "a2dpd_output_alsa.h"
#include "a2dpd_output_sco.h"
#include "a2dpd_protocol.h"
#include "a2dpd_dbus.h"
#include "a2dpd_mixer.h"
#include "a2dpd_tools.h"
#include "a2dpd_timer.h"
#include "a2dpd_ipc.h"
#include "a2dpd_avrcp.h"

#define MAXBLUETOOTHDEVICES    (3)
#define MAXCLIENTSPERDEVICE    (8)

#define POOLENTRYSIZE_A2DP     (A2DPD_BLOCK_SIZE)
#define POOLENTRYSIZE_SCO      (48)

#define A2DPD_CONFIG_FILE      ".a2dpdrc"

enum { DEVICE_STATE_NOSOUND, DEVICE_STATE_SOUND };
enum { CLIENT_STATE_DISCONNECTED, CLIENT_STATE_NEW, CLIENT_STATE_STREAMINGSETUP, CLIENT_STATE_STREAMING, CLIENT_STATE_CAPTURESETUP, CLIENT_STATE_CAPTURE, CLIENT_STATE_GETDELAY };
enum { CLIENT_FLAGS_TYPE_1_2 = 0, CLIENT_FLAGS_TYPE_3 = 1 };

static char g_sOutputFilename[512] = "";
static char g_srcfilename[512] = "";
static int g_brereadconfig = -1;
static int g_breversestereo = -1;
static int g_autoconnect = -1;
char* g_prefix="A2DPD";
FILE* g_fdout = DEFAULTFDOUT;
int g_bdebug = -1;
static int g_stdin = -1;

// if 1 then quit gently
static sig_atomic_t bSigINTReceived = 0;

// Data used to mix audio
typedef struct {
	void* lpVoid;
	uint32_t index_to_construct;
	uint32_t index_0;
	uint32_t size;
} CONVERTBUFFER;

typedef struct {
	AUDIOPACKETHEADER hdr;
	int len;
	char* buf;
} RINGINFO;

typedef struct {
	int socket;
	int lives;
	int state;
	int flags;
	int pcm_state;
	int pcm_start_threshold;
	int last_client_type;
	AUDIOPACKETHEADER last_hdr;
	int disabled;
	int blocked;
	int timeoutcount;
	AUDIOSTREAMINFOS StreamInfos;
	CONVERTBUFFER conv;
	int max_ring; // bytes
	int ring_in; // bytes
	int ring_out; // bytes
	char* ring_buffer;
	int ring_buffer_len;
	uint64_t app_ptr;
	uint64_t hw_ptr;
} BTA2DPPERCLIENTDATA;

#define WRITE_SIDE 0
#define READ_SIDE  1

enum {
	REDIRECT_A2DP=0,
	REDIRECT_ALSA=1,
	REDIRECT_NONE,
	REDIRECT_AUTO,
	REDIRECT_SCO,
};

// Data to keep per Bluetooth device
typedef struct {
	char addr[20];
	char plug[20];
	AUDIOMIXERDATA mixer;
	LPAVRCP lpAVRCP;
	int ctl_socket[2];
	int nb_clients;
	int bredirect;
	int a2dp_rate;
	int a2dp_channels;
	int a2dp_bitspersample;
	int a2dp_framesize;
	int a2dp_timeout;
	int sbcbitpool;
	int ring_in;
	int ring_out;
	RINGINFO ring[MAXCLIENTSRINGSIZE];
	BTA2DPPERCLIENTDATA clients[MAXCLIENTSPERDEVICE];
} BTA2DPPERDEVICEDATA;

// Data needed per Audio Streaming Client
typedef struct {
	BTA2DPPERDEVICEDATA* lpDevice;
	int sockfd;
	pthread_t thread;
} A2DPDCLIENT, *LPA2DPDCLIENT;

typedef struct {
	BTA2DPPERDEVICEDATA* lpDevice;
	LPA2DP a2dp;
	LPSCO sco;
	LPALSA alsa;
	TIMERINFO TimerInfos;
	int state_previous;
	time_t last_transfer_time;
	char* pcm_buffer;
	int count1;
	int count2;
	int count3;
} A2DP_HANDLER_STATE;

// Allocate a new device
BTA2DPPERDEVICEDATA* bta2dpdevicenew(char *addr)
{
	BTA2DPPERDEVICEDATA* lpDevice = mymalloc(sizeof(BTA2DPPERDEVICEDATA));
	DBG("");
	if (lpDevice) {
		memset(lpDevice, 0, sizeof(BTA2DPPERDEVICEDATA));
		strncpy(lpDevice->addr, addr, sizeof(lpDevice->addr));
		lpDevice->addr[sizeof(lpDevice->addr) - 1] = 0;
		a2dpd_signal_address_changed(lpDevice->addr);
		lpDevice->mixer.volume_speaker_left = A2DPD_VOLUME_MAX;
		lpDevice->mixer.volume_speaker_right = A2DPD_VOLUME_MAX;
		lpDevice->mixer.volume_micro_left = A2DPD_VOLUME_MAX;
		lpDevice->mixer.volume_micro_right = A2DPD_VOLUME_MAX;
		if(socketpair(AF_UNIX, SOCK_STREAM, 0, lpDevice->ctl_socket)<0) {
			DBG("Failed to create ctl_socket pair");
			memset(lpDevice->ctl_socket, 0, sizeof(lpDevice->ctl_socket));
		} else {
			a2dpd_signal_set_socket(lpDevice->ctl_socket[WRITE_SIDE]);
		}
		lpDevice->a2dp_timeout = read_config_int(g_srcfilename, "a2dpd", "timeout", 30 /*seconds */);
		lpDevice->bredirect = read_config_int(g_srcfilename, "a2dpd", "enableredirectalsa", REDIRECT_A2DP);
		// A2DP specific settings
		lpDevice->sbcbitpool = read_config_int(g_srcfilename, "a2dpd", "sbcbitpool", 32);
		lpDevice->a2dp_framesize = POOLENTRYSIZE_A2DP;
		lpDevice->a2dp_rate = read_config_int(g_srcfilename, "a2dpd", "rate", A2DPD_FRAME_RATE);
		lpDevice->a2dp_channels = read_config_int(g_srcfilename, "a2dpd", "channels", 2);
		lpDevice->a2dp_bitspersample = 16/8;
	}
	return lpDevice;
}

// Free a device
void bta2dpdevicefree(BTA2DPPERDEVICEDATA* lpDevice)
{
	int i = 0;
	if (lpDevice) {
		for (i = 0; i < MAXCLIENTSPERDEVICE; i++) {
			BTA2DPPERCLIENTDATA* lpClient = lpDevice->clients+i;
			close_socket(&lpClient->socket);
			safefree(lpClient->ring_buffer);
			lpClient->ring_buffer_len = 0;
		}
		a2dpd_signal_set_socket(-1);
		close_socket(&lpDevice->ctl_socket[READ_SIDE]);
		close_socket(&lpDevice->ctl_socket[WRITE_SIDE]);
		safefree(lpDevice);
	}
}

// handle sigterm to terminate properly
void sigint_handler(int sig)
{
	// Now we must quit properly
	DBG("handling SIGINT");
	bSigINTReceived = 1;
}

int client_ring_avail(BTA2DPPERCLIENTDATA* lpClient)
{
	int result = lpClient->max_ring - lpClient->ring_buffer_len;
	DBG4("ring_out:%d ring_in:%d max_ring:%d len:%d result:%d", lpClient->ring_out, lpClient->ring_in, lpClient->max_ring, lpClient->ring_buffer_len, result);

	return result;
}

int client_ring_toread(BTA2DPPERCLIENTDATA* lpClient)
{
	return lpClient->max_ring-client_ring_avail(lpClient);
}

void disconnect_client(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	DBG("");
	if(lpClient->lives) {
		close_socket(&lpClient->socket);
		lpClient->state = CLIENT_STATE_DISCONNECTED;
		lpClient->lives = 0;
		safefree(lpClient->ring_buffer);
		lpClient->ring_buffer_len = 0;
		lpClient->ring_out = 0;
		lpClient->ring_in = 0;
		lpClient->hw_ptr = 0;
		lpClient->app_ptr = 0;
		lpDevice->nb_clients--;
	}
}

void display_ring(BTA2DPPERCLIENTDATA* lpClient, int i)
{
	int j;
	char szLength[128] = "";

	if(lpClient->state != CLIENT_STATE_STREAMING)
		return;

	if(lpClient->max_ring == 0) {
		DBG("Invalid max_ring detected");
		return;
	}

	sprintf(szLength, "%d [in:%05d, out:%05d, len:%05d/%05d, hw:%ld, app:%ld]: ", i, lpClient->ring_in, lpClient->ring_out, lpClient->ring_buffer_len, lpClient->max_ring, (long int)lpClient->hw_ptr, (long int)lpClient->app_ptr);
	for(j = 0; j < 16; j++) {
		if(j < lpClient->ring_buffer_len * 16 / lpClient->max_ring)
			strcat(szLength, "#");
		else
			strcat(szLength, "=");
	}

	if(lpClient->hw_ptr > lpClient->app_ptr) {
		strcat(szLength, "ERROR HW>APP");
	}
	DBG7("%s", szLength);

	if(lpClient->hw_ptr > lpClient->app_ptr) {
		exit(1);
	}

	if(lpClient->app_ptr - lpClient->hw_ptr != lpClient->ring_buffer_len) {
		DBG2("hw:%ld - app:%ld = %05d != len:%05d", (long int)lpClient->app_ptr, (long int)lpClient->hw_ptr, (int)(lpClient->app_ptr - lpClient->hw_ptr), lpClient->ring_buffer_len);
		exit(1);
	}
}

// This function append data received from a client to the device ring buffer
void append_to_ring_buffer(BTA2DPPERCLIENTDATA* lpClient, CONVERTBUFFER* lpConvert, AUDIOPACKETHEADER* lpHdr)
{
	if(lpConvert->lpVoid != NULL) {
		// Append data to ring
		if (client_ring_avail(lpClient) >= lpConvert->size) {
			int step1 = min(lpConvert->size, lpClient->max_ring - lpClient->ring_in);
			int step2 = lpConvert->size - step1;
			int next_ring = (lpClient->ring_in + lpConvert->size) % lpClient->max_ring;
			
			if(step2 != 0) {
				DBG4("Overlapping buffer at append (step1:%d, step2:%d, size:%d, ring_in:%d, ring_out:%d)\n", step1, step2, lpConvert->size, lpClient->ring_in, lpClient->ring_out);
			}
			memcpy(lpClient->ring_buffer + lpClient->ring_in, lpConvert->lpVoid,         step1);
			memcpy(lpClient->ring_buffer,                     lpConvert->lpVoid + step1, step2);

			if(step2 > lpClient->ring_out) {
				DBG("Inconsistency 0");
			}

			if(step2 != 0 && next_ring != step2) {
				DBG("Inconsistency 1");
			}
			lpClient->ring_buffer_len += lpConvert->size;
			lpClient->app_ptr += lpConvert->size;
			DBG4("Appent orig:%d converted:%d bytes at ring_in:%d, next_ring:%d", lpHdr->pcm_buffer_size, lpConvert->size, lpClient->ring_in, next_ring);
			lpClient->ring_in = next_ring;
			display_ring(lpClient, 0);
			lpConvert->size = 0;
			lpConvert->index_to_construct = 0;
			lpConvert->index_0 = 0;
		} else {
			DBG("Ring buffer is full (avail:%d < converted:%d)", client_ring_avail(lpClient), lpConvert->size);
		}
	}
	// Reintegrate data in pool if not transmitted via bthandler thread
	safefree(lpConvert->lpVoid);
}

// Convert individual sample
inline void convert_sample(AUDIOSTREAMINFOS* lpStreamInfos, void* lpSample, void* lpConvertedSample, BTA2DPPERDEVICEDATA* lpDevice)
{
	// Signed 32bits pivot
	int32_t channel_1=0;
	int32_t channel_2=0;
	// Convert to pivot format
	if(lpStreamInfos->channels==1) {
		if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S8) {
			channel_1 = (*(((int8_t*)lpSample)+0))*256;
			channel_2 = (*(((int8_t*)lpSample)+0))*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_U8) {
			channel_1 = ((*(((int8_t*)lpSample)+0))-(int)128)*256;
			channel_2 = ((*(((int8_t*)lpSample)+0))-(int)128)*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S16_LE) {
			channel_1 = *(((int16_t*)lpSample)+0);
			channel_2 = *(((int16_t*)lpSample)+0);
		}
	} else if(lpStreamInfos->channels==2) {
		if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S8) {
			channel_1 = (*(((int8_t*)lpSample)+0))*256;
			channel_2 = (*(((int8_t*)lpSample)+1))*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_U8) {
			channel_1 = ((*(((int8_t*)lpSample)+0))-(int)128)*256;
			channel_2 = ((*(((int8_t*)lpSample)+1))-(int)128)*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S16_LE) {
			channel_1 = *(((int16_t*)lpSample)+0);
			channel_2 = *(((int16_t*)lpSample)+1);
		}
	} else {
		if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S8) {
			channel_1 = (*(((int8_t*)lpSample)+0))*256;
			channel_2 = (*(((int8_t*)lpSample)+1))*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_U8) {
			channel_1 = ((*(((int8_t*)lpSample)+0))-(int)128)*256;
			channel_2 = ((*(((int8_t*)lpSample)+1))-(int)128)*256;
		} else if(lpStreamInfos->format==A2DPD_PCM_FORMAT_S16_LE) {
			channel_1 = *(((int16_t*)lpSample)+0);
			channel_2 = *(((int16_t*)lpSample)+1);
		}
	}

	// Convert to destination format
	if(lpDevice->a2dp_channels==1) {
		if(lpDevice->a2dp_bitspersample==1) {
			*(int8_t*)lpConvertedSample=(channel_1+channel_2)/(2*256);
		} else if(lpDevice->a2dp_bitspersample==2) {
			*(int16_t*)lpConvertedSample=(channel_1+channel_2)/(2);
		}
	} else if(lpDevice->a2dp_channels==2) {
		if(lpDevice->a2dp_bitspersample==1) {
			*(((int8_t*)lpConvertedSample)+0)=channel_1/256;
			*(((int8_t*)lpConvertedSample)+1)=channel_2/256;
		} else if(lpDevice->a2dp_bitspersample==2) {
			*(((int16_t*)lpConvertedSample)+0)=channel_1;
			*(((int16_t*)lpConvertedSample)+1)=channel_2;
		}
	} else {
		memset(lpConvertedSample, 0, lpDevice->a2dp_bitspersample*lpDevice->a2dp_channels);
		if(lpDevice->a2dp_bitspersample==1) {
			*(((int8_t*)lpConvertedSample)+0)=channel_1/256;
			*(((int8_t*)lpConvertedSample)+1)=channel_2/256;
		} else if(lpDevice->a2dp_bitspersample==2) {
			*(((int16_t*)lpConvertedSample)+0)=channel_1;
			*(((int16_t*)lpConvertedSample)+1)=channel_2;
		}
	}
}

// This function convert a buffer to sample rate and format needed for device
void convert_rate(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClientData, void* pcm_buffer, int pcm_buffer_size, AUDIOSTREAMINFOS* lpStreamInfos, AUDIOPACKETHEADER* lpHdr)
{
	// We need this structure accross calls
	CONVERTBUFFER* lpConvert = &lpClientData->conv;

	if(lpConvert && lpStreamInfos && lpStreamInfos->bitspersample) {
		unsigned int pcm_buffer_index = 0;
		unsigned int pcm_buffer_index_0 = 0;
		unsigned int pcm_buffer_frame_bytes = (lpStreamInfos->channels*lpStreamInfos->bitspersample);
		unsigned int pcm_buffer_nframes = pcm_buffer_size/pcm_buffer_frame_bytes;
		unsigned int rate_multiplier = ((unsigned int)lpStreamInfos->rate)*256 / ((unsigned int)lpDevice->a2dp_rate);
		unsigned int convert_frame_bytes = (lpDevice->a2dp_channels*lpDevice->a2dp_bitspersample);
		void* lpConvertedSample = mymalloc(convert_frame_bytes);
		void* lpSample = NULL;
		//int i;

		lpConvert->index_0 = lpConvert->index_to_construct;
		lpConvert->index_to_construct = 0;
		while(pcm_buffer_index<pcm_buffer_nframes) {
			// Allocate destination if needed
			if(lpConvert->lpVoid==NULL) {
				lpConvert->lpVoid = mymalloc(POOLENTRYSIZE_A2DP);
				lpConvert->size = lpDevice->a2dp_framesize;
				lpConvert->index_to_construct = 0;
				lpConvert->index_0 = 0;
			}

			// Get pointer to sample to convert
			lpSample = pcm_buffer+(pcm_buffer_index*pcm_buffer_frame_bytes);

			// Conversion of individual samples
			convert_sample(lpStreamInfos, lpSample, lpConvertedSample, lpDevice);

			// Append converted sample to constructed blocks
			void* lpDest = lpConvert->lpVoid+((lpConvert->index_0+lpConvert->index_to_construct)*convert_frame_bytes);
			memcpy(lpDest, lpConvertedSample, convert_frame_bytes);

			// Fill next index
			lpConvert->index_to_construct++;

			// The index to fill will be mapped according to rates
			pcm_buffer_index = pcm_buffer_index_0 + ((lpConvert->index_to_construct*rate_multiplier)/256);

			// If constructed block is full, enqueue and allocate new
			if(((lpConvert->index_0+lpConvert->index_to_construct)*convert_frame_bytes)>=lpConvert->size) {
				DBG4("Can append (index_0:%d+to_construct:%d)*frame:%d=%d >= %d", lpConvert->index_0, lpConvert->index_to_construct, convert_frame_bytes, ((lpConvert->index_0+lpConvert->index_to_construct)*convert_frame_bytes), lpConvert->size);
				// Enqueue in ring buffer
				append_to_ring_buffer(lpClientData, lpConvert, lpHdr);

				// Store next index to read
				pcm_buffer_index_0 = pcm_buffer_index;
				pcm_buffer_index = pcm_buffer_index_0;
			}
		}

		safefree(lpConvertedSample);
	}
}

// This function manage volume change wanted by clients
void a2dpd_plugin_ctl_write(LPA2DPDCLIENT lpClient)
{
	AUDIOMIXERDATA AudioMixerData = INVALIDAUDIOMIXERDATA;

	DBG("CTL WRITE thread %d started", lpClient->sockfd);

	if (recv_socket(lpClient->sockfd, &AudioMixerData, sizeof(AudioMixerData)) == sizeof(AudioMixerData)) {
		if (AudioMixerData.volume_speaker_left != -1)
			lpClient->lpDevice->mixer.volume_speaker_left = AudioMixerData.volume_speaker_left;
		if (AudioMixerData.volume_speaker_left != -1)
			lpClient->lpDevice->mixer.volume_speaker_right = AudioMixerData.volume_speaker_right;
		if (AudioMixerData.volume_micro_left != -1)
			lpClient->lpDevice->mixer.volume_micro_left = AudioMixerData.volume_micro_left;
		if (AudioMixerData.volume_micro_left != -1)
			lpClient->lpDevice->mixer.volume_micro_right = AudioMixerData.volume_micro_right;

		// Notify other clients
		int notifyfd = make_udp_socket();
		send_socket(notifyfd, &AudioMixerData, sizeof(AudioMixerData));
		close_socket(&notifyfd);
	}
}

// This function manage volume read for client
void a2dpd_plugin_ctl_read(LPA2DPDCLIENT lpClient)
{
	AUDIOMIXERDATA AudioMixerData = INVALIDAUDIOMIXERDATA;
	DBG("CTL READ thread %d started", lpClient->sockfd);

	AudioMixerData = lpClient->lpDevice->mixer;

	send_socket(lpClient->sockfd, &AudioMixerData, sizeof(AudioMixerData));
}

// Retrieve an index in client table for device
int get_index_for_client(BTA2DPPERDEVICEDATA* lpDevice, A2DP_HANDLER_STATE* lpState)
{
	int i;

	// When a new client connects, clear sbc buffer
	if(lpDevice->nb_clients == 0)
		a2dp_reset_encoding_buffer(lpState->a2dp);

	// Find an index in clients table for the mixer
	for (i = 0; i < MAXCLIENTSPERDEVICE; i++) {
		BTA2DPPERCLIENTDATA* lpClient = lpDevice->clients+i;
		if (lpClient->lives == 0) {
			// FIXME Not sure this is safe but this is very unlikely to happen (should be fixed now due to monothreading)
			lpClient->lives = 1;
			lpClient->state = CLIENT_STATE_NEW;
			lpClient->ring_in = 0;
			lpClient->ring_out = 0;
			lpClient->max_ring = 0;
			lpClient->pcm_start_threshold = 0;
			safefree(lpClient->ring_buffer);
			lpClient->ring_buffer_len = 0;
			lpClient->hw_ptr = 0;
			lpClient->app_ptr = 0;
			lpClient->pcm_state = A2DPD_PCM_STATE_OPEN;
			lpClient->disabled = 0;
			lpClient->blocked = 0;
			lpDevice->nb_clients++;
			break;
		}
	}

	return i;
}

int stream_client_state_new(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 1;
	int client_type = INVALID_CLIENT_TYPE;

	lpClient->flags = 0;

	// Receive type of client
	if(recv_socket(lpClient->socket, &client_type, sizeof(client_type))==sizeof(client_type)) {
		// This client wants to send us pcm control data
		if (client_type == A2DPD_PLUGIN_CTL_WRITE) {
			DBG("Plugin ctl write ignored");
			//a2dpd_plugin_ctl_write(&lpDevice->clients[i]);
		}
		// This client wants to read our control status
		if (client_type == A2DPD_PLUGIN_CTL_READ) {
			DBG("Plugin ctl read ignored");
			//a2dpd_plugin_ctl_read(&lpDevice->clients[i]);
		}
		// This client wants to send us pcm stream
		if (client_type == A2DPD_PLUGIN_PCM_WRITE) {
			lpClient->state = CLIENT_STATE_STREAMINGSETUP;
			lpClient->flags = CLIENT_FLAGS_TYPE_1_2;
			bError = 0;
		}
		if (client_type == A2DPD_PLUGIN_PCM_WRITE_3) {
			lpClient->state = CLIENT_STATE_STREAMINGSETUP;
			lpClient->pcm_state = A2DPD_PCM_STATE_SETUP;
			lpClient->flags = CLIENT_FLAGS_TYPE_3;
			bError = 0;
		}
		// This client wants to send us pcm stream
		if (client_type == A2DPD_PLUGIN_PCM_READ) {
			lpClient->state = CLIENT_STATE_CAPTURESETUP;
			bError = 0;
		}
		// This client wants to send us pcm stream
		if (client_type == A2DPD_PLUGIN_GET_DELAY) {
			lpClient->state = CLIENT_STATE_GETDELAY;
			bError = 0;
		}
	}

	return bError;
}

int stream_client_state_streamingsetup(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient, int idx)
{
	int bError = 1;
	AUDIOSTREAMINFOS StreamInfos;
	if(recv_socket(lpClient->socket, &StreamInfos, sizeof(StreamInfos))==sizeof(StreamInfos)) {
		int bytes_per_frame = StreamInfos.bitspersample * StreamInfos.channels;
		lpClient->StreamInfos = StreamInfos;
		lpClient->state = CLIENT_STATE_STREAMING;
		lpClient->max_ring = StreamInfos.buffer_size * bytes_per_frame;
		lpClient->pcm_start_threshold = StreamInfos.start_threshold * bytes_per_frame;
		safefree(lpClient->ring_buffer);
		lpClient->ring_buffer = mymalloc(lpClient->max_ring);
		lpClient->ring_buffer_len = 0;
		lpClient->hw_ptr = 0;
		lpClient->app_ptr = 0;
		lpClient->ring_in = 0;
		lpClient->ring_out = 0;

		DBG("PLAYBACK thread client_index=%d, socket=%d, id=%d started (%d Hz, %d channels, %d bits, ring %d, threshold %d)", idx, lpClient->socket, StreamInfos.streamid, StreamInfos.rate, StreamInfos.channels, StreamInfos.bitspersample*8, lpClient->max_ring, lpClient->pcm_start_threshold);
		bError = 0;
	} else {
		DBG("Playback stream setup failed");
	}
	return bError;
}

int stream_client_send_delay_resp(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 1;
	GETDELAYRESP GetDelayResp;
	GetDelayResp.delay = client_ring_toread(lpClient) / lpDevice->a2dp_channels / lpDevice->a2dp_bitspersample;
	if(send_socket(lpClient->socket, &GetDelayResp, sizeof(GetDelayResp))==sizeof(GetDelayResp)) {
		bError = 0;
	} else {
		DBG("GetDelayResp failed");
	}
	return bError;
}

int stream_client_prepare(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient, int start_threshold)
{
	lpClient->ring_in = 0;
	lpClient->ring_out = 0;
	lpClient->hw_ptr = 0;
	lpClient->app_ptr = 0;
	lpClient->ring_buffer_len = 0;
	lpClient->pcm_start_threshold = min(lpClient->max_ring, start_threshold);
	lpClient->pcm_state = A2DPD_PCM_STATE_PREPARED;
	//FIXME check we're not leaking
	DBG3("pcm_start_threshold = %d", lpClient->pcm_start_threshold);

	return 0;
}

int stream_client_start(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	if(lpClient->pcm_state == A2DPD_PCM_STATE_PREPARED)
		lpClient->pcm_state = A2DPD_PCM_STATE_RUNNING;

	return 0;
}

int stream_client_drain(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	return 0;
}

int stream_client_pause_0(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	return 0;
}

int stream_client_pause_1(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	return 0;
}

int stream_client_resume(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	return 0;
}

int stream_client_send_pointer_resp(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 1;

	// Do not send pointer updates when prepared
	if(lpClient->state == A2DPD_PCM_STATE_PREPARED)
		return 0;

	GETPOINTERRESP GetPointerResp;

	GetPointerResp.hw_ptr   = lpClient->hw_ptr   / lpDevice->a2dp_channels / lpDevice->a2dp_bitspersample;
	GetPointerResp.app_ptr  = lpClient->app_ptr  / lpDevice->a2dp_channels / lpDevice->a2dp_bitspersample;
	GetPointerResp.pcm_state = lpClient->pcm_state;

	if(poll_out(lpClient->socket, 0)) {
		if(send_socket(lpClient->socket, &GetPointerResp, sizeof(GetPointerResp))==sizeof(GetPointerResp)) {
			bError = 0;
		} else {
			DBG("Error while sending data");
			disconnect_client(lpDevice, lpClient);
		}
	} else {
		DBG3("Cannot send pointer");
		bError = 0;
	}
	return bError;
}

int stream_client_stop(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	lpClient->pcm_state = A2DPD_PCM_STATE_SETUP;
	lpClient->ring_out = lpClient->ring_in;
	lpClient->hw_ptr = lpClient->app_ptr;
	lpClient->ring_buffer_len = 0;

	// Send pointers
	return stream_client_send_pointer_resp(lpDevice, lpClient);
}

int stream_client_state_streaming(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 1;
	int bCanReceive = 0;
	AUDIOPACKETHEADER hdr;
	GETDELAYREQ StreamReq = { A2DPD_PLUGIN_PCM_WRITE_3, 0 };

	if(lpClient->flags & CLIENT_FLAGS_TYPE_3) {
		if(lpClient->last_client_type == 0) {
			if(recv_socket(lpClient->socket, &StreamReq, sizeof(StreamReq)) != sizeof(StreamReq)) {
				StreamReq.action = 0;
				DBG("Received action failed");
			} else {
				DBG4("Received action %d", StreamReq.action);
			}
		} else {
			DBG3("Last action was %d", StreamReq.action);
			StreamReq.action = lpClient->last_client_type;
		}
	}

	if(StreamReq.action == A2DPD_PLUGIN_PCM_WRITE_3) {
		DBG4("A2DPD_PLUGIN_PCM_WRITE_3(disabled=%d, blocked=%d)", lpClient->disabled, lpClient->blocked);

		if(lpClient->flags & CLIENT_FLAGS_TYPE_3) {
			int factorClient = lpClient->StreamInfos.bitspersample * lpClient->StreamInfos.channels * lpClient->StreamInfos.rate;
			int factorServer = lpDevice->a2dp_bitspersample * lpDevice->a2dp_channels * lpDevice->a2dp_rate;

			if(lpClient->blocked || (recv_socket(lpClient->socket, &hdr, sizeof(hdr)) == sizeof(hdr))) {
				int needed;
				int avail = client_ring_avail(lpClient);

				// If blocked needed size is last size
				if(lpClient->blocked) {
					hdr = lpClient->last_hdr;
					DBG4("Blocked, using last value hdr.pcm_buffer_size:%d", hdr.pcm_buffer_size);
				} else {
					lpClient->last_hdr = hdr;
					DBG4("Not blocked, using received value hdr.pcm_buffer_size:%d", hdr.pcm_buffer_size);
				}
				needed = hdr.pcm_buffer_size * factorServer / factorClient ;

				DBG4("hdr.pcm_buffer_size:%d", hdr.pcm_buffer_size);
				DBG4("Needed (fs=%d / fc=%d) = %d", factorServer, factorClient, needed);
				DBG4("Avail (in=%d / out=%d) = %d", lpClient->ring_in, lpClient->ring_out, avail);

				if(needed <= lpClient->max_ring) {
					// Check we have place to write the data
					if(needed > avail) {
						DBG3("Client too fast(needed=%d > avail=%d), blocking", needed, avail);
						// Disabled socket for polling next time
						lpClient->disabled = 1;
						// Do not receive on this socket until there is space
						lpClient->blocked = 1;
					} else {
						lpClient->blocked = 0;
					}
					bCanReceive = 1;
				} else {
					//FIXME Well, we're dead...
					bError = 1;
				}
			} else {
				DBG("Did not receive pkt_hdr");
			}
		}

		if(!(lpClient->flags & CLIENT_FLAGS_TYPE_3)) {
			if(lpClient->pcm_state == A2DPD_PCM_STATE_SETUP) {
				stream_client_prepare(lpDevice, lpClient, lpClient->max_ring);
			}
			if(recv_socket(lpClient->socket, &hdr, sizeof(hdr)) == sizeof(hdr)) {
				lpClient->last_hdr = hdr;
				bCanReceive = 1;
			} else {
				DBG("Did not receive pkt_hdr");
			}
		}

		if(lpClient->blocked) {
			lpClient->last_client_type = StreamReq.action;
			bError = 0;
		} else {
			lpClient->last_client_type = 0;
			if(bCanReceive) {
				char *pcm_buffer = mymalloc(lpClient->last_hdr.pcm_buffer_size);
				if(pcm_buffer) {
					int result2 = recv_socket(lpClient->socket, pcm_buffer, lpClient->last_hdr.pcm_buffer_size);
					if (result2 == lpClient->last_hdr.pcm_buffer_size) {
						// Rate conversion
						DBG4("Received %d bytes", result2);
#if 0
						// No resampler version
						//FIXME Possibly a bug in append_to_ring_buffer related to overlapping buffers
						// Try with xmms that sends buffers of 576 samples
						CONVERTBUFFER Convert;
						Convert.lpVoid = pcm_buffer;
						Convert.size = result2;
						Convert.index_0 = 0;
						Convert.index_to_construct = 0;
						append_to_ring_buffer(lpClient, &Convert, &lpClient->last_hdr);
						pcm_buffer = NULL;
#else
						convert_rate(lpDevice, lpClient, pcm_buffer, result2, &lpClient->StreamInfos, &lpClient->last_hdr);
#endif

						// If start treshold is reached and we are prepared, then start stream
						if(lpClient->pcm_state == A2DPD_PCM_STATE_PREPARED) {
							if(lpClient->ring_buffer_len >= lpClient->pcm_start_threshold) {
								//DBG2("Start threshold reached (len:%d, thr:%d) when prepared", lpClient->ring_buffer_len, lpClient->pcm_start_threshold);
								//stream_client_start(lpDevice, lpClient);
							}
						}
						bError = 0;
					} else {
						DBG("Receive stream failed (%d/%d)", result2, lpClient->last_hdr.pcm_buffer_size);
					}
					safefree(pcm_buffer);
				} else {
					DBG("Couldn't alloc buffer %d", lpClient->last_hdr.pcm_buffer_size);
				}
			}
		}
	}

	if(StreamReq.action == A2DPD_PLUGIN_GET_DELAY) {
		DBG3("A2DPD_PLUGIN_GET_DELAY");
		bError = stream_client_send_delay_resp(lpDevice, lpClient);
	}

	if(StreamReq.action == A2DPD_PLUGIN_GET_POINTER) {
		DBG3("A2DPD_PLUGIN_GET_POINTER");
	}

	if(StreamReq.action == A2DPD_PLUGIN_PREPARE) {
		DBG3("A2DPD_PLUGIN_PREPARE");
		bError = stream_client_prepare(lpDevice, lpClient, StreamReq.param * lpDevice->a2dp_bitspersample * lpDevice->a2dp_channels);
	}

	if(StreamReq.action == A2DPD_PLUGIN_START) {
		DBG3("A2DPD_PLUGIN_START");
		bError = stream_client_start(lpDevice, lpClient);
	}

	if(StreamReq.action == A2DPD_PLUGIN_STOP) {
		DBG3("A2DPD_PLUGIN_STOP");
		bError = stream_client_stop(lpDevice, lpClient);
	}

	if(StreamReq.action == A2DPD_PLUGIN_DRAIN) {
		DBG3("A2DPD_PLUGIN_DRAIN");
		bError = stream_client_drain(lpDevice, lpClient);
	}

	if(StreamReq.action == A2DPD_PLUGIN_PAUSE_0) {
		DBG3("A2DPD_PLUGIN_PAUSE_0");
		bError = stream_client_pause_0(lpDevice, lpClient);
	}

	if(StreamReq.action == A2DPD_PLUGIN_PAUSE_1) {
		DBG3("A2DPD_PLUGIN_PAUSE_1");
		bError = stream_client_pause_1(lpDevice, lpClient);
	}
	
	if(StreamReq.action == A2DPD_PLUGIN_RESUME) {
		DBG3("A2DPD_PLUGIN_RESUME");
		bError = stream_client_resume(lpDevice, lpClient);
	}
	
	if(lpClient->flags & CLIENT_FLAGS_TYPE_3) {
		if(!bError) {
			bError = stream_client_send_pointer_resp(lpDevice, lpClient);
		}
	}

	return bError;
}

int stream_client_state_capturesetup(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient, int idx)
{
	int bError = 1;
	AUDIOSTREAMINFOS StreamInfos;
	if(recv_socket(lpClient->socket, &StreamInfos, sizeof(StreamInfos))==sizeof(StreamInfos)) {
		lpClient->StreamInfos = StreamInfos;
		lpClient->state = CLIENT_STATE_CAPTURE;
		DBG("CAPTURE thread %d.%d started (%d Hz, %d channels, %d bits)", idx, lpClient->socket, StreamInfos.rate, StreamInfos.channels, StreamInfos.bitspersample*8);
		bError = 0;
	} else {
		DBG("Capture stream setup failed");
	}
	return bError;
}

int stream_client_state_capture(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 0;
	char* tmpcapture = NULL;
	int tmpcapturelen = 0;
	uint32_t dummy = 0;

	// Read ring buffer data and write to client
	DBG("Capturing");

	if (lpDevice->ring_in != lpDevice->ring_out) {

		tmpcapture = lpDevice->ring[lpDevice->ring_out].buf;
		tmpcapturelen = lpDevice->ring[lpDevice->ring_out].len;

		DBG("Dequeue sco incoming stream %d", lpDevice->ring_out);

		// Move to next ring
		int next_ring = ((lpDevice->ring_out + 1) % MAXCLIENTSRINGSIZE);
		lpDevice->ring_out = next_ring;
	}

	if(recv_socket(lpClient->socket, &dummy, sizeof(dummy))==sizeof(dummy)) {
		DBG("RECEIVED sync from plugin");
		if(send_socket(lpClient->socket, &tmpcapturelen, sizeof(tmpcapturelen)) == sizeof(tmpcapturelen)) {
			if(tmpcapture != NULL) {
				if(send_socket(lpClient->socket, tmpcapture, tmpcapturelen) == tmpcapturelen) {
					bError = 0;
					DBG("Wrote stream %d", tmpcapturelen);
				} else {
					DBG("Couldn't write capture stream");
				}
				safefree(tmpcapture);
			} else {
				DBG("No capture sent");
			}
		} else {
			DBG("Couldn't send");
		}
	}
	return bError;
}

int stream_client_state_getdelay(BTA2DPPERDEVICEDATA* lpDevice, BTA2DPPERCLIENTDATA* lpClient)
{
	int bError = 1;
	GETDELAYREQ  GetDelayReq;

	if(recv_socket(lpClient->socket, &GetDelayReq, sizeof(GetDelayReq))==sizeof(GetDelayReq)) {
		bError = stream_client_send_delay_resp(lpDevice, lpClient);
	} else {
		DBG("GetDelayReq failed");
	}
	return bError;
}

void pollfd_cb_clients(struct pollfd* pollfds, void* param1, void* param2)
{
	BTA2DPPERDEVICEDATA* lpDevice = (BTA2DPPERDEVICEDATA*)param1;
	int i = (int)param2, bError = 1;
	BTA2DPPERCLIENTDATA* lpClient = lpDevice->clients+i;

	// Some data to read?
	if (pollfds->revents & (POLLIN|POLLPRI)) {
		// Already streaming?
		switch(lpDevice->clients[i].state) {
		case CLIENT_STATE_NEW:
			bError = stream_client_state_new(lpDevice, lpClient);
			break;
		case CLIENT_STATE_STREAMINGSETUP:
			bError = stream_client_state_streamingsetup(lpDevice, lpClient, i);
			break;
		case CLIENT_STATE_STREAMING:
			bError = stream_client_state_streaming(lpDevice, lpClient);
			break;
		case CLIENT_STATE_CAPTURESETUP:
			bError = stream_client_state_capturesetup(lpDevice, lpClient, i);
			break;
		case CLIENT_STATE_CAPTURE:
			bError = stream_client_state_capture(lpDevice, lpClient);
			break;
		case CLIENT_STATE_GETDELAY:
			bError = stream_client_state_getdelay(lpDevice, lpClient);
			break;
		default:
			DBG("Invalid state");
			break;
		}
	} else {
		// Increase client timeout counter
		//DBG("No data to read");
		bError = 0;
		lpClient->timeoutcount++;
	}

	// Reset client timeout
	if(!bError) {
		lpClient->timeoutcount=0;
	} else {
		lpClient->timeoutcount++;
	}

	// Remove client for poll at next iteration if an error is detected
	if ((lpClient->timeoutcount>100) || bError || (pollfds->revents & (POLLERR|POLLHUP|POLLNVAL))) {
		DBG("Client %d: Error detected (err=%d, toc=%d, err=%d, hup=%d)", i, bError,
			lpClient->timeoutcount,
			(pollfds->revents & POLLERR)?1:0,
			(pollfds->revents & POLLHUP)?1:0
			);
		disconnect_client(lpDevice, lpClient);
	}
}

int retrieve_data_from(BTA2DPPERDEVICEDATA* lpDevice, char** pcm_buffers, int nbuffers, int* pcm_buffers_size, AUDIOPACKETHEADER* hdr)
{
	int i;
	int new_state = DEVICE_STATE_NOSOUND;

	// Retrieve data for client where it is available
	for (i = 0; i < nbuffers; i++) {
		BTA2DPPERCLIENTDATA* lpClient = lpDevice->clients+i;
		if (lpClient->lives) {
			if(lpClient->pcm_state == A2DPD_PCM_STATE_RUNNING) {
				if (lpClient->ring_buffer_len > 0) {
					int block_size = min(A2DPD_BLOCK_SIZE, lpClient->ring_buffer_len);
					int step1 = min(block_size, lpClient->max_ring - lpClient->ring_in);
					int step2 = block_size - step1;
					int next_ring = ((lpClient->ring_out + block_size) % lpClient->max_ring);
					DBG4("Running: %d %ld %ld, taking %d bytes", lpClient->ring_buffer_len, (long)lpClient->hw_ptr, (long)lpClient->app_ptr, block_size);
					pcm_buffers[i] = mymalloc(block_size);
					pcm_buffers_size[i] = block_size;
					memcpy(pcm_buffers[i],         lpClient->ring_buffer + lpClient->ring_out, step1);
					memcpy(pcm_buffers[i] + step1, lpClient->ring_buffer,                      step2);
					lpClient->ring_buffer_len -= block_size;
					lpClient->hw_ptr += block_size;

					if(lpClient->hw_ptr > lpClient->app_ptr) {
						DBG("Inconsistency 2 (blocksize=%ld, ring_buffer_len=%ld, hw=%ld > app=%ld)", (long)block_size, (long)lpClient->ring_buffer_len, (long)lpClient->hw_ptr, (long)lpClient->app_ptr);
					}

					// Move to next block
					lpClient->ring_out = next_ring;
	
					// Remember we got some sound
					new_state = DEVICE_STATE_SOUND;

					// Allow the client for polling later
					lpClient->disabled = 0;
				} else {
					lpClient->pcm_state = A2DPD_PCM_STATE_XRUN;
					DBG("Underrun: len=%d hw=%ld app=%ld", lpClient->ring_buffer_len, (long)lpClient->hw_ptr, (long)lpClient->app_ptr);
				}
			}
			display_ring(lpClient, i);
			stream_client_send_pointer_resp(lpDevice, lpDevice->clients+i);
		}
	}

	return new_state;
}

void a2dpd_save_setup(BTA2DPPERDEVICEDATA* lpDevice) {
	
	DBG("Saving configuration to %s", g_srcfilename);
	if(write_config_string(g_srcfilename, "a2dpd", "address", lpDevice->addr)==0) {
		DBG("Saved configuration sucessfully");
	} else {
		DBG("Failed saving configuration");
	}
}

void auto_connect(A2DP_HANDLER_STATE* state, int bForce)
{
	// Try to connect only when there is a stream
	if(g_autoconnect || bForce) {
		if(state->lpDevice->bredirect == REDIRECT_A2DP)
			a2dp_state_connect(state->a2dp);
		if(state->lpDevice->bredirect == REDIRECT_SCO)
			sco_state_connect(state->sco);
		if(state->lpDevice->bredirect == REDIRECT_ALSA)
			alsa_state_connect(state->alsa);
		if(state->lpDevice->bredirect == REDIRECT_AUTO) {
			a2dp_state_connect(state->a2dp);
			alsa_state_connect(state->alsa);
		}
	}
}

void auto_disconnect(A2DP_HANDLER_STATE* state, int bForce)
{
	if(bForce || (g_autoconnect && (state->last_transfer_time>0) && ((time(NULL)-state->last_transfer_time)>state->lpDevice->a2dp_timeout))) {
		if(a2dp_is_connecting(state->a2dp)) {
			state->last_transfer_time = 0;
			a2dp_state_disconnect(state->a2dp);
		}
		if(sco_is_connecting(state->sco)) {
			state->last_transfer_time = 0;
			sco_state_disconnect(state->sco);
		}
		if(alsa_is_connecting(state->alsa)) {
			state->last_transfer_time = 0;
			alsa_state_disconnect(state->alsa);
		}
	}
}

void pollfd_cb_ctl_socket(struct pollfd* pollfds, A2DP_HANDLER_STATE* state, void* param2)
{
	// Manual connection with or without stream
	if(pollfds->revents & POLLIN) {
		unsigned char order, param;
		int lparam = 0;
		char addr[32];
		char* redirect = NULL;
		read(pollfds->fd, &order, sizeof(order));
		DBG("Processing '%c' %d", order, order);
		switch(order) {
		case 'a':
			read(pollfds->fd, &param, sizeof(param));
			if(param == 'a')
				g_autoconnect = !g_autoconnect;
			else
				g_autoconnect = (param=='1')?1:0;
			DBG("Autoconnect %s to %s", (param == 'a')?"swapped":"set", g_autoconnect?"on":"off");
			break;
		case 'c':
			auto_connect(state, 1);
			break;
		case 'd':
			auto_disconnect(state, 1);
			break;
		case 'D':
			avrcp_disconnect(state->lpDevice->lpAVRCP);
			break;
		case 'p':
			a2dp_state_suspend(state->a2dp);
			sco_state_suspend(state->sco);
			break;
		case 's':
			a2dp_state_startstream(state->a2dp);
			sco_state_startstream(state->sco);
			break;
		case 'w':
			a2dpd_save_setup(state->lpDevice);
			break;
		case 't':
			auto_disconnect(state, 1);
			// FIXME disconnect sco or a2dpd when changing
			read(pollfds->fd, &addr, sizeof(addr));
			if(strcasecmp(addr, "none")==0) {
				state->lpDevice->bredirect = REDIRECT_NONE;
				redirect = "None";
			} else if(strcasecmp(addr, "auto")==0) {
				state->lpDevice->bredirect = REDIRECT_AUTO;
				redirect = "Auto";
			} else if(strcasecmp(addr, "alsa")==0) {
				state->lpDevice->bredirect = REDIRECT_ALSA;
				redirect = "Alsa";
			} else if(strcasecmp(addr, "sco")==0) {
				state->lpDevice->bredirect = REDIRECT_SCO;
				redirect = "Sco";
			} else if(strcasecmp(addr, "a2dp")==0) {
				state->lpDevice->bredirect = REDIRECT_A2DP;
				redirect = "A2dp";
			} else {
				redirect = &addr[0];
				strcpy(state->lpDevice->addr, addr);
				a2dp_set_dst_addr(state->a2dp, addr);
				sco_set_dst_addr(state->sco, addr);
				a2dpd_signal_address_changed(addr);
			}
			DBG("SetAddress to %s", redirect);
			auto_disconnect(state, 1);
			break;
		case 'v':
			lparam = A2DPD_VOLUME_MAX;
			read(pollfds->fd, &lparam, sizeof(lparam));
			lparam = min(max(0, lparam), A2DPD_VOLUME_MAX);
			state->lpDevice->mixer.volume_speaker_left = lparam;
			state->lpDevice->mixer.volume_speaker_right = lparam;
			DBG("SetVolume to %d", lparam);
			break;
		case 'f':
			read(pollfds->fd, &lparam, sizeof(lparam));
			a2dp_set_user_flags(state->a2dp, lparam);
			DBG("SetFlags to %d", lparam);
			break;
		case 'r':
			read(pollfds->fd, &lparam, sizeof(lparam));
			g_brereadconfig = lparam;
			DBG("SetRereadConfig to %d", lparam);
			break;
		default:
			DBG("Invalid command [%c] [%d] on ctl_socket", order, order);
			break;
		}
	}
}	

void display_daemon_state(A2DP_HANDLER_STATE* state, AUDIOPACKETHEADER* hdr, int state_current)
{
	if(state->TimerInfos.display == 0)
		return;

	if(g_brereadconfig) {
		char addr[20];
		char plug[20];
		int bredirect = read_config_int(g_srcfilename, "a2dpd", "enableredirectalsa", REDIRECT_A2DP);

		read_config_string(g_srcfilename, "a2dpd", "address", addr, sizeof(addr), "");
		read_config_string(g_srcfilename, "a2dpd", "alsaoutput", plug, sizeof(plug), "");
		if((strcmp(addr, state->lpDevice->addr) != 0) || (strcmp(plug, state->lpDevice->plug) != 0) || (bredirect != state->lpDevice->bredirect)) {
			DBG("File change detected");
			// Force destroy, device will be recreated upon audio incoming
			strcpy(state->lpDevice->addr, addr);
			strcpy(state->lpDevice->plug, plug);
			state->lpDevice->bredirect = bredirect;
			a2dp_set_dst_addr(state->a2dp, addr);
			sco_set_dst_addr(state->sco, addr);
			a2dpd_signal_address_changed(addr);
			auto_disconnect(state, 1);
		}
	}

	if(a2dp_get_user_flags(state->a2dp) & A2DPD_FLAGS_DISPLAYSTATE) {
		struct timeval now, lat;
		gettimeofday(&now, NULL);
		timersub(&now, &hdr->packet_date, &lat);

		DBG("[%d,%d|%d,%d] %s %s clients=%d freq=%d cnt=%d (%d/%d) poll=%d latency=%d %s",
			state->lpDevice->mixer.volume_speaker_left,
			state->lpDevice->mixer.volume_speaker_right,
			state->lpDevice->mixer.volume_micro_left,
			state->lpDevice->mixer.volume_micro_right,
			(state_current==DEVICE_STATE_SOUND)?"playing":"silent",
			a2dp_is_connected(state->a2dp)?"connected":"disconnected",
			state->lpDevice->nb_clients,
			state->TimerInfos.display,
			state->TimerInfos.itotalcount,
			state->count1,
			state->count2,
			state->count3,
			(lat.tv_sec==0)?(int)lat.tv_usec:(int)lat.tv_sec,
			(lat.tv_sec==0)?"us.":"sec.");
		state->count1=0;
		state->count2=0;
		state->count3=0;
	}
}

void handle_state_change(BTA2DPPERDEVICEDATA* lpDevice, LPA2DP a2dp, LPSCO sco)
{
	if(a2dp_is_connected(a2dp) || sco_is_connected(sco)) {
		a2dpd_signal_state(CONNECTED, lpDevice->addr);
	} else if(a2dp_is_connecting(a2dp) || sco_is_connecting(sco)) {
		a2dpd_signal_state(CONNECTING, lpDevice->addr);
	} else {
		a2dpd_signal_state(DISCONNECTED, "");
	}
}

void setup_resampler(BTA2DPPERDEVICEDATA* lpDevice)
{
	// Setup the resampler to use the correct rate
	if(lpDevice->bredirect != REDIRECT_SCO) {
		// Alsa or A2DP ...
		if(lpDevice->a2dp_framesize != POOLENTRYSIZE_A2DP) {
			lpDevice->a2dp_framesize = POOLENTRYSIZE_A2DP;
			lpDevice->a2dp_rate = read_config_int(g_srcfilename, "a2dpd", "rate", A2DPD_FRAME_RATE);
			lpDevice->a2dp_channels = read_config_int(g_srcfilename, "a2dpd", "channels", 2);
			lpDevice->a2dp_bitspersample = 16/8;
		}
	} else {
		// ... or SCO
		if(lpDevice->a2dp_framesize != POOLENTRYSIZE_SCO) {
			lpDevice->a2dp_framesize = POOLENTRYSIZE_SCO;
			lpDevice->a2dp_rate = 8000;
			lpDevice->a2dp_channels = 1;
			lpDevice->a2dp_bitspersample = 2;
		}
	}
}

void send_sco_incoming_packets(BTA2DPPERDEVICEDATA* lpDevice, char* scoFrame, int scoFrameSize)
{
	if(scoFrameSize>0 && (lpDevice->bredirect == REDIRECT_SCO)) {
		char* tmp = mymalloc(scoFrameSize);
		if(tmp) {
			memcpy(tmp, scoFrame, scoFrameSize);
			int next_ring = (lpDevice->ring_in+1)%MAXCLIENTSRINGSIZE;
			if(next_ring != lpDevice->ring_out) {
				lpDevice->ring[lpDevice->ring_in].len = scoFrameSize;
				lpDevice->ring[lpDevice->ring_in].buf = tmp;
				lpDevice->ring_in = next_ring;
			} else {
				safefree(tmp);
			}
		}
	}
}

int transfer_outgoing_packets(A2DP_HANDLER_STATE* state, int pcm_buffer_filed_size, AUDIOPACKETHEADER* hdr)
{
	// Transfer takes place by lpDevice->a2dp_framesize bytes blocks
	int bError = 0;
	int blockstart = 0;
	int blocksize = state->lpDevice->a2dp_framesize;

	// Send data to BT headset
	while (!bError && blockstart < pcm_buffer_filed_size) {
		int transfer = -1;

		blocksize = (pcm_buffer_filed_size < state->lpDevice->a2dp_framesize) ? pcm_buffer_filed_size : state->lpDevice->a2dp_framesize;

		if(state->lpDevice->bredirect == REDIRECT_ALSA) {
			transfer = alsa_transfer_raw(state->alsa, state->pcm_buffer + blockstart, blocksize);
		} else if(state->lpDevice->bredirect == REDIRECT_A2DP) {
			transfer = a2dp_transfer_raw(state->a2dp, state->pcm_buffer + blockstart, blocksize, hdr);
		} else if(state->lpDevice->bredirect == REDIRECT_SCO) {
			transfer = sco_transfer_raw(state->sco, state->pcm_buffer + blockstart, blocksize);
		} else if(state->lpDevice->bredirect == REDIRECT_AUTO) {
			// transfer to a2dp if a2dp is connected, else transfer to alsa
			if(a2dp_is_connected(state->a2dp)) {
				transfer = a2dp_transfer_raw(state->a2dp, state->pcm_buffer + blockstart, blocksize, hdr);
			} else {
				transfer = alsa_transfer_raw(state->alsa, state->pcm_buffer + blockstart, blocksize);
			}
		} else {
			// Fake transfer
			transfer = blocksize;
		}

		// If 0, then we must retry the encoding
		if (transfer >= 0) {
			state->last_transfer_time = time(NULL);
			blockstart += transfer;
			//DBG("transfered");
			a2dp_timer_notifyframe(&state->TimerInfos);
		} else {
			DBG("Error in transfer (transfer=%d)", transfer);
			bError = 1;
		}
	}

	return !bError;
}

/////////////////////////////////
// This function handle one stream
int a2dp_handler(A2DP_HANDLER_STATE* state)
/////////////////////////////////
{
	BTA2DPPERDEVICEDATA* lpDevice = state->lpDevice;
	int i;
	int result = 0;
	int pcm_buffer_filed_size = 0;
	char *pcm_buffers[MAXCLIENTSPERDEVICE];
	int pcm_buffers_size[MAXCLIENTSPERDEVICE];
	int state_current = DEVICE_STATE_NOSOUND;
	AUDIOPACKETHEADER hdr;

	memset(&hdr, 0, sizeof(hdr));
	memset(pcm_buffers, 0, sizeof(pcm_buffers));
	memset(pcm_buffers_size, 0, sizeof(pcm_buffers_size));

	// Retrieve streams
	state_current = retrieve_data_from(lpDevice, pcm_buffers, MAXCLIENTSPERDEVICE, pcm_buffers_size, &hdr);

	// Mix incoming packets
	pcm_buffer_filed_size = audio_mixer_16bits(state->pcm_buffer, pcm_buffers, MAXCLIENTSPERDEVICE, pcm_buffers_size, lpDevice->mixer.volume_speaker_left, lpDevice->mixer.volume_speaker_right, A2DPD_VOLUME_MAX, g_breversestereo);

	// Free no longer used audio blocks
	for (i = 0; i < MAXCLIENTSPERDEVICE; i++) {
		safefree(pcm_buffers[i]);
	}

	// Send mixed audio stream to clients
	switch (state_current) {
	case DEVICE_STATE_SOUND:
		state->count1++;
		auto_connect(state, 0);

		if(!transfer_outgoing_packets(state, pcm_buffer_filed_size, &hdr)) {
			auto_disconnect(state, 1);
		}

		result = 1;
		break;
	case DEVICE_STATE_NOSOUND:
		state->count2++;
		if (state->state_previous == DEVICE_STATE_SOUND) {
			if (a2dp_get_user_flags(state->a2dp) & A2DPD_FLAGS_DISPLAYDRYSTREAM) {
				DBG("Sound stream dry");
			}
		}

		// auto disconnect when there is no stream for a longtime
		auto_disconnect(state, 0);
		break;
	}

	// Display daemon state if enabled
	display_daemon_state(state, &hdr, state_current);

	// Remember state
	state->state_previous = state_current;

	return result;
}

void pollfd_cb_stdin(struct pollfd* pollfds, BTA2DPPERDEVICEDATA* lpDevice, void* param2)
{
	if(pollfds->revents & POLLIN) {
		DBG("Data on stdin");
		char c [512];
		fgets(c, sizeof(c), stdin);
		if(c[0]) {
			// write valid commands to ctl_socket
			write(lpDevice->ctl_socket[WRITE_SIDE], &c, sizeof(c));
			DBG("Sent %c %d", c[0], c[0]);
		}
	}
}

void pollfd_cb_sockfd(struct pollfd* pollfds, BTA2DPPERDEVICEDATA* lpDevice, A2DP_HANDLER_STATE* a2dp_handler_state)
{
	// poll new stream clients
	if(pollfds->revents & POLLIN) {
		DBG2("Polling new stream client");
		int new_fd = accept_socket(pollfds->fd);

		// Handle connection if it is not the final dummy client
		if (!bSigINTReceived && new_fd > 0) {
			int client_index = get_index_for_client(lpDevice, a2dp_handler_state);
			DBG("Client %d got index %d", new_fd, client_index);

			if(client_index<MAXCLIENTSPERDEVICE) {
				lpDevice->clients[client_index].socket = new_fd;
			} else {
				DBG("Too many clients");
				close_socket(&new_fd);
			}
		} else {
			close_socket(&new_fd);
		}
	}
}

void pollfd_cb_avdtpfd(struct pollfd* pollfds, BTA2DPPERDEVICEDATA* lpDevice, A2DP_HANDLER_STATE* a2dp_handler_state)
{
	// poll new avdtp clients
	if(pollfds->revents & POLLIN) {
		DBG2("Polling new avdtp client");
		char szRemote[20];
		int incoming_socket = a2dp_wait_connection(pollfds->fd, szRemote, sizeof(szRemote), NULL);
		DBG("AVDTP Accepted %s on socket %d", szRemote, incoming_socket);

		if(incoming_socket>0) {
			avrcp_new_client(lpDevice->lpAVRCP);
		}

		// Give the socket to state machine if possible
		if(incoming_socket>0) {
			// A2DP is not already connected
			if(a2dp_state_use_socket(a2dp_handler_state->a2dp, incoming_socket)<0) {
				DBG("State machine is busy, closing socket %d", incoming_socket);
				close_socket(&incoming_socket);
			} else {
				// Someone connected, if we don't send him data, we will disconnect him
				a2dp_handler_state->last_transfer_time = time(NULL);
			}
		}
	}
}

// server processing loop
int main_loop(char *addr)
{
	int i, p, polltimeout, delay, sockfd, avdtpfd, result = 0;
	char scoFrame[512];
	ssize_t scoFrameSize = 0;
	BTA2DPPERDEVICEDATA* lpDevice = NULL;
	A2DP_HANDLER_STATE a2dp_handler_state = {0};
	A2DPSETTINGS settings;

	DBG("");
	// Master socket
	sockfd = make_server_socket();
	if(sockfd<0) {
		DBG("Cannot get UNIX socket");
		return -1;
	}

	avdtpfd = a2dp_make_listen_socket(25);
	if(avdtpfd<0) {
		DBG("*** Cannot get AVDTP socket, incoming AVDTP connections are impossible ***");
	}

	lpDevice = bta2dpdevicenew(addr);

	memset(&settings, 0, sizeof(settings));
	strncpy(settings.bdaddr, lpDevice->addr, sizeof(settings.bdaddr)-1);
	settings.framerate = lpDevice->a2dp_rate;
	settings.channels = lpDevice->a2dp_channels;
	settings.sbcbitpool = lpDevice->sbcbitpool;
	settings.flags = read_config_int(g_srcfilename, "a2dpd", "flags", 0);

	a2dp_handler_state.lpDevice = lpDevice;
	a2dp_handler_state.last_transfer_time = -1;
	a2dp_handler_state.pcm_buffer = mymalloc(lpDevice->a2dp_framesize);
	a2dp_handler_state.state_previous = DEVICE_STATE_NOSOUND;

	a2dpd_signal_state(DISCONNECTED, "");

	// Connect to the A2DP device
	a2dp_handler_state.a2dp = a2dp_new(&settings);
	a2dp_handler_state.alsa = alsa_new(lpDevice->plug, lpDevice->a2dp_rate);
	a2dp_handler_state.sco = sco_new(lpDevice->addr);

	DBG("Bluetooth Device Settings [%d hz, %d channels, %d bits]", lpDevice->a2dp_rate, lpDevice->a2dp_channels, lpDevice->a2dp_bitspersample*8);

	// This timer is used to sync bluetooth sound emission
	// This is because not all device have a queue for incoming sample
	// And device who don't have a queue won't react correctly
	// They require precise timing for sending samples
	memset(&a2dp_handler_state.TimerInfos, 0, sizeof(a2dp_handler_state.TimerInfos));
	a2dp_handler_state.TimerInfos.fpsX = lpDevice->a2dp_rate * lpDevice->a2dp_channels * lpDevice->a2dp_bitspersample * TIMERFACTOR / lpDevice->a2dp_framesize;

	lpDevice->lpAVRCP = avrcp_new(g_srcfilename);

	while (!bSigINTReceived) {
		struct pollfd pollfds[MAXCLIENTSPERDEVICE + 3/*stdin/sockfd/avdtpfd*/+3/*a2dp*/+3/*sco*/+0/*alsa*/+17/*avrcp*/+2/*dbus*/];
		struct pollfd_cb_inf pollfds_cb[MAXCLIENTSPERDEVICE + 3/*stdin/sockfd/avdtpfd*/+3/*a2dp*/+3/*sco*/+0/*alsa*/+17/*avrcp*/+2/*dbus*/];
		struct pollinfo pollinfos = {
			.pollfds=pollfds,
			.pollfds_cb=pollfds_cb,
			.maxfd=ARRAY_SIZE(pollfds),
			.pollfdcount=0,
			.polltimeout=-1
			};

		if(g_stdin)
			add_fd_to_poll(&pollinfos, fileno(stdin), POLLIN, -1, (fnpollfd_cb)pollfd_cb_stdin, lpDevice, 0);
		add_fd_to_poll(&pollinfos, sockfd, POLLIN, -1, (fnpollfd_cb)pollfd_cb_sockfd, lpDevice, &a2dp_handler_state);
		// If the listening AVDTP socket wasn't created, it is possible not to poll it
		if(avdtpfd>=0)
			add_fd_to_poll(&pollinfos, avdtpfd, POLLIN, -1, (fnpollfd_cb)pollfd_cb_avdtpfd, lpDevice, &a2dp_handler_state);

		for(i=0; i<MAXCLIENTSPERDEVICE; i++) {
			BTA2DPPERCLIENTDATA* lpClient = lpDevice->clients+i;
			if(lpClient->lives && !lpClient->disabled) {
				add_fd_to_poll(&pollinfos, lpClient->socket, POLLIN, -1, (fnpollfd_cb)pollfd_cb_clients, lpDevice, (void*)i);
			}
		}
		a2dp_add_fd_to_poll(a2dp_handler_state.a2dp, &pollinfos);
		sco_add_fd_to_poll(a2dp_handler_state.sco, &pollinfos);
		avrcp_add_fd_to_poll(lpDevice->lpAVRCP, &pollinfos);
		a2dpd_signal_add_fd_to_poll(&pollinfos);

		if(lpDevice->ctl_socket[READ_SIDE]>0)
			add_fd_to_poll(&pollinfos, lpDevice->ctl_socket[READ_SIDE], POLLIN, -1, (fnpollfd_cb)pollfd_cb_ctl_socket, &a2dp_handler_state, NULL);

		if(lpDevice->nb_clients == 0) {
				memset(&a2dp_handler_state.TimerInfos, 0, sizeof(a2dp_handler_state.TimerInfos));
				a2dp_handler_state.TimerInfos.fpsX = lpDevice->a2dp_rate * lpDevice->a2dp_channels * lpDevice->a2dp_bitspersample * TIMERFACTOR / lpDevice->a2dp_framesize;
				//DBG("Reset client delay");
				polltimeout = pollinfos.polltimeout;
		} else {
				// Use usleep if a client is connected
				polltimeout = 0;
		}

		// This strange construct based on poll and usleep is due
		// to some problem with poll()
		//FIXME
		// I hope this is the implementation of poll doing active loop to wait for small timeouts (arm)
		// Possibly I'm polling an always enabled socket?
		// So when there is no stream, we will wait using poll(),
		// But when there is a stream, we wait using usleep();
		DBG6("MainPoll: %d sockets for %d/%d ms", pollinfos.pollfdcount, pollinfos.polltimeout, polltimeout);
		p = poll(pollinfos.pollfds, pollinfos.pollfdcount, polltimeout);

		if(p<0) {
			DBG("Poll ERROR");
			//result = -1;
			continue;
		}

		a2dp_handler_state.count3++;
		DBG6("Polling %d sockets for %d/%d ms => %d", pollinfos.pollfdcount, pollinfos.polltimeout, polltimeout, p);

		if(p == 0) {
			// This func no longer wait, it is still used to provide some statistics
			delay = a2dp_timer_sleep(&a2dp_handler_state.TimerInfos, A2DPTIMERPREDELAY);

			// If more packets are enqueud then we must empty the buffer
			pollinfos.polltimeout = (pollinfos.polltimeout != -1)?min(pollinfos.polltimeout, delay):delay;

			if(pollinfos.polltimeout>0)
				usleep(pollinfos.polltimeout);

			// Call the stream handler (send/discard)
			a2dp_handler(&a2dp_handler_state);
		}

		// Call cb defined if something happens on socket or on timeout
		for(i = 0; i<pollinfos.pollfdcount; i++) {
			if(pollinfos.pollfds[i].revents != 0) {
				//DBG("Polled sockets %d", i);
				if(pollinfos.pollfds_cb[i].pollfd_cb != NULL) {
					pollinfos.pollfds_cb[i].pollfd_cb(pollinfos.pollfds+i, pollinfos.pollfds_cb[i].param1, pollinfos.pollfds_cb[i].param2);
				}
			}
		}

		// Handle states machines
		a2dp_state_machine(a2dp_handler_state.a2dp);
		setup_resampler(lpDevice);
		sco_state_machine(a2dp_handler_state.sco, scoFrame, sizeof(scoFrame), &scoFrameSize);
		send_sco_incoming_packets(lpDevice, scoFrame, scoFrameSize);

		// Notify state changes
		handle_state_change(lpDevice, a2dp_handler_state.a2dp, a2dp_handler_state.sco);
	}

	// Free allocated objects
	alsa_destroy(&a2dp_handler_state.alsa);
	a2dp_destroy(&a2dp_handler_state.a2dp);
	sco_destroy(&a2dp_handler_state.sco);
	avrcp_destroy(lpDevice->lpAVRCP);

	// Free informations on the device
	bta2dpdevicefree(lpDevice);

	safefree(a2dp_handler_state.pcm_buffer);
	close_server_socket(&sockfd);
	close_socket(&avdtpfd);

	return result;
}


static struct option main_lopts[] = {
	{ "nofork",		0, 0, 'n' },
	{ "daemon",		0, 0, 'd' },
	{ "verbose",		0, 0, 'v' },
	{ "silent",		0, 0, 's' },
	{ "kill",		0, 0, 'k' },
	{ "realtime",		0, 0, 'r' },
	{ "file",		1, 0, 'f' },
	{ "address",		1, 0, 'a' },
	{ "logfile",		1, 0, 'l' },
	{ "configread",		0, 0, 'c' },
	{ "autoconnect",	0, 0, 'o' },
	{ "debug",		0, 0, 'g' },
	{ "sessionbus",		0, 0, 'b' },
	{ "help",		0, 0, 'h' },
	{ 0, 0, 0, 0 }
};

static char main_sopts[] = "ndvskrf:a:l:cogbh";

char* opts_comment[] = {
	"Do not fork",
	"Fork and run as daemon",
	"Use stdout instead of /dev/null",
	"Do not redirect output",
	"Kill currently running daemon",
	"Set realtime priority, use with care",
	"Read config from file <param>",
	"Connect to bluetooth address <param>",
	"Redirect output in file <param>",
	"Read config file periodically",
	"Automatically connect to headset",
	"Display debug traces",
	"Use session bus",
	// keep last
	"Display usage",
	NULL
};

void usage(int argc, char** argv)
{
	int i = 0;
	printf("%s", argv[0]);
	while(opts_comment[i]) {
		printf("\t--%-15s -%c\t%s\t%s\r\n", main_lopts[i].name, main_lopts[i].val, main_lopts[i].has_arg?"<param>":"", opts_comment[i]);
 		i++;
	}
}

// main function
int main(int argc, char *argv[])
{
	int opt = 0, lockfd = 0;
//	struct timespec timer_resolution = { 0, 0 };
	char address[256] = "";
	char *addr = &address[0];
#if 0
	struct sched_param schedparam = { sched_get_priority_max(SCHED_FIFO) };
	int res = 0, bRealtime = 0;
#endif
	int bFork = 0, bVerbose = 1, bKill = 0, bSessionBus = -1;
	int ipc_inet = 0;
	char ipc_addr[256] = "";
	char ipc_bcst[256] = "";
	int ipc_port = 0;

	get_config_filename(g_srcfilename, sizeof(g_srcfilename));

	// parse arguments on cmdline
	while ((opt=getopt_long(argc, argv, main_sopts, main_lopts, NULL)) != -1) {
		switch(opt) {
		case 'n':
			bFork = 0;
			break;
		case 'd':
			bFork = 1;
			break;
		case 'v':
			bVerbose = 1;
			break;
		case 's':
			bVerbose = 0;
			break;
		case 'k':
			bKill = 1;
			break;
#ifdef REALTIME
		case 'r':
			bRealtime = 1;
			break;
#endif
		case 'f':
			strncpy(g_srcfilename, optarg, sizeof(g_srcfilename)-1);
			g_srcfilename[sizeof(g_srcfilename)-1] = 0;
			DBG("Config file %s", g_srcfilename);
			break;
		case 'a':
			strncpy(address, optarg, sizeof(address)-1);
			address[sizeof(address)-1] = 0;
			break;
		case 'l':
			strncpy(g_sOutputFilename, optarg, sizeof(g_sOutputFilename)-1);
			g_sOutputFilename[sizeof(g_sOutputFilename)-1] = 0;
			break;
		case 'c':
			g_brereadconfig = 1;
			break;
		case 'o':
			g_autoconnect = 1;
			break;
		case 'g':
			g_bdebug = 1;
			break;
		case 'b':
			bSessionBus = 1;
			break;
		case 'h':
		default:
			usage(argc, argv);
			exit(0);
			break;
		}
	}

	// Read config values from config file
	if(address[0]=='\0') read_config_string(g_srcfilename, "a2dpd", "address", address, sizeof(address), "");
	if(g_sOutputFilename[0]=='\0') read_config_string(g_srcfilename, "a2dpd", "logfile", g_sOutputFilename, sizeof(g_sOutputFilename), "/dev/null");
	if(g_brereadconfig<0) g_brereadconfig = read_config_int(g_srcfilename, "a2dpd", "enablerereadconfig", 0);
	if(g_breversestereo<0) g_breversestereo = read_config_int(g_srcfilename, "a2dpd", "enablereversestereo", 0);
	if(g_autoconnect<0) g_autoconnect = read_config_int(g_srcfilename, "a2dpd", "enableautoconnect", 1);
	if(g_bdebug<0) g_bdebug = read_config_int(g_srcfilename, "a2dpd", "enabledebug", 0);
	if(g_stdin<0) g_stdin = read_config_int(g_srcfilename, "a2dpd", "enablestdin", 0);
	if(bSessionBus<0) bSessionBus = read_config_int(g_srcfilename, "a2dpd", "sessionbus", 0);
	ipc_inet = read_config_int(g_srcfilename, "a2dpd", "ipc_inet", 0);
	read_config_string(g_srcfilename, "a2dpd", "ipc_addr", ipc_addr, sizeof(ipc_addr), "");
	read_config_string(g_srcfilename, "a2dpd", "ipc_bcst", ipc_bcst, sizeof(ipc_bcst), "");
	ipc_port = read_config_int(g_srcfilename, "a2dpd", "ipc_port", 0);

//	clock_getres(CLOCK_REALTIME, &timer_resolution);

	ignore_child_processes_return_values();

	init_ipc(ipc_inet, ipc_addr, ipc_bcst, ipc_port);

	// Redirect outputs (if a file is specified, redirect to it and not stdout)
	if(strcmp(g_sOutputFilename, "/dev/null") != 0) bVerbose = 0;
	make_daemon_process(bFork, bVerbose, g_sOutputFilename);

	// Lockfile must be acquired after daemonisation
	lockfd = lockfile(bKill);
	if(lockfd<0)
		RETURNERROR("Lockfile acquisition failed");

	DBG("%s addr=%s timer=%d us [%s %s]", argv[0], addr, (int) (0/*timer_resolution.tv_nsec*/ / 1000), __DATE__, __TIME__);

#ifdef REALTIME
	// If we can be realtime it will be better
	if(bRealtime)
	{
		DBG("Setting realtime priority to process, use with care!");
		// After some trouble while coding, a2dpd started spining 100%cpu
		// In realtime, this led me with the only option of rebooting my PC
		res = sched_setscheduler(0, SCHED_FIFO, &schedparam);
		if(res != 0)
			DBG("Setscheduler failed");
	}
#endif

	// set up the handler
	signal(SIGINT, sigint_handler);
	signal(SIGTERM, sigint_handler);
	signal(SIGUSR1, SIG_IGN);

	// global initialisations
	a2dpd_signal_init(bSessionBus);
	a2dp_init();

	// Run main loop
	while(main_loop(addr)<0 && !bSigINTReceived) {
		sleep(1);
	}

	// global termination
	a2dp_exit();
	a2dpd_signal_kill();
	unlockfile(lockfd);

	DBG("Terminated succesfully");
	return 0;
}
