# -*- coding: utf-8 -*-

#  Bluemindo
#  explorer.py

#    Bluemindo: A really simple but powerful audio player in Python/PyGTK.
#    Copyright (C) 2007-2008  Erwan Briand

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

from gettext import gettext as _
from thread import start_new_thread
from gobject import (idle_add, timeout_add, TYPE_STRING as gString,
                     TYPE_INT as gInt, GError, markup_escape_text)
from gtk.gdk import Pixbuf as gPixbuf, pixbuf_new_from_file, INTERP_BILINEAR
from gtk import (STOCK_DIRECTORY, STOCK_CDROM, STOCK_FILE, TreeStore,
                 CellRendererText, TreeViewColumn, CellRendererPixbuf,
                 ICON_SIZE_MENU, ListStore, TREE_VIEW_COLUMN_FIXED,
                 WIN_POS_CENTER_ALWAYS, TreeStore, MenuItem, STOCK_DND_MULTIPLE,
                 STOCK_ORIENTATION_PORTRAIT, ICON_SIZE_LARGE_TOOLBAR,
                 FileChooserDialog, FILE_CHOOSER_ACTION_SAVE, STOCK_CANCEL,
                 RESPONSE_CANCEL, STOCK_SAVE, RESPONSE_OK)
from gtk.glade import XML as glade_XML
from os.path import join, isdir, isfile, expanduser
from os import makedirs, walk, remove
from re import compile as re_compile
from urllib import urlopen as urllib_urlopen, urlretrieve
from random import shuffle as random_shuffle, randrange as random_randrange
from xml.parsers.expat import ExpatError
from xml.sax.saxutils import escape, unescape
from string import upper as str_upper
import tagpy
import ConfigParser

try:
    # python 2.5
	import xml.etree.cElementTree as ElementTree
except ImportError:
	try:
	    # python =< 2.4
		from elementtree import ElementTree
	except ImportError:
		raise SystemExit('This program needs ElementTree for python.')

from libs.audioscrobbler import AudioScrobblerQuery

from common.config import ConfigLoader
from common.functions import Functions
from common.sqlite import SQLite
from common.playlists import Playlists

playlists_management = Playlists()

class Explorer:
    def __init__(self, module):
        self.module = {'name': 'Explorer',
                       'logo': STOCK_DIRECTORY,
                       'configurable': True}

        # Global hooks
        module.connect('OnBluemindoStarted', self.loadmodule)
        module.connect('OnModuleConfiguration', self.loadconfig)
        module.connect('OnModuleConfigurationSave', self.saveconfig)
        module.connect('OnPlaylistRequested', self.openplaylist)

        module.connect('OnMenuRefreshPressed', self.refresh)
        module.connect('OnPlayPressedWithoutQueue', self.play_item)

        self.instance_module = module
        
        config = ConfigLoader()
        self.module_conf = join(config.confdir, 'modules', 'explorer')
        if not isdir(self.module_conf):
            makedirs(self.module_conf)

        self.configfile = ConfigParser.ConfigParser()
        self.configfile_ = self.module['name'] + '.cfg'

        self.module_data = join(config.datadir, 'modules', 'explorer')
        if not isdir(self.module_data):
            makedirs(self.module_data)

        self.player_data = join(config.datadir, 'modules', 'player')

        self.functions = Functions()

        try:
            self.configfile.read(join(self.module_conf, self.configfile_))
            _rpt_ = self.configfile.get(self.module['name'], 'repeat')

            if _rpt_ == 'True':
                self.is_repeat = True
            elif _rpt_ == 'False':
                self.is_repeat = False
        except:
            self.is_repeat = False

        self.playlist_loaded_name = False

    # This function is called when Bluemindo is starting
    def loadmodule(self, glade):
        self.widgets = glade

        if isfile(join(self.module_conf, self.configfile_)):
            self.list_songs = []

            self.widgets.get_widget('label4').hide()
            pbar = self.widgets.get_widget('progressbar1')
            pbar.set_text(_('Loading music'))

            # The folder that contains all music
            music_folder = self.configfile.get(self.module['name'], 'folder')

            # Scan music at each startup
            scan_at_startup = self.configfile.get(self.module['name'], 'scan')
            if scan_at_startup == 'False':
                scan_at_startup = False
            else:
                scan_at_startup = True

            # Type of view
            view = self.configfile.get(self.module['name'], 'mode')

            if view not in ('lightweight', 'basic', 'normal', 'full'):
                raise SystemExit('~/.config/ is not a playground.')

            # Show a progress bar
            pbar = self.widgets.get_widget('progressbar1')
            pbar.show()
            pbar.set_text(_('Loading'))
            start_new_thread(timeout_add, (100, self.timeout, pbar))

            self.load_songs(music_folder, scan_at_startup, view)
        else:
            self.widgets.get_widget('hpaned1').hide()
            self.widgets.get_widget('hbox4').hide()
            self.widgets.get_widget('menu-refresh').set_sensitive(False)
            self.widgets.get_widget('menu-playlist').set_sensitive(False)
            self.widgets.get_widget('label4').set_markup(
                _('<b>Welcome in Bluemindo!</b>\n\n'
                  'First of all, you need to configure the explorer module '
                  'in order to choose your root music directory. This is easy! '
                  'Go in the File menu, click on Preferences and you will be '
                  'able to configure all Bluemindo\'s modules.'))

    # This function start a new thread and reload songs
    def refresh(self, glade):
        # Configuration
        music_folder = self.configfile.get(self.module['name'], 'folder')
        view = self.configfile.get(self.module['name'], 'mode')

        # Show a progress bar
        pbar = self.widgets.get_widget('progressbar1')
        pbar.show()
        pbar.set_text(_('Loading'))
        start_new_thread(timeout_add, (100, self.timeout, pbar))

        # Rescan music
        start_new_thread(self.load_songs, (music_folder, True, view))

    # This function makes pulse the progressbar
    def timeout(self, pbar):
        pbar.pulse()
        return True

    # Load songs in memory
    def load_songs(self, music_folder, reload_songs, view):
        # Connect to SQLite database
        sqlite = SQLite()

        self.widgets.get_widget('label4').hide()
        self.widgets.get_widget('menu-refresh').set_sensitive(True)
        self.widgets.get_widget('menu-playlist').set_sensitive(True)

        # GTK widgets for the left
        left_box = self.widgets.get_widget('vbox2')
        lef_box_tree = self.widgets.get_widget('treeview1')

        # GTK widgets for the three treeview in the center
        center_box = self.widgets.get_widget('hbox6')

        first_win = self.widgets.get_widget('scrolledwindow4')
        first_win_tree = self.widgets.get_widget('treeview3')

        second_win = self.widgets.get_widget('scrolledwindow5')
        second_win_tree = self.widgets.get_widget('treeview4')

        third_win = self.widgets.get_widget('frame2')

        # Load music from music_folder
        if reload_songs:
            # The hard work of reloading the complete music's folder
            i = 0
            reload_list_songs = list()
            fileregxp = re_compile('.+\.(flac|ogg|oga|mp3|mpc)$')
            for (dir_, rep, files) in walk(music_folder):
                for file_ in files:
                    if fileregxp.match(file_):
                        filename = join(dir_, file_)

                        filetags = tagpy.FileRef(filename)

                        tags_ = filetags.tag()
                        audi_ = filetags.audioProperties()

                        if tags_ is None:
                            continue

                        song = [tags_.title, tags_.artist, tags_.album,
                                tags_.track, self.functions.great_length(
                                audi_.length), tags_.genre, filename,
                                audi_.length]

                        reload_list_songs.append(song)

                        i += 1
                        self.widgets.get_widget(
                        'progressbar1').set_text(
                        _('Song #%s') % str(i))

            # Delete previous songs list
            sqlite.execute('delete from songs')

            # Add songs list to database
            sqlite.executemany('insert into songs ('
                               'title, artist, album, track, length, genre, '
                               'filename, reallength) values (?,?,?,?,?,?,?,?)',
                               reload_list_songs)

        # Get list of music in the database
        cursor = sqlite.execute('select * from songs order by '
                                'artist, album, title')
        self.list_songs = sqlite.fetchall(cursor)

        # Tell user that songs are imported
        if len(self.list_songs) == 0:
            text = _('No songs found.')
        else:
            text = (_('All your %u songs have been imported!') %
                     (len(self.list_songs) - 1))

        self.widgets.get_widget('statusbar1').push(0, text)

        # We can close database connection, now
        sqlite.close()

        # ----------------------------------------------------------------------
        # STARTS PLAYLIST
        # ----------------------------------------------------------------------

        # This function handle the select of a song in the playlist
        def select_song_in_playlist(widget, a, b):
            self.play_item()

        # This function determines if the row have to be shown or not
        def visible_function(model, iter_):
            # Ensure that we have a playlist
            if len(model) > 0:
                search = self.widgets.get_widget('entry1').get_text()

                typ = self.widgets.get_widget('combobox1').get_active()
                if typ:    
                    searchin = model.get_value(iter_, typ + 1)
                else:
                    searchin = model.get_value(iter_, 1)

                if search and searchin:
                    if search.upper() in searchin.upper():
                        return True
                else:
                    return True

        # This function reorders the playlist by column
        def reorderbycolumn(treeviewcolumn, columnid, model, model_):
            def little_int(int_):
                try:
                    if int(int_) < 10:
                        return '0' + int_
                    else:
                        return int_
                # A string, not a int
                except ValueError:
                    return str_upper(int_)

            # Create a list with all songs
            songs = []
            for sg in model:
                songs.append((sg[0], sg[1], sg[2], sg[3],
                              sg[4], sg[5], sg[6], sg[7]))

            # Sort the playlist
            songs.sort(lambda x,y:cmp(
                little_int(self.functions.clear_bold(x[columnid])),
                little_int(self.functions.clear_bold(y[columnid]))
            ))

            # Empty-ise playlist
            model_.clear()

            # Re-fill the playlist
            for sg in songs:
                model_.append(sg)

        # This function show the right click menu
        def right_click_context(widget, event, selection):
            if event.button == 3:
                (mod, iter_) = selection.get_selected()

                # If a song is selected
                if iter_:
                    filename = mod.get_value(iter_, 6)

                    # This function removes the song from playlist
                    def remove_item(widgetmenu):
                        if self.playlist_loaded_name:
                            playlists_management.remove_item_from_playlist(
                                                 filename,
                                                 self.playlist_loaded_name)

                        del mod[iter_]

                    # This function creates a new playlist and adds the
                    # current item to it
                    def create_and_add(widgetmenu):
                        newwin = glade_XML(self.functions.datadir +
                                           '/glade/playlistmenu.glade',
                                           'dialog1', domain='bluemindo')
                        new_win = newwin.get_widget('dialog1')
                        response = new_win.run()

                        text = newwin.get_widget('entry1').get_text()
                        if response and text:
                            playlists_management.create_new_playlist(text)
                            playlists_management.add_item_to_playlist(filename,
                                                                      text)

                        new_win.destroy()

                    # This function adds an item to a playlist
                    def add_to_playlist(widget, playlist_name):
                        playlists_management.add_item_to_playlist(filename,
                                                                  playlist_name)

                    widgets = glade_XML(self.functions.datadir + 
                                        '/glade/playlistmenu.glade',
                                        'menu1', domain='bluemindo')

                    menu = widgets.get_widget('menu1')
                    menu.attach_to_widget(widget, None)
                    menu.show_all()
                    menu.popup(None, None, None, event.button, event.time)

                    widgets.get_widget('menu-play').connect('activate',
                                                             self.play_item)

                    widgets.get_widget('menu-playlist-remove').connect(
                                       'activate', remove_item)

                    widgets.get_widget('menu-playlist-new').connect(
                                       'activate', create_and_add)

                    playlist_menu = widgets.get_widget('menu-playlist-add_menu')
                    for plist in playlists_management.get_playlists_list():
                        menu_item = MenuItem(label=plist, use_underline=False)

                        playlist_menu.append(menu_item)
                        menu_item.show_all()
                        menu_item.connect('activate', add_to_playlist, plist)

        # This function clears all songs in the playlist
        def clear_playlist(widget):
            self.liststore.clear()

            if self.playlist_loaded_name:
                self.playlist_loaded_name = False

        # This function shuffles all songs in the playlist
        def shuffle_playlist(widget):
            n = 0
            for a in self.liststore_filter:
                n += 1

            index = range(0, n)
            random_shuffle(index)

            # Reorder the TreeModel
            if len(index) > 1:
                self.liststore.reorder(index)

        # This function turns on/off repeat mode
        def repeat(widget):
            _repeat = widget.get_active()
        
            self.configfile.set(self.module['name'], 'repeat', _repeat)
            self.configfile.write(open(join(self.module_conf, self.configfile_),
                                       'w'))
        
            if _repeat == 'True':
                self.repeat_mode = True
            elif _repeat == 'False':
                self.repeat_mode = False

        # This function changes the filter
        def playlistfilter(entry, event):
            self.liststore_filter.refilter()

        # This function deletes the filter
        def clearfilter(widget):
            self.widgets.get_widget('entry1').set_text('')
            self.liststore_filter.refilter()

        # GTK widgets for the playlist
        playlist_win = self.widgets.get_widget('scrolledwindow3')
        playlist_win_tree = self.widgets.get_widget('treeview2')

        # Drag'n'Drop
        # WE WANT TO DRAG'N'DROP SONGS FROM TREEVIEWS TO PLAYLIST.

        try:
            print '%s reloaded.' % self.liststore
        except AttributeError:
            # Customize the TreeView
            playlist_win_tree.set_rules_hint(True)
            playlist_win_tree.connect('row-activated', select_song_in_playlist)
            playlist_win_tree.connect('button-press-event', right_click_context,
                                       playlist_win_tree.get_selection())

            # Create the TreeStore
            self.liststore = ListStore(gString, gString, gString, gString,
                                       gString, gString, gString, gInt)

            # Create the TreeModelFilter
            self.liststore_filter = self.liststore.filter_new(None)
            self.liststore_filter.set_visible_func(visible_function)

            # Set this model
            playlist_win_tree.set_model(self.liststore_filter)
            playlist_win_tree.set_headers_clickable(True)


            # Playlist buttons
            btn_clear = self.widgets.get_widget('tool-clear')
            btn_shuffle = self.widgets.get_widget('tool-shuffle')
            btn_repeat = self.widgets.get_widget('tool-repeat')

            # Playlist filter
            combo = self.widgets.get_widget('combobox1')
            combo.append_text(_('Title'))
            combo.append_text(_('Artist'))
            combo.append_text(_('Album'))
            combo.set_active(0)

            self.widgets.get_widget('entry1').connect('key-press-event',
                                                      playlistfilter)
            self.widgets.get_widget('toolbutton5').connect(
                                    'clicked', clearfilter)

            # GTK handlers
            btn_clear.connect('clicked', clear_playlist)
            btn_shuffle.connect('clicked', shuffle_playlist)
            btn_repeat.connect('clicked', repeat)

            # Active or deactive repeat button
            if self.is_repeat:
                btn_repeat.set_active(True)
            else:
                btn_repeat.set_active(False)

            # This function add a column to the playlist treeview
            def add_column(infos, show):
        
                # This function calculate the real width of a column
                def width_calculator(percent):
                    return int(
                    playlist_win_tree.get_allocation().width / 100. * percent)

                title = infos[0]
                expand = infos[1]
                width = width_calculator(infos[2])
                position = infos[3]

                column = TreeViewColumn()
                column.set_title(title)

                if expand:
                    column.set_min_width(width)
                    column.set_expand(True)
                    column.set_sizing(TREE_VIEW_COLUMN_FIXED)
                else:
                    column.set_max_width(width)

                render_text = CellRendererText()
                column.pack_start(render_text, expand=True)
                column.add_attribute(render_text, 'markup', position)
                column.set_clickable(True)
                column.connect('clicked', reorderbycolumn, position,
                               self.liststore_filter, self.liststore)

                if not show:
                    column.set_visible(False)

                playlist_win_tree.append_column(column)

            # Get playlist configuration
            columns_ = self.configfile.get(self.module['name'], 'columns')
            show_columns = columns_.split(',')

            # Column settings
            # For each colum, a list contains: title, expand, width and position
            columns_infos = {'track': [_('#'), False, 4, 0],
                             'title': [_('Title'), True, 31, 1],
                             'artist': [_('Artist'), True, 20, 2],
                             'album': [_('Album'), True, 20, 3],
                             'genre': [_('Genre'), True, 14, 4],
                             'length': [_('Length'), False, 9, 5],
                             'filename': ['', False, 0, 6],
                             'reallength': ['', False, 0, 7]
                            }
            columns_list = ['track', 'title', 'artist', 'album',
                            'genre', 'length', 'filename', 'reallength']

            # Add columns
            for column in columns_list:
                if column in show_columns:
                    add_column(columns_infos[column], True)
                else:
                    add_column(columns_infos[column], False)

        # ----------------------------------------------------------------------
        # ENDS PLAYLIST
        # ----------------------------------------------------------------------

        # We want the `LIGHTWEIGHT` view
        if view == 'lightweight':
            left_box.hide()
            center_box.hide()

            start_new_thread(self.add_songs_in_playlist, (self.list_songs, True))

        elif view in ('basic', 'normal', 'full'):
            # This function ensures there are only unique values
            def unique(seq, keepstr=True):
                t = type(seq)
                if t in (str, unicode):
                    t = (list, ''.join)[bool(keepstr)]
                seen = []
                return t(c for c in seq if not (c in seen or seen.append(c)))

            # This function sorts songs
            def song_sort(x, y):
                if x[3] == y[3]:
                    return 0
                elif x[3] < y[3]:
                    return -1
                else:
                    return 1

            songs_tree = {}
            artists = []
            albums = []

            for result in self.list_songs:
                artists.append(result[1])
                albums.append((result[1], result[2]))

            artists = unique(artists)
            albums = unique(albums)

            for a in artists:
                songs_tree[a] = {}

            for a in albums:
                songs_tree[a[0]][a[1]] = []

            for result in self.list_songs:
                songs_tree[result[1]][result[2]].append((result[0],
                                                         result[1],
                                                         result[2],
                                                         result[3],
                                                         result[4],
                                                         result[5],
                                                         result[6],
                                                         result[7]
                                                        ))

            for artist in songs_tree.values():
                for album in artist.values():
                    album.sort(song_sort)

            # ------------------------------------------------------------------
            # BASIC VIEW
            # ------------------------------------------------------------------
            if view == 'basic':
                center_box.hide()

                # This function handles row activation
                def add_to_playlist(tview, path, column):

                    (mod, iter_) = tview.get_selection().get_selected()
                    iter_has_depth = tview.get_model().iter_depth(iter_)

                    if not iter_has_depth:
                        artist_ = unescape(mod.get_value(iter_, 1))[3:-4]

                        songs = []
                        for album in songs_tree[artist_]:
                            for sgs in songs_tree[artist_][album]:
                                songs.append(sgs)

                    elif iter_has_depth == 1:
                        artist = tview.get_model().iter_parent(iter_)
                        artist_ = unescape(mod.get_value(artist, 1))[3:-4]
                        album_ = unescape(mod.get_value(iter_, 1))[3:-4]

                        songs = []
                        for sgs in songs_tree[artist_][album_]:
                            songs.append(sgs)

                    elif iter_has_depth == 2:
                        album = tview.get_model().iter_parent(iter_)
                        artist = tview.get_model().iter_parent(album)

                        song_ = unescape(mod.get_value(iter_, 1))
                        artist_ = unescape(mod.get_value(artist, 1))[3:-4]
                        album_ = unescape(mod.get_value(album, 1))[3:-4]
    
                        songs = []
                        for sgs in songs_tree[artist_][album_]:
                            if song_ == sgs[0]:
                                songs.append(sgs)

                    try:
                        self.add_songs_in_playlist(songs)
                    except NameError:
                        pass

                treestore = TreeStore(gPixbuf, gString)
                column = TreeViewColumn()

                render_pixbuf = CellRendererPixbuf()
                column.pack_start(render_pixbuf, expand=False)
                column.add_attribute(render_pixbuf, 'pixbuf', 0)

                render_text = CellRendererText()
                column.pack_start(render_text, expand=True)
                column.add_attribute(render_text, 'markup', 1)

                tview = self.widgets.get_widget('treeview1')
                tview.set_model(treestore)
                tview.append_column(column)
                tview.connect('row-activated', add_to_playlist)
                treestore.clear()

                items = songs_tree.items()
                items.sort()
                items_sort = [key for key, value in items]

                for artist in items_sort:
                    logo = tview.render_icon(STOCK_DIRECTORY, ICON_SIZE_MENU)
                    t = treestore.append(None, (logo, '<b>%s</b>' %
                                         escape(artist)))

                    for album in songs_tree[artist]:
                        logo = tview.render_icon(STOCK_CDROM, ICON_SIZE_MENU)
                        u = treestore.append(t, (logo, '<i>%s</i>' %
                                            escape(album)))

                        for song in songs_tree[artist][album]:
                            logo = tview.render_icon(STOCK_FILE, ICON_SIZE_MENU)
                            v = treestore.append(u, (logo,
                                          escape(song[0])))

            # ------------------------------------------------------------------
            # NORMAL VIEW
            # ------------------------------------------------------------------
            elif view == 'normal':
                left_box.hide()
                third_win.hide()

                # This function show albums by artist
                def browse_artist(selection, treestore, column):
                    if selection:
                        (mod, iter_) = selection.get_selected()
                        if iter_:
                            artist = unescape(mod.get_value(iter_, 0))

                        try:
                            treestore_.clear()
                            items = songs_tree[artist].items()
                            items.sort()
                            items_sort = [key for key, value in items]
                        except UnboundLocalError:
                            return

                        nb = 0
                        for album_ in items_sort:
                            nb = nb + len(songs_tree[artist][album_])

                            _file = join(self.player_data,
                                         self.functions.get_hash(album_, artist)
                                                                + '.covers')
                            if isfile(_file):
                                icon = pixbuf_new_from_file(_file)
                            else:
                                icon = pixbuf_new_from_file(
                                       self.functions.datadir +
                                       '/image/logo_head_big.png')
    
                            icon = icon.scale_simple(32, 32, INTERP_BILINEAR)
                            treestore_.append((icon,
                            '%s' % markup_escape_text(unicode(album_)),
                            ('<b>%s</b>' %
                            len(songs_tree[artist][album_]))
                            ))
    
                        column.set_title('%s albums (%s songs)' %
                                        (len(songs_tree[artist]), nb))
                    else:
                        albums = []

                        i = 0
                        n = -1
                        for art in songs_tree:
                            for alb in songs_tree[art]:
                                albums.append({'album': alb, 'artist': art})
                                i = i + 1
                                n = n + len(songs_tree[art][alb])

                        albums.sort()

                        column.set_title('%s albums (%s songs)' % (i, n))

                        for alb in albums:
                            album_ = alb['album']
                            artist = alb['artist']

                            _file = join(self.player_data, 
                            self.functions.get_hash(album_, artist) + '.covers')

                            if isfile(_file):
                                icon = pixbuf_new_from_file(_file)
                            else:
                                icon = pixbuf_new_from_file(
                                       self.functions.datadir +
                                       '/image/logo_head_big.png')

                            icon = icon.scale_simple(32, 32, INTERP_BILINEAR)
                            treestore_.append((icon,
                            '%s' % markup_escape_text(unicode(album_)),
                            ('<b>%s</b>' % len(songs_tree[artist][album_])
                            )))

                # This function adds artist's songs to playlist
                def load_artist(tview, path, column):
                    (mod, iter_) = tview.get_selection().get_selected()
                    artist = unescape(mod.get_value(iter_, 0))
                    songs = []

                    for sg in self.list_songs:
                        if artist == sg[1]:
                            songs.append(sg)

                    self.add_songs_in_playlist(songs)

                # This function adds album's songs to playlist
                def load_album(tview, path, column):
                    try:
                        tart = self.widgets.get_widget('treeview3')
                        (mod, iter_) = tart.get_selection().get_selected()
                        artist = unescape(mod.get_value(iter_, 0))
                        by_artist = True
                    except TypeError:
                        by_artist = False

                    (mod, iter_) = tview.get_selection().get_selected()
                    album = unescape(mod.get_value(iter_, 1))
                    songs = []

                    for sg in self.list_songs:
                        # There is an artist selected
                        if by_artist and artist == sg[1] and album == sg[2]:
                            songs.append(sg)
                        # There is no artist selected
                        elif album == sg[2]:
                            songs.append(sg)

                    if len(songs) > 0:
                        self.add_songs_in_playlist(songs)

                render_pixbuf_ = CellRendererPixbuf()
                render_text_ = CellRendererText()

                if self.widgets.get_widget('treeview4').get_model() is None:
                    # Album
                    treestore_ = ListStore(gPixbuf, gString, gString)
                    albumview = self.widgets.get_widget('treeview4')
                    albumview.set_model(treestore_)
                    albumview.set_rules_hint(True)
                    albumview.connect('row-activated', load_album)

                    column0 = TreeViewColumn()
                    column0.pack_start(render_pixbuf_, expand=False)
                    column0.add_attribute(render_pixbuf_, 'pixbuf', 0)
                    albumview.append_column(column0)
                
                    column1 = TreeViewColumn()
                    column1.pack_start(render_text_, expand=True)
                    column1.add_attribute(render_text_, 'markup', 1)
                    column1.set_min_width(250)
                    column1.set_expand(True)
                    column1.set_sizing(TREE_VIEW_COLUMN_FIXED)
                    albumview.append_column(column1)
                    self.basic_view_col = column1

                    column2 = TreeViewColumn()
                    column2.pack_start(render_text_, expand=False)
                    column2.add_attribute(render_text_, 'markup', 2)
                    albumview.append_column(column2)

                    # Artist
                    treestore = ListStore(gString, gString)
                    artistview = self.widgets.get_widget('treeview3')
                    artistview.set_model(treestore)
                    artistview.set_rules_hint(True)
                    artistview.connect('row-activated', load_artist)
                    artistview.get_selection().connect('changed', browse_artist,
                                                       treestore_, column1)

                    column0 = TreeViewColumn()
                    column0.pack_start(render_text_, expand=True)
                    column0.add_attribute(render_text_, 'markup', 0)
                    column0.set_min_width(280)
                    column0.set_expand(True)
                    column0.set_sizing(TREE_VIEW_COLUMN_FIXED)
                    artistview.append_column(column0)
                    self.basic_view_col_ = column0

                    column1 = TreeViewColumn()
                    column1.pack_start(render_text_, expand=False)
                    column1.add_attribute(render_text_, 'markup', 1)
                    artistview.append_column(column1)
                else:
                    treestore = self.widgets.get_widget('treeview3').get_model()
                    treestore_ = self.widgets.get_widget('treeview4').get_model()

                # Clear all treestore
                treestore.clear()
                treestore_.clear()

                # Fills artists
                items = songs_tree.items()
                items.sort()
                items_sort = [key for key, value in items]

                ni = -1
                for artist in items_sort:
                    nb = 0
                    for alb in songs_tree[artist]:
                        nb = nb + len(songs_tree[artist][alb])
                    treestore.append(('%s' % markup_escape_text(unicode(artist)),
                                      '<b>%s</b>' % nb))
                    ni = ni + nb

                self.basic_view_col_.set_title('%s artists (%s songs)' %
                                 (len(songs_tree), ni))
                start_new_thread(browse_artist, (None,
                                                 treestore_,
                                                 self.basic_view_col))

            # ------------------------------------------------------------------
            # FULL VIEW
            # ------------------------------------------------------------------
            elif view == 'full':
                first_win.hide()
                second_win.hide()
                self.widgets.get_widget('hbox6').hide()

                # This function downloads pictures for an artist
                def download_artists_picture(tstore_artists_list, artistn):
                    if artistn == -1:
                        return

                    pbar = self.widgets.get_widget('progressbar1')

                    # Show the progress bar
                    idle_add(pbar.show)

                    artistid = 1
                    for row in tstore_artists_list:
                        # Only artists
                        if row[2] != 'artist':
                            continue

                        _artist_ = row[1]

                        # Tell user what artist are we fetching
                        idle_add(pbar.set_text,
                        _('Fetching artist %(id)u/%(n)u') % {'id': artistid,
                                                             'n': artistn})
                        artistid += 1

                        # Filename
                        artist_picture = join(self.module_data,
                                              self.functions.get_hash(_artist_,
                                                                     'picture'))

                        # Artist's picture downloading
                        if not isfile(artist_picture):
                            artist_url = _artist_.replace(' ',
                                                          '+').replace('/',
                                                                       '+')

                            try:
                                lastfm_ = urllib_urlopen(
                                      'http://ws.audioscrobbler.com/1.0/artist/'
                                      + artist_url.encode('iso-8859-15')
                                      + '/similar.xml')
                                lastfm = lastfm_.read()
                            except UnicodeEncodeError:
                                pass

                            try:
                                tree = ElementTree.fromstring(lastfm)
                                file_ = tree.attrib.get("picture")
                                if (file_ != 'http://cdn.last.fm/depth/'
                                             'catalogue/noimage/noartist_140px'
                                             '.jpg'):
                                    urlretrieve(file_, artist_picture)

                                # Show the new picture
                                try:
                                    picture = artist_picture
                                    if isfile(picture):
                                        logo = pixbuf_new_from_file(picture)
                                        logo = logo.scale_simple(24, 24,
                                                                INTERP_BILINEAR)
                                    else:
                                        logo = None
                                except GError:
                                    logo = None
                                    if isfile(artist_picture):
                                        remove(artist_picture)

                                def add_logo(row, logo):
                                    row = logo

                                if logo is not None:
                                    idle_add(add_logo, (row[0], logo))

                            except (ExpatError, UnicodeError, SyntaxError):
                                pass

                    # Hide the progress bar
                    idle_add(pbar.hide)

                # This function shows all artist's arts
                def show_albums_covers(artist):
                    artist = unescape(artist)

                    # This function loads an album in the playlist
                    def load_album_in_playlist(widget, alb):
                        songs = []
                        for song in songs_tree[artist][alb]:
                            songs.append(song)

                        self.add_songs_in_playlist(songs)

                    # Show the albums container
                    self.widgets.get_widget('hbox6').show()
                    self.widgets.get_widget('label7').set_markup(
                           '<b>' + markup_escape_text(unicode(artist)) + '</b>')

                    # Delete all buttons
                    container = self.widgets.get_widget('hbuttonbox1')
                    for wdg in container:
                        container.remove(wdg)

                    # Add a button for each albums, based on the model
                    for album in songs_tree[artist]:
                        widgets_ = glade_XML(self.functions.datadir +
                                             '/glade/mainwindow.glade',
                                             'button2', domain='bluemindo')
                        album_button = widgets_.get_widget('button2')

                        # Add the album name and the cover
                        widgets_.get_widget('label8').set_markup(
                            '<b>' + markup_escape_text(unicode(album)) + '</b>')

                        _file = join(self.player_data, 
                        self.functions.get_hash(album, artist) + '.covers')

                        if isfile(_file):
                            icon = pixbuf_new_from_file(_file)
                        else:
                            icon = pixbuf_new_from_file(self.functions.datadir +
                                                     '/image/logo_head_big.png')

                        icon = icon.scale_simple(120, 100, INTERP_BILINEAR)
                        widgets_.get_widget('image5').set_from_pixbuf(icon)

                        album_button.show_all()
                        self.widgets.get_widget('hbuttonbox1').add(album_button)

                        # GTK handler
                        album_button.connect('clicked',
                                             load_album_in_playlist, album)

                # This function handles the artists or playlists loading
                def load_item(tview, path, column):
                    (mod, iter_) = tview.get_selection().get_selected()

                    value = mod.get_value(iter_, 1)
                    typ = mod.get_value(iter_, 2)

                    if typ in 'artist':
                        show_albums_covers(value)

                        artist = unescape(value)
                        songs = []

                        _alb = {}

                        for sg in self.list_songs:
                            # Create a list of albums
                            if artist == sg[1]:
                                if sg[2] not in _alb:
                                    _alb[sg[2]] = []

                                _alb[sg[2]].append(sg)

                        # Order songs by track number and by albums
                        for alb in _alb:
                            _alb[alb].sort(lambda x,y:cmp(x[3],y[3]))

                            # Add songs in the list
                            for sg in _alb[alb]:
                                songs.append(sg)

                        # Finally insert songs in playlist :)
                        self.add_songs_in_playlist(songs)

                    elif typ in 'playlist':
                        self.widgets.get_widget('hbox6').hide()
                        self.instance_module.load_event('OnPlaylistRequested',
                                                        value)

                # This function handles right click
                def click_on_treeview(widget, event, selection):
                    (mod, iter_) = selection.get_selected()

                    if iter_:
                        value = mod.get_value(iter_, 1)
                        typ = mod.get_value(iter_, 2)

                        # Context menu for playlists
                        if event.button == 3 and typ == 'playlist':

                            # This function loads a playlist
                            def ct_playlist_play(widgetmenu, playlist):
                                self.widgets.get_widget('hbox6').hide()
                                self.instance_module.load_event(
                                              'OnPlaylistRequested', value)

                            # This function deletes a playlist
                            def ct_playlist_delete(widgetmenu, playlist):
                                playlists_management.delete_playlist(playlist)
                                del mod[iter_]

                            # This function exports a playlist
                            def ct_playlist_export(widgetmenu, playlist):
                                fcd_playlist = FileChooserDialog(
                                               _('Export a playlist'), None,
                                               FILE_CHOOSER_ACTION_SAVE,
                                                 (STOCK_CANCEL, RESPONSE_CANCEL,
                                                  STOCK_SAVE, RESPONSE_OK
                                                 ))
                                new_playlist = fcd_playlist.run()

                                if new_playlist == RESPONSE_OK:
                                    filename = fcd_playlist.get_filename()
                                    playlists_management.export_playlist(
                                                         playlist, filename)

                                fcd_playlist.destroy()

                            widgets = glade_XML(self.functions.datadir +
                                                '/glade/mainwindow.glade',
                                                'menu-playlist',
                                                domain='bluemindo')

                            menu = widgets.get_widget('menu-playlist')
                            menu.attach_to_widget(widget, None)
                            menu.show_all()
                            menu.popup(None, None, None,
                                       event.button, event.time)

                            # GTK handlers for the context menu
                            widgets.get_widget('menu-playlist-play').connect(
                                    'activate', ct_playlist_play, value)
                            widgets.get_widget('menu-playlist-delete').connect(
                                    'activate', ct_playlist_delete, value)
                            widgets.get_widget('menu-playlist-export').connect(
                                    'activate', ct_playlist_export, value)

                # This function loads covers for an artist
                def load_artist_covers(selection):
                    if selection:
                        (mod, iter_) = selection.get_selected()
                        if iter_ and mod.get_value(iter_, 2) not in ('title',
                                                                     'playlist'):
                            show_albums_covers(mod.get_value(iter_, 1))

                # Don't recreate the treestore if we are reloading
                if self.widgets.get_widget('treeview1').get_model() is None:
                    treestore = ListStore(gPixbuf, gString, gString)
                    tview = self.widgets.get_widget('treeview1')
                    tview.set_model(treestore)
                    tview.set_rules_hint(True)

                    # GTK Handlers
                    tview.connect('row-activated', load_item)
                    tview.connect('button-press-event', click_on_treeview,
                                  tview.get_selection())
                    tview.get_selection().connect('changed', load_artist_covers)

                    column = TreeViewColumn()
                    render_pixbuf = CellRendererPixbuf()
                    column.pack_start(render_pixbuf, expand=False)
                    column.add_attribute(render_pixbuf, 'pixbuf', 0)
                    tview.append_column(column)

                    column = TreeViewColumn()
                    render_text = CellRendererText()
                    column.pack_start(render_text, expand=True)
                    column.add_attribute(render_text, 'markup', 1)
                    tview.append_column(column)

                    column = TreeViewColumn()
                    render_text = CellRendererText()
                    column.pack_start(render_text, expand=False)
                    column.add_attribute(render_text, 'markup', 2)
                    column.set_visible(False)
                    tview.append_column(column)
                else:
                    tview = self.widgets.get_widget('treeview1')
                    treestore = tview.get_model()
                    treestore.clear()

                # Add playlists
                logo = tview.render_icon(STOCK_DND_MULTIPLE,
                                         ICON_SIZE_LARGE_TOOLBAR)
                treestore.append((logo, _('<b>Playlists</b>'), 'title'))

                playlists_list = playlists_management.get_playlists_list()

                for playlist_name in playlists_list:
                    treestore.append((None, playlist_name, 'playlist'))

                # Add artists
                logo = tview.render_icon(STOCK_ORIENTATION_PORTRAIT,
                                         ICON_SIZE_LARGE_TOOLBAR)
                treestore.append((logo, _('<b>Artists</b>'), 'title'))

                items = songs_tree.items()
                items.sort()
                items_sort = [key for key, value in items]

                artists_list = []
                for artist in items_sort:
                    artists_list.append(artist)

                    artist_ = escape(artist)

                    artist_picture = join(self.module_data,
                                          self.functions.get_hash(artist,
                                                                  'picture'))
                    try:
                        if isfile(artist_picture):
                            logo = pixbuf_new_from_file(artist_picture)
                            logo = logo.scale_simple(24, 24, INTERP_BILINEAR)
                        else:
                            logo = None
                    except GError:
                        logo = None
                        if isfile(artist_picture):
                            remove(artist_picture)

                    treestore.append((logo, artist_, 'artist'))

                # Download a picture for each artist
                start_new_thread(download_artists_picture,
                                (treestore, len(artists_list) - 1))

        # Hide progress bar
        self.widgets.get_widget('progressbar1').hide()

    # This function opens a playlist
    def openplaylist(self, playlist_name):
        self.liststore.clear()
        self.playlist_loaded_name = playlist_name
        songs_for_playlist = playlists_management.load_playlist(playlist_name)

        for filesong in songs_for_playlist:
            filesong = filesong[:-1]

            if isfile(filesong):
                tag = tagpy.FileRef(filesong)
                tags = tag.tag()
                audi = tag.audioProperties()

                track = tags.track
                title = tags.title
                artist = tags.artist
                album = tags.album
                genre = tags.genre
                length = self.functions.great_length(audi.length)
                reallength = audi.length

                self.liststore.append((track, title, artist, album, genre,
                                       length, filesong, reallength))

    # This function fills the playlist
    def add_songs_in_playlist(self, songs_list, is_in_thread=False):
        tview = self.widgets.get_widget('treeview2')
        tstore = self.liststore
        n_occurs = 0
        n_length = 0

        for song in songs_list:
            # Database song list to playlist
            title = escape(song[0])
            artist = escape(song[1])
            album = escape(song[2])
            track = song[3]
            length = song[4]
            genre = escape(song[5])
            filename = escape(song[6])
            reallength = song[7]

            has_already = False
            for sg in tstore:
                if sg[6] == filename:
                    has_already = True

            if not has_already:

                if is_in_thread:
                    print n_occurs

                tstore.append((track, title, artist, album,
                               genre, length, filename, reallength))

                n_occurs += 1
                n_length += reallength



        # Tell user the work is completed
        self.widgets.get_widget('statusbar1').push(
            0, _('Added %(nbsongs)u songs (%(totalength)s).') %
            {'nbsongs': n_occurs,
             'totalength': self.functions.great_length(n_length)
            })

    # This function play a song
    def play_item(self, contextmenu=None):
        treeview = self.widgets.get_widget('treeview2')
        (mod, iter_) = treeview.get_selection().get_selected()

        # Test if a song is selected
        if treeview.get_selection().get_selected()[1]:
            track = self.functions.clear_bold(unescape(mod.get_value(iter_, 0)))
            title = self.functions.clear_bold(unescape(mod.get_value(iter_, 1)))
            artist = self.functions.clear_bold(unescape(mod.get_value(iter_, 2)))
            album = self.functions.clear_bold(unescape(mod.get_value(iter_, 3)))
            genre = self.functions.clear_bold(unescape(mod.get_value(iter_, 4)))
            length = self.functions.clear_bold(unescape(mod.get_value(iter_, 5)))
            filename = unescape(mod.get_value(iter_, 6))
            reallength = mod.get_value(iter_, 7)

            self.widgets.get_widget('hscale1').set_value(0)
            self.widgets.get_widget('hscale1').set_range(0, float(reallength))
            self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                              artist, album,
                                                              genre, length,
                                                              filename,
                                                              reallength))

        # No song in the playlist, select randomly a song
        # in the whole database
        else:
            index = random_randrange(len(self.list_songs) - 1)
            song = self.list_songs[index]
            # Database song list to playlist format
            title = song[0]
            artist = song[1]
            album = song[2]
            track = song[3]
            length = song[4]
            genre = song[5]
            filename = song[6]
            reallength = song[7]

            self.widgets.get_widget('hscale1').set_value(0)
            self.widgets.get_widget('hscale1').set_range(0, float(reallength))

            # Sending start song event
            self.instance_module.load_event('OnPlayNewSong', (track, title,
                                                              artist, album,
                                                              genre, length,
                                                              filename,
                                                              reallength))

    # This function loads the configuration interface for explorer and playlist
    def loadconfig(self, (module, confglade)):
        if module == self.module['name']:
            # Load the glade and put the vertical box in the module's
            # configuration one
            self.conf_widgets = glade_XML('modules/explorer/configuration.glade',
                                          'vbox1', domain='bluemindo')

            hbox = confglade.get_widget('hbox1')

            try:
                kids = hbox.get_children()
                hbox.remove(kids[2])
            except:
                pass

            hbox.add(self.conf_widgets.get_widget('vbox1'))

            # Add view's icon
            self.conf_widgets.get_widget('image-l').set_from_file(
                               'modules/explorer/views/view_lightweight.png')
            self.conf_widgets.get_widget('image-b').set_from_file(
                               'modules/explorer/views/view_basic.png')
            self.conf_widgets.get_widget('image-n').set_from_file(
                               'modules/explorer/views/view_normal.png')
            self.conf_widgets.get_widget('image-f').set_from_file(
                               'modules/explorer/views/view_full.png')

            radio_l = self.conf_widgets.get_widget('radio-view-lightweight')
            radio_b = self.conf_widgets.get_widget('radio-view-basic')
            radio_n = self.conf_widgets.get_widget('radio-view-normal')
            radio_f = self.conf_widgets.get_widget('radio-view-full')

            radio_l.set_active(False)
            radio_b.set_group(radio_l)
            radio_b.set_active(False)
            radio_n.set_group(radio_l)
            radio_n.set_active(False)
            radio_f.set_group(radio_l)
            radio_f.set_active(False)

            # Adapt GUI to configuration
            self.configfile.read(join(self.module_conf, self.configfile_))
            try:
                folder = self.configfile.get(self.module['name'], 'folder')
                scan = self.configfile.get(self.module['name'], 'scan')
                viewmode = self.configfile.get(self.module['name'], 'mode')
                columns_ = self.configfile.get(self.module['name'], 'columns')
                file_exist = True
            except:
                file_exist = False

            if file_exist:
                self.conf_widgets.get_widget(
                       'filechooserbutton1').set_current_folder(folder)

                if scan == 'True':
                    self.conf_widgets.get_widget('check-scan').set_active(True)

                columns = columns_.split(',')
                for column in columns:
                    self.conf_widgets.get_widget('check-column-' +
                                        column).set_active(True)

                self.conf_widgets.get_widget('radio-view-' +
                                             viewmode).set_active(True)

    # This functions saves the module configuration
    def saveconfig(self, extension):
        if extension == self.module['name']:
            try:
                radio_l = self.conf_widgets.get_widget('radio-view-lightweight')
                radio_b = self.conf_widgets.get_widget('radio-view-basic')
                radio_n = self.conf_widgets.get_widget('radio-view-normal')
                radio_f = self.conf_widgets.get_widget('radio-view-full')

                check_track = self.conf_widgets.get_widget('check-column-track')
                check_title = self.conf_widgets.get_widget('check-column-title')
                check_artist = self.conf_widgets.get_widget('check-column-artist')
                check_album = self.conf_widgets.get_widget('check-column-album')
                check_genre = self.conf_widgets.get_widget('check-column-genre')
                check_length = self.conf_widgets.get_widget('check-column-length')

                check_scan = self.conf_widgets.get_widget('check-scan')

                file_chooser = self.conf_widgets.get_widget('filechooserbutton1')

                widgets_retrieved = True
            except:
                widgets_retrieved = False

            if widgets_retrieved:
                try:
                    self.configfile.add_section(self.module['name'])
                except:
                    pass

                form0 = file_chooser.get_filename()
                form1 = check_scan.get_active()

                if radio_l.get_active():
                    form2 = 'lightweight'
                elif radio_b.get_active():
                    form2 = 'basic'
                elif radio_n.get_active():
                    form2 = 'normal'
                elif radio_f.get_active():
                    form2 = 'full'

                form3_ = []
                if check_track.get_active():
                    form3_.append('track')
                if check_title.get_active():
                    form3_.append('title')
                if check_artist.get_active():
                    form3_.append('artist')
                if check_album.get_active():
                    form3_.append('album')
                if check_genre.get_active():
                    form3_.append('genre')
                if check_length.get_active():
                    form3_.append('length')

                form3 = ''
                i = 0
                for col in form3_:
                    i += 1

                    form3 += col
                    if i < len(form3_):
                        form3 += ','

                self.configfile.set(self.module['name'], 'folder', form0)
                self.configfile.set(self.module['name'], 'scan', form1)
                self.configfile.set(self.module['name'], 'mode', form2)
                self.configfile.set(self.module['name'], 'columns', form3)

                self.configfile.write(open(join(self.module_conf,
                                                self.configfile_), 'w'))
