import java.io.FileWriter;
import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.File;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.regex.Matcher;


/**
 * Class that initializes the required properties for
 * BioMAJ to work.
 * 
 * @author rsabas
 *
 */
public class Configurator {
	
	private static final String MYSQL_DRIVER = "com.mysql.jdbc.Driver";
	
	private static final String HSQLDB_DRIVER = "org.hsqldb.jdbcDriver";
	private static final String HSQLDB_LOGIN = "sa";
	private static final String HSQLDB_PASSWD = "";
	private static final String HSQLDB_URL = "jdbc:hsqldb:hsql://localhost/bmajdb";
	
	private static final String TOMCAT_DIR = "apache-tomcat-6.0.24";
	private static final int MESSAGE_MAX = 30;
	
	private static String biomajRoot = "";
	
	private static PrintWriter logger = null;
	
	public static void main(String[] args) {

		boolean useTomcat = true;
		
		Map<String, String> params = new HashMap<String, String>();
		
		for (String arg : args) {
			String[] pair = arg.split(":=");
			if (pair.length > 1)
				params.put(pair[0], pair[1]);
			else
				params.put(pair[0], "");
		}

		biomajRoot = params.get("biomaj_root");
		
		try {
			logger = new PrintWriter(biomajRoot + "/install_log.log");
		} catch (FileNotFoundException e1) {
			log(e1.toString());
		}
		
		if (params.get("java_home").trim().isEmpty())
			params.put("java_home","/usr/lib/jvm/java-6-openjdk");


		if( !params.get("tomcat_port").trim().isEmpty() && params.get("tomcat_port").equals("0")) {
			useTomcat = false;
		}

		if(useTomcat) {
		/*
		 * Context.xml
		 */
		log("Updating Context.xml...");
		try {
			PrintWriter pw = new PrintWriter(biomajRoot + "/" + TOMCAT_DIR + "/conf/Catalina/localhost/BmajWatcher.xml");
			
			pw.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			pw.println("<Context reloadable=\"false\">");

			pw.println("<!-- ENV VARIABLES -->");
			pw.println("<Parameter name=\"BIOMAJ_ROOT\" value=\"" + biomajRoot + "\" override=\"false\"/>");
			
			pw.println("<!-- WEB INTERFACE CONF -->");
			pw.println("<Parameter name=\"MESSAGE_MAX\" value=\"" + MESSAGE_MAX + "\" override=\"false\"/>");

			pw.println("<Parameter name=\"USELOCAL\" value=\"1\" override=\"false\"/>");
			pw.println("<!-- LDAP CONF -->");
			if (params.get("use_ldap").equals("true"))
				pw.println("<Parameter name=\"USELDAP\" value=\"1\" override=\"false\"/>");
			else
				pw.println("<Parameter name=\"USELDAP\" value=\"0\" override=\"false\"/>");

			log("Creating authentication file...");
			createPasswdFile(params.get("app_login"), params.get("app_passwd"));
			
			pw.println("<Parameter name=\"LDAPHOST\" value=\"" + params.get("ldap_host") + "\" override=\"false\"/>");
			pw.println("<Parameter name=\"LDAPDN\" value=\"" + params.get("ldap_dn") + "\" override=\"false\"/>");
			pw.println("<Parameter name=\"OPTFILTER\" value=\"" + params.get("opt_filter") + "\" override=\"false\"/>");
			
			pw.println("<!-- LOCAL/REMOTE EXECUTION -->");
			pw.println("<Parameter name=\"EXECUTION\" value=\"local\" override=\"false\"/>");

			pw.println("<!-- SSH CONFIG -->");
			pw.println("<Parameter name=\"SSH_AUTH_TYPE\" value=\"\" override=\"false\"/>");
			pw.println("<Parameter name=\"SSH_LOGIN\" value=\"\" override=\"false\"/>");
			pw.println("<Parameter name=\"SSH_PASSWD\" value=\"\" override=\"false\"/>");
			pw.println("<Parameter name=\"SSH_KEY_PATH\" value=\"\" override=\"false\"/>");
			pw.println("<Parameter name=\"SSH_PASSPHRASE\" value=\"\" override=\"false\"/>");
			pw.println("<Parameter name=\"SSH_HOSTS\" value=\"\" override=\"false\"/>");
			
			
			pw.println("</Context>");
			
			pw.close();
			
			/*
			 * Log handling
			 */
			String logBase = "/var/log/biomaj";
			pw = new PrintWriter(biomajRoot + "/" + TOMCAT_DIR + "/conf/Catalina/localhost/BmajWatcher#logs.xml");
			
			pw.println("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
			pw.println("<Context docBase=\"" + logBase + "\" debug=\"0\" privileged=\"true\">");
			pw.println("</Context>");
			
			pw.close();
		} catch (FileNotFoundException e) {
			log(e.toString());
		}

		}
		
		/*
		 * Global.properties
		 */
		log("Updating global.properties...");
		String globalLocation = "/etc/biomaj/db_properties/global.properties";
		StringBuilder sb = new StringBuilder();
		try {
			BufferedReader br = new BufferedReader(new FileReader(globalLocation));
			String line = "";
			while ((line = br.readLine()) != null) {
				if (line.startsWith("database."))
					continue;
				sb.append(line + "\n");
			}
		} catch (FileNotFoundException e) {
			log(e.toString());
		} catch (IOException e) {
			log(e.toString());
		}
		
		if (params.get("db_type").equals("MySQL")) {
			log("Using MySQL");
			sb.append("database.type=mysql\n");
			sb.append("database.driver=" + MYSQL_DRIVER + "\n");
			sb.append("database.url=" + params.get("db_url") + "\n");
			sb.append("database.login=" + params.get("db_login") + "\n");
			sb.append("database.password=" + params.get("db_passwd") + "\n");
		} else {
			log("Using HSQLDB");
			sb.append("database.type=hsqldb\n");
			sb.append("database.driver=" + HSQLDB_DRIVER + "\n");
			sb.append("database.url=" + HSQLDB_URL + "\n");
			sb.append("database.login=" + HSQLDB_LOGIN + "\n");
			sb.append("database.password=" + HSQLDB_PASSWD + "\n");
		}
		
		try {
			PrintWriter pw = new PrintWriter(globalLocation);
			pw.println(sb.toString());
			pw.close();
		} catch (FileNotFoundException ex) {
			log(ex.toString());
		}
		
		if(useTomcat) {
		/*
		 * Server.xml
		 */
		rewriteServerXml(params.get("tomcat_port"));

		}
		
		/*
		 * Creer env.sh
		 */
		createEnv(params.get("ant_home"), params.get("java_home"));
		
		/*
		 * Run scripts
		 */
		runScripts(params.get("db_type").equalsIgnoreCase("HSQLDB"),  params);
		
		log("Configuration complete");
		logger.close();
	}
	
	private static void rewriteServerXml(String newPort) {
		if (newPort.trim().isEmpty())
			newPort = "8080";
		String location = "/usr/share/biomaj/apache-tomcat-6.0.24/conf/server.xml";
		try {
			BufferedReader br = new BufferedReader(new FileReader(location));
			String line = null;
			StringBuilder sb = new StringBuilder();
			Pattern p = Pattern.compile("<Connector port=\"(\\d+)\" protocol=\"HTTP/1\\.1\"");
			while ((line = br.readLine()) != null) {
				Matcher m = p.matcher(line);
				if (m.find()) {
					sb.append(line.replaceFirst(m.group(1), newPort) + "\n");
				}
				else
					sb.append(line + "\n");
			}
			br.close();
			
			FileWriter fw = new FileWriter(location);
			fw.write(sb.toString());
			fw.close();
			
		} catch (IOException e) {
			log(e.toString());
		}
	}
	
	private static void createPasswdFile(String login, String passwd) {
		try {
			PrintWriter pw = new PrintWriter(biomajRoot + "/.bmajwatcherpasswd");
			MessageDigest digest = MessageDigest.getInstance("SHA1");
			digest.update(login.getBytes());
			byte[] hashedLogin = digest.digest();

			digest.reset();
			digest.update(passwd.getBytes());
			byte[] hashedPasswd = digest.digest();
			
			pw.println(getHexString(hashedLogin) + ":" + getHexString(hashedPasswd));
			pw.close();
		} catch (FileNotFoundException e) {
			log(e.toString());
		} catch (NoSuchAlgorithmException e) {
			log(e.toString());
		}
	}
	
	public static String getHexString(byte[] buf) {
		char[] TAB_BYTE_HEX = { '0', '1', '2', '3', '4', '5', '6', '7', '8',
				'9', 'a', 'b', 'c', 'd', 'e', 'f' };

		StringBuilder sb = new StringBuilder(buf.length * 2);

		for (int i = 0; i < buf.length; i++) {
			sb.append(TAB_BYTE_HEX[(buf[i] >>> 4) & 0xf]);
			sb.append(TAB_BYTE_HEX[buf[i] & 0x0f]);
		}
		return sb.toString();
	}
	
	private static void createEnv(String antHome, String javaHome) {

		try {
			PrintWriter pw = new PrintWriter(biomajRoot + "/bin/env.sh");
			pw.println("#!/bin/sh");
			pw.println("export JAVA_HOME=" + javaHome);
			pw.println("export ANT_HOME=" + antHome);
			pw.println("export BIOMAJ_ROOT=" + biomajRoot);
			pw.close();
		} catch (FileNotFoundException ex) {
			log(ex.toString());
		}
		
	}
	
	/*
	private static void copyTo(String source, String dest) {
		try {
			InputStream  in = new FileInputStream(source);
			OutputStream out = new FileOutputStream(dest);
			byte[] buf = new byte[1024];
			int len;
			while ((len = in.read(buf)) > 0) {
				out.write(buf, 0, len);
			}
			in.close();
			out.close(); 
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		}
	}*/
	
	/**
	 * Starts hsql server if required.
	 * Creates hsql database if required.
	 * Starts tomcat server.
	 * 
	 * @param startHsql whether to init hsqldb server
	 */
	private static void runScripts(boolean startHsql, Map<String,String> params) {
		
		String[] env = {"ANT_HOME=" + params.get("ant_home"), "JAVA_HOME=" + params.get("java_home"), "BIOMAJ_ROOT=" + biomajRoot};
		
		try {
			log("Building jar with ant...");
			Process p = Runtime.getRuntime().exec("ant", env, new File(biomajRoot));
			p.waitFor();
			log("Done.");
		} catch (IOException e1) {
			log(e1.toString());
		} catch (InterruptedException e2) {
			log(e2.toString());
		}
				
		if (startHsql) {
			try {
				if (!new File(biomajRoot + "/tmp/db.pid").exists()) {
					new File(biomajRoot + "/tmp").mkdir();
					log("Starting database server...");
					Runtime.getRuntime().exec("bash " + biomajRoot + "/bin/start_dbserver.sh", env, null);
					try {
						TimeUnit.SECONDS.sleep(2);
					} catch (InterruptedException e) {
						log(e.toString());
					}
					
					log("Creating database...");
					Runtime.getRuntime().exec("bash " + biomajRoot + "/bin/create_database.sh", env, null);
				}
			} catch (IOException e) {
				log(e.toString());
			}
		} else { // Create schema
			// createMySQLSchema(params);
		}
	}
	
	
	private static void createMySQLSchema(Map<String, String> params) {
		try {
			Class.forName("com.mysql.jdbc.Driver");
		} catch (ClassNotFoundException e2) {
			log(e2.toString());
		}
		try {
			String url = params.get("db_url");
			Connection conn = DriverManager.getConnection(url.substring(0, url.lastIndexOf("/")), params.get("db_login"), params.get("db_passwd"));
			Statement s = conn.createStatement();
			try {
				BufferedReader br = new BufferedReader(new FileReader(biomajRoot + "/sql/mysql.sql"));
				StringBuilder sb = new StringBuilder();
				String line = "";
				try {
					while ((line = br.readLine()) != null) {
						if (!line.startsWith("--") && !line.trim().isEmpty())
							sb.append(line);
						if (line.endsWith(";")) {
							s.executeUpdate(sb.toString());
							sb = new StringBuilder();
						}
						
					}
				} catch (IOException e) {
					log(e.toString());
				}
					
			} catch (FileNotFoundException e) {
				log(e.toString());
			}
			conn.close();
		} catch (SQLException e1) {
			log(e1.toString());
		}
	}
	
	private static void log(String message) {
		SimpleDateFormat sdf = new SimpleDateFormat("[yyyy-MM-dd HH:mm:ss]");
		logger.println(sdf.format(new Date()) + ": " + message);
		logger.flush();
	}
}

