/*=====================================================================*/
/*    .../prgm/project/bigloo/api/pthread/src/Posix/bglpthread.c       */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Fri Feb 22 12:12:04 2002                          */
/*    Last change :  Tue May  3 14:49:34 2005 (serrano)                */
/*    Copyright   :  2002-05 Manuel Serrano                            */
/*    -------------------------------------------------------------    */
/*    C utilities for native Bigloo fair threads implementation.       */
/*=====================================================================*/
#include <pthread.h>
#include <sched.h>
#include <stdlib.h>
#include <string.h>

#define GC_PRIVATE_H
#include <gc.h>
#include <bglpthread.h>

/*---------------------------------------------------------------------*/
/*    Imports                                                          */
/*---------------------------------------------------------------------*/
BGL_RUNTIME_DECL void bgl_thread_getspecific_register();
BGL_RUNTIME_DECL void bgl_thread_setspecific_register();
BGL_RUNTIME_DECL void bgl_dynamic_env_register();
BGL_RUNTIME_DECL void bgl_dynamic_env_set_register();

/*---------------------------------------------------------------------*/
/*    pthread_key_t                                                    */
/*    bgldenv_key ...                                                  */
/*---------------------------------------------------------------------*/
static pthread_key_t bgldenv_key;

/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglpth_dynamic_env ...                                           */
/*---------------------------------------------------------------------*/
bgldenv_t
bglpth_dynamic_env() {
   return (bgldenv_t)pthread_getspecific( bgldenv_key );
}
 
/*---------------------------------------------------------------------*/
/*    bgldenv_t                                                        */
/*    bglpth_dynamic_env_set ...                                       */
/*---------------------------------------------------------------------*/
bgldenv_t
bglpth_dynamic_env_set( bgldenv_t env ) {
   pthread_setspecific( bgldenv_key, env );

   return env;
}

/*---------------------------------------------------------------------*/
/*    bglpthread_t                                                     */
/*    bglpth_thread_new ...                                            */
/*---------------------------------------------------------------------*/
bglpthread_t
bglpth_thread_new( obj_t thunk ) {
   bglpthread_t t = (bglpthread_t)GC_MALLOC( sizeof( struct bglpthread ) );

   pthread_mutex_init( &(t->mutex), 0L );
   pthread_cond_init( &(t->condvar), 0L );
   
   t->thunk = thunk;
   t->specific = BUNSPEC;
   t->cleanup = BUNSPEC;
   t->status = 0;
   
   return t;
}

/*---------------------------------------------------------------------*/
/*    static void                                                      */
/*    bglpth_thread_cleanup ...                                        */
/*---------------------------------------------------------------------*/
static void
bglpth_thread_cleanup( void *arg ) {
   bglpthread_t self = (bglpthread_t)arg;
   obj_t cleanup = self->cleanup;

   /* lock the internal state of the thread */
   pthread_mutex_lock( &(self->mutex) );
   
   /* mark the thread terminated */
   self->status = 2;
   
   /* unlock all locked mutexes */
   bglpth_mutexes_unlock( self );
   
   /* unlock the internal state of the thread */
   pthread_mutex_unlock( &(self->mutex) );
   
   /* invoke user cleanup */
   if( PROCEDUREP( cleanup ) ) {
      PROCEDURE_ENTRY( cleanup )( cleanup, self->bglthread, BEOA );
   }
}

/*---------------------------------------------------------------------*/
/*    static void *                                                    */
/*    bglpth_thread_run ...                                            */
/*---------------------------------------------------------------------*/
static void *
bglpth_thread_run( void *arg ) {
   bglpthread_t self = (bglpthread_t)arg;
   obj_t thunk = self->thunk;

   bglpth_dynamic_env_set( self->env );
   self->env->stack_bottom = (char *)&arg;
   self->env->current_thread = self;
   
   bgl_init_trace();

   pthread_setcanceltype( PTHREAD_CANCEL_ASYNCHRONOUS, 0 );
   
   pthread_cleanup_push( bglpth_thread_cleanup, arg );
   
   /* mark the thread started */
   pthread_mutex_lock( &(self->mutex) );
   self->status = 1;
   pthread_cond_broadcast( &(self->condvar) );
   pthread_mutex_unlock( &(self->mutex) );

   /* enter the user code */
   PROCEDURE_ENTRY( thunk )( thunk, BEOA );
   pthread_cleanup_pop( 1 );
       
   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_thread_start ...                                          */
/*---------------------------------------------------------------------*/
void
bglpth_thread_start( bglpthread_t thread, obj_t bglthread, bool_t dt ) {
   pthread_attr_t a;

   pthread_attr_init( &a );

   if( dt ) pthread_attr_setdetachstate( &a, PTHREAD_CREATE_DETACHED );

   thread->bglthread = bglthread;
   thread->env = bgl_dup_dynamic_env( bgl_dynamic_env() );

   if( pthread_create( &(thread->pthread), &a, bglpth_thread_run, thread ) )
      FAILURE( string_to_bstring( "thread-start!" ),
	       string_to_bstring( "Cannot start thread" ),
	       string_to_bstring( strerror( errno ) ) );
}

/*---------------------------------------------------------------------*/
/*    bglpthread_t                                                     */
/*    bglpth_current_pthread ...                                       */
/*---------------------------------------------------------------------*/
bglpthread_t
bglpth_current_pthread() {
   return bgl_dynamic_env()->current_thread;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    bglpth_current_thread ...                                        */
/*---------------------------------------------------------------------*/
obj_t
bglpth_current_thread() {
   bglpthread_t cur = bglpth_current_pthread();
   
   return cur ? cur->bglthread : BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_thread_join ...                                           */
/*---------------------------------------------------------------------*/
void
bglpth_thread_join( bglpthread_t t ) {
   pthread_mutex_lock( &(t->mutex) );
   if( !t->status ) {
      pthread_cond_wait( &(t->condvar), &(t->mutex) );
   }
   pthread_mutex_unlock( &(t->mutex) );

   if( pthread_join( t->pthread, 0L ) ) {
      FAILURE( string_to_bstring( "thread-join!" ),
	       string_to_bstring( "Cannot join thread" ),
	       string_to_bstring( strerror( errno ) ) );
   }
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    bglpth_thread_terminate ...                                      */
/*---------------------------------------------------------------------*/
bool_t
bglpth_thread_terminate( bglpthread_t t ) {
   pthread_mutex_lock( &(t->mutex) );
   if( t->status != 2 ) {
      pthread_cancel( t->pthread );
      pthread_mutex_unlock( &(t->mutex) );
      return 1;
   } else {
      pthread_mutex_unlock( &(t->mutex) );
      return 0;
   }
}

/*---------------------------------------------------------------------*/
/*    void *                                                           */
/*    bglpth_thread_getspecific ...                                    */
/*---------------------------------------------------------------------*/
void *
bglpth_thread_getspecific( void **key ) {
   return pthread_getspecific( *((pthread_key_t *)key) );
}

/*---------------------------------------------------------------------*/
/*    void *                                                           */
/*    bglpth_thread_setspecific ...                                    */
/*---------------------------------------------------------------------*/
void *
bglpth_thread_setspecific( void **key, void *val ) {
   if( !pthread_setspecific( *((pthread_key_t *)key), val ) ) {
      return val;
   } else {
      return 0L;
   }
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    bglpth_setup_thread ...                                          */
/*---------------------------------------------------------------------*/
void
bglpth_setup_thread() {
   pthread_key_create( &bgldenv_key, 0L );

   bgl_thread_getspecific_register( &bglpth_thread_getspecific );
   bgl_thread_setspecific_register( &bglpth_thread_setspecific );
   
   bglpth_dynamic_env_set( bgl_dynamic_env() );
			   
   bgl_dynamic_env_register( &bglpth_dynamic_env );
   bgl_dynamic_env_set_register( &bglpth_dynamic_env_set );

   bglpth_dynamic_env_set( make_dynamic_env() );
}
