/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "mapping.h"
#include <config.h>
#include "utilities.h"
#include "bible.h"
#include "shell.h"
#include "gwrappers.h"
#include "books.h"
#include "sqlite_reader.h"
#include "directories.h"


ustring mapping_get_filename (ustring name)
{
  replace_text (name, " ", "_");
  return gw_build_filename (directories_get_templates (), "map_" + name.casefold() + ".sql");
}


ustring mapping_get_name (ustring path)
{
  path = gw_path_get_basename (path);
  ustring name (path);
  name.erase (0, 4);
  name.erase (name.length() - 4, 4);
  replace_text (name, "_", " ");
  size_t pos = 0;
  while (pos != string::npos) {
    name.replace (pos, 1, upperCase (name.substr (pos, 1)));
    pos = name.find (" ", pos);
    if (pos != string::npos) pos++;
  }
  return name;
}


vector<ustring> mapping_get_raw_templates_available ()
{
  vector <ustring> paths;
  ReadFiles rf (PACKAGE_DATA_DIR, "map_", ".txt");
  for (unsigned int i = 0; i < rf.files.size(); i++) {
    paths.push_back (gw_build_filename (PACKAGE_DATA_DIR, rf.files[i]));
  }
  ReadFiles rf2 (directories_get_templates_user (), "map_", ".txt");
  for (unsigned int i = 0; i < rf2.files.size(); i++) {
    paths.push_back (gw_build_filename (directories_get_templates_user (), rf2.files[i]));
  }
  return paths;
}


Mapping::Mapping (const ustring& name, unsigned int book)
{
  // Open database.
  error = NULL;
  ustring database = mapping_get_filename (name);
  rc = sqlite3_open (database.c_str (), &db);
  sqlite3_busy_timeout (db, 1000);
  // Save book.
  mybook = book;
}


Mapping::~Mapping ()
{
  sqlite3_close (db);
}


void Mapping::me_to_original (int mychapter, const ustring& myverse,
                              vector<int>& original_chapter, vector<int>& original_verse)
/*
Input:
- a chapter (int).
- a verse (ustring) : can be a sequence or a range too.
Output:
- containers of mapped chapters/verses.
*/
{
  // Storage for the containers with chapters and verses to map.
  vector<int> chapters;
  vector<int> verses;
  // Convert the verse to a container with half verses.
  vector<int> half_verses = verses_encode (myverse);
  set<int> encoded;
  // Divide the half-verses by 2 and add them to the containers with values to map.
  for (unsigned int i = 0; i < half_verses.size(); i++) {
    int verse = half_verses[i] / 2;
    if (encoded.find (verse) == encoded.end()) {
      chapters.push_back (mychapter);
      verses.push_back (verse);
      encoded.insert (verse);
    }
  }
  // Do the mapping.
  me_to_original (chapters, verses, original_chapter, original_verse);
}


void Mapping::me_to_original (vector<int> mychapter, vector<int> myverse,
                              vector<int>& original_chapter, vector<int>& original_verse)
/*
Input:
- containers of chapters/verses.
Output:
- containers of mapped chapters/verses.
Containers as input and as output are needed as there may be a verse that 
maps to one verse in a chapter, and one verse in the next chapter.
*/
{
  for (unsigned int i = 0; i < mychapter.size(); i++) {
    vector<int> origchapter;
    vector<int> origverse;
    get_original (mychapter[i], myverse[i], origchapter, origverse);
    for (unsigned int i2 = 0; i2 < origchapter.size(); i2++) {
      original_chapter.push_back (origchapter[i2]);
      original_verse.push_back (origverse[i2]);
    }    
  }
}


void Mapping::original_to_me (int originalchapter, const ustring& originalverse,
                              vector<int>& mychapter, vector<int>& myverse)
/*
Input:
- a chapter (int).
- a verse (ustring) : can be a sequence or a range too.
Output:
- containers of mapped chapters/verses.
*/
{
  // Storage for the containers with chapters and verses to map.
  vector<int> chapters;
  vector<int> verses;
  // Convert the verse to a container with half verses.
  vector<int> half_verses = verses_encode (originalverse);
  set<int> encoded;
  // Divide the half-verses by 2 and add them to the containers with values to map.
  for (unsigned int i = 0; i < half_verses.size(); i++) {
    int verse = half_verses[i] / 2;
    if (encoded.find (verse) == encoded.end()) {
      chapters.push_back (originalchapter);
      verses.push_back (verse);
      encoded.insert (verse);
    }
  }
  // Do the mapping.
  original_to_me (chapters, verses, mychapter, myverse);
}


void Mapping::original_to_me (vector<int> original_chapter, vector<int> original_verse,
                              vector<int>& mychapter, vector<int>& myverse)
{
  for (unsigned int i = 0; i < original_chapter.size(); i++) {
    vector<int> chapter;
    vector<int> verse;
    get_me (original_chapter[i], original_verse[i], chapter, verse);
    for (unsigned int i2 = 0; i2 < chapter.size(); i2++) {
      mychapter.push_back (chapter[i2]);
      myverse.push_back (verse[i2]);
    }    
  }
}


void Mapping::get_original (int mychapter, int myverse,
                            vector<int>& originalchapter, vector<int>& originalverse)
{
  try 
  {
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select origchapter, origverse from map where book = %d and chapter = %d and verse = %d;", mybook, mychapter, myverse);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    vector <ustring> lines;
    for (unsigned int i = 0; i < sqlitereader.ustring0.size(); i++) {
      originalchapter.push_back (convert_to_int (sqlitereader.ustring0[i]));
      originalverse.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
  }
  catch (exception& ex) {
    gw_critical (ex.what ());
  }    
  if (originalchapter.empty()) {
    originalchapter.push_back (mychapter);
    originalverse.push_back (myverse);
  }
}


void Mapping::get_me (int originalchapter, int originalverse,
                      vector<int>& mychapter, vector<int>& myverse)
{
  try 
  {
    SqliteReader sqlitereader (0);
    char * sql;
    sql = g_strdup_printf ("select chapter, verse from map where origbook = %d and origchapter = %d and origverse = %d;", mybook, originalchapter, originalverse);
    rc = sqlite3_exec(db, sql, sqlitereader.callback, &sqlitereader, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    vector <ustring> lines;
    for (unsigned int i = 0; i < sqlitereader.ustring0.size(); i++) {
      mychapter.push_back (convert_to_int (sqlitereader.ustring0[i]));
      myverse.push_back (convert_to_int (sqlitereader.ustring1[i]));
    }
  }
  catch (exception& ex) {
    gw_critical (ex.what ());
  }    
  if (mychapter.empty()) {
    mychapter.push_back (originalchapter);
    myverse.push_back (originalverse);
  }
}


void mapping_import_textfile (const ustring& name, const ustring& textfile)
// Imports a mapping textfile.
{
  sqlite3 *db;
  int rc;
  char *error = NULL;
  try 
  {
    // Read the text.
    ReadText rt (textfile);
    // Open and create the database.
    unlink (mapping_get_filename (name).c_str());
    rc = sqlite3_open (mapping_get_filename (name).c_str (), &db);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sqlite3_busy_timeout (db, 1000);
    char * sql;
    sql = g_strdup_printf ("create table map (book integer, chapter integer, verse integer, origbook integer, origchapter integer, origverse integer);");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    sql = g_strdup_printf ("PRAGMA synchronous=OFF;");
    rc = sqlite3_exec (db, sql, NULL, NULL, &error);
    g_free (sql);
    if (rc) throw runtime_error (sqlite3_errmsg(db));
    // Go through the lines.
    cout << name << " " << rt.lines.size() << " lines" << endl;
    for (unsigned int i = 0; i < rt.lines.size(); i++) {
      // Skip comments.
      if (rt.lines[i].find ("#") != string::npos)
        continue;
      // Convert the line.
      ustring line = rt.lines[i];
      size_t pos = line.find_last_of (":");
      if (pos == string::npos) throw runtime_error (line);
      int origverse = convert_to_int (line.substr (pos + 1, 5));
      line.erase (pos, 6);
      pos = line.find_last_of (" ");
      if (pos == string::npos) throw runtime_error (line);
      int origchapter = convert_to_int (line.substr (pos +1, 5));
      line.erase (pos, 6);
      pos = line.find ("=");
      if (pos == string::npos) throw runtime_error (line);
      ustring s = trim (line.substr (pos + 1, 1000));
      int recognized_orig_id = book_find_valid (s);
      line.erase (pos, 1000);
      line = trim (line);
      pos = line.find_last_of (":");
      if (pos == string::npos) throw runtime_error (line);
      int verse = convert_to_int (line.substr (pos + 1, 5));
      line.erase (pos, 6);
      pos = line.find_last_of (" ");
      if (pos == string::npos) throw runtime_error (line);
      int chapter = convert_to_int (line.substr (pos +1, 5));
      line.erase (pos, 6);
      line = trim (line);
      int recognized_id = book_find_valid (line);
      sql = g_strdup_printf ("insert into map values (%d, %d, %d, %d, %d, %d);", recognized_id, chapter, verse, recognized_orig_id, origchapter, origverse);
      rc = sqlite3_exec (db, sql, NULL, NULL, &error);
      g_free (sql);
      if (rc) throw runtime_error (sqlite3_errmsg(db));
      if ((recognized_id == 0) || (recognized_orig_id == 0)) 
        throw runtime_error ("Unknown book");
    }    
  }
  catch (exception& ex) {
    unlink (mapping_get_filename (name).c_str());
    gw_critical (ex.what ());
  }    
  sqlite3_close (db);
}
