/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DebugLayout.h"
#include "DebugConnection.h"
#include "RefCountable.h"
#include "RefCounter.h"
#include <list>
#include <limits>
#include <algorithm>
#include <iostream>

using namespace std;

namespace GtkGUI
{

class DebugLayout::Column :
	public RefCountable<RefCounter<ACE_NULL_SYNCH> >,
	public sigc::trackable
{
public:
	Column(DebugLayout& layout, int left);
	
	virtual ~Column();
	
	int getLeft() const { return m_left; }
	
	int getRight() const { return m_left + CONN_WIDTH; }
	
	int getTop() const;
	
	bool isInUse() const;
	
	bool isEmpty() const { return m_connections.empty(); }
	
	IntrusivePtr<DebugConnection> createConnection();
	
	void growBy(int increment);
	
	void trimTopBy(int value);
private:
	typedef std::list<DebugConnection*> ConnectionList;
	
	void onDestroyConnection(ConnectionList::iterator const& it);
	
	DebugLayout& m_rLayout;
	int m_left;
	ConnectionList m_connections;
};


DebugLayout::DebugLayout()
:	m_width(LAYOUT_PADDING*2),
	m_height(LAYOUT_PADDING*2),
	m_nextObjectBeginPos(LAYOUT_PADDING),
	m_nextObjectEndPos(LAYOUT_PADDING),
	m_connectionsToTrack(20),
	m_trafficLimit(200000)
{
	modify_bg_pixmap(Gtk::STATE_NORMAL, "<none>");
	set_size(m_width, m_height);
}

DebugLayout::~DebugLayout()
{
}

void
DebugLayout::growBy(int increment)
{
	if (increment <= 0) {
		return;
	}
	
	m_height += increment;
	set_size(m_width, m_height);
	
	ColumnQueue::iterator it = m_columns.begin();
	ColumnQueue::iterator const end = m_columns.end();
	for (; it != end; ++it) {
		(*it)->growBy(increment);
	}
}

DebugConnection*
DebugLayout::findOpenConnection(ACE_thread_t thread_id)
{
	ConnectionMap::iterator it = m_openConnections.find(thread_id);
	if (it != m_openConnections.end()) {
		return it->second;
	}
	return 0;
}

void
DebugLayout::clientConnectionBegin(ACE_thread_t thread_id)
{
	removeOldConnections(getConnectionsToTrack() - 1);
	
	// find a column that's not in use
	IntrusivePtr<Column> col;
	ColumnQueue::iterator it = m_columns.begin();
	ColumnQueue::iterator const end = m_columns.end();
	for (; it != end; ++it) {
		if (!(*it)->isInUse()) {
			col = *it;
			break;
		}
	}
	
	if (!col) {
		// create a new column
		int left = LAYOUT_PADDING;
		if (!m_columns.empty()) {
			left = m_columns.back()->getRight() + CONN_HOR_SPACING;
		}
		col.reset(new Column(*this, left));
		m_columns.push_back(col);
		m_width = col->getRight() + LAYOUT_PADDING;
		set_size(m_width, m_height);
	}
	
	IntrusivePtr<DebugConnection> conn = col->createConnection();
	m_orderedConnections.push_back(conn);
	typedef std::pair<ConnectionMap::iterator, bool> MapPos;
	MapPos map_pos = m_openConnections.insert(
		ConnectionMap::value_type(thread_id, conn.get())
	);
	if (!map_pos.second) {
		map_pos.first->second->finish(true);
		map_pos.first->second = conn.get();
	}
	conn->destroySignal().connect(
		sigc::bind(
			sigc::mem_fun(*this, &DebugLayout::onDestroyConnection),
			conn.get(), map_pos.first
		)
	);
}

void
DebugLayout::clientConnectionEnd(ACE_thread_t thread_id)
{
	ConnectionMap::iterator it = m_openConnections.find(thread_id);
	if (it != m_openConnections.end()) {
		it->second->finish();
		m_openConnections.erase(it);
	}
}

void
DebugLayout::httpMessageBegin(
	ACE_thread_t thread_id, int request_id,
	HttpMessageType type, std::string const& headers)
{
	DebugConnection* conn = findOpenConnection(thread_id);
	if (conn) {
		conn->httpMessageBegin(request_id, type, headers);
	}
}

void
DebugLayout::httpMessageEnd(ACE_thread_t thread_id,
	HttpMessageType type, bool error)
{
	DebugConnection* conn = findOpenConnection(thread_id);
	if (conn) {
		conn->httpMessageEnd(type, error);
	}
}

void
DebugLayout::logMessage(ACE_thread_t thread_id, std::string const& msg)
{
	DebugConnection* conn = findOpenConnection(thread_id);
	if (conn) {
		conn->logMessage(msg);
	} else {
		std::cerr << msg << std::endl;
	}
}

void
DebugLayout::logTraffic(ACE_thread_t thread_id,
	SplittableBuffer& traf, TrafficDirection dir)
{
	DebugConnection* conn = findOpenConnection(thread_id);
	if (conn) {
		conn->logTraffic(traf, dir);
	}
}

void
DebugLayout::removeOldConnections(int num2keep)
{
	int num2remove = int(m_orderedConnections.size()) - num2keep;
	if (num2remove > 0) {
		for (int i = 0; i < num2remove; ++i) {
			m_orderedConnections.pop_front();
		}
		adjustColumns();
	}
}

void
DebugLayout::adjustColumns()
{
	// remove unused columns from the back of the queue
	bool size_changed = false;
	while (!m_columns.empty() && m_columns.back()->isEmpty()) {
		m_columns.pop_back();
		size_changed = true;
	}
	if (size_changed) {
		if (m_columns.empty()) {
			m_width = LAYOUT_PADDING*2;
		} else {
			m_width = m_columns.back()->getRight() + LAYOUT_PADDING;
		}
	}
	
	// move everything to the top if some empty space exists there
	int lowest_top = std::numeric_limits<int>::max();
	ColumnQueue::iterator it = m_columns.begin();
	ColumnQueue::iterator const end = m_columns.end();
	for (; it != end; ++it) {
		lowest_top = std::min(lowest_top, (*it)->getTop());
	}
	if (lowest_top == std::numeric_limits<int>::max()) {
		// zero columns
		m_height = LAYOUT_PADDING*2;
		size_changed = true;
	} else if (lowest_top > LAYOUT_PADDING) {
		int trim_by = lowest_top - LAYOUT_PADDING;
		for (it = m_columns.begin(); it != end; ++it) {
			(*it)->trimTopBy(trim_by);
		}
		m_height -= trim_by;
		m_nextObjectBeginPos -= trim_by;
		m_nextObjectEndPos -= trim_by;
		size_changed = true;
	}
	
	if (size_changed) {
		set_size(m_width, m_height);
	}
}

void
DebugLayout::onDestroyConnection(
	DebugConnection* conn, ConnectionMap::iterator const& map_pos)
{
	if (!conn->isClosed()) {
		m_openConnections.erase(map_pos);
	} // otherwise it has already been removed from m_openConnections
}


/*======================== DebugLayout::Column =========================*/

DebugLayout::Column::Column(DebugLayout& layout, int left)
:	m_rLayout(layout),
	m_left(left)
{
}

DebugLayout::Column::~Column()
{
}

int
DebugLayout::Column::getTop() const
{
	if (m_connections.empty()) {
		return LAYOUT_PADDING;
	}
	return m_connections.front()->getTop();
}

bool
DebugLayout::Column::isInUse() const
{
	if (m_connections.empty()) {
		return false;
	}
	return !m_connections.back()->isClosed();
}

IntrusivePtr<DebugConnection>
DebugLayout::Column::createConnection()
{	
	int top = m_rLayout.getNextObjectBeginPos();
	if (!m_connections.empty()) {
		DebugConnection* last_conn = m_connections.back();
		if (last_conn) {
			top = std::max(top, last_conn->getBottom() + CONN_VER_SPACING);
		}
	}
	int bottom = top + CONN_INITIAL_HEIGHT;
	int grow_by = bottom - (m_rLayout.getHeight() - LAYOUT_PADDING);
	if (grow_by > 0) {
		m_rLayout.growBy(grow_by);
	}
	
	IntrusivePtr<DebugConnection> conn(
		new DebugConnection(
			m_rLayout, top, m_left, m_left + CONN_WIDTH
		)
	);
	ConnectionList::iterator pos = m_connections.insert(m_connections.end(), conn.get());
	conn->destroySignal().connect(
		sigc::bind(
			sigc::mem_fun(*this, &Column::onDestroyConnection),
			pos
		)
	);
	m_rLayout.put(*conn, m_left, top);
	
	m_rLayout.setNextObjectBeginPos(bottom);
	m_rLayout.setNextObjectEndPos(bottom);
	
	return conn;
}

void
DebugLayout::Column::growBy(int increment)
{
	if (m_connections.empty()) {
		return;
	}
	DebugConnection* last_conn = m_connections.back();
	if (!last_conn->isClosed()) {
		last_conn->growBy(increment);
	}
}

void
DebugLayout::Column::trimTopBy(int value)
{
	ConnectionList::iterator it = m_connections.begin();
	ConnectionList::iterator const end = m_connections.end();
	for (; it != end; ++it) {
		int new_top = (*it)->getTop() - value;
		m_rLayout.move(**it, m_left, new_top);
		(*it)->setTop(new_top);
	}
}

void
DebugLayout::Column::onDestroyConnection(ConnectionList::iterator const& iter)
{
	m_connections.erase(iter);
}

} // namespace GtkGUI

