/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RegexResponseFilter.h"
#include "RegexFilterDescriptor.h"
#include "HttpResponseMetadata.h"
#include "HttpHeadersCollection.h"
#include "TextPattern.h"
#include "BString.h"
#include "JsFilterContext.h"
#include "RequestTag.h"
#include <ace/config-lite.h>
#include <boost/regex.hpp>
#include <string>
#include <vector>
#include <stdexcept>
#include <cassert>

using namespace std;

template<>
std::auto_ptr<std::string>
RegexResponseFilter::getReplacement(
	boost::match_results<SplittableBuffer::ByteIterator> const& mr)
{
	namespace rc = boost::regex_constants;
	auto_ptr<string> res;
	
	assert(m_ptrFilterDescriptor->replacement().get());
	std::string const& replacement = *m_ptrFilterDescriptor->replacement();
	
	switch (m_ptrFilterDescriptor->replacementType()) {
		case RegexFilterDescriptor::TEXT: {
			res.reset(new string(replacement));
			break;
		}
		case RegexFilterDescriptor::EXPRESSION: {
			res.reset(new string(
				mr.format(replacement, rc::format_all)
			));
			break;
		}
		case RegexFilterDescriptor::JS: {
			vector<BString> params;
			params.reserve(mr.size());
			for (unsigned i = 0; i < mr.size(); ++i) {
				params.push_back(SplittableBuffer::toBString(
					mr[i].first, mr[i].second
				));
			}
			JsFilterContext& cx = getFilterChain().getJsContextFor(
				m_ptrFilterDescriptor->getGroupTag()
			);
			res = cx.performAction(
				m_ptrFilterDescriptor->getTag(),
				m_ptrFilterDescriptor->getGroupTag(),
				replacement, params
			);
			break;
		}
	}
	
	return res;
}

RegexResponseFilter::RegexResponseFilter(ResponseFilterChain& chain,
 	IntrusivePtr<RegexFilterDescriptor const> const& filter_desc)
:	ResponseFilterBase(chain),
	m_ptrFilterDescriptor(filter_desc),
	m_matchCount(0),
	m_isDataBegin(true),
	m_isDataEnd(false)
{
}

RegexResponseFilter::~RegexResponseFilter()
{
}

void
RegexResponseFilter::processMetadata(
	RequestStatus& status, std::auto_ptr<HttpResponseMetadata> metadata)
{
	metadata->headers().removeHeader(BString("Content-MD5"));
	metadata->setBodyStatus(HttpResponseMetadata::BODY_SIZE_UNKNOWN);
	getRequestTag()->flags().set(RequestTag::RESPONSE_MODIFIED);
	outputMetadata(status, metadata);
}

void
RegexResponseFilter::processBodyData(
	RequestStatus& status, SplittableBuffer& data, bool eof)
{
	assert(m_ptrFilterDescriptor->searchPattern().get());
	
	if (m_isDataEnd) {
		assert(data.empty() && eof);
		return;
	}
	m_isDataEnd = eof;
	
	m_data.appendDestructive(data);
	while (true) {
		namespace rc = boost::regex_constants;
		bool matched = false;
		boost::match_results<SplittableBuffer::ByteIterator> mr;
		if (isMatchingAllowed()) {
			boost::match_flag_type flags = rc::match_default|rc::match_single_line;
			if (!m_isDataBegin) {
				flags |= rc::match_not_bob|rc::match_not_bol;
			}
			if (!m_isDataEnd) {
				flags |= rc::match_not_eob|rc::match_not_eol|rc::match_partial;
			}
			try {
				matched = boost::regex_search(
					m_data.begin(), m_data.end(), mr,
					m_ptrFilterDescriptor->searchPattern()->regex(), flags
				);
			} catch (std::runtime_error& e) {}
		}
		
		if (!matched) {
			// no match
			if (!m_data.empty()) {
				m_isDataBegin = false;
				m_outStream.data().appendDestructive(m_data);
			}
			break;
		} else if (!mr[0].matched) {
			// partial match
			if (mr[0].first != m_data.begin()) {
				m_isDataBegin = false;
				m_data.splitFrontAppendBack(mr[0].first, m_outStream.data());
			}
			break;
		} else {
			// full match
			auto_ptr<string> replacement = getReplacement<auto_ptr<string> >(mr);
			if (!replacement.get()) {
				if (mr[0].first != m_data.begin()) {
					m_isDataBegin = false;
				}
				if (mr[0].first == m_data.end()) {
					m_outStream.data().appendDestructive(m_data);
					break;
				} else {
					SplittableBuffer::ByteIterator iter(mr[0].first);
					++iter;
					m_data.splitFrontAppendBack(iter, m_outStream.data());
					if (m_data.empty()) {
						break;
					} else {
						continue;
					}
				}
			}
			
			bool const empty_match = (mr[0].first == mr[0].second);
			if (mr[0].first != m_data.begin()) {
				m_data.splitFrontAppendBack(mr[0].first, m_outStream.data());
				m_isDataBegin = false;
			}
			
			registerMatch();
			
			m_outStream << *replacement;
			m_data.trimFront(mr[0].second);
			if (m_data.empty()) {
				break;
			} else if (empty_match) {
				m_data.splitFrontAppendBack(1, m_outStream.data());
				m_isDataBegin = false;
			}
		}
	}
	if (!m_outStream.data().empty() || eof) {
		outputBodyData(status, m_outStream.data(), eof);
	}
}

bool
RegexResponseFilter::isMatchingAllowed() const
{
	if (m_ptrFilterDescriptor->matchCountLimit() == m_matchCount) {
		// matchCountLimit may be -1 which means no limit
		return false;
	}
	
	string if_flag = m_ptrFilterDescriptor->ifFlag();
	if (if_flag.empty() || getFilterChain().isFlagSet(
	    m_ptrFilterDescriptor->getGroupTag(), if_flag)) {
		return true;
	}
	
	return false;
}

void
RegexResponseFilter::registerMatch()
{
	++m_matchCount;
	
	string set_flag = m_ptrFilterDescriptor->setFlag();
	if (!set_flag.empty()) {
		getFilterChain().groupLocalFlags().set(
			m_ptrFilterDescriptor->getGroupTag(), set_flag
		);
	}
	
	string clear_flag = m_ptrFilterDescriptor->clearFlag();
	if (!clear_flag.empty()) {
		getFilterChain().groupLocalFlags().clear(
			m_ptrFilterDescriptor->getGroupTag(), clear_flag
		);
	}
}
