/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef INTRUSIVELIST_H_
#define INTRUSIVELIST_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "NonCopyable.h"
#include <iterator>
#include <cassert>
#include <stddef.h>

class IntrusiveListBase;
template<typename T> class IntrusiveList;
template<typename T, typename Node> class IntrusiveListIterator;
template<typename T> class IntrusiveListConstIterator;

class IntrusiveListNode
{
	DECLARE_NON_COPYABLE(IntrusiveListNode)
	friend class IntrusiveListBase;
	template<typename T> friend class IntrusiveList;
	template<typename T, typename Node> friend class IntrusiveListIterator;
public:
	IntrusiveListNode() : prev(0), next(0), object(0) {}
	
	~IntrusiveListNode() { unlink(); }

	void unlink();
private:
	IntrusiveListNode* prev;
	IntrusiveListNode* next;
	void const* object;
};


template<typename T, typename Node>
class IntrusiveListIterator : public std::iterator<
	std::bidirectional_iterator_tag, T, ptrdiff_t>
{
	friend class IntrusiveListConstIterator<T>;
	friend class IntrusiveList<T>;
public:
	typedef T& reference;
	typedef T* pointer;
	
	IntrusiveListIterator() : m_pNode(0) {}
	
	IntrusiveListIterator(IntrusiveListIterator const& other)
	: m_pNode(other.m_pNode) {}
		
	// Default assignment operator OK

	reference operator*() const;
	
	pointer operator->() const;
		
	IntrusiveListIterator& operator++();
	
	IntrusiveListIterator& operator--();
	
	IntrusiveListIterator operator++(int);
	
	IntrusiveListIterator operator--(int);
		
	bool operator==(IntrusiveListIterator const& rhs) const;
		
	bool operator!=(IntrusiveListIterator const& rhs) const;
private:
	explicit IntrusiveListIterator(Node* node) : m_pNode(node) {}

	Node* m_pNode;
};


template<typename T>
class IntrusiveListConstIterator :
	public IntrusiveListIterator<T const, IntrusiveListNode const>
{
	friend class IntrusiveList<T>;
public:
	IntrusiveListConstIterator() {}
	
	IntrusiveListConstIterator(IntrusiveListConstIterator const& other)
	: IntrusiveListIterator<T const, IntrusiveListNode const>(other.m_pNode) {}

	IntrusiveListConstIterator(IntrusiveListIterator<T, IntrusiveListNode> const& other)
	: IntrusiveListIterator<T const, IntrusiveListNode const>(other.m_pNode) {}
private:
	explicit IntrusiveListConstIterator(IntrusiveListNode const* node)
	: IntrusiveListIterator<T const, IntrusiveListNode const>(node) {}
};


class IntrusiveListBase
{
	DECLARE_NON_COPYABLE(IntrusiveListBase)
	template<typename T> friend class IntrusiveList;
public:
	IntrusiveListBase();

	~IntrusiveListBase();
	
	void clear();
	
	bool empty() const;
	
	size_t size() const;
	
	void pop_front();

	void pop_back();
private:
	IntrusiveListNode m_sentinel;
};


template<typename T>
class IntrusiveList : public IntrusiveListBase
{
	DECLARE_NON_COPYABLE(IntrusiveList)
public:
	typedef IntrusiveListIterator<T, IntrusiveListNode> iterator;
	typedef IntrusiveListConstIterator<T> const_iterator;
	
	IntrusiveList() {}
	
	T& front() const;

	T& back() const;
	
	void push_front(T& obj);
	
	void push_front(T& obj, IntrusiveListNode& node);
	
	void push_back(T& obj);
	
	void push_back(T& obj, IntrusiveListNode& node);
	
	iterator insert(iterator loc, T& obj);
	
	iterator insert(iterator loc, T& obj, IntrusiveListNode& node);
	
	iterator begin();

	const_iterator begin() const;

	iterator end();

	const_iterator end() const;
};


/*=========================== IntrusiveListNode ==========================*/

inline void
IntrusiveListNode::unlink()
{
	if (object) {
		assert(next && prev);
		prev->next = next;
		next->prev = prev;
		prev = 0;
		next = 0;
		object = 0;
	} // Otherwise it could be an unlinked node or a sentinel node.
}


/*======================== IntrusiveListIterator =========================*/

template<typename T, typename Node>
inline typename IntrusiveListIterator<T, Node>::reference
IntrusiveListIterator<T, Node>::operator*() const
{
	assert(m_pNode);
	return *(T*)m_pNode->object;
}

template<typename T, typename Node>
inline typename IntrusiveListIterator<T, Node>::pointer
IntrusiveListIterator<T, Node>::operator->() const
{
	assert(m_pNode);
	return (T*)m_pNode->object;
}

template<typename T, typename Node>
inline IntrusiveListIterator<T, Node>&
IntrusiveListIterator<T, Node>::operator++()
{
	assert(m_pNode);
	m_pNode = m_pNode->next;
	return *this;
}

template<typename T, typename Node>
inline IntrusiveListIterator<T, Node>&
IntrusiveListIterator<T, Node>::operator--()
{
	assert(m_pNode);
	m_pNode = m_pNode->prev;
	return *this;
}

template<typename T, typename Node>
inline IntrusiveListIterator<T, Node>
IntrusiveListIterator<T, Node>::operator++(int)
{
	IntrusiveListIterator old(*this);
	++*this;
	return old;
}

template<typename T, typename Node>
inline IntrusiveListIterator<T, Node>
IntrusiveListIterator<T, Node>::operator--(int)
{
	IntrusiveListIterator old(*this);
	--*this;
	return old;
}

template<typename T, typename Node>
inline bool
IntrusiveListIterator<T, Node>::operator==(IntrusiveListIterator const& rhs) const
{
	return m_pNode == rhs.m_pNode;
}

template<typename T, typename Node>
inline bool
IntrusiveListIterator<T, Node>::operator!=(IntrusiveListIterator const& rhs) const
{
	return m_pNode != rhs.m_pNode;
}


/*========================== IntrusiveListBase ===========================*/

inline
IntrusiveListBase::IntrusiveListBase()
{
	m_sentinel.prev = &m_sentinel;
	m_sentinel.next = &m_sentinel;
}

inline
IntrusiveListBase::~IntrusiveListBase()
{
	clear();
}

inline void
IntrusiveListBase::clear()
{
	IntrusiveListNode* node;
	while ((node = m_sentinel.next) != &m_sentinel) {
		node->unlink();
	}
}

inline bool
IntrusiveListBase::empty() const
{
	return m_sentinel.next == &m_sentinel;
}

inline size_t
IntrusiveListBase::size() const
{
	size_t counter = 0;
	IntrusiveListNode* node = m_sentinel.next;
	for (; node != &m_sentinel; node = node->next) {
		++counter;
	}
	return counter;
}

inline void
IntrusiveListBase::pop_front()
{
	assert(!empty());
	m_sentinel.next->unlink();
}

inline void
IntrusiveListBase::pop_back()
{
	assert(!empty());
	m_sentinel.prev->unlink();
}


/*=========================== IntrusiveList ==============================*/

template<typename T>
inline T&
IntrusiveList<T>::front() const
{
	assert(!empty());
	return *(T*)m_sentinel.next->object;
}

template<typename T>
inline T&
IntrusiveList<T>::back() const
{
	assert(!empty());
	return *(T*)m_sentinel.prev->object;
}

template<typename T>
inline void
IntrusiveList<T>::push_front(T& obj)
{
	push_front(obj, static_cast<IntrusiveListNode&>(obj));
}

template<typename T>
void
IntrusiveList<T>::push_front(T& obj, IntrusiveListNode& node)
{
	assert(!node.prev && !node.next && !node.object);
	node.object = &obj;
	
	node.prev = &m_sentinel;
	node.next = m_sentinel.next;
	m_sentinel.next->prev = &node;
	m_sentinel.next = &node;
}

template<typename T>
inline void
IntrusiveList<T>::push_back(T& obj)
{
	push_back(obj, static_cast<IntrusiveListNode&>(obj));
}

template<typename T>
void
IntrusiveList<T>::push_back(T& obj, IntrusiveListNode& node)
{
	assert(!node.prev && !node.next && !node.object);
	node.object = &obj;

	node.prev = m_sentinel.prev;
	node.next = &m_sentinel;
	m_sentinel.prev->next = &node;
	m_sentinel.prev = &node;
}

template<typename T>
inline typename IntrusiveList<T>::iterator
IntrusiveList<T>::insert(iterator loc, T& obj)
{
	return insert(loc, obj, static_cast<IntrusiveListNode&>(obj));
}

template<typename T>
typename IntrusiveList<T>::iterator
IntrusiveList<T>::insert(iterator loc, T& obj, IntrusiveListNode& node)
{
	assert(!node.prev && !node.next && !node.object);
	node.object = &obj;

	node.prev = loc.m_pNode->prev;
	node.next = loc.m_pNode;
	loc.m_pNode->prev->next = &node;
	loc.m_pNode->prev = &node;
	
	return iterator(&node);
}

template<typename T>
inline typename IntrusiveList<T>::iterator
IntrusiveList<T>::begin()
{
	return iterator(m_sentinel.next);
}

template<typename T>
inline typename IntrusiveList<T>::const_iterator
IntrusiveList<T>::begin() const
{
	return const_iterator(m_sentinel.next);
}

template<typename T>
inline typename IntrusiveList<T>::iterator
IntrusiveList<T>::end()
{
	return iterator(&m_sentinel);
}

template<typename T>
inline typename IntrusiveList<T>::const_iterator
IntrusiveList<T>::end() const
{
	return const_iterator(&m_sentinel);
}

#endif
