/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2005  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "DeflateDecompressor.h"
#include "AbstractContentIdentifier.h"
#include "SplittableBuffer.h"
#include "DataChunk.h"
#include <memory>
#include <cstring>
#include <algorithm>
#include <cassert>

using namespace std;

struct DeflateDecompressor::OperationContext
{
	std::auto_ptr<DataChunk> chunk;
	size_t chunk_data_size;
	size_t input_consumed;
	size_t output_produced;
	size_t const output_limit;
	
	OperationContext(size_t const limit)
	: chunk(), chunk_data_size(0), input_consumed(0),
	  output_produced(0), output_limit(limit) {}
};


const char DeflateDecompressor::ZLIB_SIGNATURE[] = { 0x78, 0x9c };

DeflateDecompressor::DeflateDecompressor(Flavor flavor)
:	m_flavor(flavor),
	m_state(flavor == AUTODETECT ? ST_DETECTING_FLAVOR : ST_DECOMPRESSING),
	m_isEOF(false),
	m_isStreamReady(false)
{
	if (m_flavor != AUTODETECT) {
		initStream(m_flavor == ZLIB);
	}
}

DeflateDecompressor::~DeflateDecompressor()
{
	if (m_isStreamReady) {
		inflateEnd(&m_strm);
	}
}

void
DeflateDecompressor::reset()
{
	if (m_isStreamReady) {
		if (m_flavor == AUTODETECT) {
			inflateEnd(&m_strm);
			m_isStreamReady = false;
		} else {
			inflateReset(&m_strm);
		}
	}
	m_state = (m_flavor == AUTODETECT ? ST_DETECTING_FLAVOR : ST_DECOMPRESSING);
	m_bufferedInput.clear();
	m_isEOF = false;
}

bool
DeflateDecompressor::isError() const
{
	return (m_state == ST_ERROR);
}

void
DeflateDecompressor::consume(SplittableBuffer& data, bool eof)
{
	if (m_state == ST_ERROR) {
		data.clear();
		return;
	}
	if (m_isEOF) {
		assert(data.empty() && eof);
		return;
	}
	m_bufferedInput.appendDestructive(data);
	m_isEOF = eof;
	if (m_state == ST_DETECTING_FLAVOR) {
		typedef AbstractContentIdentifier ACI;
		ACI::Status status = ACI::checkSignature(
			m_bufferedInput,
			ZLIB_SIGNATURE, ZLIB_SIGNATURE+sizeof(ZLIB_SIGNATURE)
		);
		if (status == ACI::MATCH) {
			initStream(true);
			m_state = ST_DECOMPRESSING;
		} else if (status == ACI::NO_MATCH) {
			m_state = ST_DECOMPRESSING;
			initStream(false);
		}
	}
}

size_t
DeflateDecompressor::retrieveOutput(SplittableBuffer& buf, size_t limit)
{
	OperationContext ctx(limit);
	SplittableBuffer::SpanIterator iter = m_bufferedInput.spanBegin();
	SplittableBuffer::SpanIterator const iter_end = m_bufferedInput.spanEnd();
	for (; m_state == ST_DECOMPRESSING && iter != iter_end; ++iter) {
		uint8_t const* begin = (uint8_t const*)iter->begin();
		uint8_t const* const end = (uint8_t const*)iter->end();
		
		decompress(begin, end, buf, ctx);
		if (ctx.output_produced == limit) {
			break;
		}
		if (begin != end) {
			break;
		}
	}
	
	if (ctx.chunk_data_size > 0) {
		buf.append(DataChunk::resize(ctx.chunk, ctx.chunk_data_size));
	}
	
	m_bufferedInput.trimFront(ctx.input_consumed);
	
	return ctx.output_produced;
}

void
DeflateDecompressor::initStream(bool zlib_flavor)
{
	m_strm.zalloc = Z_NULL;
	m_strm.zfree = Z_NULL;
	m_strm.opaque = Z_NULL;
	m_strm.next_in = Z_NULL;
	if (zlib_flavor) {
		inflateInit(&m_strm);
	} else { // m_flavor == RAW
		inflateInit2(&m_strm, -MAX_WBITS);
	}
	m_isStreamReady = true;
}

void
DeflateDecompressor::decompress(
	uint8_t const*& begin, uint8_t const* end,
	SplittableBuffer& out, OperationContext& ctx)
{
	while (true) {
		if (!ctx.chunk.get()) {
			size_t avail = ctx.output_limit - ctx.output_produced;
			if (avail == 0) {
				return;
			}
			ctx.chunk = DataChunk::create(
				std::min<size_t>(MAX_OUTBUF_SIZE, avail)
			);
		}
		m_strm.next_in = (Bytef*)begin;
		m_strm.avail_in = end - begin;
		uint8_t* const out_begin = (uint8_t*)ctx.chunk->getDataAddr()
			+ ctx.chunk_data_size;
		m_strm.next_out = out_begin;
		m_strm.avail_out = ctx.chunk->getDataSize() - ctx.chunk_data_size;
		int res = inflate(&m_strm, Z_SYNC_FLUSH);
		size_t consumed = m_strm.next_in - begin;
		ctx.input_consumed += consumed;
		begin = m_strm.next_in;
		if (out_begin != m_strm.next_out) {
			size_t produced = m_strm.next_out - out_begin;
			ctx.chunk_data_size += produced;
			ctx.output_produced += produced;
			if (ctx.chunk_data_size == ctx.chunk->getDataSize()) {
				out.append(ctx.chunk);
				ctx.chunk_data_size = 0;
			}
		}
		
		if (res == Z_OK && m_strm.avail_out == 0) {
			continue;
		}
		// With m_flavor == RAW, we may not get Z_STREAM_END at all,
		// so we don't insist on getting it.
		if (res != Z_OK && res != Z_BUF_ERROR && res != Z_STREAM_END) {
			m_state = ST_ERROR;
		}
		break;
	} 
}
