/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2004  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef BSTRING_H_
#define BSTRING_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "IntrusivePtr.h"
#include "DataChunk.h"
#include <stddef.h>
#include <memory>
#include <iosfwd>
#include <string>
#include <cstring>

class BString
{
public:
	struct DataRange
	{
		char const* begin;
		char const* end;
		
		DataRange(char const* b, char const* e) : begin(b), end(e) {}
		
		void set(char const* b, char const* e) { begin = b; end = e; }
	};
	
	typedef IntrusivePtr<DataChunk const> ChunkPtr;
	
	// Default copy ctor and assignment operator are OK
	
	BString();
	
	BString(char const* begin, size_t len);
	
	explicit BString(std::string const& str);
	
	template<int N> explicit BString(char const (&str)[N]);
	// str is supposed to be a static, null-terminated array of chars
	
	BString(std::auto_ptr<DataChunk> chunk, size_t offset = 0);
	
	BString(IntrusivePtr<DataChunk const> const& chunk, char const* b, char const* e);
	
	BString(BString const& other, char const* b, char const* e);
	
	DataRange& data() { return m_data; }
	
	DataRange const& data() const { return m_data; }
	
	ChunkPtr const& chunk() const { return m_ptrChunk; }
	
	char const* begin() const { return m_data.begin; }
	
	char const* end() const { return m_data.end; }
	
	bool empty() const { return m_data.begin == m_data.end; }
	
	bool null() const { return !m_ptrChunk && !m_data.begin && !m_data.end; }
	
	size_t size() const { return m_data.end - m_data.begin; }
	
	void clear(); // the string becomes null
	
	void trimFront(ptrdiff_t by) { m_data.begin += by; }
	
	void trimBack(ptrdiff_t by) { m_data.end -= by; }
	
	void toStream(std::ostream& strm) const;
	
	std::string toStdString() const { return std::string(begin(), size()); }
	
	char const operator[](ptrdiff_t idx) const { return m_data.begin[idx]; }
	
	BString operator+(BString const& rhs) const;
	
	static bool less(BString const& lhs, BString const& rhs, bool or_equal);
	
	void swap(BString& other);
private:
	ChunkPtr m_ptrChunk;
	DataRange m_data;
};


inline
BString::BString()
:	m_data(0, 0)
{
}

template<int N> inline
BString::BString(char const (&str)[N])
:	m_data(&str[0], &str[N-1])
{
}

inline
BString::BString(std::auto_ptr<DataChunk> chunk, size_t offset)
:	m_ptrChunk(chunk.release()),
	m_data(
		m_ptrChunk->getDataAddr() + offset,
		m_ptrChunk->getDataAddr() + m_ptrChunk->getDataSize()
	)
{
}

inline
BString::BString(IntrusivePtr<DataChunk const> const& chunk,
                 char const* b, char const* e)
:	m_ptrChunk(chunk),
	m_data(b, e)
{
}

inline
BString::BString(BString const& other, char const* b, char const* e)
:	m_ptrChunk(other.m_ptrChunk),
	m_data(b, e)
{
}

inline void
BString::clear()
{
	m_ptrChunk.reset(0);
	m_data.set(0, 0);
}

inline
bool operator==(BString const& lhs, BString const& rhs)
{
	using namespace std;
	return (lhs.size() == rhs.size() && memcmp(lhs.begin(), rhs.begin(), lhs.size()) == 0);
}

inline
bool operator!=(BString const& lhs, BString const& rhs)
{
	return !(lhs == rhs);
}

inline
bool operator<(BString const& lhs, BString const& rhs)
{
	return BString::less(lhs, rhs, false);
}

inline
bool operator>(BString const& lhs, BString const& rhs)
{
	return BString::less(rhs, lhs, false);
}

inline
bool operator<=(BString const& lhs, BString const& rhs)
{
	return BString::less(lhs, rhs, true);
}

inline
bool operator>=(BString const& lhs, BString const& rhs)
{
	return BString::less(rhs, lhs, true);
}

inline
std::ostream& operator<<(std::ostream& strm, BString const& str)
{
	str.toStream(strm);
	return strm;
}

inline void swap(BString& o1, BString& o2)
{
	o1.swap(o2);
}

#endif
