/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "RequestLogHandler.h"
#include "Application.h"
#include "RequestLog.h"
#include "RequestTag.h"
#include "DownloadProgress.h"
#include "ArraySize.h"
#include "CompiledImages.h"
#include "types.h"
#include <gtkmm/liststore.h>
#include <gtkmm/treeview.h>
#include <gtkmm/treeiter.h>
#include <gtkmm/treemodelcolumn.h>
#include <gtkmm/cellrenderertext.h>
#include <gtkmm/cellrendererpixbuf.h>
#include <gtkmm/clipboard.h>
#include <gdkmm/color.h>
#include <gdkmm/pixbuf.h>
#include <glibmm/ustring.h>
#include <glibmm/convert.h>
#include <sstream>
#include <string>
#include <map>

using namespace std;

class RequestLogHandler::ListModel : public Gtk::ListStore
{
public:
	enum { MAX_ROWS = 400 };
	
	struct Columns : public Gtk::TreeModelColumnRecord
	{
		Gtk::TreeModelColumn<RequestId> ID;
		Gtk::TreeModelColumn<Glib::ustring> SIZE;
		Gtk::TreeModelColumn<Glib::ustring> URL;
		Gtk::TreeModelColumn<RequestType> TYPE;
		Gtk::TreeModelColumn<int> STATUS_CODE;
		Gtk::TreeModelColumn<Gdk::Color> BACKGROUND;
		Gtk::TreeModelColumn<Glib::RefPtr<Gdk::Pixbuf> > IMAGE;
		
		Columns() {
			add(ID); add(SIZE); add(URL); add(TYPE);
			add(STATUS_CODE); add(BACKGROUND); add(IMAGE);
		}
	};
	
	ListModel();
	
	virtual ~ListModel();
	
	Columns const& getCols() const { return m_cols; }
	
	void processRequest(
		RequestId const& req_id, std::string const& uri, RequestType type);
	
	void processResponseBegin(
		RequestId const& req_id, int status_code,
		DownloadProgress const& progress);
	
	void processResponseEnd(
		RequestId const& req_id, int req_flags,
		DownloadProgress const& progress, bool error);
	
	void processRequestCancel(
		RequestId const& req_id, DownloadProgress const& progress);
	
	void copyToClipboard();
	
	void copyUrlAtRow(Gtk::TreePath const& path);
private:
	typedef std::map<RequestId, Gtk::TreeIter> RowsById;
	
	static Glib::ustring formatSize(uintmax_t size);
	
	static Glib::ustring urlToUtf8(std::string const& url);
	
	Columns m_cols;
	RowsById m_rowsById;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconAd;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconCache;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconCancel;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconError;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconOK;
	Glib::RefPtr<Gdk::Pixbuf> m_ptrIconRedirect;
};


RequestLogHandler::RequestLogHandler(InterthreadCommandQueue& command_queue)
:	m_rCommandQueue(command_queue),
	m_ptrModel(new ListModel)
{
	RequestLog::setHandler(this);
}

RequestLogHandler::~RequestLogHandler()
{
}

void
RequestLogHandler::attachView(Gtk::TreeView* view)
{
	view->set_model(m_ptrModel);
	view->remove_all_columns();
	view->set_headers_visible(false);
	view->get_selection()->set_mode(Gtk::SELECTION_NONE);
	
	ListModel::Columns const& cols = m_ptrModel->getCols();
	int ncols = 0;
	
	Gtk::CellRendererPixbuf* img_renderer = Gtk::manage(new Gtk::CellRendererPixbuf);
	ncols = view->append_column(Glib::ustring(), *img_renderer);
	Gtk::TreeViewColumn* img_col = view->get_column(ncols-1);
	img_col->add_attribute(img_renderer->property_pixbuf(), cols.IMAGE);
	img_col->add_attribute(img_renderer->property_cell_background_gdk(), cols.BACKGROUND);
	
	Gtk::CellRendererText* size_renderer = Gtk::manage(new Gtk::CellRendererText);
	ncols = view->append_column(Glib::ustring(), *size_renderer);
	size_renderer->property_xalign() = 1.0;
	Gtk::TreeViewColumn* size_col = view->get_column(ncols-1);
	size_col->add_attribute(size_renderer->property_text(), cols.SIZE);
	size_col->add_attribute(size_renderer->property_cell_background_gdk(), cols.BACKGROUND);
	
	Gtk::CellRendererText* url_renderer = Gtk::manage(new Gtk::CellRendererText);
	ncols = view->append_column(Glib::ustring(), *url_renderer);
	url_renderer->property_editable() = true;
	Gtk::TreeViewColumn* url_col = view->get_column(ncols-1);
	url_col->add_attribute(url_renderer->property_text(), cols.URL);
	url_col->add_attribute(url_renderer->property_cell_background_gdk(), cols.BACKGROUND);
#if 0
	view->set_search_column(ncols-1);
#else
	view->set_enable_search(false);
#endif

	if (!m_ptrModel->children().empty()) {
		view->scroll_to_row(Gtk::TreePath(--m_ptrModel->children().end()));
	}
}

void
RequestLogHandler::clearLog()
{
	m_ptrModel->clear();
}

void
RequestLogHandler::copyToClipboard()
{
	m_ptrModel->copyToClipboard();
}

void
RequestLogHandler::copyUrlAtRow(Gtk::TreePath const& path)
{
	m_ptrModel->copyUrlAtRow(path);
}

Gdk::Color
RequestLogHandler::getColorFor(RequestType type)
{
	switch (type) {
		case RequestLog::SUBST_REQUEST: {
			return Gdk::Color("#fdf79e");
		}
		case RequestLog::ANALYZE_REQUEST: {
			return Gdk::Color("#ffe1be");
		}
		case RequestLog::INTERNAL_REQUEST: {
			return Gdk::Color("#d6d1ee");
		}
		default: {
			return Gdk::Color("#ffffff");
		}
	}
}

InterthreadCommandQueue&
RequestLogHandler::getCommandQueue()
{
	return m_rCommandQueue;
}

void
RequestLogHandler::processRequest(
	RequestId const& req_id, std::string const& uri, RequestType type)
{
	m_ptrModel->processRequest(req_id, uri, type);
}

void
RequestLogHandler::processResponseBegin(
	RequestId const& req_id, int status_code,
	DownloadProgress const& progress)
{
	m_ptrModel->processResponseBegin(req_id, status_code, progress);
}

void
RequestLogHandler::processResponseEnd(
	RequestId const& req_id, int req_flags,
	DownloadProgress const& progress, bool error)
{
	m_ptrModel->processResponseEnd(req_id, req_flags, progress, error);
}

void
RequestLogHandler::processRequestCancel(
	RequestId const& req_id, DownloadProgress const& progress)
{
	m_ptrModel->processRequestCancel(req_id, progress);
}


/*=================== RequestLogHandler::ListModel ======================*/

RequestLogHandler::ListModel::ListModel()
:	m_ptrIconAd(CompiledImages::resp_ad_png.getPixbuf()),
	m_ptrIconCache(CompiledImages::resp_cache_png.getPixbuf()),
	m_ptrIconCancel(CompiledImages::resp_cancel_png.getPixbuf()),
	m_ptrIconError(CompiledImages::resp_error_png.getPixbuf()),
	m_ptrIconOK(CompiledImages::resp_ok_png.getPixbuf()),
	m_ptrIconRedirect(CompiledImages::resp_redirect_png.getPixbuf())
{
	set_column_types(m_cols);
}

RequestLogHandler::ListModel::~ListModel()
{
}

void
RequestLogHandler::ListModel::processRequest(
	RequestId const& req_id, std::string const& uri, RequestType type)
{
	if (children().size() == MAX_ROWS) {
		Gtk::TreeIter it = *children().begin();
		Gtk::TreeRow row = *it;
		m_rowsById.erase(row[m_cols.ID]);
		erase(it);
	}
	
	Gtk::TreeIter iter = append();
	m_rowsById.insert(RowsById::value_type(req_id, iter));
	
	Gtk::TreeRow row = *iter;
	row[m_cols.ID] = req_id;
	row[m_cols.TYPE] = type;
	row[m_cols.URL] = urlToUtf8(uri);
	row[m_cols.BACKGROUND] = getColorFor(type);
}

void
RequestLogHandler::ListModel::processResponseBegin(
	RequestId const& req_id, int status_code,
	DownloadProgress const& progress)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	Gtk::TreeRow row = *it->second;
	row[m_cols.STATUS_CODE] = status_code;
}

void
RequestLogHandler::ListModel::processResponseEnd(
	RequestId const& req_id, int req_flags,
	DownloadProgress const& progress, bool error)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	Gtk::TreeRow row = *it->second;
	row[m_cols.SIZE] = formatSize(progress.received());
	
	Glib::RefPtr<Gdk::Pixbuf> img = m_ptrIconOK;
	RequestType const type = row[m_cols.TYPE];
	if (error) {
		img = m_ptrIconError;
	} else if (type == RequestLog::SUBST_REQUEST ||
		   (req_flags & RequestTag::RESPONSE_CRAFTED)) {
		img = m_ptrIconAd;
	} else {
		int const scode = row[m_cols.STATUS_CODE];
		if (scode >= 400) {
			img = m_ptrIconError;
		} else if (scode >= 300) {
			if (scode == 304) {
				img = m_ptrIconCache;
			} else {
				img = m_ptrIconRedirect;
			}
		}
	}
	row[m_cols.IMAGE] = img;
}

void
RequestLogHandler::ListModel::processRequestCancel(
	RequestId const& req_id, DownloadProgress const& progress)
{
	RowsById::iterator it = m_rowsById.find(req_id);
	if (it == m_rowsById.end()) {
		return;
	}
	
	Gtk::TreeRow row = *it->second;
	row[m_cols.SIZE] = formatSize(progress.received());
	row[m_cols.IMAGE] = m_ptrIconCancel;
}

void
RequestLogHandler::ListModel::copyToClipboard()
{
	static char const padding[5] = {' ', ' ', ' ', ' ', ' ' };
	ostringstream strm;
	
	Gtk::TreeIter it = children().begin();
	Gtk::TreeIter const end = children().end();
	for (; it != end; ++it) {
		Gtk::TreeRow row = *it;
		Glib::ustring size = row[m_cols.SIZE];
		Glib::ustring url = row[m_cols.URL];
		int const pad_width = sizeof(padding) - size.size();
		if (pad_width > 0) {
			strm.write(padding, pad_width);
		}
		strm << size.raw() << "  " << url.raw() << '\n';
	}
	
	Glib::RefPtr<Gtk::Clipboard> clipboard = Gtk::Clipboard::get();
	clipboard->set_text(Glib::ustring(strm.str()));
}

void
RequestLogHandler::ListModel::copyUrlAtRow(Gtk::TreePath const& path)
{
	Gtk::TreeIter it = get_iter(path);
	if (!it) {
		return;
	}
	
	Gtk::TreeRow row = *it;
	
	Glib::RefPtr<Gtk::Clipboard> clipboard = Gtk::Clipboard::get();
	clipboard->set_text(row[m_cols.URL]);
}

Glib::ustring
RequestLogHandler::ListModel::formatSize(uintmax_t size)
{
	static char const suffix[] = { 'B', 'K', 'M', 'G', 'T' };
	
	ostringstream strm;
	uintmax_t divisor = 1;
	
	for (int i = 0;; ++i, divisor <<= 10) {
		if (size < divisor * 1000u || i + 1 == ARRAY_SIZE(suffix)) {
			strm << ((size + divisor - 1u) / divisor) << ' ' << suffix[i];
			break;
		}
	}
	
	return strm.str();
}

Glib::ustring
RequestLogHandler::ListModel::urlToUtf8(std::string const& url)
{
	Glib::ustring str = url;
	if (str.validate()) {
		// URL is in ASCII or UTF-8
		return str;
	}
	return Application::localeToUtf8(url);
}
