/* 
 * Copyright © 2005 Quinn Storm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 * Author: Quinn Storm <quinnstorm@beryl-project.org>
 */
#define _GNU_SOURCE
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <libintl.h>
#include <beryl-settings-backend.h>

struct _Modifier {
    gchar *name;
    gint  modifier;
} const static modifiers[] = {
    { "<Shift>",      ShiftMask		 },
    { "<Control>",    ControlMask	 },
    { "<Mod1>",	      Mod1Mask		 },
    { "<Mod2>",	      Mod2Mask		 },
    { "<Mod3>",	      Mod3Mask		 },
    { "<Mod4>",	      Mod4Mask		 },
    { "<Mod5>",	      Mod5Mask		 },
    { "<Alt>",	      CompAltMask        },
    { "<Meta>",	      CompMetaMask       },
    { "<Super>",      CompSuperMask      },
    { "<Hyper>",      CompHyperMask	 },
    { "<ModeSwitch>", CompModeSwitchMask },
};

#define N_MODIFIERS (sizeof (modifiers) / sizeof (struct _Modifier))

static unsigned int get_mods_and_endptr(gchar * src, gchar ** ret)
{
    unsigned int mods=0;
    gchar * spos=src;
    while((spos=strchr(spos,'<')) && *src)
    {
        int i;
        for (i=0;i<N_MODIFIERS;i++)
            if (strncasecmp(modifiers[i].name,
                        spos,strlen(modifiers[i].name))==0)
            {
                mods|=modifiers[i].modifier;
                spos+=strlen(modifiers[i].name);
                src=spos;
                break;
            }
        if (i==N_MODIFIERS)
            break;
    }
    *ret=src;
    return mods;
}

static gchar * mods_to_string(unsigned int mods)
{
    gchar * retstr=g_strdup("");
    gchar * tmpstr=retstr;
    gint i;
    for (i=0;i<N_MODIFIERS;i++)
    {
        if (mods & modifiers[i].modifier)
        {
            retstr=g_strconcat(retstr,modifiers[i].name,NULL);
            g_free(tmpstr);
            tmpstr=retstr;
        }
    }
    return retstr;
}

static gboolean get_binding_is_enabled(gchar * src)
{
    if (strcasecmp(src,"Disabled")==0 || !strlen(src))
        return FALSE;
    if (src[0]=='#')
        return FALSE;
    return TRUE;
}

static void set_key_binding_from_string(BerylSettingValue * value, gchar * src)
{
    gboolean enabled=get_binding_is_enabled(src);
    int keysym=0;
    int keymods=0;
    gchar * spos;
    keymods=get_mods_and_endptr(src,&spos);
    if (spos && *spos)
    {
        keysym=XStringToKeysym(spos);
    };
    beryl_setting_value_set_keysym(value,&keysym);
    beryl_setting_value_set_keymods(value,&keymods);
    beryl_setting_value_set_key_enabled(value,&enabled);
}

static void set_button_binding_from_string(BerylSettingValue * value, gchar * src)
{
    gboolean enabled=get_binding_is_enabled(src);
    int button=0;
    int buttonmods=0;
    gchar * spos;
    buttonmods=get_mods_and_endptr(src,&spos);
    if (spos && *spos)
    {
        spos=strcasestr(spos,"Button");
        if (spos && *spos)
        {
            spos+=strlen("Button");
            button=atoi(spos);
        }
    }
    beryl_setting_value_set_button(value,&button);
    beryl_setting_value_set_buttonmods(value,&buttonmods);
    beryl_setting_value_set_button_enabled(value,&enabled);
}


static void set_binding_value_from_array(BerylSettingValue * value, gint * array, gsize array_count)
{
    gboolean button_enabled;
    gboolean key_enabled;
    if (array_count==8)
    {
        //it's 8 if new-style semantics are used, allowing the enabled to be set separately
        //this should reintroduce some features gconf allowed but csm didn't
        button_enabled=array[6]?TRUE:FALSE;
        key_enabled=array[7]?TRUE:FALSE;
    }
    else if (array_count==6)
    {
        button_enabled=array[1]?TRUE:FALSE;
        key_enabled=array[3]?TRUE:FALSE;
    }
    if (array_count==6 || array_count==8)
    {
        beryl_setting_value_set_key_enabled(value,&key_enabled);
        beryl_setting_value_set_button_enabled(value,&button_enabled);
        beryl_setting_value_set_buttonmods(value,&array[0]);
        beryl_setting_value_set_button(value,&array[1]);
        beryl_setting_value_set_keymods(value,&array[2]);
        beryl_setting_value_set_keysym(value,&array[3]);
        beryl_setting_value_set_edgemask(value,&array[4]);
        gboolean bell_set=array[5]?TRUE:FALSE;
        beryl_setting_value_set_bell(value,&bell_set);
    }
}

static void set_color_value_from_array(BerylSettingValue * value, gint * array)
{
    BerylSettingColorValue color_val;
    memcpy(color_val.array.array,array,sizeof(gint)*4);
    beryl_setting_value_set_color(value,&color_val);
}
void read_setting(BerylSettingsContext * c, BerylSetting * setting)
{
    GKeyFile * f = c->backend_private_ptr;
    const gchar * groupname;
    gchar * keyname;
    gboolean bool_val;
    gint int_val;
    gdouble float_val;
    gchar * str_val;
    gchar ** str_ret;
    gsize ret_count;
    gint * int_ret;
    gboolean * bool_ret;
    GError * e = NULL;
    if (setting->parent->name)
        groupname=setting->parent->name;
    else
        groupname="_"; // pseudo-plugin for core
    if (setting->is_screen)
        keyname=g_strconcat("s_",setting->name,NULL);
    else
        keyname=g_strconcat("a_",setting->name,NULL);
    if (setting->type != BERYL_SETTING_TYPE_BINDING && !g_key_file_has_key(f,groupname,keyname,NULL))
    {
        g_free(keyname);
        return;
    }
    e=NULL;
    setting->is_default=FALSE;
    switch(setting->type)
    {
        case BERYL_SETTING_TYPE_INT:
            int_val=g_key_file_get_integer(f,groupname,keyname,&e);
            if (!e)
                beryl_setting_value_set_int(&setting->value,&int_val);
            break;
        case BERYL_SETTING_TYPE_FLOAT:
            str_val=g_key_file_get_string(f,groupname,keyname,&e);
            if (!e && str_val)
            {
                float_val=g_ascii_strtod(str_val,NULL);
                beryl_setting_value_set_float(&setting->value,&float_val);
                g_free(str_val);
            }
            break;
        case BERYL_SETTING_TYPE_BOOL:
            bool_val=g_key_file_get_boolean(f,groupname,keyname,&e);
            if (!e)
                beryl_setting_value_set_bool(&setting->value,&bool_val);
            break;
        case BERYL_SETTING_TYPE_COLOR:
            int_ret=g_key_file_get_integer_list(f,groupname,keyname,&ret_count,&e);
            if (!e && int_ret)
            {
                if (ret_count==4)
                    set_color_value_from_array(&setting->value,int_ret);
                g_free(int_ret);
            }
            break;
        case BERYL_SETTING_TYPE_STRING:
            str_val=g_key_file_get_string(f,groupname,keyname,&e);
            if (!e && str_val)
            {
                //FIXME probably need _set_raw_string here
                //but since
                beryl_setting_value_set_raw_string(&setting->value,(const gchar **) &str_val);
                //setting->value.value.as_string=g_strdup(str_val);
                //setting->is_default=FALSE;
                g_free(str_val);
            }
            break;
        case BERYL_SETTING_TYPE_BINDING:
            {
                setting->is_default=TRUE;
                gchar * nkeyname;
                nkeyname=g_strconcat(keyname,"__keyboard",NULL);
                str_val=g_key_file_get_string(f,groupname,nkeyname,&e);
                if (!e && str_val)
                {
                    setting->is_default=FALSE;
                    set_key_binding_from_string(&setting->value,str_val);
                    g_free(str_val);
                }
                if (e) {
                    g_error_free(e);
                    e = NULL;
                }
                g_free(nkeyname);
                nkeyname=g_strconcat(keyname,"__mouse",NULL);
                str_val=g_key_file_get_string(f,groupname,nkeyname,&e);
                if (!e && str_val)
                {
                    setting->is_default=FALSE;
                    set_button_binding_from_string(&setting->value,str_val);
                    g_free(str_val);
                }
                if (e) {
                    g_error_free(e);
                    e = NULL;
                }
                g_free(nkeyname);
                nkeyname=g_strconcat(keyname,"__edge",NULL);
                int_val=g_key_file_get_integer(f,groupname,nkeyname,&e);
                if (!e)
                {
					beryl_setting_value_set_edgemask(&setting->value,&int_val);
                }
                g_free(nkeyname);
                if (e) {
                    g_error_free(e);
                    e = NULL;
                }
                nkeyname=g_strconcat(keyname,"__bell",NULL);
                bool_val=g_key_file_get_boolean(f,groupname,nkeyname,&e);
                if (!e)
                {
                    setting->is_default=FALSE;
                    beryl_setting_value_set_bell(&setting->value,&bool_val);
                } else {
                    g_error_free(e);
                    e = NULL;
                }
                g_free(nkeyname);
            }
            break;
        case BERYL_SETTING_TYPE_LIST:
            switch(setting->info.for_list.list_of_type)
            {
                case BERYL_SETTING_TYPE_STRING:
                    str_ret=g_key_file_get_string_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && str_ret)
                    {
                        beryl_setting_list_clear(setting);
                        int i;
                        for (i=0;i<ret_count;i++)
                        {
                            BerylSettingValue * value=beryl_setting_list_append(setting);
                            value->value.as_string=g_strdup(str_ret[i]);
                            value->parent->is_default=FALSE;
                            //beryl_setting_value_set_string(value,(const gchar **) &str_ret[i]);
                        }
                        g_strfreev(str_ret);
                    }
                    break;
                case BERYL_SETTING_TYPE_FLOAT:
                    str_ret=g_key_file_get_string_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && str_ret)
                    {
                        beryl_setting_list_clear(setting);
                        int i;
                        for (i=0;i<ret_count;i++)
                        {
                            BerylSettingValue * value=beryl_setting_list_append(setting);
                            float_val=g_ascii_strtod(str_ret[i],NULL);
                            beryl_setting_value_set_float(value,&float_val);
                        }
                        g_strfreev(str_ret);
                    }
                    break;
                case BERYL_SETTING_TYPE_INT:
                    int_ret=g_key_file_get_integer_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && int_ret)
                    {
                        beryl_setting_list_clear(setting);
                        int i;
                        for (i=0;i<ret_count;i++)
                        {
                            BerylSettingValue * value=beryl_setting_list_append(setting);
                            beryl_setting_value_set_int(value,&int_ret[i]);
                        }
                        g_free(int_ret);
                    }
                    break;
                case BERYL_SETTING_TYPE_BOOL:
                    bool_ret=g_key_file_get_boolean_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && bool_ret)
                    {
                        beryl_setting_list_clear(setting);
                        int i;
                        for (i=0;i<ret_count;i++)
                        {
                            BerylSettingValue * value=beryl_setting_list_append(setting);
                            beryl_setting_value_set_bool(value,&bool_ret[i]);
                        }
                        g_free(bool_ret);
                    }
                    break;
                case BERYL_SETTING_TYPE_COLOR:
                    int_ret=g_key_file_get_integer_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && int_ret)
                    {
                        if (ret_count%4==0)
                        {
                            beryl_setting_list_clear(setting);
                            int i;
                            for (i=0;i<ret_count/4;i++)
                            {
                                BerylSettingValue * value=beryl_setting_list_append(setting);
                                set_color_value_from_array(value,&int_ret[i*4]);
                            }
                        }
                        g_free(int_ret);
                    }
                    break;
                case BERYL_SETTING_TYPE_BINDING:
                    int_ret=g_key_file_get_integer_list(f,groupname,keyname,&ret_count,&e);
                    if (!e && int_ret)
                    {
                        if (ret_count%6==0 || ret_count%8==0)
                        {
                            beryl_setting_list_clear(setting);
                            int num_per=(ret_count%6==0)?6:8;
                            int i;
                            for (i=0;i<ret_count/num_per;i++)
                            {
                                BerylSettingValue * value=beryl_setting_list_append(setting);
                                set_binding_value_from_array(value,&int_ret[i*num_per],num_per);
                            }
                        }
                        g_free(int_ret);
                    }
                    break;
                default:
                    break;
            }
            break;
        default:
            break;
    }
    g_free(keyname);
    if (e)
        g_error_free(e);
}

static void set_array_from_binding(gint * array, BerylSettingValue * value)
{
    array[0]=value->value.as_binding.button_mod_mask;
    array[1]=value->value.as_binding.button;
    array[2]=value->value.as_binding.key_mod_mask;
    array[3]=value->value.as_binding.keysym;
    array[4]=value->value.as_binding.edge_mask;
    array[5]=value->value.as_binding.on_bell?1:0;
    array[6]=value->value.as_binding.enabled.value.button?1:0;
    array[7]=value->value.as_binding.enabled.value.key?1:0;
}

static void set_array_from_color(BerylSettingValue * value, gint * array)
{
    memcpy(array,value->value.as_color.array.array,sizeof(gint)*4);
}

static void copy_bool(BerylSettingValue * value, gboolean ** ptr)
{
    *(*ptr++)=value->value.as_bool;
}

static void copy_int(BerylSettingValue * value, gint ** ptr)
{
    *(*ptr++)=value->value.as_int;
}

static void copy_float(BerylSettingValue * value, gchar *** ptr)
{
    gchar buffer[G_ASCII_DTOSTR_BUF_SIZE];
    *(*ptr++)=g_strdup(g_ascii_dtostr(buffer,G_ASCII_DTOSTR_BUF_SIZE,value->value.as_float));
}

static void copy_binding(BerylSettingValue * value, gint ** ptr)
{
    set_array_from_binding(*ptr,value);
    *ptr+=8;
}

static void copy_string(BerylSettingValue * value, gchar *** ptr)
{
    *(*ptr)=g_strdup(value->value.as_string);
    *ptr=*ptr+1;
}

static void copy_color(BerylSettingValue * value, gint ** ptr)
{
    set_array_from_color(value,*ptr);
    *ptr+=4;
}

void write_setting(BerylSettingsContext * c, BerylSetting * setting)
{
    GKeyFile * f = c->backend_private_ptr;
    gchar * groupname=setting->parent->name?setting->parent->name:"_";
    gchar * keyname=g_strconcat(setting->is_screen?"s_":"a_",setting->name,NULL);
    if (setting->is_default)
    {
        g_key_file_remove_key(f,groupname,keyname,NULL);
        g_free(keyname);
        return;
    }
    switch(setting->type)
    {
        case BERYL_SETTING_TYPE_STRING:
            g_key_file_set_string(f,groupname,keyname,setting->value.value.as_string);
            break;
        case BERYL_SETTING_TYPE_FLOAT:
            {
                gchar buffer[G_ASCII_DTOSTR_BUF_SIZE];
                g_ascii_dtostr(buffer,G_ASCII_DTOSTR_BUF_SIZE,setting->value.value.as_float);
                g_key_file_set_string(f,groupname,keyname,buffer);
            }
            break;
        case BERYL_SETTING_TYPE_INT:
            g_key_file_set_integer(f,groupname,keyname,setting->value.value.as_int);
            break;
        case BERYL_SETTING_TYPE_BOOL:
            g_key_file_set_boolean(f,groupname,keyname,setting->value.value.as_bool);
            break;
        case BERYL_SETTING_TYPE_BINDING:
            {
                gchar * nkeyname;
                gboolean bool_val;
                if (beryl_setting_get_can_set_key(setting,&bool_val)
                        &&bool_val)
                {
                    gchar * pre;
                    if (beryl_setting_value_get_key_enabled(
                                &setting->value,&bool_val)&&bool_val)
                        pre="";
                    else
                        pre="#";
                    nkeyname=g_strconcat(keyname,"__keyboard",NULL);
                    gint ire;
                    gchar * smod=g_strdup(""), *ksym="";
                    if (beryl_setting_value_get_keymods(&setting->value,&ire))
                    {
                        g_free(smod);
                        smod=mods_to_string(ire);
                    }
                    if (beryl_setting_value_get_keysym(&setting->value,&ire))
                        ksym=XKeysymToString(ire);
                    if (!ksym) ksym="None";
                    pre=g_strconcat(pre,smod,ksym,NULL);
                    g_free(smod);
                    g_key_file_set_string(f,groupname,nkeyname,pre);
                    g_free(pre);
                    g_free(nkeyname);
                }
                if (beryl_setting_get_can_set_button(setting,&bool_val)
                        &&bool_val)
                {
                    gchar * pre;
                    if (beryl_setting_value_get_button_enabled(
                                &setting->value,&bool_val)&&bool_val)
                        pre="";
                    else
                        pre="#";
                    nkeyname=g_strconcat(keyname,"__mouse",NULL);
                    gint ire;
                    gchar * smod=g_strdup(""), *bnum=g_strdup("Any");
                    if (beryl_setting_value_get_buttonmods(&setting->value,&ire))
                    {
                        g_free(smod);
                        smod=mods_to_string(ire);
                    }
                    if (beryl_setting_value_get_button(&setting->value,&ire))
                    {
                        g_free(bnum);
                        bnum=g_strdup_printf("Button%d",ire);
                    }
                    pre=g_strconcat(pre,smod,bnum,NULL);
                    g_free(smod);
                    g_free(bnum);
                    g_key_file_set_string(f,groupname,nkeyname,pre);
                    g_free(pre);
                    g_free(nkeyname);
                }
                if (beryl_setting_get_can_set_edgemask(setting,&bool_val)
                        &&bool_val)
                {
                    nkeyname=g_strconcat(keyname,"__edge",NULL);
                    int ema;
                    if (beryl_setting_value_get_edgemask(&setting->value,&ema))
                    {
						g_key_file_set_integer(f,groupname,nkeyname,ema);
                    }
                    g_free(nkeyname);
                }
                if (beryl_setting_get_can_set_bell(setting,&bool_val)
                        &&bool_val)
                {
                    nkeyname=g_strconcat(keyname,"__bell",NULL);
                    if (beryl_setting_value_get_bell(&setting->value,&bool_val))
                        g_key_file_set_boolean(f,groupname,nkeyname,bool_val);
                    g_free(nkeyname);
                }
            }
            break;
        case BERYL_SETTING_TYPE_COLOR:
            {
                gint colorval[4];
                set_array_from_color(&setting->value,colorval);
                g_key_file_set_integer_list(f,groupname,keyname,colorval,4);
            }
            break;
        case BERYL_SETTING_TYPE_LIST:
            {
                gsize len=g_slist_length(setting->value.value.as_list);
                switch(setting->info.for_list.list_of_type)
                {
                    case BERYL_SETTING_TYPE_BOOL:
                        {
                            gboolean * list = malloc(len*sizeof(gboolean));
                            gboolean * ptr = list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_bool,&ptr);
                            g_key_file_set_boolean_list(f,groupname,keyname,list,len);
                            free(list);
                        }
                        break;
                    case BERYL_SETTING_TYPE_INT:
                        {
                            gint * list = malloc(len*sizeof(gint));
                            gint * ptr = list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_int,&ptr);
                            g_key_file_set_integer_list(f,groupname,keyname,list,len);
                            free(list);
                        }
                        break;
                    case BERYL_SETTING_TYPE_STRING:
                        {
                            gchar ** list = malloc((len+1)*sizeof(gchar *));
                            list[len]=NULL;
                            gchar ** ptr=list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_string,&ptr);
                            g_key_file_set_string_list(f,groupname,keyname,(const gchar * const *)list,
                                    len);
                            g_strfreev(list); // TODO - I think this is safe with malloc()'d arrays
                        }
                        break;
                    case BERYL_SETTING_TYPE_FLOAT:
                        {
                            gchar ** list = malloc((len+1)*sizeof(gchar *));
                            list[len]=NULL;
                            gchar ** ptr=list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_float,&ptr);
                            g_key_file_set_string_list(f,groupname,keyname,(const gchar * const *)list,
                                    len);
                            g_strfreev(list); // TODO - I think this is safe with malloc()'d arrays
                        }
                        break;
                    case BERYL_SETTING_TYPE_COLOR:
                        {
                            gint * list = malloc(len*sizeof(gint)*4);
                            gint * ptr = list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_color,&ptr);
                            g_key_file_set_integer_list(f,groupname,keyname,list,len);
                            free(list);
                        }
                        break;
                    case BERYL_SETTING_TYPE_BINDING:
                        {
                            gint * list = malloc(len*sizeof(gint)*8);
                            gint * ptr = list;
                            g_slist_foreach(setting->value.value.as_list,(GFunc)copy_binding,&ptr);
                            g_key_file_set_integer_list(f,groupname,keyname,list,len);
                            free(list);
                        }
                        break;
                    default:
                        break;
                }
            }
            break;
        default:
            break;
    }
    g_free(keyname);
}
gboolean read_init(BerylSettingsContext * c)
{
    GKeyFile * f;
	gchar * s = beryl_settings_context_get_profile(c);
	if (!s)
		s=g_strdup("settings");
	else
		s=g_strdup_printf("%s.settings",s);
    gchar * path = g_strconcat(g_get_home_dir(),"/.beryl/",s,NULL);
	g_free(s);
    f=g_key_file_new();
    if (!g_key_file_load_from_file(f,path,0,NULL))
    {
        g_key_file_free(f);
        g_free(path);
        return FALSE;
    }
    g_free(path);
    c->backend_private_ptr=f;
    return TRUE;
}
void read_done(BerylSettingsContext * c)
{
    g_key_file_free(c->backend_private_ptr);
}
gboolean write_init(BerylSettingsContext * c)
{
    GKeyFile * f;
    gchar * path = g_strconcat(g_get_home_dir(),"/.beryl/",NULL);
    g_mkdir_with_parents(path,00755); // ensure path exists
    g_free(path);
	gchar * s = beryl_settings_context_get_profile(c);
	if (!s)
		s=g_strdup("settings");
	else
		s=g_strdup_printf("%s.settings",s);
    path = g_strconcat(g_get_home_dir(),"/.beryl/",s,NULL);
	g_free(s);
    f=g_key_file_new();
    g_key_file_load_from_file(f,path,0,NULL);
    g_free(path);
    c->backend_private_ptr=f;
    return TRUE;
}

void write_done(BerylSettingsContext * c)
{
    GKeyFile * f = c->backend_private_ptr;
	gchar * s = beryl_settings_context_get_profile(c);
	if (!s)
		s=g_strdup("settings");
	else
		s=g_strdup_printf("%s.settings",s);
    gchar * path = g_strconcat(g_get_home_dir(),"/.beryl/",s,NULL);
	g_free(s);
    gchar * data;
    if (g_file_test(path,G_FILE_TEST_IS_SYMLINK))
    {
        gchar * tmp = path;
        path=g_file_read_link(path,NULL);
        g_free(tmp);
    }
    data = g_key_file_to_data(f,NULL,NULL);
    g_file_set_contents(path,data,-1,NULL);
    g_free(path);
    g_free(data);
    g_key_file_free(f);
}

GSList * get_existing_profiles(void)
{
	GDir * d;
	gchar * s;
	GSList * ret = NULL;
	s = g_strconcat(g_get_home_dir(),"/.beryl/",NULL);
	d = g_dir_open(s,0,NULL);
	g_free(s);
	if (!d)
		return NULL;
	while ((s=(gchar *)g_dir_read_name(d)))
	{
		if (g_str_has_suffix(s,".settings"))
		{
			gchar * p = g_strndup(s,strlen(s)-strlen(".settings"));
			ret=g_slist_append(ret,p);
		}
	}
	g_dir_close(d);
	return ret;
}

gboolean get_setting_is_integrated(BerylSetting * setting)
{
	return FALSE;
}

gboolean get_setting_is_read_only(BerylSetting * setting)
{
	return FALSE;
}

gboolean backend_init(BerylSettingsContext * context)
{
	return TRUE;
}

gboolean backend_fini(BerylSettingsContext * context)
{
	return TRUE;
}

const gchar * get_short_desc()
{
	return _("Flat-File Backend");
}

gboolean get_supports_integration()
{
	return FALSE;
}

gboolean delete_profile(gchar * profile)
{
	gchar * s;
	if (!profile || !strlen(profile))
		s = g_strconcat(g_get_home_dir(),"/.beryl/settings",NULL);
	else
		s = g_strconcat(g_get_home_dir(),"/.beryl/",profile,".settings",NULL);
	int ret = g_unlink(s);
	g_free(s);
	return !ret;
}
