// TLVBuffer.cpp: implementation of the CTLVBuffer class.
//
//////////////////////////////////////////////////////////////////////

#include "TLVBuffer.h"

#ifdef _WIN32
#include <windows.h>
#endif

extern unsigned short BytesToShort(unsigned char *, int, bool); 

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

char CTLVBuffer::hexChars[] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'};

CTLVBuffer::CTLVBuffer()
{

}

CTLVBuffer::~CTLVBuffer()
{
    ITMap it;
    for (it = m_oMapTLV.begin(); it != m_oMapTLV.end(); ++it)
    {
        CTLV *pTemp = (*it).second;
        if(pTemp != NULL)
        {
            delete pTemp;
        }
    }
    m_oMapTLV.clear();
}

int CTLVBuffer::ParseTLV(unsigned char *pucData, unsigned long ulLen)
{
    int iRet = 0;
    
    if(pucData != NULL && ulLen > 0)
    {
        m_oMapTLV.clear();

        unsigned long ulIndex = 0;
        while (ulIndex < ulLen)
        {
            unsigned char ucTag = pucData[ulIndex++];
            unsigned long ulFieldLen = pucData[ulIndex];
            
            if(ucTag == 0x00 && ulIndex > 2)
            {
                break;
            }
            while(0xFF == pucData[ulIndex])
            {
                ulFieldLen += pucData[++ulIndex];
            }
            ++ulIndex;
            m_oMapTLV[ucTag] = new CTLV(ucTag, pucData + ulIndex, ulFieldLen);
            ulIndex += ulFieldLen;
        }
        iRet = 1;
    }
    return iRet;
}

CTLV *CTLVBuffer::GetTagData(unsigned char ucTag)
{
    ITMap it = m_oMapTLV.find(ucTag);
    if(it != m_oMapTLV.end())
    {
        return(*it).second;
    }
    return NULL;
}

void CTLVBuffer::FillDateData(unsigned char ucTag, char *pData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        // Format Data to YYYYMMDD
        unsigned char *pTemp = NULL;
        unsigned char *pDate = pTagData->GetData(); 
        int iLen = pTagData->GetLength();
        if(BEID_MAX_DATE_BEGIN_LEN == iLen && pDate[1] == ' ')
        {
            pTemp = new unsigned char[iLen + 2];
            memset(pTemp, 0, iLen + 2);
            pTemp[0] = '0';
            memcpy(pTemp + 1, pDate, iLen);
            pDate = pTemp;
            iLen++;
        }
        if (BEID_MAX_DATE_BEGIN_LEN == iLen)
        {
            // DD.MM.YYYY
            pData[0] = pDate[6];    
            pData[1] = pDate[7];    
            pData[2] = pDate[8];    
            pData[3] = pDate[9];    
            pData[4] = pDate[3];    
            pData[5] = pDate[4];    
            pData[6] = pDate[0];    
            pData[7] = pDate[1];    
        }
        else if (iLen > BEID_MAX_DATE_BEGIN_LEN)
        {
            // DD mmmm YYYY or DD.mmm.YYYY
            pData[0] = pDate[iLen-4];    
            pData[1] = pDate[iLen-3];    
            pData[2] = pDate[iLen-2];    
            pData[3] = pDate[iLen-1];       
            pData[6] = pDate[0];    
            pData[7] = pDate[1];
            FillMonth(pDate + 3, pData + 4);
        }
        else
        {
            memcpy(pData, pTagData->GetData(), pTagData->GetLength());
        }
        if(pTemp != NULL)
        {
            delete[] pTemp;
        }
    }
}

void CTLVBuffer::FillASCIIData(unsigned char ucTag, char *pData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        memcpy(pData, pTagData->GetData(), pTagData->GetLength());
    }
}

void CTLVBuffer::FillUTF8Data(unsigned char ucTag, char *pData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
/*#ifdef _UNICODE
        int iSize = 0;
        iSize = MultiByteToWideChar(CP_UTF8, 0, (const char *)pTagData->GetData(), pTagData->GetLength(), pData, iSize);
        memset(pData, 0, sizeof(TCHAR)*iSize);
        MultiByteToWideChar(CP_UTF8, 0, (const char *)pTagData->GetData(), pTagData->GetLength(), pData, iSize);
#else*/
        memcpy(pData, pTagData->GetData(), pTagData->GetLength());
//#endif
    }
}

void CTLVBuffer::FillBinaryStringData(unsigned char ucTag, char *pData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        char *pszTemp = Hexify(pTagData->GetData(), pTagData->GetLength());
        memcpy(pData, pszTemp, strlen(pszTemp));
        delete pszTemp;
    }
}

void CTLVBuffer::FillShortData(unsigned char ucTag, short *psData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        *psData = BytesToShort(pTagData->GetData(), 0, true);
    }
}

void CTLVBuffer::FillIntData(unsigned char ucTag, int *piData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        char *pszTemp = new char[pTagData->GetLength() + 1];
        memset(pszTemp, 0, pTagData->GetLength() + 1);
        memcpy(pszTemp, pTagData->GetData(), pTagData->GetLength());
        *piData = atoi(pszTemp);
        delete []pszTemp;
    }
}

void CTLVBuffer::FillLongData(unsigned char ucTag, long *piData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        char *pszTemp = new char[pTagData->GetLength() + 1];
        memset(pszTemp, 0, pTagData->GetLength() + 1);
        memcpy(pszTemp, pTagData->GetData(), pTagData->GetLength());
        *piData = atol(pszTemp);
        delete []pszTemp;
    }
}

void CTLVBuffer::FillBinaryData(unsigned char ucTag, unsigned char *pData)
{
    CTLV *pTagData = NULL;
    if(NULL != (pTagData = GetTagData(ucTag)))
    {
        memcpy(pData, pTagData->GetData(), pTagData->GetLength());
    }
}

char *CTLVBuffer::Hexify(unsigned char * pData, unsigned long ulLen) 
{
    char *pszHex = new char[ulLen*2 + 1];
    memset(pszHex, 0, ulLen*2 + 1);
    if(pData != NULL)
    {
        int j = 0;
        for(unsigned long i = 0; i < ulLen; i++) 
        {
            pszHex[j++] = hexChars[pData[i]>>4 & 0x0F];
            pszHex[j++] = hexChars[pData[i] & 0x0F];
         }
    }
    return pszHex;
}

int CTLVBuffer::BytesToInt(unsigned char * pData, unsigned long ulLen) 
{ 
    int iOut = 0; 
    for (unsigned int i = 0; i < ulLen; i++) 
    { 
        iOut = iOut << 4; 
        iOut = iOut | ((pData[i] >> 4) & 0x0F); 
        iOut = iOut << 4; 
        iOut = iOut | (pData[i] & 0x0F); 
    }
    return iOut; 
}  

unsigned long CTLVBuffer::GetLengthForSignature()
{
    unsigned long lRet = 0;
    ITMap it;
    for (it = m_oMapTLV.begin(); it != m_oMapTLV.end(); ++it)
    {
        CTLV *pTemp = (*it).second;
        if(pTemp != NULL)
        {
            lRet += pTemp->GetLength();
            lRet += 2;
        }
    }
    return lRet;
}

void CTLVBuffer::FillMonth(unsigned char *pDate, char *pMonth)
{
    if(0 == memcmp(pDate, "JAN", 3))
    {
        pMonth[0] = '0';
        pMonth[1] = '1';
        return;
    }
    if(0 == memcmp(pDate, "FEB", 3) || 0 == memcmp(pDate, "FEV", 3))
    {
        pMonth[0] = '0';
        pMonth[1] = '2';
        return;
    }
    if(0 == memcmp(pDate, "MARS", 4) || 0 == memcmp(pDate, "MAAR", 4) || 0 == memcmp(pDate, "\x4D\xC3\x84\x52", 4))
    {
        pMonth[0] = '0';
        pMonth[1] = '3';
        return;
    }
    if(0 == memcmp(pDate, "APR", 3) || 0 == memcmp(pDate, "AVR", 3))
    {
        pMonth[0] = '0';
        pMonth[1] = '4';
        return;
    }
    if(0 == memcmp(pDate, "MAI", 3) || 0 == memcmp(pDate, "MEI", 3))
    {
        pMonth[0] = '0';
        pMonth[1] = '5';
        return;
    }
    if(0 == memcmp(pDate, "JUN", 3) || 0 == memcmp(pDate, "JUIN", 4))
    {
        pMonth[0] = '0';
        pMonth[1] = '6';
        return;
    }
    if(0 == memcmp(pDate, "JUL", 3) || 0 == memcmp(pDate, "JUIL", 4))
    {
        pMonth[0] = '0';
        pMonth[1] = '7';
        return;
    }
    if(0 == memcmp(pDate, "AUG", 3) || 0 == memcmp(pDate, "AOUT", 4))
    {
        pMonth[0] = '0';
        pMonth[1] = '8';
        return;
    }
    if(0 == memcmp(pDate, "SEP", 3) || 0 == memcmp(pDate, "SEPT", 4))
    {
        pMonth[0] = '0';
        pMonth[1] = '9';
        return;
    }
    if(0 == memcmp(pDate, "OKT", 3) || 0 == memcmp(pDate, "OCT", 3))
    {
        pMonth[0] = '1';
        pMonth[1] = '0';
        return;
    }
    if(0 == memcmp(pDate, "NOV", 3))
    {
        pMonth[0] = '1';
        pMonth[1] = '1';
        return;
    }
    if(0 == memcmp(pDate, "DEC", 3) || 0 == memcmp(pDate, "DEZ", 3))
    {
        pMonth[0] = '1';
        pMonth[1] = '2';
        return;
    }
}
