<?php 

/***************************************************************************
****************************************************************************

FSPageStore.php

A class for storing and retrieving pages using
a files system as storage. Defines abstract class
PageStore. Singleton.

Each page is represented as a directory with this structure:

page_name/        -- the enclosing directory
  index.php       -- dummy php file to make apache configuration easier
                     on some setups.
  basic.prop      -- basic page properties which cannot be inherited. 
                     ie title, page type, timestamp, nav info.
  inherit.prop    -- properties which may be inherited
                     ie permissions, visibility.
  content.en.txt  -- static text data in english
  content.es.txt  -- static text data in spanish
  content.en.html -- static html data in english, and so on...
  order           -- if this pages contains other pages, this file
                     lists their order.
  RCS/            -- directory holding version info
  data/           -- directory for dynamic module data
  sub_page_1/     -- a storage directory for a sub page.
  sub_page_2/     -- ditto.
  
To the $page object, we add the variable $storage. $storage points to the
absolute path of the enclosing directory which stores data for this page.

***************************************************************************
**************************************************************************/

require_once('common.php');
require_once('Properties.php');
require_once('PageStore.php');

class FSPageStore extends PageStore {

var $ignore = array('RCS','data','img','files','es','en','bamboo','order');
var $ignore_match ='/^content\.|\.prop$/';

// initializes the $page to be able
// to talk to this PageStore.
function init(&$page) {
	$page->storage = $this->locateStorage($page);
	$page->fnf = !is_dir($page->storage);
}

/*
 * returns an array of basic page info for page object
 */
 
function loadbasic($siteroot, $path) {
	$storage = $this->locateStorageFromPath($siteroot,$path);
	$file = "$storage/basic.prop";
	$ret = array();
	if (is_file($file)) 
		$ret = $this->loadProperties($file);
	$ret['name'] = deurlize($path); // name is special, can't be set, based on path.
	return $ret;
}

function loadinherit($siteroot, $path) {
	$storage = $this->locateStorageFromPath($siteroot,$path);
	$file = "$storage/inherit.prop";
	if (is_file($file))
		return $this->loadProperties($file);
	else
		return null;
}

function savebasic($siteroot, $path, &$data) {
	$storage = $this->locateStorageFromPath($siteroot,$path);
	$file = "$storage/basic.prop";
	$err = $this->saveProperties($file, $data);
	return $err;
}

function saveinherit($siteroot, $path, &$data) {
	$storage = $this->locateStorageFromPath($siteroot,$path);
	$file = "$storage/basic.prop";
	$err = $this->saveProperties($file, $data);
	return $err;
}


/*
 * returns the contents of a page
 * (only for static pages, obviously)
 */
function fetch(&$_the_page_) {
	# the wacky $_the_page is so that it will be less likely to
	# collide with anything we might include if the page type is php.
	# in the future, we need a way to reset the symbol table if the php
	# page type is going to be useful.

	if ($_the_page_->fnf) return;

	$contentfile = $_the_page_->storage . '/content.' . lang() . '.' . $_the_page_->type;
	if (!is_file($contentfile)) {
		$contentfile = $_the_page_->storage . '/content.' . default_lang() . '.' . $_the_page_->type;
	}
	if (!is_file($contentfile)) {
		return;
	}
/*	if ($_the_page_->type == 'php') {
		ob_start();
			// make page be $_the_page_ to lessen the chance that the
			// included file will write over the page.
			include($contentfile);
  			$data = &ob_get_contents();
		ob_end_clean();
	}
*/	else {
		ob_start();
			readfile($contentfile);
  			$data = &ob_get_contents();
		ob_end_clean();
	}	
	$_the_page_->content = &$data;
}

/* 
 * saves changes to 'content' or 'name'
 * (property saving is handled by Properties class.
 */
function commit(&$page) {
	if (!$page->dirty()) return;
	
	if ($page->isdirty('content')) {
		$contentfile = $page->storage . '/content.' . lang() . '.' . $page->type;
		
		#if (strlen(trim($page->content))==0)
		#	trigger_error("There is probably an error: content is null");
		
		$handle = fopen($contentfile,'w');
		if ($handle != FALSE) {
			fwrite($handle,$page->content);
			fclose($handle);
			$page->undirty('content');
		}
	}
	if ($page->isdirty('name')) {
		$oldpath = $page->path;
		$newpath = dirname($page->path) . '/' . $page->name;
		$ok = $this->movePage($page, $newpath);
		$page->undirty('name');
		return $ok;
	}
	return true;
}

function getContentSize(&$page) {
	$contentfile = $page->storage . '/content.' . lang() . '.' . $page->type;
	if (is_file($contentfile))
		return filesize($contentfile); 
	else
		return 0;
}

function rebuildOrder(&$page) {
	$listing = $this->getDirListing($page->storage);
	$newlist = array();
	foreach($listing as $name) {
		$child = $page->child($name);
		if (!$child->fnf)
			$newlist[] = $name;
	}
	$this->writeOrderFile($page->storage, $newlist);
}

function getOrder(&$page) {
	$name = basename($page->path);
	$parent = $page->parent();
	$listing = $this->getDirListing($parent->storage);
	return array_search($name, $listing);
}

function setOrder(&$page, $order) {
	$parent = $page->parent();
	$name = basename($page->path);
	$listing = $this->getDirListing($parent->storage);
	$files = array();

	$count = count($listing);
	if ($order >= $count || $order < 0)
		return;
			
	unset($listing[array_search($name,$listing)]);
	reset($listing);
	for($i=0; $i<$count; $i++) {
		if ($i == $order) {
			$files[$i] = $name;
		}
		else {
			$files[$i] = current($listing);
			next($listing);
		}
	}
	$this->writeOrderFile($parent->storage, $files);
}

/*
 * Creates a new page as a child of $page.
 */
	#$skeleton = dirname(__FILE__) . "/skeleton/page";
	#$err = `cp -ar $skeleton $newpage->storage 2>&1`;

function &newChild(&$page, $name) {
	$name = urlize($name);
	$newpath = "$page->path/$name";
	$newpage = new Page($this, $newpath);
	$err = `mkdir $newpage->storage 2>&1`;
	if (!empty($err)) {
		$page->error("Could not create $newpage->storage: " . $err);
	}
	$newpage->init($this,$newpath);
	return $newpage;
}

function destroyPage(&$page) {
	$err = `rm -r $page->storage 2>&1`;
	if (!empty($err)) {
		$page->error("Could not rm $page->storage: " . $err);
	}
}

function movePage(&$page, $newpath) {
	$newpage = new Page($this,$newpath);
	if (!$newpage->dir && is_dir($newpage->storage)) {
		$page->error($newpage->path . _(" already exists. "));
		return false;
	}
	else {
		$err = `mv $page->storage/ $newpage->storage/ 2>&1`;
		if (!empty($err)) {
			$page->error("Could not move page $page->path to $newpage->path: " . $err);
			return false;
		}
		$this->rebuildOrder($page->parent());
		$page->init($this, $newpath);
		return true;
	}
}

function duplicatePage(&$page, $newpath) {
	$newpage = new Page($this,$newpath);
	$err = `cp -ar $page->storage $newpage->storage 2>&1`;
	if (!empty($err)) {
		$page->error("Could not duplicate page $page->path to $newpage->path: " . $err);
	}
}

function getNextPage($page) {
	$filepath = $this->getFilePath($page->path);
	$filename = basename($filepath);
	
	if (is_dir($filepath))
		return $this->getFirstChild($page);
	else {
		$dir = dirname($filepath);
		$listing = $this->getDirListing($dir);
		for($i=0; $i<count($listing); $i++) {
			if ($listing[$i] == $filename) {
				if ( isset($listing[$i+1]) ) {
					return $this->getPage( $this->dirname($page->path) . '/'. $listing[$i+1]);
				}
				else {
					return null;
				}	
			}
		}
	}
}

function getPrevPage($page) {
	$filepath = $this->getFilePath($page->path);
	$filename = basename($filepath);
	$dir = dirname($filepath);
	$listing = $this->getDirListing($dir);
	for($i=0; $i<count($listing); $i++) {
		if ($listing[$i] == $filename) {
			if ( isset($listing[$i-1]) ) {
				return $this->getPage( $this->dirname($page->path) . '/'. $listing[$i-1]);
			}
			else {
				return null;
			}	
		}
	}
}

function getFirstChild($page) {
	$filepath = $this->getFilePath($page->path);
	if (!is_dir($filepath)) 
		return null;
		
	$files = $this->getDirListing($filepath);
	if (isset($files[0]))
		return $this->getPage($page->path . '/' . $files[0]);
	else
		return null;
}

function getChildren(&$page) {		
	$ret = array();	
	$files = $this->getDirListing($page->storage);
	foreach($files as $file) {
		$ret[] = $this->getPage($page->path . '/' . $file);
	}
	return $ret;
}	


/**
 * returns the navigation title from just the path.
 * does not require a page object.
 **/

function getNavTitle($path) {
	// may be optimized in the future have a better way of doing this
	
	$page = $this->getPage($path);
	if ($page->fnf)
		return deurlize($path);
	else
		return $page->get('navtitle');
}

################################################

function getAttachments(&$page) {
	$path = $page->storage;
	$handle = opendir($path);
	$files = array();
	while ($filename = readdir($handle)) { 
		if ($filename{0} == '.' || is_dir("$path/$filename")
		  || in_array($filename, $this->ignore)
		  || preg_match($this->ignore_match,$filename)
		) {
			continue;
		}
		$files[$filename] = array(
			'file' => "$path/$filename",
			'size' => filesize("$path/$filename"),
			'type' => mime_type("$path/$filename"),
			'mtime' => filemtime("$path/$filename"),
			'name' => $filename
		);
	}
	return $files;
}

function addAttachment(&$page,$filedata) {
	$path = $page->storage;
	$err = '';
	if ($filedata['error']) {
		switch ($filedata['error']) {
			case UPLOAD_ERR_INI_SIZE: $err = "The uploaded file exceeds the upload_max_filesize directive in php.ini."; break;
			case UPLOAD_ERR_FORM_SIZE: $err = "The uploaded file exceeds the MAX_FILE_SIZE directive that was specified in the HTML form."; break;
			case UPLOAD_ERR_PARTIAL: $err = "The uploaded file was only partially uploaded."; break;
			case UPLOAD_ERR_NO_FILE: $err = "No file was uploaded."; break;
		}
	}
	elseif (file_exists("$path/$filedata[name]")) {
		$err = _("That file already exists");
	}
	
	if (!$err && move_uploaded_file($filedata['tmp_name'],"$path/$filedata[name]" )) {
		return true;
	} else {
		$err = $err ? $err : 'unknown';
		error_log("file upload failed! $err");
		$page->error("file upload failed! $err");
		return true;
	}
}

function removeAttachment(&$page,$filename) {
	$file = escapeshellcmd("$page->storage/$filename");
	if (is_dir($file)) {
		$page->error("Something is wrong: that is a directory");
		return;
	}
	$err = `rm -r $file 2>&1`;
	if (!empty($err)) {
		$page->error("Could not remove $file: " . $err);
	}
}


#################################################
## Private Functions
#################################################

// returns an array of data in file.
// file                  -->     array
// sample_prop = value   -->     $array['sample_prop'] == 'value'
function &loadProperties($filename) {
	$ret = parse_ini_file($filename);
	if ($ret == null)
		return array();
	else	
		return $ret;
}

function saveProperties($filename, &$array) {
	$out = fopen($filename, 'w');
	if ($out != false) {	
		foreach($array as $key => $value) {
			if ($this->prop->proptype($key) == 'boolean') {
				if ($value == false)
					$value = 'false';
				fwrite($out, "$key = $value\n");
			}
			else {
				$value = preg_replace('/"/', '&quot;', $value);
				fwrite($out, "$key = \"$value\"\n");
			}
		}
		fclose($out);
	}
}

function getDirListing($dir) {
	#debug($this);
	#printvar($dir,'$dir');
	$handle = opendir($dir);
	$subdirs = array();
	while ($filename = readdir($handle)) { 
		if ($filename{0} == '.' || !is_dir("$dir/$filename") || in_array($filename, $this->ignore) )
			continue;
		$subdirs[] = $filename;
	}
	closedir($handle);
	if (is_file("$dir/order")) {
		$order = file("$dir/order");
		$order = array_map("trim",$order);
		$subdirs = array_flip($subdirs);
		$order = array_flip($order);
		$subdirs = array_merge($subdirs, $order);
		asort($subdirs, SORT_NUMERIC);
		$subdirs = array_keys($subdirs);
	}
	else
		sort($subdirs);
	return $subdirs;
}

function writeOrderFile($dir, $filelist) {
	$out = fopen("$dir/order", 'w');
	if ($out != false) {
		foreach($filelist as $file) {
			fwrite($out, "$file\n");
		}
		fclose($out);
	}
}

function ignore($filename)
{
	$suffix = substr( strrchr($filename,"."), 1);
	$suffixexists = $suffix !== FALSE;
	$suffix = strtolower($suffix);
	
	$ret = (
	  $suffixexists
	  && !in_array($suffix, $this->suffix)
	  #&& (strcasecmp($suffix,".html") != 0)
	  #&& (strcasecmp($suffix,".php") != 0)
	);
	$ret = $ret || in_array($filename, $this->ignore);
	$ret = $ret || $filename{0} == '.';

	return $ret;
}

//////////////////////////////


/*
 * same as dirname, but if only / is left, return ''.
 */
function dirname($path) {
	$ret = dirname($path);
	if ($ret=='/') return '';
	else return $ret;
}


/*
 returns the absolute path to the file system directory
 where $page is stored. 
*/
function locateStorage(&$page) {
	$ret = $_SERVER['DOCUMENT_ROOT']
		. '/' . $page->prop->getGlobal('siteroot')
		. '/' . $page->path;
	$ret = preg_replace(
		array("'/{2,}'","'/$'"),
		array('/',''),
		$ret
	);
	return $ret;
}

function locateStorageFromPath($siteroot, $path) {
	$ret = $_SERVER['DOCUMENT_ROOT'] . '/' . $siteroot . '/' . $path;
	$ret = preg_replace(
		array("'/{2,}'","'/$'"),
		array('/',''),
		$ret
	);
	return $ret;
}

// this is easy and nice, but a little hackish
function storage($path) {
	global $prop;
	$ret = $_SERVER['DOCUMENT_ROOT'].'/'.$prop->getGlobal('siteroot').'/'.$path;
	$ret = preg_replace(array("'/{2,}'","'/$'"),array('/',''),$ret);
	return $ret;
}

function mkdirectory($path) {
	$filename = $this->storage($path);
	if (!is_dir($filename)) {
		if (!is_writeable(dirname($filename))) {
			$user = posix_getpwuid(posix_getuid());
			die(dirname($filename) . " must be writeable by " . $user['name']);
		}
		else {
			mkdir($filename);
		}
	}
}

// returns true if there exists a page with path equal to $path
function exists($path) {
	$filename = $this->storage($path);
	return is_dir($filename);
}

} // end class

// the system call mime_content_type seems to be broken, so we use our own:

function mime_type($file) {
	return exec("file -bi " . escapeshellcmd($file));
}

?>