<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    xml-defs.php                                            */
/* Author:      Paul Waite                                              */
/* Description: Basic definitions pertaining to XML processing.         */
/*                                                                      */
/* ******************************************************************** */
/** @package xml */

// ----------------------------------------------------------------------
/** The XML tag class. This class encapsulates the functionality of
* an XML tag. It can accept child classes, usually of the same class,
* which will be rendered as child XML elements. A whole XML hierarchy
* can thus be represented in a single xmltag object.
* @package xml
*/
class xmltag {
  /** Name of the XML tag */
  var $tagname;
  /** Value of this XML tag */
  var $value;
  /** Array of tag attributes */
  var $attributes = array();
  /** Array of child tags of this tag */
  var $childtags = array();
  /** If "cdata", we use CDATA to encode the value string */
  var $encode_with = "cdata";
  // .....................................................................
  /** Constructor
  * @param $name Name of this XML tag
  * @param $value Optional value for this XML tag
  */
  function xmltag($name, $value="", $encode_with="cdata") {
    $this->tagname = $name;
    $this->encode_with = $encode_with;
    if ($value != "") {
      $this->value = $value;
    }
  } // xmltag
  // .....................................................................
  /** Set a tag attribute key=value pair.
  * @param $name Name of the tag attribute as in name="value"
  * @param $value Value of the tag attribute
  */
  function setattribute($name, $value) {
    $this->attributes[$name] = $value;
  } // setattribute
  // .....................................................................
  /** Add a child tag.
  * @param $tag XMl tag object to add to this tag as a child
  */
  function childtag($tag) {
    if (is_object($tag)) {
      $this->childtags[] = clone($tag);
    }
  } // childtag
  // .....................................................................
  /** Render the tag as XML.
  * @return string The XML for this tag
  */
  function render($indent=0) {
    $s = "";
    $margin = str_repeat(" ", $indent);
    $s .= "$margin<$this->tagname";
    if (isset($this->attributes) && count($this->attributes) > 0) {
      foreach ($this->attributes as $name => $value) {
        $s .= " $name=\"$value\"";
      }
    }
    if (isset($this->value)) {
      $tagval = ($this->encode_with == "cdata") ? xmldata($this->value) : $this->value;
      $s .= ">$tagval</" . $this->tagname . ">\n";
    }
    else {
      if (count($this->childtags) > 0) {
        $s .= ">\n";
        foreach ($this->childtags as $child) {
          $s .= $child->render($indent + 2);
        }
        $s .= "$margin</" . $this->tagname . ">\n";
      }
      else {
        // A simple singleton tag..
        $s .= "/>\n";
      }
    }
    return $s;
  } // render
} // xmltag class

// ----------------------------------------------------------------------
/**
* Class comprising the functionality of an XML parser. You should
* build specific parsers by inheriting this class and building your
* custom handler methods to handle the XML content.
* @package xml
*/
class xmlparser  {
  /** Parser object */
  var $parser;
  /** Error message string if any */
  var $error_message = "";
  /** True if XML parsed correctly */
  var $valid_xml = false;
  // .....................................................................
  /** Construct a new parser. */
  function xmlparser() {
    $this->parser = xml_parser_create();
    xml_set_object($this->parser, &$this);
    xml_parser_set_option($this->parser, XML_OPTION_CASE_FOLDING, false);
    xml_set_element_handler($this->parser, "tag_open", "tag_close");
    xml_set_character_data_handler($this->parser, "cdata");
  } // xmlparser
  // .....................................................................
  /**
  * Parse the given XML document.
  * @param string $xml The content of the XML document to parse.
  * @return boolean True if the XML was parsed as valid
  */
  function parse($xml) {
    $this->error_message = "";
    $this->valid_xml = false;
    $xml = trim($xml);
    if ($xml != "") {
      $this->valid_xml = xml_parse($this->parser, $xml);
      if (!$this->valid_xml) {
        $errmsg  = xml_error_string(xml_get_error_code($this->parser));
        $errline = xml_get_current_line_number($this->parser);
        $this->error_message = "XML error: $errmsg at line $errline\n";
      }
    }
    return $this->valid_xml;
  } // parse
  // .....................................................................
  /** Method invoked when a tag is opened */
  function tag_open($parser, $tag, $attributes) {
  }
  // .....................................................................
  /** Method invoked when character data is available */
  function cdata($parser, $cdata) {
  }
  // .....................................................................
  /** Method invoked when a tag is closed */
  function tag_close($parser, $tag) {
  }

} // xmlparser class

// ----------------------------------------------------------------------
/**
* Returns a value for XML compliant data transfer. If the
* value has any of the chars: "<", ">" or "&" in it, then
* we enclose it in a CDATA construct.
* @param string $val The value to return as an XML data value
* @return string The XML-compliant data value
*/
function xmldata($val) {
  if (is_bool($val)) {
    $val = ($val ? "true" : "false");
  }
  elseif (preg_match("/[<&>]/", $val)) {
    $val = "<![CDATA[$val]]>";
  }
  return $val;
} // xmldata

// ----------------------------------------------------------------------
/**
* Returns the XML header string.
* @param string $version The XML version to declare in the header.
* @param string $encoding The XML encoding standard to declare.
* @return string The <?xml...> header string.
*/
function xmlheader($version="1.0", $encoding="utf-8") {
  return "<?xml version=\"$version\" encoding=\"$encoding\"?>\n";
} // xmlheader

// ----------------------------------------------------------------------
/**
* Processes an assumed XML string to a format where it can be safely
* viewed as a debugging dump in a browser et al.
* @param string $xml The XML string to process
* @return string The safe-to-display XML debugging representation
*/
function xmldump($xml) {
  $xml = trim($xml);
  if ($xml != "") {
    $xml = preg_replace("/(<\!\[CDATA\[(.*)\]\]>)/", "\\2", $xml);
    $xml = htmlspecialchars($xml);
  }
  return $xml;
} // xmldump
// ----------------------------------------------------------------------
?>