<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    story-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing and using Axyl stories/news    */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Media catalog */
include_once("catalog-defs.php");
/** Wysiwyg Editor Widget */
include_once("wysiwyg-editor-defs.php");
include_once("dojo-widget-defs.php");

// Wysiwyg plugin settings..
//add_wysiwyg_plugins("all");

// Some global widths for form elements etc..
$width = 590;
$width_img_preview = 125;
$height_img_preview = 125;
$width_icon_preview = 90;
$height_icon_preview = 50;
$widthpx    = $width . "px";
$smlwidthpx = ceil($width/3) . "px";
$stdwidthpx = ceil($width/2) . "px";
$bigwidthpx = ceil((2 * $width)/3) . "px";

/** New story ID indicator */
define("NEW_STORY", -1);

/**
 * Some simple stop-words which are used in highlighting and
 * simple searches.
 */
$stop_words = array(
  "a", "about", "all", "am", "an", "and", "any", "are", "as",
  "at", "be", "been", "but", "by", "do", "down", "each", "for",
  "from", "had", "has", "have", "he", "her", "here", "him", "his",
  "how", "i", "if", "in", "into", "is", "it", "its", "me", "more",
  "my", "no", "nor", "not", "of", "off", "on", "only", "or", "other",
  "our", "ours", "out", "over", "own", "same", "she", "so", "some",
  "such", "than", "that", "the", "their", "theirs", "them", "then",
  "there", "these", "they", "this", "those", "through", "to", "too",
  "under", "until", "up", "very", "was", "we", "were", "what", "when",
  "where", "which", "while", "who", "whom", "why", "with", "you",
  "your", "yours", "yourself", "yourselves"
  );

/**
* A class which encapsulates a story or article item. Provides methods
* to get/save to database, edit the story in a popup window, and view it.
* Also provides methods to index/unindex to the search engine.
* @package cm
*/
class story extends RenderableObject {
  var $story_id = NEW_STORY;         // Our unique DB key for the story
  var $story_category = false;       // The group of stories this belongs to
  var $story_category_desc = "";     // Wordy descriptive version of above
  var $has_media = true;             // By category: associated media
  var $has_multimedia = false;       // By category: more then one assoc. media
  var $has_precis = true;            // By category: has a precis
  var $has_expiry = true;            // By category: has an expiry option
  var $has_multilang = true;         // By category: can be translated
  var $language = 0;                 // Language this story is in (0 = default)
  var $story_headline = "";          // Headline of this story
  var $story_precis = "";            // The lead-in section of this story
  var $story_content = "";           // The main story body content
  var $story_columns = 1;            // Number of columns to display story in
  var $story_author = "";            // The story author - FK from ax_user.user_id
  var $story_author_name = "";       // The story author full name
  var $story_type = "a";             // The story type - 'a' - Article, 'f' - Feature/Editorial
  var $story_date = "";              // Date published, NICE_DATE format
  var $story_time = "";              // Time published, NICE_TIME_ONLY format
  var $story_date_ts = 0;            // Unix timestamp of datetime story was written
  var $expiry_date = "";             // Datetime to expire, NICE_DATE format
  var $expiry_time = "";             // Datetime to expire, NICE_TIME_ONLY format
  var $expiry_date_ts = 0;           // Unix timestamp of datetime story should expire
  var $lastmodified = "";            // Datetime last modified, NICE_FULLDATETIME format
  var $lastmodified_ts = 0;          // Unix timestamp of datetime story last modified
  var $story_media = array();        // An array of media associated with this story
  var $story_icon;                   // The catalogitem object of the icon image for this story
  var $story_icon_url;               // The URL for the icon image for this story
  var $visible = false;              // True if story is visible on the website
  var $story_locs = array();         // An array of locations this story is published to
  var $story_translations = array(); // An array of media associated with this story
  var $root_translation_id = -1;     // Story ID of root (original) of translated stories
  var $root_translation_lang;        // Language of root (original) of translated stories
  var $highlights;                   // Array of highlight words to apply
  var $highlight_class = "axhl";     // Highlight which was applied

  // Internal Flags and Vars..
  var $deleted = false;              // True if story has been flagged as deleted
  var $info_msg = "";                // Contains info/error message as appropriate
  var $newstory = false;             // True if we just created this story
  var $valid = false;                // True if story was retreived successfully
  var $storymode = "";               // Mode of action on this story
  var $formname = "";                // Name of the form we use
  var $bytesize = 0;                 // Size of article + media in bytes
  var $wordcount = 0;                // Number of words written
  var $json_thumbdata_url;           // URL to get catalog thumbnail JSON data from
  // .....................................................................
  /** Constructor
  * @param mixed $id Story ID, or false if not known
  * @param mixed $category String category identifier, or false if unknown
  * @param mixed $language Integer language code, or false if default
  */
  function story($id=false, $category=false, $language=false) {
    global $RESPONSE;
    global $storymode;
    global $story_id, $cat, $lang;

    // Set up our vars..
    $this->initialise();

    // Form..
    $this->formname = "storyfm";

    // Default the mode..
    if (!isset($storymode)) $this->storymode = "view";
    else $this->storymode = $storymode;

    // Set the story ID..
    if ($id === false) {
      if (isset($story_id)) {
        $this->story_id = $story_id;
      }
    }
    else {
      $this->story_id = $id;
    }

    // Set the category..
    if ($category === false) {
      if (isset($cat)) {
        $this->story_category = $cat;
      }
    }
    else {
      $this->story_category = $category;
    }

    // Set the language..
    if ($language === false) {
      if (isset($lang)) {
        $this->language = $lang;
      }
    }
    else {
      $this->language = $language;
    }

    // Detect new story creation..
    if ($this->story_id === false || $this->story_id == NEW_STORY) {
      // Create a brand new one
      $this->story_id = get_next_sequencevalue("seq_story_id", "ax_story", "story_id");
      $this->story_date_ts = time();
      $this->story_date = timestamp_to_displaydate(NICE_DATE, $this->story_date_ts);
      $this->story_time = timestamp_to_displaydate(NICE_TIME_ONLY, $this->story_date_ts);
      $this->newstory = true;
      $this->valid = true;
      $this->get_author_info();
      $this->get_category_info();
      $this->get_default_locations();
      $this->storymode = "adding";
    }
    // Further processing for existing stories..
    if (!$this->newstory) {
      if ($this->storymode == "adding") {
        $this->newstory = true;
        $this->valid = true;
      }
      else {
        // Attempt to get the story
        $this->get_story();
      }
      // Process POST from form..
      $this->POSTprocess();

      // Final get for display..
      $this->get_story();
    }
  } // story
  // .....................................................................
  /** Initialise local vars.. */
  function initialise() {
    global $RESPONSE;
    $this->language = 0;
    $this->story_headline = "";
    $this->story_content = "";
    $this->story_precis = "";
    $this->story_columns = 1;
    if (isset($RESPONSE)) {
      $this->story_author = $RESPONSE->userid;
      $this->story_author_name = $RESPONSE->name;
    }
    else {
      $this->story_author = "";
      $this->story_author_name = "";
    }
    $this->story_type = "a";  // standard article
    $this->story_date = "";
    $this->story_date_ts = 0;
    $this->expiry_date = "";
    $this->expiry_date_ts = 0;
    $this->lastmodified = "";
    $this->lastmodified_ts = 0;
    $this->deleted = false;
    $this->visible = true;
    $this->newstory = false;
    $this->info_msg = "";
    $this->json_thumbdata_url = "/axyl-catalog-json.php";
    if (isset($this->highlights)) {
      unset($this->highlights);
    }
    $this->valid = false;
  } // story initialise
  // .....................................................................
  /**
  * Get a story in total. We always access stories by their ID.
  * @param mixed $id Story ID, or false if not known
  */
  function get_story($id=false) {
    global $RESPONSE;
    $res = false;
    if ($id !== false) $this->story_id = $id;
    if ($this->story_id !== false) {
      $q  = "SELECT * FROM ax_story";
      $q .= " WHERE story_id=$this->story_id";
      $storyQ = dbrecordset($q);
      if ($storyQ->hasdata) {
        // Main story content..
        $this->language       = $storyQ->field("lang_id");
        $this->story_category = $storyQ->field("category_id");
        $this->story_headline = $storyQ->field("story_headline");
        $this->story_precis   = $storyQ->field("story_precis");
        $this->story_content  = $storyQ->field("story_content");
        $this->story_columns  = $storyQ->field("story_columns");
        $this->story_author   = $storyQ->field("story_author");
        $this->story_type     = $storyQ->field("story_type");
        $this->story_icon_url = $storyQ->field("story_icon_url");
        if ($storyQ->field("story_icon") != "") {
          $iconitem = new catalogitem($storyQ->field("story_icon"));
          if ($iconitem->valid) {
            $this->story_icon = new story_media($this->story_id, $iconitem);
          }
        }
        // Dates and flags..
        $story_date = $storyQ->field("story_date");
        if ($story_date != "") {
          $this->story_date = datetime_to_displaydate(NICE_DATE, $story_date);
          $this->story_time = datetime_to_displaydate(NICE_TIME_ONLY, $story_date);
          $this->story_date_ts = datetime_to_timestamp($story_date);
        }
        $expiry_date = $storyQ->field("expiry_date");
        if ($expiry_date != "") {
          $this->expiry_date = datetime_to_displaydate(NICE_DATE, $expiry_date);
          $this->expiry_time = datetime_to_displaydate(NICE_TIME_ONLY, $expiry_date);
          $this->expiry_date_ts = datetime_to_timestamp($expiry_date);
        }
        $this->lastmodified = datetime_to_displaydate(NICE_FULLDATETIME, $storyQ->field("last_modified"));
        $this->lastmodified_ts = datetime_to_timestamp($storyQ->field("last_modified"));
        $this->deleted = $storyQ->istrue("deleted");
        $this->visible = $storyQ->istrue("visible");
        $res = true;

        // Now go grab sundry other associated story info..
        $this->get_author_info();
        $this->get_category_info();
        if ($this->has_media) {
          $this->get_story_media();
        }
        $this->get_story_locations();
        $this->get_story_metrics();
      }
      else {
        $this->info_msg = "No record of story ID: $this->story_id";
      }
    }
    else {
      $this->info_msg = "No story ID given";
    }
    // Did we succeed..?
    $this->valid = $res;
    return $res;
  } // story get_story
  // .....................................................................
  /**
  * Determine wwhether user can edit this story.
  * @return boolean True if user can edit the story, else false.
  */
  function user_can_edit() {
    global $RESPONSE;
    $can = false;
    if ($RESPONSE->ismemberof_group("Editor")
    || ($RESPONSE->ismemberof_group("Author") && $this->story_author == $RESPONSE->userid)
    ) {
      $can = true;
    }
    return $can;
  } // user_can_edit
  // .....................................................................
  /**
  * Get story author info. Allow override of user id via argument
  * passed in, otherwise use the resident story author ID.
  * @param string $userid Override user_id to use to get info
  * @access private
  */
  function get_author_info($userid=false) {
    if ($userid !== false) {
      $story_author = $userid;
    }
    else {
      $story_author = $this->story_author;
    }
    if ($story_author != "" && $story_author !== false) {
      $su = dbrecordset("SELECT * FROM ax_user WHERE user_id='" . escape_string($story_author) . "'");
      if ($su->hasdata) {
        $this->story_author_name = $su->field("full_name");
        $this->story_author = $story_author;
      }
    }
  } // get_author_info
  // .....................................................................
  /**
  * Get story category info. Allow override of category id via argument
  * passed in, otherwise use the resident story category ID.
  * @param integer $category_id Override category_id to use to get info
  * @access private
  */
  function get_category_info($category_id=false) {
    if ($category_id !== false) {
      $story_category = $category_id;
    }
    else {
      $story_category = $this->story_category;
    }
    if ($story_category != "" && $story_category !== false) {
      $cat = dbrecordset("SELECT * FROM ax_story_category WHERE category_id=$story_category");
      if ($cat->hasdata) {
        $this->story_category_desc = $cat->field("category_desc");
        $this->has_media       = $cat->istrue("has_media");
        $this->has_multimedia  = $cat->istrue("has_multimedia");
        $this->has_precis      = $cat->istrue("has_precis");
        $this->has_expiry      = $cat->istrue("has_expiry");
        $this->has_multilang   = $cat->istrue("has_multilang");
        $this->story_category = $story_category;
      }
    }
  } // get_category_info
  // .....................................................................
  /**
  * Get media associated with this story. This should be called after the
  * story category info has been ascertained. This method populates the
  * class variable 'story_media', an array which contains media catalog
  * ID and the filename separated by "|".
  * @access private
  */
  function get_story_media() {
    $this->story_media = array();
    $q  = "SELECT * FROM ax_story_media";
    $q .= " WHERE story_id=$this->story_id";
    $q .= " ORDER BY display_order";
    $sm = dbrecordset($q);
    while ($sm->get_next_row()) {
      $cat_id = $sm->field("cat_id");
      $media = new story_media($this->story_id);
      $media->get_catalogitem($cat_id);
      $media->justify = $sm->field("justify");
      $media->caption = $sm->field("caption");
      $media->width = $sm->field("width");
      $media->height = $sm->field("height");
      $this->story_media[$cat_id] = $media;
      debugbr("adding story media: $cat_id " . $media->catalogitem->filepath);
    }
  } // get_story_media
  // .....................................................................
  /**
  * Get the story locations defined for this story. This method
  * is an internal one designed to be used to populate the currently
  * defined locations the story will be published in.
  * @access private
  */
  function get_story_locations() {
    $this->story_locs = array();
    $q  = "SELECT * FROM ax_story_location";
    $q .= " WHERE story_id=$this->story_id";
    $loc = dbrecordset($q);
    if ($loc->hasdata) {
      do {
        $locid = $loc->field("location_id");
        $this->story_locs[] = $locid;
      } while ($loc->get_next());
    }
  } // get_story_locations
  // .....................................................................
  /**
  * Get the default locations for this story category. This method
  * is an internal one designed to be used to populate the initial
  * locations to publish a new story to.
  * @access private
  */
  function get_default_locations() {
    $this->story_locs = array();
    if ($this->story_category) {
      $q = "SELECT location_id FROM ax_story_category_locs"
         . " WHERE category_id=$this->story_category";
    }
    else {
      $q = "SELECT location_id FROM ax_content_location";
    }
    $loc = dbrecordset($q);
    if ($loc->hasdata) {
      do {
        $locid = $loc->field("location_id");
        $this->story_locs[] = $locid;
      } while ($loc->get_next());
    }
  } // get_default_locations
  // .....................................................................
  /**
  * Get the story locations defined for this story. This method
  * is an internal one designed to be used to populate the current
  * locations to publish the story in.
  * @access private
  */
  function get_story_metrics() {
    global $RESPONSE;
    $words = $this->story_headline . $this->story_precis . $this->story_content;
    $bytesize = strlen($words);
    if (count($this->story_media) > 0) {
      foreach ($this->story_media as $media) {
        if (isset($media->catalogitem) && $media->catalogitem->filepath != "") {
          if (file_exists($RESPONSE->site_docroot . $media->catalogitem->filepath)) {
            $bytesize += filesize($RESPONSE->site_docroot . $media->catalogitem->filepath);
          }
        }
      } // foreach
    }
    $this->bytesize = $bytesize;
    $this->wordcount = $this->word_count();
  } // get_story_metrics
  // .....................................................................
  /**
  * Get the stories which are translated versions of this one.
  * @access private
  */
  function get_story_translations() {
    $this->story_translations = array();
     // Find root story info for this set of translations..
    debugbr("translation family: this story ID is $this->story_id");
    $this->root_translation_id = -1;
    $q  = "SELECT st.story_id as sid, s.lang_id as lang";
    $q .= "  FROM ax_story_translation st, ax_story s";
    $q .= " WHERE st.translated_story_id=$this->story_id";
    $q .= "   AND s.story_id=st.story_id";
    $q .= "   AND s.deleted=FALSE";
    $q .= " UNION ";
    $q .= "SELECT st.story_id as sid, s.lang_id as lang";
    $q .= "  FROM ax_story_translation st, ax_story s";
    $q .= " WHERE st.story_id=$this->story_id";
    $q .= "   AND s.story_id=st.story_id";
    $q .= "   AND s.deleted=FALSE";
    $trans = dbrecordset($q);
    if ($trans->hasdata) {
      $this->root_translation_id = $trans->field("sid");
      $this->root_translation_lang = $trans->field("lang");
      debugbr("translation story id root = $this->root_translation_id");
      // Add root story if it's not us..
      if ($this->root_translation_id != $this->story_id) {
        debugbr("ADDING translation story id (root)=$this->root_translation_id lang=$this->root_translation_lang");
        $this->story_translations[$this->root_translation_id] = $this->root_translation_lang;
      }
    }
    // Now get all translations of the root..
    $q  = "SELECT st.translated_story_id as sid, s.lang_id as lang";
    $q .= "  FROM ax_story_translation st, ax_story s";
    $q .= " WHERE st.story_id=$this->root_translation_id";
    $q .= "   AND s.story_id=st.translated_story_id";
    $q .= "   AND s.deleted=FALSE";
    $trans = dbrecordset($q);
    if ($trans->hasdata) {
      do {
        $storyid = $trans->field("sid");
        $langid  = $trans->field("lang");
        // Add story if it's not us..
        if ($storyid != $this->story_id) {
          debugbr("ADDING translation story id=$storyid lang=$langid");
          $this->story_translations[$storyid] = $langid;
        }
      } while ($trans->get_next());
    }
  } // get_story_translations
  // .....................................................................
  /** Index this story to the search engine, if enabled for this website. */
  function index() {
    global $SE_AVAILABLE;
    // Deal with indexing if enabled. In this case we then use the unique
    // story_id as the index ID, and index the story heading and body text.
    // We also categorise it.
    if ($SE_AVAILABLE) {
      include_once("search-lucene-defs.php");
      include_once("search-index-defs.php");
      $allcontent[] = $this->story_headline;
      $allcontent[] = $this->story_precis;
      $allcontent[] = $this->story_content;
      $I = new searchengine_indexer();
      $I->index_field("category:Text",       "article");
      $I->index_field("title:Text",          $this->story_headline);
      $I->index_field("story_date:Date",     $this->story_date_ts);
      $I->index_field("story_author:Text",   $this->story_author);
      $I->index_field("story_lang:Text",     $this->language);
      $I->index_field("story_category:Text", $this->story_category);
      $I->index_field("story_type:Text",     $this->story_type);
      if ($this->story_url != "") {
        $I->index_field("story_url:Text", $this->story_url);
      }
      $I->index_content($this->story_id, strip_tags(implode(" ", $allcontent)));
      $I->execute();
    }
  } // story index
  // .....................................................................
  /** Un-Index this story from the search engine, if enabled for this website. */
  function unindex() {
    global $SE_AVAILABLE;
    if ($SE_AVAILABLE) {
      include_once("search-lucene-defs.php");
      include_once("search-index-defs.php");
      $UI = new searchengine_unindexer();
      $UI->unindex($this->story_id);
      $UI->execute();
    }
  } // unindex
  // .....................................................................
  /**
   * Given a list of words, we scan the story body and highlight any we
   * find in there. The parm '$words' is an array of words to highlight.
   * @param array $words Array of words to highlight in the story.
   * @param string $hlclass Highlight class to use when highlighting words
   */
  function highlight_words($words, $hlclass="axhl") {
    global $stop_words;
    if (is_array($words) && count($words) > 0) {
      $this->highlights = array();
      $this->highlight_classs = $hlclass;
      if ($this->valid && $this->story_content != "") {
        foreach ($words as $word) {
          if ( eregi("/", $word) == 0 ) {
            $word = str_replace("\"", "", trim($word));
            $word = preg_replace("/~[0-9]*/", "", trim($word));
            $word = preg_replace("/[+-]/", "", trim($word));
            if ($word != "" && !in_array($word, $stop_words)) {
              $this->highlights[] = $word;
              $pattern = "/\b(" . $word . ")\b/i";
              $this->story_headline = preg_replace($pattern, "<span class=$hlclass>\$1</span>", $this->story_headline);
              $this->story_precis   = preg_replace($pattern, "<span class=$hlclass>\$1</span>", $this->story_precis);
              $this->story_content  = preg_replace($pattern, "<span class=$hlclass>\$1</span>", $this->story_content);
            }
          }
        } // foreach
      }
    }
  } // highlight_words
  // .....................................................................
  /**
   *  Removes the highlighting tags from the story body.
   * @param string $hlclass Highlight class to match when removing
   */
  function unhighlight($hlclass="") {
    if (!$this->valid) return;
    if (isset($this->highlights) && count($this->highlights) > 0) {
      if ($hlclass == "") {
        $hlclass = $this->highlight_class;
      }
      foreach ($this->highlights as $word) {
        $pattern = "/(<span class=$hlclass>)($word)(<\/span>)/i";
        $this->story_headline = preg_replace($pattern, "\$2", $this->story_headline);
        $this->story_precis   = preg_replace($pattern, "\$2", $this->story_precis);
        $this->story_content  = preg_replace($pattern, "\$2", $this->story_content);
      } // foreach
    }
  } // unhighlight
  // .....................................................................
  /**
   * Method which saves the whole story to the database including the
   * story media and story locations information.
   * Note that this method does not
   */
  function save_story() {
    // Save story media
    $this->save_story_details();
      
    // Save story media
    $this->save_story_media();
      
    // Save story locations
    $this->save_story_locations();
  } // save_story
  // .....................................................................
  /**
  * Save the story details to the database.
  * @return boolean True if no error occurred
  */
  function save_story_details() {
    if ($this->story_id) {
      if ($this->newstory) {
        $sup = new dbinsert("ax_story");
        $sup->set("story_id", $this->story_id);
      }
      else {
        $sup = new dbupdate("ax_story");
      }
      $this->unhighlight();
      $sup->set("lang_id",        $this->language);
      $sup->set("story_headline", $this->story_headline);
      $sup->set("story_precis",   $this->story_precis);
      $sup->set("story_content",  $this->story_content);
      $sup->set("story_columns",  $this->story_columns);
      $sup->set("story_author",   ($this->story_author != "") ? $this->story_author : NULLVALUE);
      $sup->set("story_type",     $this->story_type);
      $sup->set("story_icon_url", $this->story_icon_url);
      $sup->set("category_id",    ($this->story_category !== false) ? $this->story_category : NULLVALUE);
      if (isset($this->story_icon) && $this->story_icon !== "") {
        $sup->set("story_icon", $this->story_icon->catalogitem->cat_id);
      }
      else {
        $sup->set("story_icon", NULLVALUE);
      }
      if ($this->story_date_ts == 0) {
        $sup->set("story_date", NULLVALUE);
      }
      else {
        $sup->set("story_date", timestamp_to_datetime($this->story_date_ts));
      }
      if ($this->expiry_date_ts == 0) {
        $sup->set("expiry_date", NULLVALUE);
      }
      else {
        $sup->set("expiry_date", timestamp_to_datetime($this->expiry_date_ts));
      }
      $sup->set("last_modified",  'now()');
      $sup->set("visible",        $this->visible);
      $sup->set("deleted",        $this->deleted);
      $sup->where("story_id=$this->story_id");
      if ($sup->execute()) {
        // Index to search engine..
        $this->index();
        $this->newstory = false;
      }
    }
  } // save_story_details
  // .....................................................................
  /**
  * Save the story media to the database. There may be zero or more media
  * items stored against this story, so we delete existing ones from the
  * database and recreate the records from the latest set.
  * @return boolean True if no error occurred
  */
  function save_story_media() {
    $res = true;
    if ($this->story_id) {
      $mytran = start_transaction();
      $smdel = new dbdelete("ax_story_media");
      $smdel->where("story_id=$this->story_id");
      $smdel->execute();
      if (is_array($this->story_media) && count($this->story_media > 0)) {
        foreach ($this->story_media as $cat_id => $media) {
          $smin = new dbinsert("ax_story_media");
          $smin->set("story_id", $this->story_id);
          $smin->set("cat_id",   $cat_id);
          $smin->set("caption",  $media->caption);
          $smin->set("justify",  $media->justify);
          $smin->set("width",    (($media->width  != "") ? $media->width  : 0));
          $smin->set("height",   (($media->height != "") ? $media->height : 0));
          if (!$smin->execute()) {
            $res = false;
            break;
          }
        } // foreach
      }
      if ($mytran) commit();
    }
    return $res;
  } // save_story_media
  // .....................................................................
  /**
  * Save the story locations to the database. A story can be published
  * to zero or more locations. These are just FKs to ax_content_location.
  * @return boolean True if no error occurred
  */
  function save_story_locations() {
    $res = true;
    if ($this->story_id) {
      $mytran = start_transaction();
      $sldel = new dbdelete("ax_story_location");
      $sldel->where("story_id=$this->story_id");
      $sldel->execute();
      if (is_array($this->story_locs) && count($this->story_locs > 0)) {
        foreach ($this->story_locs as $loc_id) {
          if ($loc_id != "") {
            $slin = new dbinsert("ax_story_location");
            $slin->set("story_id", $this->story_id);
            $slin->set("location_id", $loc_id);
            if (!$slin->execute()) {
              $res = false;
              break;
            }
          }
        } // foreach
      }
      if ($mytran) commit();
    }
    return $res;
  } // save_story_locations  
  // .....................................................................
  /** Remove the story from the system. We actually just flag it as
  * deleted on the database, and keep the record.
  */
  function delete_story() {
    global $RESPONSE, $CONTEXT;
    if ($this->valid && !$this->deleted) {
      $del = new dbupdate("ax_story");
      $del->set("deleted", true);
      $del->where("story_id=$this->story_id");
      $this->deleted = $del->execute();
      if ($this->deleted) {
        $this->unindex();
        $this->info_msg = "Story has been marked as deleted";
      }
    }
  }  // delete_story
  // .....................................................................
  /**
  * Add a new or existing media item to the story. Since the media item
  * array is associative and keyed on cat_id, duplicate catalog items
  * should simply overwrite, which is what we want.
  * @param object Media item object to add to this story.
  */
  function add_media_item($mediaobj) {
    if (is_object($mediaobj) && isset($mediaobj->catalogitem)) {
      $cat_id = $mediaobj->catalogitem->cat_id;
      $this->story_media[$cat_id] = $mediaobj;
    }
  } // add_media_item
  // .....................................................................
  /**
   * Allows setting of the URL where we can obtain thumbnail images to
   * populate the story thumbnailpicker, to enable editors to select an
   * image to go with this story. Usually this is a script which emits
   * a JSON stream used to populate the picker.
   * @param string $url URL which returns JSON image catalog data
   */
  function set_json_thumbdata_url($url) {
    $this->json_thumbdata_url = $url;
  } // set_json_thumbdata_url
  // .....................................................................
  /** Process the POST from form. This method deals with POSTed content
  * from the edit form.
  */
  function POSTprocess() {
    global $RESPONSE;
    global $storysave_x;     // Clicked save
    global $storyedit_x;     // Clicked edit
    global $storycancel_x;   // Clicked cancel
    global $translate_x;     // Clicked translate
    global $story_headline;  // Story headline
    global $story_precis;    // Story precis/lead-in
    global $story_content;   // Story content
    global $story_columns;   // Display columns
    global $story_media;     // Reference to picture, movie etc.
    global $story_icon;      // Reference to icon catalog item
    global $story_icon_url;  // URL for story icon
    global $uploadmedia;     // Uploaded media file present
    global $story_locs;      // List of locations to publish to
    global $caption;         // New image caption
    global $media_justify;   // Image justify setting 'left' or 'right'
    global $media_width;     // Image width px
    global $media_height;    // Image height px
    global $media_resize;    // Resize uploaded image to width pixels
    global $story_author;    // Story author
    global $story_type;      // Story type
    global $story_date;      // Story publish date setting
    global $story_time;      // Story publish time setting
    global $story_language;  // Story language setting
    global $new_language;    // New story langauage - translated
    global $expiry_date;     // Expiry date setting
    global $expiry_time;     // Expiry time setting
    global $visible;         // Visible flag

    debugbr("POSTprocess: storymode initial: $this->storymode", DBG_DEBUG);

    // Save story
    if (isset($storysave_x)) {
      if (isset($story_headline)) $this->story_headline = $story_headline;
      if (isset($story_precis))   $this->story_precis   = $story_precis;
      if (isset($story_content))  $this->story_content  = $story_content;
      if (isset($story_columns))  $this->story_columns  = $story_columns;
      if (isset($story_author))   $this->story_author   = $story_author;
      if (isset($story_type))     $this->story_type     = $story_type;
      if (isset($story_icon_url)) $this->story_icon_url = $story_icon_url;
      if ($this->has_multilang && isset($story_language)) {
        $this->language = $story_language;
      }

      // Story dates..
      if (isset($story_date)) {
        debugbr("story_date=[$story_date] story_time=[$story_time]");
        if ( $story_date != "" || $story_time != "") {
          if ($story_date == "") {
            $story_date = timestamp_to_displaydate(NICE_DATE, time());
          }
          if ($story_time == "") {
            $story_time = timestamp_to_displaydate(NICE_TIME_ONLY, time());
          }
          else {
            $story_time = str_replace("T", "", $story_time);
          }
          debugbr("story_datetime=[$story_date $story_time]");
          $this->story_date_ts = strtotime("$story_date $story_time");
        }
        else {
          // Supply default of 'now'. Must always have a story date.
          $this->story_date_ts = time();
        }
        $this->story_date = timestamp_to_displaydate(NICE_DATE, $this->story_date_ts);
        $this->story_time = timestamp_to_displaydate(NICE_TIME_ONLY, $this->story_date_ts);
        debugbr("this->story_date=[$this->story_date] this->story_time=[$this->story_time]");
      }
      if ($this->has_expiry && isset($expiry_date)) {
        debugbr("expiry_date=[$expiry_date] expiry_time=[$expiry_time]");
        if ( $expiry_date != "" || $expiry_time != "") {
          if ($expiry_date == "") {
            $expiry_date = timestamp_to_displaydate(NICE_DATE, (time() + (2 * SECS_1_WEEK)));
          }
          if ($expiry_time == "") {
            $expiry_time = timestamp_to_displaydate(NICE_TIME_ONLY, time());
          }
          else {
            $expiry_time = str_replace("T", "", $expiry_time);
          }
          debugbr("expiry_datetime=[$expiry_date $expiry_time]");
          $this->expiry_date_ts = strtotime("$expiry_date $expiry_time");
        }
        else {
          // Supply default of 2 weeks in the future, since the 'has_expiry'
          // flag means we must have an expiry date specified.
          $this->expiry_date_ts = time() + (2 * SECS_1_WEEK);
        }
        $this->expiry_date = timestamp_to_displaydate(NICE_DATE, $this->expiry_date_ts);
        $this->expiry_time = timestamp_to_displaydate(NICE_TIME_ONLY, $this->expiry_date_ts);
        debugbr("this->expiry_date=[$this->expiry_date] this->expiry_time=[$this->expiry_time]");
      }

      // Visible flag..
      $this->visible = isset($visible);

      // Story icon
      if (isset($story_icon) && $story_icon != "") {
        $icon_bits = explode("|", $story_icon);
        $cat_id = $icon_bits[0];
        $newci = new catalogitem($cat_id);
        if ($newci->valid) {
          $this->story_icon = new story_media($this->story_id, $newci);
        }
      }
      else {
        unset($this->story_icon);
      }

      // Save it if changed..
      $this->save_story_details();

      // Media data POSTings..
      if ($this->has_media) {
        // Some defaults..
        if ($media_width  == "") $media_width  = 0;
        if ($media_height == "") $media_height = 0;

        // Deal with a new media file upload. In this case we
        // assume we are just adding new media to existing & save
        if (isset($uploadmedia) && $uploadmedia != "none" && $uploadmedia != "") {
          // Start from scratch and rebuild media, then save it
          $this->story_media = array();
          $newci = new catalogitem();
          $errmsgs = $newci->upload($caption, $this->category);
          if ($newci->valid) {
            if (isset($media_resize) && intval($media_resize) > 0) {
              $newci->resize(intval($media_resize));
            }
            $media = new story_media($this->story_id, $newci);
            $media->caption = $caption;
            $media->justify = $media_justify;
            $media->width   = 0;
            $media->height  = 0;
            $this->story_media[$newci->cat_id] = $media;
            $this->save_story_media();
          }
          else {
            if (count($errmsgs) > 0) {
              $this->info_msg = implode("<br>", $errmsgs);
            }
          }
        }
        // Otherwise, totally re-create the media from the posted
        // list of media items..
        else {
          // Turn incoming value(s) into an array..
          $new_story_media = array();
          if (is_array($story_media)) {
            $new_story_media = $story_media;
          }
          elseif ($story_media != "") {
            $new_story_media = array($story_media);
          }
          // Start from scratch and rebuild media, then save it
          $media_width   = ceil($media_width);
          $media_height  = ceil($media_height);
          $this->story_media = array();
          foreach ($new_story_media as $cat_info) {
            if ($cat_info != "") {
              $cat_bits = explode("|", $cat_info);
              $cat_id = $cat_bits[0];
              if ($cat_id != "") {
                $newci = new catalogitem($cat_id);
                if ($newci->valid) {
                  if ($media_width == $newci->width && $media_height == $newci->height) {
                    $media_width = 0;
                    $media_height = 0;
                  }
                  // Stash it in the story media array
                  $media = new story_media($this->story_id, $newci);
                  $media->caption = $caption;
                  $media->justify = $media_justify;
                  $media->width   = $media_width;
                  $media->height  = $media_height;
                  $this->story_media[$cat_id] = $media;
                  $media_to_save = false;
                }
              }
            }
          } // foreach
          
          // Now save media
          $this->save_story_media();
        }
      } // has_media

      // Story publish to locations..
      $this->story_locs = array();
      if (isset($story_locs)) {
        if (!is_array($story_locs)) {
          $story_locs = explode(",", $story_locs);
        }
        if (is_array($story_locs)) {
          $this->story_locs = $story_locs;
          $this->save_story_locations();
        }
      }
      
      $this->info_msg = "Story was saved";
      $this->storymode = "viewagain";
    } // storysave_x

    // Edit story
    elseif (isset($storyedit_x)) {
      $this->storymode = "edit";
    } // storyedit_x

    // Cancel current mode
    elseif (isset($storycancel_x)) {
      $this->storymode = "viewagain";
    } // storycancel_x

    // Translate current story into new language.
    elseif (isset($translate_x)) {
      $translation = $this->get_translation($new_language);
      if ($translation !== false) {
        $this->get_story($translation);
      }
    } // translate_x

    // Remove story
    elseif ($this->storymode == "remove") {
      $this->delete_story();
      $this->storymode = "viewagain";
    } // remove
    debugbr("POSTprocess: storymode final: $this->storymode", DBG_DEBUG);
  } // story POSTprocess
  // .....................................................................
  /**
  * Returns the story_id of a translation of the current story in the
  * given language. If it already exists, then it just returns the story
  * ID. If it doesn't exist, then it simply makes a copy of this story,
  * assigns it the language it _will_ be translated into, and records a
  * relationship to the other associated translations in the database table
  * 'story_tranlsation'. This latter table allows us to put a list of
  * languages (or little country flags) on any stories which have alternatives
  * in another language.
  * @param integer $language Language to get translated story in
  */
  function get_translation($language) {
    // Check if this story already has a translation available, and
    // we just return the translated story id if so..
    $this->get_story_translations();
    foreach ($this->story_translations as $translated_storyid => $chklang) {
      if ($chklang == $language) {
        return $translated_storyid;
        break;
      }
    }

    // Preserve some info for use later on..
    $original_story_id = $this->story_id;
    $original_translations = $this->story_translations;

    // Create new story..
    start_transaction();
    $this->story_id = get_next_sequencevalue("seq_story_id", "ax_story", "story_id");
    // Set the new language..
    $this->language = $language;
    $this->story_date_ts = time();
    $this->story_date = timestamp_to_displaydate(NICE_DATE, $this->story_date_ts);
    $this->newstory = true;
    $this->valid = true;
    $this->save_story();

    // Duplicate media..
    $ord = 1;
    foreach ($this->story_media as $cat_id => $media) {
      $in = new dbinsert("ax_story_media");
      $in->set("story_id",      $this->story_id);
      $in->set("cat_id",        $cat_id);
      $in->set("caption",       $media->caption);
      $in->set("width",         $media->width);
      $in->set("height",        $media->height);
      $in->set("justify",       $media->justify);
      $in->set("display_order", $ord++);
      $in->execute();
    }
    // Duplicate sports..
    foreach ($this->story_sports as $sport_id) {
      $in = new dbinsert("ax_story_sport");
      $in->set("story_id",      $this->story_id);
      $in->set("sport_id",      $sport_id);
      $in->execute();
    }
    // Duplicate locations..
    $ord = 1;
    foreach ($this->story_locs as $loc_id) {
      $in = new dbinsert("ax_story_location");
      $in->set("story_id",      $this->story_id);
      $in->set("location_id",   $loc_id);
      $in->set("display_order", $ord++);
      $in->execute();
    }
    if (commit()) {
      // Create translated story relationship..
      if ($this->root_translation_id == -1) {
        $root_trans_id = $original_story_id;
      }
      else {
        $root_trans_id = $this->root_translation_id;
      }
      $in = new dbinsert("ax_story_translation");
      $in->set("story_id",            $root_trans_id);
      $in->set("translated_story_id", $this->story_id);
      $in->execute();
    }
    return $this->story_id;
  } // translate_into
  // .....................................................................
  /** Do a re-count of the story words. Set our local variable
  * and also return the value as a by-product..
  * @return integer Count of words in the story
  */
  function word_count() {
    $words = explode(" ",
          $this->story_headline . " "
        . $this->story_precis . " "
        .  $this->story_content
        );
    return count($words);
  } // word_count
  // .....................................................................
  /** Generate a precis from the story content. This is provided so that
   * you can show the first few words of a story to give an indication
   * to the reader what the rest migh contain.
   * @param integer $maxwords Maximum number of words for the precis
   * @param integer $minwords Minimum number of words for the precis
   * @return string The precis
   */
  function make_precis($maxwords=50, $minwords=5) {
    $precis = "";
    $patt = "(([\S]+[\s]+){" . $minwords . "," . $maxwords . "})";
    $matches = array();
    preg_match("/$patt/", strip_tags($this->story_content), $matches);
    if (isset($matches[1])) {
      $precis = $matches[1];
    }
    $precis = str_replace("\x0d\x0a", " ", $precis);
    return $precis;
  } // make_precis
  // .....................................................................
  /**
  * Return the rendering of the story icon (if one exists) either as a
  * standard HTML anchor tag if an icon URL exists, or as an image.
  * @return string HTML for anchor or image of the story icon
  */
  function render_story_icon() {
    $s = "";
    if (isset($this->story_icon) && is_object($this->story_icon)) {
      $s = $this->story_icon->catalogitem->Insitu();
      if ($this->story_icon_url != "") {
        $a = new anchor($this->story_icon_url, $s);
        if (stristr($this->story_icon_url, "http")) {
          $a->settarget("_new");
        }
        $s = $a->render();
      }
    }
    return $s;
  } // render_story_icon
  // .....................................................................
  /**
   * Return the first 'image' media object associated with this story
   * or false if there isn't one.
   * @param string $version Version of image to return: 'thumb', 'medium', 'flash'
   * @param boolean $createifabsent If true (default) will create missing version
   * @return object Story image object based on the 'img' class.
   */
   function story_image($version="", $createifabsent=true) {
      $pic = false;
      foreach ($this->story_media as $cat_id => $media) {
        $ci = $media->catalogitem;
        if ($ci->mime_category == "image") {
          $filepath = "";
          if ($version == "") {
            $width = ($media->width  > 0) ? $media->width  : $ci->width;
            $height = ($media->height > 0) ? $media->height : $ci->height;
            $filepath = $ci->filepath;
            $name = $ci->mime_category . $ci->cat_id;
          }
          else {
            // Make the requested version, but only if missing
            $ci->make_version($version);
            if (isset($ci->versions[$version])) {
              $ver = $ci->versions[$version];
              $filepath = $ver->filepath;
              $name = $ci->mime_category . $version . $ci->cat_id;
              $width = $ver->width;
              $height = $ver->height;
            }
          }
          if ($filepath != "") {
            $caption = ($media->caption != "") ? $media->caption : $ci->cat_name;
            $pic = new img(
                    $filepath,
                    $name,
                    $caption,
                    ($width  > 0 ? $width  : false),
                    ($height > 0 ? $height : false)
                    );
          }
        }
      } // foreach
      
      // Image object, or false
      return $pic;
   } // story_image
  // .....................................................................
  /** Render the story details as an edit form.
  * @return string The editform complete.
  */
  function editform() {
    global $RESPONSE;
    global $LIBDIR, $CMDIR, $IMAGESDIR;
    global $width, $width_img_preview, $height, $height_img_preview;
    global $width_icon_preview, $height_icon_preview;
    global $widthpx, $smlwidthpx, $stdwidthpx, $bigwidthpx;

    // HIDDEN FIELDS
    // These are rendered at the end of the form in the foot
    $hidFlds = array();

    // CONTROL BUTTONS
    $s = "";
    if ($this->user_can_edit()) {
      $savb = new form_imagebutton("storysave",   "Save",   "", "$LIBDIR/img/_save.gif",   "Save changes", 57, 15);
      $canb = new form_imagebutton("storycancel", "Cancel", "", "$LIBDIR/img/_cancel.gif", "Cancel",       57, 15);
      if ($this->newstory) {
        $canb->set_onclick("window.close()");
      }
      $s .= $savb->render() . "&nbsp;&nbsp;" . $canb->render();
    }
    $CONTROL_BUTTONS = $s;

    $tabwidthpx = ($width + 110) . "px";
    $tabheightpx = "800px";
    $TAB = new dojo_tabcontainer("mainTabContainer");
    $TAB->setcss(
        "padding:0px;top:0px;left:0px;width:$tabwidthpx;height:$tabheightpx;"
        );

    // COMMON HEADER
    $rowbg = "axbgdark";
    $Thd = new table("editor heading");
    $Thd->tr($rowbg);
    $title = "$this->story_category_desc ";
    if ($this->newstory) {
      $title .= "- New Article";
    }
    $Thd->td("<h3>$title</h3>", "padding-left:6px;");
    $Thd->td($CONTROL_BUTTONS, "text-align:right;vertical-align:bottom;padding-right:6px;");
    $Thd->td_alignment("right");

    // -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    // STORY EDITOR TAB
    $Tst = new table("story");
    $Tst->setpadding(2);

    $rowbg = "axbgdark";
    $Tst->tr($rowbg);
    $Tst->td($Thd->render(), "border-bottom:1px solid black");
    $Tst->td_colspan(2);

    // ERRMSG
    if ($this->info_msg != "") {
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tst->tr($rowbg);
      $Tst->td($this->info_msg, "axerror");
      $Tst->td_alignment("center");
    }

    // HEADLINE
    $Fld = new form_textfield("story_headline", "Headline", $this->story_headline);
    $Fld->setstyle("width:$widthpx;height:30px;font-weight:bold;font-size:14pt;");
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td($Fld->render());
    $Tst->td_css("padding-top:10px;");

    // CATEGORY
    $Fld = new form_combofield("story_category", "Category", $this->story_category);
    $Fld->setstyle("width:$stdwidthpx;");
    $Fld->set_autocomplete();
    $Fld->settitle("Pick a story category");
    $q  = "SELECT * FROM ax_story_category";
    $q .= " ORDER BY category_desc";
    $cat = dbrecordset($q);
    if ($cat->hasdata) {
      $Fld->add_querydata($cat, "category_id", "category_desc");
    }
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td($Fld->render());

    // STORY AUTHOR
    if ($RESPONSE->ismemberof_group("Editor")) {
      // Editors can change the author..
      $Fld = new form_combofield("story_author", "Author", $this->story_author);
      $Fld->setstyle("width:$stdwidthpx;");
      $Fld->set_autocomplete();
      $Fld->settitle("Pick an author of this story");
      $Fld->additem("");
      $q  = "SELECT * FROM ax_user u, ax_user_group ug, ax_group g";
      $q .= " WHERE ug.user_id=u.user_id";
      $q .= "   AND g.group_id=ug.group_id";
      $q .= "   AND g.group_desc IN ('Editor','Author','Columnist')";
      $q .= "   AND u.enabled";
      $q .= " ORDER BY u.full_name";
      $authors = dbrecordset($q);
      if ($authors->hasdata) {
        $Fld->add_querydata($authors, "user_id", "full_name");
      }
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tst->tr($rowbg);
      $Tst->td($Fld->render());
    }
    else {
      // Standard authors can't change by-line..
      $Fld = new form_labelfield("story_author", "<b>$this->story_author_name ($this->story_author)</b>");
      $hidFlds[] = new form_hiddenfield("story_author", $this->story_author);
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tst->tr($rowbg);
      $Tst->td($Fld->render());
    }

    // PRECIS/LEAD IN
    if ($this->has_precis) {
      $Fld = new form_memofield("story_precis", "Story Lead-In", $this->story_precis);
      $Fld->setclass("axmemo");
      $Fld->setstyle("width:$widthpx;height:60px;");
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tst->tr($rowbg);
      $Tst->td("Story Lead Paragraph:", "font-weight:bold");
      $Tst->tr($rowbg);
      $Tst->td($Fld->render());
    }

    // STORY CONTENT
    $Fld = new form_wysiwygfield("story_content", "Article", $this->story_content);
    $Fld->setclass("axmemo");
    $Fld->setwidth($width);
    $Fld->setheight(500);
    $Fld->setstyle("width:$widthpx;height:500px;");
    $Fld->register_plugins("all");
    $Fld->set_toolbar("full");
    $Fld->set_styles($RESPONSE->head->stylesheet);
    $Fld->set_statusbar(false);
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td($Fld->render());

    # ADD TO TAB
    $storyed_pane = new dojo_contentpane("storyeditor", "Story Editor");
    $storyed_pane->add_content($Tst->render());
    $TAB->add_contentpane($storyed_pane);

    // -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    // STORY MEDIA
    if ($this->has_media) {
      $rowbg = "axbgdark";
      $Tst = new table("media");
      $Tst->setpadding(2);
      $Tst->tr($rowbg);
      $Tst->td($Thd->render(), "border-bottom:1px solid black");
      $Tst->td_colspan(2);

      // Media details table..
      $Tmed = new table("storymedia");
      $Tmed->setwidth("100%");

      // Find the first (selected) media object $selmedia_first. This object
      // is used extensively below to populate the various form fields. We
      // currently only properly support one object (the first one).
      if (count($this->story_media) > 0) {
        reset($this->story_media);
        list($catid, $selmedia_first) = each($this->story_media);
      }
      else {
        $selmedia_first = new story_media();
      }
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tmed->tr($rowbg);

      // Width and Height override fields
      $sizes_defaulted = ($selmedia_first->width == 0 || $selmedia_first->height == 0);
      $real_width  = $selmedia_first->catalogitem->width;
      $real_height = $selmedia_first->catalogitem->height;
      if ($sizes_defaulted) {
        $width  = $real_width;
        $height = $real_height;
      }
      else {
        $width  = $selmedia_first->width;
        $height = $selmedia_first->height;
      }

      // Image stuff detail table
      $Tin = new table();
      $Tin->setpadding(4);

      // Media selector..
      switch ($RESPONSE->ui_toolkit) {
        case "dojo":
          // Thumbnail picker
          $thumbpicker = new dojo_thumbpicker("thumbs", 700);
          $thumbpicker->set_clickable();
          $thumbpicker->subscribe_script("imgPreview");
          $thumbpicker->set_data_url("");
          $thumbpicker->set_data_url($this->json_thumbdata_url);
          $thumbpicker->set_fetch_count(10);
          
          if ($RESPONSE->browser == BROWSER_IE) {
            $shob = new form_imagebutton("showtp", "Show", "", "$LIBDIR/img/_show.gif", "Show all thumbnails", 57, 15);
            $shob->set_onclick("init_thumbpicker()");
            $Tin->tr($rowbg);
            $Tin->td($shob->render(), "text-align:right;padding-right:6px;padding-top:2px;");
            $Tin->td_colspan(3);
          }
          $hid = new form_hiddenfield("story_media", $selmedia_first->keyinfo());
          $hidFlds[] = $hid;

          $Tin->tr($rowbg);
          $Tin->td($thumbpicker->render());
          $Tin->td_colspan(3);
          break;
          
        default:
          $catFld = new form_combofield("story_media", "Media", $selmedia_first->keyinfo());
          $catFld->setstyle("width:$bigwidthpx;");
          $catFld->additem("", "None");
          $catalog = new catalog();
          $catalog->search("", array("image"));
          foreach ($catalog->catalogitems as $catid => $catitem) {
            if (isset($this->story_media[$catid])) {
              $media = $this->story_media[$catid];
            }
            else {
              $media = new story_media($this->story_id, $catitem);
              $media->caption = $catitem->cat_name;
            }
            $key = $media->keyinfo();
            $label = $media->catalogitem->cat_name;
            if ($label == "") {
              $label = $media->catalogitem->filepath;
            }
            $catFld->additem($key, $label);
          }
          $catFld->set_onchange("imgPreview(this)");
          $Tin->tr($rowbg);
          $Tin->td("Use existing:", "font-weight:bold;");
          $Tin->td($catFld->render());
          $Tin->td_colspan(2);
      } // switch

      // File upload field
      $upFld = new form_fileuploadfield("uploadmedia", "Upload");
      $upFld->setclass("axtxtbox");
      $upFld->setstyle("width:$smlwidthpx;");

      // Image upload auto-resizer
      $rzFld = new form_combofield("media_resize", "Resize", 300);
      $rzFld->setstyle("width:80px");
      $rzFld->additem(150, "100px");
      $rzFld->additem(150, "150px");
      $rzFld->additem(200, "200px");
      $rzFld->additem(250, "250px");
      $rzFld->additem(300, "300px");
      $rzFld->additem(325, "325px");
      $rzFld->additem(350, "350px");
      $rzFld->additem(400, "400px");
      $rzFld->additem(450, "450px");
      $rzFld->additem(500, "500px");
      $rzFld->additem(600, "600px");
      $rzFld->additem(600, "700px");
      $rzFld->additem(600, "800px");
      
      $Tin->tr($rowbg); 
      $Tin->td("Upload new:", "font-weight:bold;");
      $Tin->td($upFld->render());
      $Tin->td($rzFld->render() . "&nbsp;Width limit", "axfg");

      $Tin->set_width_profile("15%,55%,30%");

      $Tmed->tr($rowbg);
      $Tmed->td($Tin->render());
      $Tmed->td_alignment("", "top");

      // Image manipulation fields - only shown when an
      // image is associated with the story
      $Tin = new table();
      $Tin->setpadding(4);

      // Image justify setting
      $Fld = new form_combofield("media_justify", "Justify", $selmedia_first->justify);
      $Fld->setstyle("width:100px");
      $Fld->additem("", "default");
      $Fld->additem("left", "Left");
      $Fld->additem("right", "Right");
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tin->tr($rowbg); 
      $Tin->td("Justify:", "font-weight:bold;");
      $Tin->td($Fld->render());
      $Tin->td_colspan(2);

      // Media display size adjustment
      $mwFld = new form_numberslider("media_width", "Width", $width);
      $mwFld->setstyle("width:300px;");
      $mwFld->set_minmax(0, 800);
      $mwFld->set_discrete_steps(200);
      $mwFld->set_orientation("horizontal");
      $mwFld->show_intermediate_changes();
      $mwFld->set_onchange("setHeightFromWidth(this);");
      
      $hidFlds[] = new form_hiddenfield("media_height", $height);
      if ($sizes_defaulted) {
        $wFld->disabled = true;
      }
      // Image default size toggle
      $mdFld = new form_checkbox("media_size_default");
      $mdFld->setclass("axchkbox");
      $mdFld->checked = ($sizes_defaulted === true);
      $mdFld->set_onclick("defSizingToggle(this);");

      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tin->tr($rowbg);
      $Tin->td("Display size:", "font-weight:bold;");
      $Tin->td($mwFld->render());
      $Tin->td($mdFld->render() . "&nbsp;Default");

      // Image removal button
      $imgRem = new img("$LIBDIR/img/_redx16.gif", "imageremove", "Remove image", 16, 16);
      $imgRem->set_onclick("imgPreview(this);return false;");
      
      // Size info field
      $sziFld = new form_textfield("media_sizeinfo");
      $sziFld->setcss("background-color:#E4DDCA;border:0;width:80px;text-align:center;");
      $sziFld->setvalue("$width x $height");
      $sziFld->editable = false;;

      // Create a preview image..
      $imgFld = new img(
                  $selmedia_first->catalogitem->filepath,
                  "preview",
                  "Preview",
                  $width,
                  $height
                  );
      $imgFld->setstyle("padding-left:1px;padding-right:4px;");
      $imgFld->setalign("left");
      $Tin->tr($rowbg);
      $Tin->td($sziFld->render(), "text-align:right;vertical-align:bottom;");
      $Tin->td($imgFld->render() . $imgRem->render(), "vertical-align:bottom;");
      $Tin->td_colspan(2);

      $Fld = new form_textfield("caption", "Caption", $selmedia_first->caption);
      $Fld->setclass("axtxtbox");
      $Fld->setstyle("width:$stdwidthpx;");
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tin->tr($rowbg);
      $Tin->td("Caption:", "font-weight:bold;");
      $Tin->td($Fld->render());
      $Tin->td_colspan(2);

      $Tin->set_width_profile("15%,55%,30%");

      $Tmed->tr($rowbg);
      $Tmed->td($Tin->render());
      $Tmed->td_alignment("", "top");

      // Now we render the above sub-table $Tmed inside the main table
      $Tst->tr($rowbg);
      $Tst->td($Tmed->render());

      $img_pane = new dojo_contentpane("imagemanager", "Image Manager");
      $img_pane->add_content($Tst->render());
      
      $TAB->add_contentpane($img_pane);
      
      if ($RESPONSE->browser != BROWSER_IE) {
        // This is a hack so that we re-initialise the thumbpicker when it is shown.
        // The current version of the picker doesn't populate itself properly when it
        // is hidden - it just shows the first thumb with no controls. The init script
        // only resets the thumbpicker once, when the user clicks its tab.
        $RESPONSE->head->add_named_script(
            "dojo.connect(dijit.byId('dijit_layout__TabButton_1'), 'onclick', 'init_thumbpicker');\n",
            "dojo_connects"
            );
      }
      // The javascript functions for this page
      $RESPONSE->head->add_script(
            "var realW=$real_width;\n"
          . "var realH=$real_height;\n"
          . "var resizing=false;\n"
          . "function imgPreview(obj) {\n"
          . " var key='';\n"
          . " if(obj.name!=null && obj.name=='imageremove') {\n"
          . "  key='||0|0';\n"
          . " }\n"
          . " else {\n"
          . "  key=obj.link;\n"
          . " }\n"
          . " var sm=dojo.byId('story_media');\n"
          . " sm.value=key;\n"
          . " d=getDopeFromKey(key);\n"
          . " var catid=d[0];\n"
          . " var imgfile=d[1];\n"
          . " var w=d[2];\n"
          . " var h=d[3];\n"
          . " document.$this->formname.preview.src=imgfile;\n"
          . " imgResize(w,h);\n"
          . " realW=w; realH=h;\n"
          . " chkbox=dijit.byId('media_size_default');\n"
          . " chkbox.setAttribute('checked', true);\n"
          . " defSizingToggle(chkbox);\n"
          . "}\n"
          . "function getDopeFromKey(key) {\n"
          . " var dope=new Array('','',0,0);\n"
          . " if(key!=null) {\n"
          . "   var parts=key.split('|');\n"
          . "   dope=new Array(parts[0],parts[1],parts[2],parts[3]);\n"
          . " }\n"
          . " return dope;\n"
          . "}\n"
          . "function imgResize(w,h) {\n"
          . " resizing=true;\n"
          . " var img=document.$this->formname.preview;\n"
          . " img.width=w; img.height=h;\n"
          . " var mh=dojo.byId('media_height');\n"
          . " var mw=dijit.byId('media_width');\n"
          . " var si=dijit.byId('media_sizeinfo');\n"
          . " si.setValue(Math.ceil(w) + ' x ' + Math.ceil(h));\n"
          . " mw.setValue(w);\n"
          . " mh.value=h;\n"
          . " resizing=false;\n"
          . "}\n"
          . "function defSizingToggle(chkbox) {\n"
          . " var mw=dijit.byId('media_width');\n"
          . " var mh=dojo.byId('media_height');\n"
          . " if (chkbox.checked) {\n"
          . "   mw.setValue(realW);\n"
          . "   mh.value=realH;\n"
          . "   mw.setAttribute('disabled', true);\n"
          . "   imgResize(realW,realH);\n"
          . " }\n"
          . " else {\n"
          . "   mw.setAttribute('disabled', false);\n"
          . " }\n"
          . "}\n"
          . "function setHeightFromWidth(mw) {\n"
          . " if(resizing) return;\n"
          . " var mh=dojo.byId('media_height');\n"
          . " var w=mw.getValue();\n"
          . " var h=mh.value;\n"
          . " if(realW > 0) {\n"
          . "   h=Math.round(w * realH/realW);\n"
          . "   mh.value=h;\n"
          . "   imgResize(w,h);\n"
          . " }\n"
          . "}\n"
          . "var init_tp_count=0;\n"
          . "function init_thumbpicker() {\n"
          . " if(init_tp_count<2) {\n"
          . "  var tp=dijit.byId('thumbs');\n"
          . "  var preventCacheOld=tp.preventCache;\n"
          . "  tp.preventCache=true;\n"
          . "  tp.reset(); tp.init();\n"
          . "  init_tp_count+=1;\n"
          . "  tp.preventCache=preventCacheOld;\n"
          . " }\n"
          . "}\n"
          );
    }
    // -=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
    // STORY SETTINGS
    $rowbg = "axbgdark";
    $Tst = new table("storysettings");
    $Tst->setpadding(2);
    $Tst->tr($rowbg);
    $Tst->td($Thd->render(), "border-bottom:1px solid black");
    $Tst->td_colspan(2);

    // STORY DATE & TIME
    $Fld = new form_datefield("story_date", "Story date");
    $Fld->setvalue(timestamp_to_displaydate(ISO_8601_DATE_ONLY, $this->story_date_ts));
    $Fld->setclass("axdatetime");
    $Fld->setstyle("width:150px;");
    $Fld->set_required();
    $Fld->set_format("dd-MM-yyy");
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tdt = new table("sdt");
    $Tdt->tr($rowbg);
    $Tdt->td("Date:");
    $Tdt->td($Fld->render());
    $Fld = new form_timefield("story_time", "Story time");
    $Fld->setvalue(timestamp_to_displaydate(ISO_8601_TIME_ONLY, $this->story_date_ts));
    $Fld->setclass("axdatetime");
    $Fld->setstyle("width:150px;");
    //$Fld->set_format("H:mm");
    $Tdt->td("Time:");
    $Tdt->td($Fld->render());
    $Tst->tr($rowbg);
    $Tst->td("Publish on:", "font-weight:bold;");
    $Tst->td($Tdt->render());

    // EXPIRY DATE
    if ($this->has_expiry) {
      $Fld = new form_datefield("expiry_date", "Expiry date");
      $Fld->setvalue(timestamp_to_displaydate(ISO_8601_DATE_ONLY, $this->expiry_date_ts));
      $Fld->setclass("axdatetime");
      $Fld->setstyle("width:150px;");
      $Fld->set_format("dd-MM-yyyy");
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tdt = new table("edt");
      $Tdt->tr($rowbg);
      $Tdt->td("Date:");
      $Tdt->td($Fld->render());
      $Fld = new form_timefield("expiry_time", "Expiry time");
      $Fld->setvalue(timestamp_to_displaydate(ISO_8601_TIME_ONLY, $this->expiry_date_ts));
      $Fld->setclass("axdatetime");
      $Fld->setstyle("width:150px;");
      //$Fld->set_format("h:mma");
      $Tdt->td("Time:");
      $Tdt->td($Fld->render());
      $Tst->tr($rowbg);
      $Tst->td("Expire on:", "font-weight:bold;");
      $Tst->td($Tdt->render());
    }

    // LANGUAGE
    if ($this->has_multilang) {
      $this->get_story_translations();
      $Tlng = new table("language");

      $Fld = new form_combofield("story_language", "", $this->language);
      $Fld->setstyle("width:$smlwidthpx;");

      // Fill the dropdown selector with all possibilities..
      $q  = "SELECT * FROM ax_language";
      $q .= " WHERE enabled=TRUE";
      $q .= " ORDER BY display_order";
      $langs = dbrecordset($q);
      if ($langs->hasdata) {
        $Fld->add_querydata($langs, "lang_id", "lang_desc");
      }
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tlng->tr($rowbg);
      $Tlng->td($Fld->render());

      $Fld = new form_combofield("new_language");
      $Fld->setstyle("width:$smlwidthpx;");
      $Fld->additem("", "&mdash; Translate into &mdash;");
      // Determine languages already translated..
      $already_translated = array($this->language);
      foreach ($this->story_translations as $sid => $langid) {
        $already_translated[] = $langid;
      }
      $q = "SELECT * FROM ax_language WHERE enabled=TRUE";
      if (count($already_translated) > 0) {
        $langlist = implode(",", $already_translated);
        if ($langlist != "") {
          $q .= " AND NOT lang_id IN (" . implode(",", $already_translated) . ")";
        }
      }
      $q .= " ORDER BY display_order";
      $tlangs = dbrecordset($q);
      if ($tlangs->hasdata) {
        $Fld->add_querydata($tlangs, "lang_id", "lang_desc");
      }
      $transbtn = new form_imagebutton("translate", "", "", "$LIBDIR/img/_translate.gif", "Translate", 77, 15);
      $Tlng->td($Fld->render());
      $Tlng->td_alignment("right");
      $Tlng->td($transbtn->render());
      $Tlng->td_alignment("left");

      if (count($this->story_translations) > 0) {
        $translist = array();
        foreach ($this->story_translations as $trans_story_id => $trans_lang_id) {
          $lq = dbrecordset("SELECT * FROM ax_language WHERE lang_id=$trans_lang_id");
          if ($lq->hasdata) {
            $translist[] = ucfirst($lq->field("lang_desc"));
          }
        }
        if (count($translist) > 0) {
          $Tlng->tr($rowbg);
          $Tlng->td("Existing Translations: " . implode("&nbsp;|&nbsp;", $translist));
          $Tlng->td_colspan(2);
        }
      }
      $Tst->tr($rowbg);
      $Tst->td("Language:", "font-weight:bold;");
      $Tst->td_alignment("", "top");
      $Tst->td($Tlng->render());
    }

    // STORY TYPE
    if ($RESPONSE->ismemberof_group("Editor")) {
      $FldA = new form_radiobutton("story_type", "Article", "a");
      $FldA->setid("story_typeA");
      $FldA->checked = $this->story_type == "a";
      $FldF = new form_radiobutton("story_type", "Feature", "f");
      $FldF->setid("story_typeF");
      $FldF->checked = $this->story_type == "f";
      $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
      $Tin = new table();
      $Tin->setwidth("");
      $Tin->tr();
      $Tin->td($FldA->render());
      $Tin->td("&nbsp;Article&nbsp;&nbsp;");
      $Tin->td($FldF->render());
      $Tin->td("&nbsp;Editorial / feature");
      $Tst->tr($rowbg);
      $Tst->td("Type:", "font-weight:bold;");
      $Tst->td($Tin->render());
    }
    else {
      $hid = new form_hiddenfield("story_type", $this->story_type);
      $hidFlds[] = $hid;
    }

    // COLUMNS
    $Fld1 = new form_radiobutton("story_columns", "Columns", "1");
    $Fld1->setid("story_columns1");
    $Fld1->checked = $this->story_columns == 1;
    $Fld2 = new form_radiobutton("story_columns", "Columns", "2");
    $Fld2->setid("story_columns2");
    $Fld2->checked = $this->story_columns == 2;
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tin = new table();
    $Tin->setwidth("");
    $Tin->tr();
    $Tin->td($Fld1->render());
    $Tin->td("&nbsp;Single&nbsp;&nbsp;");
    $Tin->td($Fld2->render());
    $Tin->td("&nbsp;Double");
    $Tst->tr($rowbg);
    $Tst->td("Columns:", "font-weight:bold;");
    $Tst->td($Tin->render());

    // VISIBLE
    $Fld = new form_checkbox("visible", "Visible");
    $Fld->checked = $this->visible;
    $Fld->setclass("axchkbox");
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td("Visible:", "font-weight:bold;");
    $Tst->td($Fld->render());

    // STORY LOCATIONS
    $Fld = new form_combofield("story_locs", "", $this->story_locs);
    $Fld->multiselect = true;
    $Fld->setclass("axlistbox");
    $Fld->size = 6;
    $Fld->setstyle("width:$stdwidthpx;");
    $q  = "SELECT * FROM ax_content_location";
    $q .= " WHERE enabled";
    $q .= " ORDER BY location_name";
    $locs = dbrecordset($q);
    if ($locs->hasdata) {
      $Fld->add_querydata($locs, "location_id", "location_name");
    }
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td("Publish to:", "font-weight:bold;");
    $Tst->td_alignment("", "top");
    $Tst->td($Fld->render());

    // LAST MODIFIED
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tst->tr($rowbg);
    $Tst->td("Last modified:", "font-weight:bold;");
    $Tst->td($this->lastmodified);

    // Rule-off row..
    $foot_content = "";
    if (count($hidFlds) > 0) {
      foreach ($hidFlds as $hid) {
        $foot_content .= $hid->render();
      }
    }
    $Tst->tr("axfoot");
    $Tst->td($foot_content, "axfoot");
    $Tst->td_colspan(2);

    $Tst->set_width_profile("20%,80%");

    $settings_pane = new dojo_contentpane("storysettings", "Story Settings");
    $settings_pane->add_content($Tst->render());
    $TAB->add_contentpane($settings_pane);

    // Add field validation scripts..
    $RESPONSE->add_scriptsrc("$LIBDIR/js/fieldvalidation.js");

    return $TAB->render();
  } // editform
  // .....................................................................
  /** Render the story as a maintainer reader would view it. Note that this
  * is not a fully dressed-up story viewer. It is designed as a view that
  * a story administrator would see, showing all the technical bits and
  * pieces such as story byte-size etc. You should create your own viewer
  * for rendering stories 'prettily' on your website.
  * @return string The HTML for the view story content.
  */
  function view() {
    global $RESPONSE;
    global $LIBDIR;

    // CONTROL BUTTONS
    $s = "";
    // Buttons for administrators and editors only..
    $doneb = new form_imagebutton("closewin", "Close", "", "$LIBDIR/img/_done.gif", "Close viewer", 57, 15);
    $doneb->set_onclick("window.close()");
    if ($this->user_can_edit()) {
      $editb = new form_imagebutton("storyedit",   "Edit",   "", "$LIBDIR/img/_edit.gif",   "Edit this article",   42, 15);
      $remvb = new form_imagebutton("storyremove", "Delete", "", "$LIBDIR/img/_delete.gif", "Delete this article", 57, 15);
      $remvb->set_onclick("remove_confirm()");
      $s .= $editb->render() . "&nbsp;&nbsp;" . $remvb->render();
      // Removal protection..
      $RESPONSE->head->add_script(
          "function remove_confirm() {\n"
        . " var msg = '\\n\\nWARNING: Do you really want to delete\\n';\n"
        . " msg += 'the article. This is irrevocable.\\n';"
        . " rc = confirm(msg);\n"
        . " if (rc) {\n"
        . "  document.$this->formname.storymode.value='remove';\n"
        . "  document.$this->formname.submit();\n"
        . " }\n"
        . " else alert('Delete is cancelled.');\n"
        . "}\n"
      );
    }
    if ($s != "") $s .= "&nbsp;&nbsp;";
    $s .= $doneb->render();
    $CONTROL_BUTTONS = $s;

    $Tvw = new table("storyviewer");
    $Tvw->setstyle("border: 2px solid #DCDCDC;");

    $rowbg = "axbgdark";

    // EDITOR HEADER
    $Thd = new table("viewerhead");
    $Thd->tr($rowbg);
    $title = $this->story_category_desc;
    $Thd->td("<h3>$title</h3>", "padding-left:6px;");
    $Thd->td($CONTROL_BUTTONS, "text-align:right;vertical-align:bottom;padding-right:6px;");

    if ($this->language != 0) {
      $lq = dbrecordset("SELECT * FROM ax_language WHERE lang_id=$this->language");
      if ($lq->hasdata) {
        $Thd->tr($rowbg);
        $Thd->td("in " . $lq->field("lang_desc"));
        $Thd->td_colspan(2);
      }
    }

    $Tvw->tr($rowbg);
    $Tvw->td($Thd->render(), "border-bottom:1px solid black");
    $Tvw->td_colspan(2);

    if ($this->info_msg != "") {
      $Tvw->tr($rowbg);
      $Tvw->td($this->info_msg, "axerror");
      $Tvw->td_colspan(2);
      $Tvw->td_alignment("center");
    }

    // HEADLINE, BY-LINE, STORY TYPE & WORDCOUNT
    // STORY TYPE
    switch ($this->story_type) {
      case "a": $type = "News article";
        break;
      case "f": $type = "Editorial / feature";
        break;
      default: $type = "";
    }

    $Thd = new table("masthead");
    $Thd->tr($rowbg);
    $Thd->td($this->story_headline, "axstoryheadline");
    $Thd->td($type, "axstoryinfo");
    $Thd->td_alignment("right");
    $Thd->tr($rowbg);
    $byline = "by ";
    $byline .= ($this->story_author_name != "") ? $this->story_author_name : "(anonymous)";
    $Thd->td($byline, "axstorybyline"
      );
    $stats = array();
    if ($this->has_media && count($this->story_media) > 0) {
      $stats[] = count($this->story_media) . " x media";
    }
    $stats[] = $this->wordcount . " words (" . nicebytesize($this->bytesize) . ")";
    $Thd->td(implode(", ", $stats), "axstoryinfo");
    $Thd->td_alignment("right");
    if (isset($this->story_icon)) {
      $Thd->tr();
      $Thd->td($this->render_story_icon());
      $Thd->td_colspan(2);
    }
    $Tvw->tr($rowbg);
    $Tvw->td($Thd->render());
    $Tvw->td_colspan(2);

    // STORY DATE & EXPIRY DATE
    $Tvw->tr($rowbg);
    $Tvw->td(timestamp_to_displaydate(DAY_AND_DATETIME, $this->story_date_ts) . "<br>&nbsp;", "axstoryinfo");
    if ($this->has_expiry && $this->expiry_date != "") {
      if ($this->expiry_date_ts - time() > 0) {
        $expmsg = "Expires: " . timestamp_to_displaydate(DAY_AND_DATETIME, $this->expiry_date_ts)
                . "<br>(in " . prose_diff_ts(time(), $this->expiry_date_ts, true) . ")";
        $Tvw->td($expmsg, "axstoryinfo");
      }
      else {
        $Tvw->td("Expired on " . timestamp_to_displaydate(DAY_AND_DATETIME, $this->expiry_date_ts), "axstoryinfo");
        $Tvw->td_contentcss("axerror");
      }
      $Tvw->td_alignment("right");
    }
    else {
      $Tvw->td("&nbsp;");
    }

    // PUBLISHING STATUS
    $status = "<b>Published to:</b>&nbsp;";
    if (!$this->visible) {
      $status .= "Currently hidden";
    }
    else {
      if (count($this->story_locs) == 0) {
        $status .= "<span class=\"axerror\">No location is selected!</span>";
      }
      else {
        $q .= "SELECT * FROM ax_content_location";
        $q .= " WHERE location_id in (" . implode(",", $this->story_locs) . ")";
        $locs = dbrecordset($q);
        if ($locs->hasdata) {
          $locnames = array();
          do {
            $locnames[] = $locs->field("location_name");
          } while ($locs->get_next());
          $status .= implode(", ", $locnames);
        }
      }
    }
    $Tvw->tr($rowbg);
    $Tvw->td($status, "axstoryinfo");
    $Tvw->td_css("border-top:1px solid black");
    $Tvw->td_colspan(2);

    // LEAD-IN & STORY CONTENT
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tvw->tr($rowbg);

    // Add media
    $media_content = "";
    if ($this->has_media) {
      if (count($this->story_media) > 0) {
        foreach ($this->story_media as $cat_id => $media) {
          $width   = ($media->width  > 0) ? $media->width  : $media->catalogitem->width;
          $height  = ($media->height > 0) ? $media->height : $media->catalogitem->height;
          $caption = $media->caption;
          $pic = new img(
                  $media->catalogitem->filepath,
                  $media->catalogitem->mime_category . $media->catalogitem->cat_id,
                  $caption,
                  ($width  > 0 ? $width  : false),
                  ($height > 0 ? $height : false)
                  );
          $pic->setstyle("border: 1px solid #636363;");
          $pic->setstyle("float: " . ($media->justify != "") ? $media->justify : "right" . ";");

          $s = "<div class=\"axstoryimage\"";
          if ($width > 0) {
            $s .= " style=\"width:$width" . "px;";
            switch ($media->justify) {
              case "left":
                $s .= "float:left;margin-left:0px;margin-right:10px;";
                break;
              default:
                $s .= "float:right;margin-right:0px;margin-left:10px;";
            } // switch
          }
          $s .= "\">";
          $s .= $pic->render();
          if ($caption != "") {
            $s .= "<p class=\"aximagecaption\" style=\"text-align:center\">$caption</p>";
          }
          $s .= "</div>";
          $media_content .= $s;
        } // foreach
      }
    }

    // Story precis & content
    $story_content = "";
    if (trim($this->story_precis) != "") {
      $story_content .= "<p style=\"padding-top:0px;margin-top:0px;\"><span style=\"font-size:116%; font-weight:bold; line-height:130%;\">"
               . $this->story_precis
               . "</span></p>";
    }
    // Add story body
    if (trim($this->story_content) != "") {
      $story_content .= " " . paras2ptags( wikitable2html($this->story_content) );
    }
    
    // Column rendering
    if ($this->story_columns > 1) {
      $Tcols = new table("columns");
      if ($story_content != "") {
        $columns = chunkify($story_content, array(57, 43));
      }
      else {
        $columns = chunkify($story_content, array(51, 49));
      }
      $Tcols->tr();
      $Tcols->td($columns[0], "vertical-align:top;width:360px;padding-right:4px;");
      $Tcols->td($media_content . $columns[1], "vertical-align:top;width:360px;;padding-left:4px;");
      $story_content = $Tcols->render();
    }
    else {
      $story_content = $media_content . $story_content;
    }
    
    $Tvw->td($story_content, "axstorytext");
    $Tvw->td_css("padding-top:15px;padding-bottom:40px;border-top:1px solid black");
    $Tvw->td_colspan(2);

    // TRANSLATIONS
    $this->get_story_translations();
    if (count($this->story_translations) > 0) {
      $RESPONSE->head->add_script(
            "function reloadViewer(url) {\n"
          . " document.location=url;\n"
          . "}\n"
          );
      $translinks = array();
      foreach ($this->story_translations as $trans_story_id => $trans_lang_id) {
        $lq = dbrecordset("SELECT * FROM ax_language WHERE lang_id=$trans_lang_id");
        if ($lq->hasdata) {
          $auth_code = $RESPONSE->get_auth_code();
          $shref = "/story-viewer.php";
          $shref = href_addparm($shref, "story_id", $trans_story_id);
          $shref = href_addparm($shref, "auth_code", $auth_code);
          $href = "javascript:reloadViewer('$shref')";
          $translink = new anchor($href, ucfirst($lq->field("lang_desc")));
          $translinks[] = $translink->render();
        }
      }
      if (count($translinks) > 0) {
        $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
        $Tvw->tr($rowbg);
        $Tvw->td("Translations: " . implode("&nbsp;|&nbsp;", $translinks), "axstoryinfo");
        $Tvw->ts_css("border-top:1px solid black");
        $Tvw->td_colspan(2);
      }
    }

    // LAST MODIFIED
    $rowbg = ($rowbg == "axbglite") ? "axbgdark" : "axbglite";
    $Tvw->tr($rowbg);
    $Tvw->td("Last modified:&nbsp;$this->lastmodified", "axstoryinfo");
    $Tvw->td_css("border-top:1px solid black");
    $Tvw->td("#$this->story_id", "axstoryinfo");
    $Tvw->td_css("border-top:1px solid black; text-align:right;");

    // Rule-off row..
    $Tvw->tr("axfoot");
    $Tvw->td("", "axfoot");
    $Tvw->td_colspan(2);

    return $Tvw->render();
  } // view
  // .....................................................................
  /**
  * Return the content of this story formatted for plaintext display
  * @param integer $wrapchars Number of characters to wrap the lines at
  */
  function plaintext_content($wrapchars=0) {
    // Join all hard-breaks into single lines..
    $content = str_replace("\n", " ", $this->story_content);
    // Split into paragraphs..
    $paras = explode("<p>", $content);
    // Wrap each paragrph if required..
    if ($wrapchars > 0) {
      $newparas = array();
      foreach ($paras as $para) {
        $para = wordwrap($para, $wrapchars, "\r\n");
        $newparas[] = $para;
      }
      $paras = $newparas;
    }
    // Join up into multiple paragraphs split by CRLF..
    $content = strip_tags( implode("\r\n\r\n", $paras) );
    return $content;
  } // plaintext_content
  // .....................................................................
  /** Render the story. We render the story as a table within a form containing all
  * the form elements required to manipulate the story content, email it to
  * someone, save it, and delete it etc...
  * @return string The HTML for edit or view.
  */
  function html() {
    global $RESPONSE;
    // HIDDEN FIELDS
    $cathid  = new form_hiddenfield("cat",       $this->story_category);
    $authhid = new form_hiddenfield("auth_code", $RESPONSE->auth_code);
    $modehid = new form_hiddenfield("storymode", $this->storymode);
    $sidhid  = new form_hiddenfield("story_id",  $this->story_id);

    // STORY FORM, VIEW or EDIT..

    switch ($this->storymode) {
      case "edit":
      case "adding":
        $story_form = new multipart_form($this->formname);
        $story_form->add_text($this->editform());
        break;
      default:
        $story_form = new form($this->formname);
        $story_form->add_text($this->view());
    } // switch

    // Render hidden fields too..
    $story_form->add($cathid);
    $story_form->add($authhid);
    $story_form->add($modehid);
    $story_form->add($sidhid);

    return $story_form->render();
  } // story html

} // story class

// -----------------------------------------------------------------------
/**
* A container class for media item associated with a story. Contains
* a single piece of media which is associated with this story.
* @package cm
*/
class story_media {
  /** ID of story this media belongs to */
  var $story_id = false;
  /** The catalogitem object */
  var $catalogitem;
  /** The caption for this item */
  var $caption = "";
  /** The way to justify this item */
  var $justify = "";
  /** Local override width */
  var $width = 0;
  /** Local override height */
  var $height = 0;
  // .....................................................................
  /**
  * Create a new piece of story media. This comprises a catalogitem
  * object, and a set of methods to access it.
  * @param mixed $id Story ID, or false if not known
  * @param mixed $item Object catalogitem, or false if initially undefined
  */
  function story_media($story_id=false, $item=false) {
    if ($story_id !== false) {
      $this->story_id = $story_id;
    }
    if ($item !== false && is_object($item)) {
      $this->catalogitem = $item;
    }
    else {
      $this->catalogitem = new catalogitem();
    }
  } // story_media
  // .....................................................................
  /**
  * Define this story media object from the given catalog item key. This
  * will obtain the given piece of catalog media from the database and
  * assign the object variables accordingly.
  * @param integer $catid Catalog item ID to obtain
  */
  function get_catalogitem($catid) {
    $this->catalogitem = new catalogitem($catid);
  } // get_catalogitem
  // .....................................................................
  /**
  * Return the keyinfo string for this media item. This is encoded
  * as follows, and is used in select combos:
  *   'cat_id|filepath|width|height|justify'
  */
  function keyinfo() {
    $info = "||||";
    if (isset($this->catalogitem)) {
      $info = $this->catalogitem->keyinfo();
      $info .= "|" . $this->justify;
    }
    return $info;
  } // keyinfo
} // story_media class

// -----------------------------------------------------------------------
?>