<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    sq-schema-defs.php                                      */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing SQLite DATABASE SCHEMAS.       */
/*                                                                      */
/*              The main use of this class is to read in database info  */
/*              for other utilities, such as table maintenance scripts. */
/*                                                                      */
/* ******************************************************************** */
/** @package database */

/** Generic schema classes */
include_once("schema-defs.php");

// ----------------------------------------------------------------------
/** Defines a database sequence.
* @package database
*/
class DB_dbsequence extends dbsequence {
  // ....................................................................
  function DB_dbsequence(&$schema, $name) {
    $this->dbsequence($schema, $name);
  }
} // class DB_dbsequence

// ----------------------------------------------------------------------
/** Defines a database function (procedure).
* @package database
*/
class DB_dbfunction extends dbfunction {
  // ....................................................................
  function DB_dbfunction(&$schema, $name, $returns="", $src="", $args="", $lang="") {
    $this->dbfunction($schema, $name, $returns, $src, $args, $lang);
  }
} // class DB_dbfunction

// ----------------------------------------------------------------------
/** Defines a database index.
* @package database
*/
class DB_dbindex extends dbindex {
  // ....................................................................
  function DB_dbindex(&$schema, $name, $tablename, $flds="", $primary=false, $unique=false) {
    $this->dbindex($schema, $name, $tablename, $flds, $primary, $unique);
  }
} // class DB_dbindex

// ----------------------------------------------------------------------
/** Defines a database constraint.
* @package database
*/
class DB_dbconstraint extends dbconstraint {
  // ....................................................................
  function DB_dbconstraint(&$schema, $name, $type="p", $table="", $fktable="", $flds="",
                        $fkflds="", $updact="", $delact="", $match="", $cksrc="") {
    $this->dbconstraint($schema, $name, $type, $table, $fktable, $flds, $fkflds, $updact, $delact, $match, $cksrc);
  }
} // class DB_dbconstraint

// ----------------------------------------------------------------------
/** Defines a database trigger.
* @package database
*/
class DB_dbtrigger extends dbtrigger {
  // ....................................................................
  function DB_dbtrigger(&$schema, $name, $bitmask=0, $table="", $funcname="", $args="") {
    $this->dbtrigger($schema, $name, $bitmask, $table, $funcname, $args);
  }
} // class DB_dbtrigger

// ----------------------------------------------------------------------
/** Class describing a database field
* @package database
*/
class DB_dbfield extends dbfield {
  // ....................................................................
  function DB_dbfield(&$schema, $name, $num, $type, $defaultval="", $notnull=false, $ispkey=false) {
    $this->dbfield($schema, $name, $num, $type, $defaultval, $notnull, $ispkey);
  }
  // ....................................................................
  /**
  * Return the generic type of the field. The generic types are as
  * follows:
  *   text       Fixed or varying length strings
  */
  function generic_type() {
    $gtype = "";
    $typematch = array(
      "text"     => "text",
      "numeric"  => "integer|integer primary key",
      "logical"  => "boolean",
      ""         => ".*"
      );

    foreach ($typematch as $gentype => $pattern) {
      if (preg_match("/$pattern/i", $this->type)) {
        $gtype = $gentype;
        break;
      }
    }
    return $gtype;
  }
  // ....................................................................
  /**
  * Return true if the field is of an integer class. We exclude the serial
  * types due to the initial purpose of this method - to determine fields
  * which are standalone integer types, without attached sequences.
  * NB: Override this function to get schema info per DB type.
  */
  function is_integer_class() {
    $pattern = "integer|integer primary key";
    return preg_match("/$pattern/i", $this->type);
  }
  // ....................................................................
  /**
  * Return true if the field is of a 'serial' class. This is a pseudo
  * class of types which encapsulates integer fields which are able
  * to auto-increment themselves when records are inserted.
  * NB: Override this function to get schema info per DB type.
  */
  function is_serial_class() {
    $pattern = "integer primary key";
    return preg_match("/$pattern/i", $this->type);
  }
} // class DB_dbfield
// ----------------------------------------------------------------------
/** Class describing a Postgres database table. Inherits the standard
* dbtable class properties and methods, but adds in the getschema
* specifics for acquiring Postgres table info from the metadata, and
* provides a specific create() method.
* @package database
*/
class DB_dbtable extends dbtable {
  // ....................................................................
  /** Construct a table of given name and array of primary key fields.
  * @param string $name The name of the table
  * @param integer $dbversion Optional database version information
  */
  function DB_dbtable(&$schema, $name) {
    $this->dbtable($schema, $name);
  }
} // class DB_dbtable

// ----------------------------------------------------------------------
/**
* Class describing a Postgres database schema.
* @package database
*/
class DB_schema extends schema {
  // ....................................................................
  /**
  * Create a schema (database) of given name & type. The name should be a
  * valid existing database name that is currently connected.
  * @param string $name Name of this particular database
  */
  function DB_schema($name) {
    $this->schema($name, "SQLite");
  }
  // ....................................................................
  /**
  * Return database capabilities. There are specific capabilities which
  * the diff code needs to query, and this method should be overridden
  * in the specific database module to answer those questions.
  */
  function capable_of($capability="") {
    $cando = false;
    switch ($capability) {
      // Supports the ALTER <tablename> DROP <colname> SQL
      // statement to remove table columns.
      case "alter_table_drop_column":
        $cando = false;
        break;
      // Supports functions or stored procedures.
      case "stored_procedures":
        $cando = false;
        break;
      // Can define check constraints on table columns.
      case "check_constraints":
        $cando = false;
        break;
      // Can define RI constraints between table/columns to
      // support foreign-keys.
      case "RI_constraints":
        $cando = false;
        break;
      // Supports indexes on table columns.
      case "indexes":
        $cando = false;
        break;
      // Unique indexes are auto-generated with unique constraints. Ie.
      // when a primary key constraint is added to a table a unique
      // index is automatically built for it.
      case "unique_index_with_constraint":
        $cando = false;
        break;
      // Supports triggers on table update, delete, insert.
      case "triggers":
        $cando = false;
        break;
      // Supports named sequences.
      case "named_sequences":
        $cando = false;
        break;
      default:
        $cando = false;
    }
    return $cando;
  }
  // ....................................................................
  /** Acquire the Postgres database version. We make some gross assumptions
  * here with regard to standard local Postgres setup. Change as necessary.
  * Currently we are assuming the Debian standard location which can be
  * over-ridden with a passed-in parameter. Then we try for the newer multi-
  * version Postgresql arrangement where the major.minor version number
  * (eg. '8.2') is used as the root directory, before falling back to the
  * older single-version arrangement.
  */
  function getversion($rootdir="") {
    $vstr = 7.4;
    $pg_root =  ($rootdir == "") ? "/usr/lib/postgresql" : $rootdir;
    $pg_bin = "$pg_root/bin";

    # Reassign pg_root if we have multiversion hierarchies..
    $dirlisting = get_dirlist($pg_root, false, "/[0-9]+.[0-9]+/");
    if (count($dirlisting) > 0) {
      sort($dirlisting, SORT_NUMERIC);
      reset($dirlisting);
      $pg_root = array_pop($dirlisting);
    }
    
    # Set the postgres binaries dir..    
    $pg_bin = "$pg_root/bin";
    
    // Get the database version..
    if (file_exists("$pg_bin/pg_config")) {
      $vbits = explode(" ", shell_exec("$pg_bin/pg_config --version"));
      if ($vbits[1] != "") {
        $vstr = $vbits[1];
      }
    }
    elseif (file_exists("$pg_bin/psql")) {
      $vbits = explode(" ", shell_exec("$pg_bin/psql --version"));
      if ($vbits[2] != "") {
        $vvbits = explode(".", $vbits[2]);
        $vstr = $vvbits[0] . "." . (isset($vvbits[1]) ? $vvbits[1] : "0");
      }
    }
    $this->set_dbversion( (float) $vstr );
    debugbr("$this->database_server version detection", DBG_DEBUG);
    debugbr("database version set to $this->database_version", DBG_DEBUG);
    return $this->database_version;
  }

} // class DB_schema

// ----------------------------------------------------------------------
?>