<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    recmaint-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for record maintenance in the browser.      */
/*                                                                      */
/* This class provides a convenient way of encapsulating the required   */
/* javascript and hiddne fields to enable in-browser maintenance of a   */
/* recordset.                                                           */
/*                                                                      */
/* The concept is that you will have a combo/select form element which  */
/* is a list of record keys called the 'keycombo' below, together with  */
/* a set of additional form fields of whatever type, to maintain the    */
/* fields in the record.                                                */
/*                                                                      */
/* When you create an object of 'recmaintainer' class, you specify the  */
/* name of the form containing your keycombo and also pass the actual   */
/* keycombo object (as a reference). After that, you associate your     */
/* form field objects using the register_field() method, and special    */
/* buttons with the register_button() method. NOTE: you *must* give     */
/* your form fields names which are the same as the DB field names of   */
/* the fields they are maintaining!                                     */
/*                                                                      */
/* The record data is then added using add_record(). This takes an      */
/* associative array of fields as a parameter and these fields should   */
/* obviously match your form fields in naming. Also make sure that      */
/* the order you add the records is the same as the order of the keys   */
/* in your keycombo select box.                                         */
/*                                                                      */
/* Somewhere at the bottom of the aforementioned form, you then call    */
/* the render() method. This puts the Javascript into the page, and     */
/* some hidden fields for when you save data and POST it.               */
/*                                                                      */
/* POST processing: when the form gets posted, some special form fields */
/* are sent and these are as follows..                                  */
/*  _recmaintpost_form:  contains the posted form name                  */
/*  _recmaintpost_flds:  contains a list of field names in each record  */
/*  _recmaintpost_data:  contains all changed records data              */
/*  _recmaintpost_dels:  contains keys of deleted records, if any       */
/*  _recmaintpost_order: contains keys in order (but only if reordered) */
/*                                                                      */
/* Lists are sent back as a string delimited by the special ASCII chars */
/* shown in the definitions below this comment block. The exception is  */
/* the recmaintpost_flds which uses commas as the delimiter.            */
/*                                                                      */
/* One additional point regarding the naming of the above vars. We      */
/* provide for multiple of these forms on a single page, by allowing    */
/* a prefix. The above names have no prefix, which is the default       */
/* option, but if more than one recmaint was present a prefix would be  */
/* added in front of the underscore, to differentiate the vars.         */
/*                                                                      */
/* ******************************************************************** */
/** @package recmaint */

// ----------------------------------------------------------------------
// Anything which uses recmaint has to have this record maintenance
// module Javascript in the page. If you are not using Axyl's
// RESPONSE object, then make sure you insert this into your page..
if (isset($RESPONSE)) {
  $RESPONSE->head->add_script(
      "var keyfield = new Array();\n"
    . "var curid = new Array();\n"
    . "var newid = new Array();\n"
    );
  $RESPONSE->head->add_scriptsrc("$LIBDIR/js/recmaint.js");
}
/**
* The record maintainer class.
* @package recmaint
*/
class recmaintainer extends RenderableObject {
  // Public
  // Private
  /** Name of the form containing maintained fields
      @access private */
  var $formname;
  /** Prefix to differentiate one set of records from anther
      @access private */
  var $prefix = "";
  /** If true, maintainer will be displayed with fields disabled
      @access private */
  var $start_disabled = false;
  /** Pointer to the key combo/select box
      @access private */
  var $keycombo;
  /** Internal array of field names being maintained
      @access private */
  var $fields = array();
  /** Record data to maintain
      @access private */
  var $records = array();
  /** Field defaults for new records
      @access private */
  var $defaults = array();
  /** Value, if defined, of the record to focus to on initialisation
      @access private */
  var $initial_record_value;
  // ....................................................................
  /**
  * Define a record maintainer object.
  * The form name, key combo (select) box, and the list of fields
  * it will be maintaining must be specified.
  * @param string $formname Name of the form containing maintenance fields
  * @param object $keycombo The form_combofield object containing record keys
  * @param string $fieldlist The name sof all fields being maintained
  * @param string $prefix Prefix to identify multiple recmaintainers uniquely
  */
  function recmaintainer($formname, &$keycombo, $prefix="_") {
    $this->formname = $formname;
    $this->keycombo = $keycombo;
    $this->prefix = $prefix;
    $keycombo->set_onchange("changeRecord('$formname',this,'$this->prefix')");
  } // recmaintainer
  // ....................................................................
  /**
  * Register a button.
  * This assigns the proper onclick event so that the button will do the
  * right thing when clicked. There are several types of button recognised:
  * "up", "down", "save", "reset" and "del".
  * NB: If the button in question already has an onclick script attached to
  * it then this is preserved, and will be executed after the new script
  * we attach here.
  * @param string $type Type of button being registered
  * @param object The button object to be registered
  */
  function register_button($type, &$button) {
    $new_onclick = "";
    switch ($type) {
      // Move record up one place in listbox..
      case "up":
        $new_onclick = "comboMove('$this->formname','" . $this->keycombo->name . "','up')";
        break;
      // Move record down one place in listbox..
      case "down":
        $new_onclick = "comboMove('$this->formname','" . $this->keycombo->name . "','down')";
        break;
      // Delete record from the maintainer..
      case "del":
      case "delete":
        $new_onclick = "deleteRecord('$this->formname','$this->prefix')";
        break;
      // Add a new record to the maintainer..
      case "add":
        $new_onclick = "addRecord('$this->formname','$this->prefix')";
        break;
      // General-purpose save-data and submit form button..
      case "save":
        $new_onclick = "saveRecords('$this->formname','$this->prefix')";
        break;
      // Special-case store-only button, for when custom behviour reqd..
      case "store":
        $new_onclick = "storeRecords('$this->formname','$this->prefix')";
        break;
      // Reset the form..
      case "reset":
        $new_onclick = "document.$this->formname.reset()";
        break;
    } // switch
    // Set new onclick script..
    $button->set_onclick($new_onclick, SCRIPT_PREFIX);
  } // register_button
  // ....................................................................
  /**
  * Register a form field in the maintainer.
  * The object passed should be a normal Axyl form object such as a
  * form_textfield, form_checkbox or whatever. Add all of the form
  * objects that you have in your maintenance screen.
  * @param object $field A form element eg: a form_textfield object
  * @param string $fieldid HTML id attribute to assign to form element
  */
  function register_field(&$field, $fieldid="fdata") {
    $this->fields[] = $field->name;
    $field->set_onchange("changedValue('$this->formname',this,'$this->prefix')");
    $field->setid($this->keycombo->name . "_" . $fieldid);
  } // register_field
  // ....................................................................
  /**
  * Add a record of data to the maintainer.
  * The record is passed as an associative array, which is directory
  * compatible with the rows returned by get_next() etc. in normal
  * database queries. The array is therefore expected to contain the
  * normal fieldname=>data pairs. The ordering of adding these records
  * should be identical to the ordering of the records in the keycombo.
  * @param string $keyval The key identifier for this record
  * @param array $record An associative arrray containing a record of data
  */
  function add_record($keyval, $record) {
    $this->records[$keyval] = $record;
  } // add_record
  // ....................................................................
  /**
  * Add an array of default values for each field.
  * The array is a key=>value associative array, with the key = fieldname,
  * and the value = default value for field.
  * @param array $defaults An associative arrray containing field defaults
  */
  function add_defaults($defaults) {
    $this->defaults = $defaults;
  } // add_defaults
  // ....................................................................
  /**
  * Cause the maintainer to be displayed initially diabled. All form fields
  * which are part of the maintainer will be readonly/disabled.
  */
  function display_disabled() {
    $this->start_disabled = true;
  } // display_disabled
  // ....................................................................
  /**
  * Tell the record maintainer to focus the keycombo to the record with
  * the given value, when it first initialises.
  */
  function initial_record($value) {
    $this->initial_record_value = $value;
  } // initial_record
  // ....................................................................
  /**
  * Return the HTML for this maintainer. This consists of Javascript
  * objects to contain the data, and hidden form fields which will be
  * used to POST changes etc. This should be rendered inside your
  * main form element.
  * @param string $prefix A prefix to use in field-naming for uniqueness
  */
  function html() {
    global $RESPONSE;

    $dat_pfx = $this->prefix . "dat_";
    $new_pfx = $this->prefix . "new_";
    $rec_pfx = $this->prefix . "rec_";

    $fieldlist = implode(",", $this->fields);
    $s = "";
    $keyvals = array();
    $data_js  = "function $rec_pfx$this->formname($fieldlist) {\n";
    foreach ($this->fields as $fieldname) {
      $data_js .= "this.$fieldname=$fieldname;\n";
    }
    $data_js .= "}\n";
    $data_js .= "function $new_pfx$this->formname(id) {\n";
    $data_js .= "$dat_pfx$this->formname[id] = new datarec(id,";
    $data_js .= "new $rec_pfx$this->formname(";
    $args = array();
    foreach ($this->fields as $fieldname) {
      $args[] = "'" . rawurlencode($this->defaults[$fieldname]) . "'";
    }
    $data_js .= implode(",", $args);
    $data_js .= "),'$this->prefix');\n";
    $data_js .= "$dat_pfx$this->formname[id].changed=true;\n";
    $data_js .= "}\n";
    $data_js .= "var $dat_pfx$this->formname = new Object();\n";
    foreach ($this->records as $keyval => $rec) {
      $data_js .= "$dat_pfx$this->formname['$keyval'] = new datarec('$keyval',";
      $data_js .= "new $rec_pfx$this->formname(";
      $args = array();
      foreach ($this->fields as $fieldname) {
        $fldval = $rec[$fieldname];
        if (isset($RESPONSE) && $RESPONSE->multilang && function_exists("utf8_encode")) {
          //debugbr("block recmaint: field $fieldname MULTILANG");
          $fldval = utf8_ensure($fldval);
          $patts = array("/[\n]/", "/[']/", "/[\012\015]/");
          $repls = array("%0d%0a", "%27", "");
          $fldval = preg_replace($patts, $repls, $fldval);
          $fldval = utf8RawUrlEncode($fldval);
          $args[] = "'$fldval'";
        }
        else {
          $fldval = rawurlencode($fldval);
          $args[] = "'$fldval'";
        }
      }
      $data_js .= implode(",", $args);
      $data_js .= "),'$this->prefix');\n";
    } // foreach
    // This contains the posted form name..
    $hid = new form_hiddenfield($this->prefix . "recmaintpost_form", $this->formname);
    $s .= $hid->render();
    // This contains the field names in each record..
    $hid = new form_hiddenfield($this->prefix . "recmaintpost_flds", $fieldlist);
    $s .= $hid->render();
    // This field is posted with data of changed records..
    $hid = new form_hiddenfield($this->prefix . "recmaintpost_data");
    $s .= $hid->render();
    // This field is posted with keys of deleted records..
    $hid = new form_hiddenfield($this->prefix . "recmaintpost_dels");
    $s .= $hid->render();
    // This field is posted with keys in new order if changed..
    $hid = new form_hiddenfield($this->prefix . "recmaintpost_order");
    $s .= $hid->render();

    // Inject the Javascript to the page..
    if (isset($RESPONSE)) {
      $RESPONSE->body->add_script(
          "keyfield['$this->prefix']='" . $this->keycombo->name . "';\n"
        . "curid['$this->prefix']=-1;\n"
        . "newid['$this->prefix']=10000;\n"
        . "$data_js"
        );
      if ($this->start_disabled || count($this->records) == 0) {
        $RESPONSE->foot->add_script(
            "disable('$this->formname',true,'" . $this->keycombo->name . "');\n"
            );
      }
      else {
        if (isset($this->initial_record_value)) {
          foreach ($this->records as $keyval => $rec) {
            if ($keyval == $this->initial_record_value)
              break;
          }
          $RESPONSE->foot->add_script(
              "focusRecordByValue('$this->formname','$this->prefix','$this->initial_record_value');\n"
              );
        }
        else {
          foreach ($this->records as $keyval => $rec) { break; }
          $RESPONSE->foot->add_script(
              "firstRecord('$this->formname','$this->prefix');\n"
              );
        }
      }
    }
    else {
      $s .= "<script language=\"javascript\">\n";
      $s .= "keyfield['$this->prefix']='" . $this->keycombo->name . "';\n";
      $s .= "curid['$this->prefix']=-1;\n";
      $s .= "newid['$this->prefix']=10000;\n";
      $s .= $data_js;
      if ($this->start_disabled || count($this->records) == 0) {
        $s .= "disable('$this->formname',true,'" . $this->keycombo->name . "');\n";
      }
      else {
        if (isset($this->initial_record_value)) {
          foreach ($this->records as $keyval => $rec) {
            if ($keyval == $this->initial_record_value)
              break;
          }
          $s .= "focusRecordByValue('$this->formname','$this->prefix','$this->initial_record_value');\n";
        }
        else {
          foreach ($this->records as $keyval => $rec) { break; }
          $s .= "firstRecord('$this->formname','$this->prefix');\n";
        }
      }
      $s .= "</script>\n";
    }
    // Return the other page content..
    return $s;
  } // html
} // class recmaintainer

// ----------------------------------------------------------------------
/**
* Utility function. Returns the value of the named field. Returns the
* value of the named field which is in the data array $rec. Uses the
* array $fields to determine the position of the named field in the
* data record.
* @param string $fldname Name of field to fetch value of.
* @param array $fields Array of fields in the record.
* @param array $rec Array of data
*/
function get_recmaintfield($fldname, $fields, $rec) {
  $val = "";
  $ix = 0;
  foreach ($fields as $fld) {
    if ($fld == $fldname) {
      $val = $rec[$ix];
      break;
    }
    $ix += 1;
  }
  return $val;
} // get_recmaintfield

// ----------------------------------------------------------------------
?>