<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    metadata-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Various bits n pieces for maintaining metadata.         */
/*                                                                      */
/* ******************************************************************** */
/** @package cm */

/** Record maintainer */
include_once("recmaint-defs.php");

// -----------------------------------------------------------------------
// Width of form elements..
$ewidth = "250px";
$cwidth = "150px";
$pwidth = "90px";

// -----------------------------------------------------------------------
/**
* A class which is derived from the generic tag class, and which is
* specifically for the HTML META tag.
* @package cm
*/
class meta_tag extends HTMLtag {
  /** Constructor. Creates an empty HTML meta tag object. */
  function meta_tag() {
    $this->HTMLtag("meta");
  }
} // meta_tag class

// -----------------------------------------------------------------------
/**
* A class which encapsulates a meta tag which itself contains particular
* information, which may be in a specified language, and may have been
* picked from a defined vocabulary/encoding scheme.
* @package cm
*/
class data_meta_tag extends meta_tag {
  /**
  * Constructor
  * @param string $name The name of the metadata item
  * @param string $content The metadata information content itself
  * @param string $lang The language the content is in
  * @param string $scheme The scheme/vocabulary the content was picked from
  */
  function data_meta_tag($name, $content, $lang="", $scheme="") {
    $this->meta_tag();
    $this->add_attribute("name", $name);
    $this->add_attribute("content", $content);
    if ($lang != "") {
      $this->add_attribute("lang", $lang);
    }
    if ($scheme != "") {
      $this->add_attribute("scheme", $scheme);
    }
  } // data_meta_tag
} // data_meta_tag class

// -----------------------------------------------------------------------
/**
* A class which encapsulates a meta tag which represents a reference to
* an item using a URL/URI.
* @package cm
*/
class uri_meta_tag extends meta_tag {
  /**
  * Constructor
  * @param string $rel The name of the metadata item for this reference
  * @param string $href The URL/URI for this reference
  * @param string $hreflang The language the refernced item is in
  */
  function uri_meta_tag($rel, $href, $hreflang="") {
    $this->meta_tag();
    $this->add_attribute("rel", $rel);
    $this->add_attribute("href", $href);
    if ($hreflang != "") {
      $this->add_attribute("hreflang", $hreflang);
    }
  } // uri_meta_tag
} // uri_meta_tag class

// -----------------------------------------------------------------------
/**
* A class which encapsulates a scheme qualifier. A qualifier is an
* attribute which is used in a metadata tag to qualify or refine it.
* Qualifiers can have default values or allowed lists of values, but
* essentially they just allow the user to enter a string of content
* against a named attribute in the meta tag.
* @package cm
*/
class enc_qualifier extends RenderableObject {
  /** Whether object contains valid data */
  var $valid = false;
  /** Unique encoding scheme ID */
  var $scheme_id = "";
  /** Name of this qualifier */
  var $qual_name = "";
  /** Label for qualifier */
  var $qual_label = "";
  /** Blurb regarding this qualifier */
  var $comments = "";
  /** Default value of this qualifier */
  var $default_value = "";
  /** Allowed list of values for this qualifier */
  var $list_of_values = "";
  /** Display order */
  var $display_order = 0;
  // .....................................................................
  /**
  * Constructor
  * @param integer $scheme_id Unique ID of this scheme
  * @param string $qualname Name of this qualifier
  */
  function enc_qualifier($scheme_id="", $qualname="") {
    $this->get($scheme_id, $qualname);
  } // enc_qualifier
  // .....................................................................
  /**
  * Get the data for this qualifier.
  * @param integer $scheme_id The unique ID of the scheme to get
  * @param string $qualname Name of this qualifier
  */
  function get($scheme_id="", $qual_name="") {
    $this->valid = false;
    if ($scheme_id != "") {
      $this->scheme_id = $scheme_id;
    }
    if ($qual_name != "") {
      $this->qual_name = $qual_name;
    }
    if ($this->scheme_id != "" && $this->qual_name != "") {
      $q  = "SELECT * FROM ax_enc_qualifier";
      $q .= " WHERE enc_scheme_id=$this->scheme_id";
      $q .= "   AND qual_name='" . escape_string($this->qual_name) . "'";
      $qual = dbrecordset($q);
      if ($qual->hasdata) {
        $this->qual_label     = $qual->field("qual_label");
        $this->comments       = $qual->field("comments");
        $this->default_value  = $qual->field("default_value");
        $this->list_of_values = $qual->field("list_of_values");
        $this->display_order  = $qual->field("display_order");
        $this->valid = true;
      }
    }
    return $this->valid;
  } // get

} // enc_qualifier class

// -----------------------------------------------------------------------
/**
* A class which encapsulates a metadata scheme. This is a set of
* rules or a set of values (a vocabulary) designed to help people
* when entering metadata. This class holds the details of the
* scheme (name, tag, description), link(s) to resources on the scheme
* and possibly values for it.
* @package cm
*/
class metadata_scheme extends RenderableObject {
  /** Whether object contains valid data */
  var $valid = false;
  /** Unique encoding scheme ID */
  var $scheme_id = "";
  /** Name of this encoding scheme */
  var $enc_scheme_name = "";
  /** Label for scheme */
  var $label = "";
  /** Scheme name to be used in meta tags */
  var $tag_name = "";
  /** A description of the scheme */
  var $description = "";
  /** URI of scheme data */
  var $datasrc_uri = "";
  /** Reference URL for this scheme */
  var $reference_url = "";
  /** Array of scheme key/value pairs */
  var $values = array();
  /** Array of scheme qualifier objects */
  var $qualifiers = array();
  /** Whether this is the 'preferred' scheme, of many */
  var $preferred = false;
  // .....................................................................
  /**
  * Constructor
  * @param integer $scheme_id Unique ID of this scheme
  */
  function metadata_scheme($scheme_id="") {
    $this->get($scheme_id);
  } // metadata_scheme
  // .....................................................................
  /**
  * Get the data for this scheme.
  * @param integer $scheme_id The unique ID of the scheme to get
  * @return boolean Whether the get succeeded and object is valid
  */
  function get($scheme_id="") {
    $this->valid = false;
    if ($scheme_id != "") {
      $this->scheme_id = $scheme_id;
    }
    if ($this->scheme_id != "") {
      $q  = "SELECT * FROM ax_enc_scheme";
      $q .= " WHERE enc_scheme_id=$this->scheme_id";
      $scheme = dbrecordset($q);
      if ($scheme->hasdata) {
        $this->enc_scheme_name = $scheme->field("enc_scheme_name");
        $this->label           = $scheme->field("label");
        $this->tag_name        = $scheme->field("tag_name");
        $this->description     = $scheme->field("description");
        $this->datasrc_uri     = $scheme->field("datasrc_uri");
        $this->reference_url   = $scheme->field("reference_url");
        $this->valid = true;
      }
      // Data values..
      $this->values = array();
      $q  = "SELECT * FROM ax_enc_value";
      $q .= " WHERE enc_scheme_id=$this->scheme_id";
      $q .= " ORDER BY enc_label";
      $enc_values = dbrecordset($q);
      if ($enc_values->hasdata) {
        do {
          $enc_label = $enc_values->field("enc_label");
          $enc_value = $enc_values->field("enc_value");
          $this->values[$enc_label] = $enc_value;
        } while ($enc_values->get_next());
      }
      // Qualifiers..
      $q  = "SELECT qual_name FROM ax_enc_qualifier";
      $q .= " WHERE enc_scheme_id=$this->scheme_id";
      $q .= " ORDER BY display_order";
      $enc_quals = dbrecordset($q);
      if ($enc_quals->hasdata) {
        do {
          $qual_name = $enc_quals->field("qual_name");
          $qual = new enc_qualifier($this->scheme_id, $qual_name);
          if ($qual->valid) {
            $this->qualifiers[$qual_name] = $qual;
          }
        } while ($enc_quals->get_next());
      }
    }
    return $this->valid;
  } // get

  // .....................................................................
  /**
  * Render this scheme as HTML. This produces a table containing the
  * schem in a format suitable for a metadata definition form. If only
  * a reference URL, then a simple URL is used, if values are available
  * then a combo is used, if a qualifier set is defined, then a special
  * qualifier assembly widget is displayed. In the latter two cases,
  * an end-result for content s produced, and you can pass in the name
  * of a javascript function to call to insert the result.
  * @param string $formname Name of the form this will go into
  * @param string $fieldname Name of form field to update when content defined
  * @param string $fieldvalue Initial value of form field we are updating
  * @param string $schemefield Name of form field of scheme to update
  */
  function definition_form($formname="", $fieldname="", $fieldvalue="", $schemefield="") {
    global $RESPONSE, $LIBDIR, $ewidth, $cwidth, $pwidth;
    $Tsch = new table("scheme_" . $this->scheme_id);
    $Tsch->setwidth("100%");
    $Tsch->setpadding(2);

    $Tsch->tr("axbglite");
    $title = "<b>$this->label</b>";

    if ($this->reference_url != "") {
      if ($this->tag_name != "") {
        $URLlabel = "($this->tag_name)";
      }
      else {
        $URLlabel = "(see reference)";
      }
      $schref = new anchor($this->reference_url, $URLlabel);
      $schref->settarget("_new");
      $title .= "&nbsp;" . $schref->render();
    }
    elseif ($this->tag_name != "") {
      $title .= " ($this->tag_name)";
    }
    if ($this->preferred) {
      $title .= " <span class=\"axhl\"><small>*preferred*</small></span>";
    }
    $Tsch->td($title, "axbglite");
    $Tsch->td_colspan(2);
    $Tsch->td_width("70%");
    $maxwords = 15;
    $words = explode(" ", $this->description);
    if (count($words) > $maxwords) {
      $pattern = "/(([\S]+[\s]+){1," . $maxwords . "})/";
      $matches = array();
      preg_match($pattern, $this->description, $matches);
      if (isset($matches[1])) {
        $desc = $matches[1];
      }
      else {
        $desc = $this->description;
      }
      $desc = "<span title=\"$this->description\">$desc</span>...";
    }
    else {
      $desc = $this->description;
    }
    $Tsch->td("<p>$desc</p>", "axbgdarker");
    $Tsch->td_alignment("", "top");
    $Tsch->td_width("30%");
    $Tsch->td_rowspan(2);
    $Tsch->td_css("padding:6px;");

    $Fldname = "elem_" . $this->scheme_id . "_scheme";
    if (count($this->values) > 0) {
      $sch_F = new form_combofield($Fldname);
      $sch_F->setclass("axcombo");
      $sch_F->setstyle("width:$ewidth");
      foreach ($this->values as $label => $val) {
        $sch_F->additem($val, $label);
      }
      $sch_F->setvalue($fieldvalue);
      if ($fieldname != "") {
        $scr = "document.forms.$formname.$fieldname.value=this.value;";
        if ($schemefield != "") {
          $scr .= "document.forms.$formname.$schemefield.value='$this->scheme_id';";
        }
        $sch_F->set_onchange($scr);
      }
      $Tsch->tr("axbgdark");
      $Tsch->td("Values:", "axfg");
      $Tsch->td_alignment("", "top");
      $Tsch->td($sch_F->render());
      $Tsch->td_alignment("", "top");
    }
    elseif (count($this->qualifiers) > 0) {
      $Tq = new table($Fldname);
      $bset = new form_imagebutton("_set", "", "", "$LIBDIR/img/_set.gif", "Set as metadata content", 57, 15 );
      $bset->set_onclick("scheme_set('$this->scheme_id','$formname','$fieldname','$schemefield')");
      if ($fieldvalue != "") {
        $fieldvalues = explode(",", $fieldvalue);
        $initvalues = array();
        foreach ($fieldvalues as $val) {
          $bits = explode("=", $val);
          $qualname = trim($bits[0]);
          $qualval  = trim($bits[1]);
          $initvalues[$qualname] = $qualval;
        }
      }
      $qual_listbox = new form_combofield($Fldname);
      $qual_listbox->setclass("axlistbox");
      $qual_listbox->setstyle("width:$ewidth;");
      $qual_listbox->size = 4;
      // Make a new record maintainer..
      $qual_maintainer = new recmaintainer($formname, $qual_listbox, "pfx$this->scheme_id");
      foreach ($this->qualifiers as $qual) {
        $qual_listbox->additem($qual->qual_name, $qual->qual_label);
        // Populate maintainer data. The maintainer add_record method
        // requires an associative array keyed on listbox key id..
        if (isset($initvalues[$qual->qual_name])) {
          $initvalue = $initvalues[$qual->qual_name];
        }
        else {
          $initvalue = "";
        }
        $rec = array(
                "qual_value" => $initvalue,
                "qual_comments" => $qual->comments
                );
        $qual_maintainer->add_record($qual->qual_name, $rec);
      }
      // ..................................................................
      // The listbox field..
      $Tq->tr("axbgdark");
      $Tq->td( "Qualifiers:", "axfg" );
      $Tq->td_alignment("", "top");
      $Tq->td( $qual_listbox->render() );
      $Tq->td_alignment("", "top");
      // ..................................................................
      // Value field
      $Fld = new form_textfield("qual_value");
      $Fld->setstyle("width:$ewidth;");
      $Fld->setclass("axtxtbox");
      if ($qual->default_value != "") {
        $Fld->setvalue($qual->default_value);
      }
      $qual_maintainer->register_field($Fld);
      $Tq->tr("axbglite");
      $Tq->td( "Value:", "axfg" );
      $Tq->td( $Fld->render() );
      // ..................................................................
      // Comments field
      $Fld = new form_memofield("qual_comments");
      $Fld->setstyle("width:$ewidth;height:80px;");
      $Fld->setclass("axmemo");
      $Fld->disabled = true;
      $qual_maintainer->register_field($Fld);
      $Tq->tr("axbgdark");
      $Tq->td( "Comments:", "axfg" );
      $Tq->td_alignment( "", "top" );
      $Tq->td( $Fld->render() );
      $Tq->tr("axbglite");
      $Tq->td( "&nbsp;" );
      $Tq->td( $bset->render() );
      $Tq->tr("axbglite");
      $Tq->td( $qual_maintainer->render() );
      $Tq->td_colspan(2);
      $Tq->set_width_profile("29%,71%");
      // Plug it in..
      $Tsch->tr("axbgdark");
      $Tsch->td($Tq->render());
      $Tsch->td_colspan(2);
    }
    elseif ($this->tag_name != "") {
      $sch_F = new form_textfield($Fldname);
      $sch_F->setclass("axtxtbox");
      $sch_F->setstyle("width:$ewidth");
      $sch_F->setvalue($fieldvalue);
      if ($fieldname != "") {
        $scr = "document.forms.$formname.$fieldname.value=this.value;";
        if ($schemefield != "") {
          $scr .= "document.forms.$formname.$schemefield.value='$this->scheme_id';";
        }
        $sch_F->set_onchange($scr);
      }
      $Tsch->tr("axbgdark");
      $Tsch->td("Value:");
      $Tsch->td_alignment("", "top");
      $Tsch->td($sch_F->render());
      $Tsch->td_alignment("", "top");
    }
    $Tsch->tr("axhdg");
    $Tsch->td();
    $Tsch->td_height(2);
    $Tsch->td_colspan(3);
    // Return rendered table..
    $Tsch->set_width_profile("20%,50%,30%");
    return $Tsch->render();
  } // definition_form

} // metadata_scheme class

// -----------------------------------------------------------------------
/**
* A class which encapsulates a metadata element. This is the actual
* object which will be rendered into the end-user content (eg. the
* webpage), and contains everything necessary for that process.
* @package cm
*/
class metadata_element extends RenderableObject {
  // CONTROL FLAGS etc.
  /** Whether info has been already got */
  var $gotinfo = false;
  /** Whether schemes have been already got */
  var $gotschemes = false;
  /** Whether object contains valid data */
  var $valid = false;
  // METADATA ELEMENT
  /** The element ID for this item */
  var $element_id = "";
  /** The element ID of the parent element of this item */
  var $parent_element_id = "";
  /** Child element_ids of this element */
  var $child_element_ids = array();
  /** Whether this element is instantiated as layout metadata */
  var $instantiated = false;
  /** The meta schema that this item was sourced from */
  var $schema_name = "";
  /** The namespace code for the schema */
  var $schema_namespace = "";
  /** The refence URi for the schema */
  var $schema_namespace_uri = "";
  /** The base name of this element */
  var $element_name = "";
  /** The full tag name of this element */
  var $tag_name = "";
  /** Label against the element value field */
  var $label = "";
  /** Element description, usage details */
  var $description = "";
  /** Whether this element should be indexed */
  var $indexed = false;
  /** Whether this element can be searched for */
  var $searchable = false;
  /** Whether optional, mandatory, conditional or recommended */
  var $obligation = "o";
  /** Obligation, descriptive */
  var $obligation_desc = "";
  /** The default value for this element */
  var $default_value = "";
  /** A list of permitted element values */
  var $list_of_values = "";
  /** Order of display */
  var $display_order = 0;
  /** Schemes associated with this metadata element. This
      is an associative array keyed by scheme ID */
  var $schemes = array();
  /** ID of preferred scheme for this metadata element */
  var $preferred_scheme_id = "";
  // METADATA DATA
  /** The content of this metadata tag. This is the actual metadata
      content which gets rendered in the HTML/XML tag itself */
  var $tag_value;
  /** Whether the tag value is actually a URL/URI */
  var $linked_uri = false;
  /** The language of the content or resource referenced by URI */
  var $language = "";
  /** The encoding scheme/vocabulary used to pick the tag content */
  var $enc_scheme_id = "";
  /** The tag used to identify the encoding scheme used */
  var $enc_scheme_tag = "";
  /** Whether this is a base element, or has a parent */
  var $base_element = true;
  // .....................................................................
  /** Constructor  */
  function metadata_element($element_id="", $schema_name="") {
    $this->get($element_id, $schema_name);
  } // metadata_element

  // .....................................................................
  /**
  * Get all the relevant data for this metadata element.
  * @param integer $element_id ID of this metadata element
  * @param string $schema_name Name of schema set this element belongs to
  * @return boolean Whether the get succeeded and object is valid
  */
  function get($element_id="", $schema_name="") {
    $this->valid = false;
    if ($element_id != "") {
      $this->element_id = $element_id;
    }
    if ($schema_name != "") {
      $this->schema_name = $schema_name;
    }
    if ($this->element_id != "" && $this->schema_name != "") {
      // Get the full metadata tag name..
      $q  = "SELECT tag_name, parent_element";
      $q .= "  FROM ax_meta_element";
      $q .= " WHERE element_id=$this->element_id";
      $elem = dbrecordset($q);
      if ($elem->hasdata) {
        $tag_name = $elem->field("tag_name");
        $this->element_name = $tag_name;
        $this->parent_element_id = $elem->field("parent_element");
        if ($this->parent_element_id != "") {
          $this->base_element = false;
        }
        $tag_name = get_full_tag_name($tag_name, $this->parent_element_id);

        // Get schema details..
        $q  = "SELECT * FROM ax_meta_schema";
        $q .= " WHERE schema_name='" . escape_string($this->schema_name) . "'";
        $schema = dbrecordset($q);
        if ($schema->hasdata) {
          $this->schema_namespace = $schema->field("namespace");
          $this->schema_namespace_uri = $schema->field("namespace_uri");
        }
        // Prefix the namespace identifier..
        if ($this->schema_namespace != "") {
          $tag_name = strtoupper($this->schema_namespace) . "." . $tag_name;
        }
        $this->tag_name = $tag_name;
        // Flag data is valid..
        $this->valid = true;
      }
    }
    return $this->valid;
  } // get

  // .....................................................................
  /**
  * Get all the info which are associated with this metadata element.
  */
  function get_info() {
    if ($this->valid && !$this->gotinfo) {
      $q  = "SELECT *";
      $q .= "  FROM ax_site_meta_element";
      $q .= " WHERE element_id=$this->element_id";
      $q .= "   AND schema_name='" . escape_string($this->schema_name) . "'";
      $elem = dbrecordset($q);
      if ($elem->hasdata) {
        $this->label = $elem->field("label");
        $this->description = $elem->field("description");
        $this->indexed = $elem->istrue("indexed");
        $this->searchable = $elem->istrue("searchable");
        $this->obligation = $elem->field("obligation");
        switch ($this->obligation) {
          case 'o': $this->obligation_desc = "Optional";    break;
          case 'm': $this->obligation_desc = "Mandatory";   break;
          case 'c': $this->obligation_desc = "Conditional"; break;
          case 'r': $this->obligation_desc = "Recommended"; break;
          default:
            $this->obligation_desc = "Unspecified";
        }
        $this->default_value = $elem->field("default_value");
        $this->list_of_values = $elem->field("list_of_values");
        $this->display_order = $elem->field("display_order");
        $this->gotinfo = true;
      }
    }
  } // get_info

  // .....................................................................
  /**
  * Get all the schemes which are associated with this metadata element.
  */
  function get_schemes() {
    if ($this->valid && !$this->gotschemes) {
      $q  = "SELECT es.preferred_enc_scheme,ese.enc_scheme_id";
      $q .= "  FROM ax_meta_element_set es, ax_element_set_enc ese";
      $q .= " WHERE es.element_id=$this->element_id";
      $q .= "   AND es.schema_name='" . escape_string($this->schema_name) . "'";
      $q .= "   AND ese.element_id=es.element_id";
      $q .= "   AND ese.schema_name=es.schema_name";
      $schemesQ = dbrecordset($q);
      if ($schemesQ->hasdata) {
        $this->schemes = array();
        $pref_scheme_id = $schemesQ->field("preferred_enc_scheme");
        if ($pref_scheme_id != "") {
          $scheme = new metadata_scheme($pref_scheme_id);
          if ($scheme->valid) {
            $scheme->preferred = true;
            $this->schemes[$pref_scheme_id] = $scheme;
          }
        }
        do {
          $scheme_id = $schemesQ->field("enc_scheme_id");
          $scheme = new metadata_scheme($scheme_id);
          if ($scheme->valid) {
            $this->schemes[$scheme_id] = $scheme;
          }
        } while ($schemesQ->get_next());
      }
      $this->gotschemes = true;
    }
  } // get_schemes

  // .....................................................................
  /**
  * Inherit schemes from parent metadata elements. This involves a
  * traversal of the lineage (parentage) of this current metadata element
  * and retreival of the schemes of each one. Where the scheme is not
  * already associated with this element, it is added.
  */
  function inherit_schemes() {
    if (!$this->base_element) {
      $parent_elemid = $this->parent_element_id;
      while ($parent_elemid != "") {
        $q  = "SELECT * FROM ax_element_set_enc ese, ax_meta_element e";
        $q .= " WHERE ese.element_id=$parent_elemid";
        $q .= "   AND ese.schema_name='" . escape_string($this->schema_name) . "'";
        $q .= "   AND e.element_id=ese.element_id";
        $parentschemes = dbrecordset($q);
        if ($parentschemes->hasdata) {
          do {
            $parent_elemid = $parentschemes->field("parent_element");
            $scheme_id = $parentschemes->field("enc_scheme_id");
            if (!in_array($scheme_id, $this->schemes)) {
              $scheme = new metadata_scheme($scheme_id);
              if ($scheme->valid) {
                $this->schemes[$scheme_id] = $scheme;
              }
            }
          } while ($parentschemes->get_next());
        }
        else {
          $parent_elemid = "";
        }
      } // while
    }
  } // inherit_schemes

  // .....................................................................
  /**
  * Set the tag value of this meta data element. This is used when the
  * content is NOT a URI/URL. If it is, then use set_uri() instead.
  * @param string $val The value for this meta data element content
  */
  function set_tag_value($val) {
    $this->tag_value = $val;
    $this->linked_uri = false;
  } // set_tag_value

  // .....................................................................
  /**
  * Set the tag value to be a URI. Same as setting tag value but we also
  * flick the uri flag as well.
  * @param string $href The URL/URI for this meta data element
  */
  function set_uri($uri) {
    $this->set_tag_value($uri);
    $this->linked_uri = true;
  } // set_uri

  // .....................................................................
  /**
  * Set the language of the content of this meta data element
  * @param string $lang The language code for the content of this element
  */
  function set_language($lang) {
    $this->language = $lang;
  } // set_language

  // .....................................................................
  /**
  * Set the scheme used to define the content of this meta data element. If
  * the tag of the scheme is not specified, then we go and find it by looking
  * up the scheme record in the DB.
  * @param integer $scheme_id The ID of the scheme used for picking the content
  * @param string $scheme_tag The tag name of the scheme
  */
  function set_scheme($scheme_id, $scheme_tag="") {
    $this->enc_scheme_id = $scheme_id;
    if ($scheme_tag != "") {
      $this->enc_scheme_tag = $scheme_tag;
    }
    else {
      $scheme = dbrecordset("SELECT * FROM ax_enc_scheme WHERE enc_scheme_id=$scheme_id");
      if ($scheme->hasdata) {
        $this->enc_scheme_tag = $scheme->field("tag_name");
      }
    }
  } // set_scheme

  // .....................................................................
  /**
  * Render a definition form for this metadata element as HTML. This is a
  * self-contained table, and it has all of the element details, and fields
  * for setting the content, language, and scheme. It is not enclosed in
  * a form.
  * @return string An HTML rendering of metadata definition fields
  */
  function definition_form($formname="") {
    // Width of form elements..
    global $RESPONSE, $LIBDIR, $cwidth, $ewidth, $cwidth, $pwidth;

    $this->get_info();
    $this->get_schemes();

    // Generic script which inserts scheme data into metadata
    // form fields automatically on scheme data-entry..
    $RESPONSE->add_script(
          "function scheme_set(schid,fm,datfld,schfld) {\n"
        . "  var datObj=eval('pfx'+schid+'dat_'+fm);\n"
        . "  if (datObj!=null) {\n"
        . "    var postData=findFieldObj(fm,datfld);\n"
        . "    if (postData!=null) {\n"
        . "      postData.value='';\n"
        . "      for (var id in datObj) {\n"
        . "        qual_value=getValue(fm,'qual_value',id,'pfx'+schid);\n"
        . "        if (qual_value!='') {\n"
        . "          if (postData.value!='') postData.value+='; ';\n"
        . "          newqual=id+'='+qual_value;\n"
        . "          postData.value+=newqual;\n"
        . "          if (schfld != '') {\n"
        . "            var postSche=findFieldObj(fm,schfld);\n"
        . "            if (postSche != null) {\n"
        . "              comboSet(postSche,schid);\n"
        . "            }\n"
        . "          }\n"
        . "        }\n"
        . "      }\n"
        . "    }\n"
        . "  }\n"
        . "}\n"
        );

    $s = "";
    $Te = new table("eledef");
    $Te->setwidth("100%");
    $Te->setpadding(2);

    // CONTENT SECTION
    $Te->tr("axbgdark");
    $Te->td("METADATA TAG COMPOSER", "axsubhdg");
    $Te->td_colspan(3);

    $Te->tr("axbglite");
    $Te->td("<h2>$this->tag_name</h2>");
    $Te->td_colspan(2);
    $info  = "<p><b>$this->label</b><br>";
    $maxwords = 30;
    $words = explode(" ", $this->description);
    if (count($words) > $maxwords) {
      $pattern = "/(([\S]+[\s]+){1," . $maxwords . "})/";
      $matches = array();
      preg_match($pattern, $this->description, $matches);
      if (isset($matches[1])) {
        $desc = $matches[1];
      }
      else {
        $desc = $this->description;
      }
      $desc = "<span title=\"$this->description\">$desc</span>...";
    }
    else {
      $desc = $this->description;
    }
    $info .= "$desc</p>";
    if ($this->obligation_desc != "" && $this->obligation_desc != "Unspecified") {
      $info .= "<p>This metadata element is $this->obligation_desc.</p>";
    }
    $Te->td($info, "axbgdarker");
    $Te->td_alignment("", "top");
    $Te->td_rowspan(6);

    // Metadata Content
    if ($this->list_of_values != "") {
      $content_F = new form_combofield("elem_" . $this->element_id . "_content");
      $content_F->setclass("axcombo");
      $content_F->setstyle("width:$ewidth");
      $vals = explode(",", $this->list_of_values);
      foreach ($vals as $val) {
        $content_F->additem($val);
      }
    }
    else {
      $content_F = new form_memofield("elem_" . $this->element_id . "_content");
      $content_F->setclass("axmemo");
      $content_F->setstyle("width:$ewidth;height:60px");
    }
    if (isset($this->tag_value)) {
      $content_F->setvalue($this->tag_value);
    }
    elseif (isset($this->linked_uri)) {
      $content_F->setvalue($this->linked_uri);
    }
    elseif ($this->default_value != "") {
      $content_F->setvalue($this->default_value);
    }
    $Te->tr("axbgdark");
    $Te->td("Content:", "axfg");
    $Te->td_alignment("", "top");
    $Te->td_css("padding-top:3px");
    $Te->td($content_F->render());
    $Te->td_alignment("", "top");

    // URI flag
    if ($this->list_of_values == "") {
      $chkUri_F = new form_checkbox("elem_" . $this->element_id . "_chkuri");
      $chkUri_F->setclass("axchkbox");
      $chkUri_F->checked = $this->linked_uri;
      $Te->tr("axbgdark");
      $Te->td("Content is a URI:", "axfg");
      $Te->td($chkUri_F->render());
    }

    // Metadata Language
    $Te->tr("axbglite");
    $Te->td("Language code:", "axfg");
    $lang_F = new form_textfield("elem_" . $this->element_id . "_lang");
    $lang_F->setclass("axtxtbox");
    $lang_F->setstyle("width:$pwidth");

    if (isset($this->language)) {
      $lang_F->setvalue($this->language);
    }
    $Te->td($lang_F->render());

    // Metadata Scheme code
    $scheme_F = new form_combofield("elem_" . $this->element_id . "_scheme");
    $scheme_F->setclass("axcombo");
    $q  = "SELECT * FROM ax_enc_scheme";
    $q .= " WHERE enabled=TRUE";
    $q .= " ORDER BY tag_name";
    $schQ = dbrecordset($q);
    $scheme_F->additem("");
    if ($schQ->hasdata) {
      do {
        $schid = $schQ->field("enc_scheme_id");
        $tag_name = $schQ->field("tag_name");
        if ($tag_name != "") {
          $scheme_F->additem($schid, $tag_name);
        }
      } while ($schQ->get_next());
    }
    if (isset($this->enc_scheme_id)) {
      $scheme_F->setvalue($this->enc_scheme_id);
    }
    $Te->tr("axbgdark");
    $Te->td("Scheme:", "axfg");
    $Te->td($scheme_F->render());

    // A save button & done button
    $bsave = new form_imagebutton("save", "", "", "$LIBDIR/img/_save.gif", "Save metadata content", 57, 15 );
    $bdone = new form_imagebutton("done", "", "", "$LIBDIR/img/_done.gif", "Exit without saving",   57, 15 );
    $bsave->set_onclick("meta_action('save','$this->element_id','$this->schema_name')");
    $bdone->set_onclick("meta_action('done','$this->element_id','$this->schema_name')");
    $Te->tr("axbglite");
    $Te->td("&nbsp;");
    $Te->td($bsave->render() . "&nbsp;" . $bdone->render());

    // LANGUAGES
    $Te->tr("axbglite");
    $Te->td("LANGUAGES", "axhdg");
    $Te->td_colspan(3);
    $q = "SELECT * FROM ax_enc_scheme WHERE enc_scheme_name LIKE 'ISO639%' ORDER BY enc_scheme_name";
    $iso639 = dbrecordset($q);
    if ($iso639->hasdata) {
      do {
        $iso639id = $iso639->field("enc_scheme_id");
        if ($iso639id != "") {
          $iso639_scheme = new metadata_scheme($iso639id);
          if ($iso639_scheme->valid) {
            $schdef = $iso639_scheme->definition_form(
                        $formname,
                        "elem_" . $this->element_id . "_lang"
                        );
            $Te->tr("axbgdark");
            $Te->td( $schdef );
            $Te->td_colspan(3);
          }
        }
      } while ($iso639->get_next());
    }

    // SCHEMES APPLICABLE TO THIS ELEMENT
    $Te->tr("axbglite");
    $Te->td("APPLICABLE SCHEMES", "axhdg");
    $Te->td_colspan(3);
    foreach ($this->schemes as $scheme) {
      $schdef = $scheme->definition_form(
                    $formname,
                    "elem_" . $this->element_id . "_content",
                    "",
                    "elem_" . $this->element_id . "_scheme"
                    );
      $Te->tr("axbgdark");
      $Te->td( $schdef );
      $Te->td_colspan(3);
    }
    $Te->set_width_profile("20%,50%,30%");
    return $Te->render();
  } // definition_form

  // .....................................................................
  /**
  * Process the POST of the form of this metadata element. We are just
  * looking for the few fields containing the relevant data.
  * NOTE: We do NOT save anything to the database, we just update our
  * class variables with the newly POSTed data.
  */
  function POSTprocess() {
    global $mode;
    if (isset($mode) && $mode == "save") {
      $pfx = "elem_" . $this->element_id;
      $contentvar = $pfx . "_content";
      $langvar    = $pfx . "_lang";
      $chkurivar  = $pfx . "_chkuri";
      $schemevar  = $pfx . "_scheme";

      global $$contentvar, $$langvar, $$chkurivar, $$schemevar;

      $content = $$contentvar;
      $language = $$langvar;
      $chkUri = isset($$chkurivar);
      $scheme_id = $$schemevar;

      if ($chkUri) {
        $this->set_uri($content);
      }
      else {
        $this->set_tag_value($content);
      }
      if ($language != "") {
        $this->set_language($language);
      }
      if ($scheme_id != "") {
        $this->set_scheme($scheme_id);
      }
    }
  } // POSTprocess

  // .....................................................................
  /**
  * Render this metadata element as a metatag object. This method
  * creates the metatag object and returns it.
  * @return object The metadata element as a metatag object
  */
  function metatag() {
    if ($this->linked_uri) {
      $metatag = new uri_meta_tag(
                    $this->tag_name,
                    $this->tag_value,
                    $this->language
                    );
    }
    else {
      $metatag = new data_meta_tag(
                    $this->tag_name,
                    $this->tag_value,
                    $this->language,
                    $this->enc_scheme_tag
                    );
    }
    return $metatag;
  } // metatag

  // .....................................................................
  /**
  * Render this metadata element as HTML.
  * @return string The HTML rendering of this metadata element
  */
  function html() {
    $meta = $this->metatag();
    return $meta->html();
  } // html

} // metadata_element class

// -----------------------------------------------------------------------
/**
* A class which holds multiple metadata elements, for a given layout.
* Acquires all the metadata elements for a given layout. Provides the
* means to render these.
* @package cm
*/
class layout_metadata_elements {
  // Public
  /** The layout these elements are for */
  var $layout_id = "";
  /** The metadata elements themselves */
  var $metadata_elements = array();

  // Private
  /** Standard form name to use
      @access private */
  var $formname = "";
  // .....................................................................
  /** Constructor
  * @param integer $layout_id The unique ID of the layout
  * @param string $formname The form name to use for form rendering
  * @param boolean $instantiated_only True means get only existing meta data
  */
  function layout_metadata_elements($layout_id="", $formname="", $instantiated_only=false) {
    $this->get($layout_id, $instantiated_only);
    if ($formname == "") {
      $formname = "layout_" . $this->layout_id . "_metadata";
    }
    $this->formname = $formname;
  } // layout_metadata_elements
  // .....................................................................
  /**
  * Get all of the metadata elements associated with this layout. We can
  * do this in two modes, determined by the $instantiated_only flag. If
  * this is true then we only grab the data for the metadata elements
  * which exist for the layout. Otherwise we grab ALL of them. The
  * latter is for editing, and the former is for page rendering where
  * we want minimal Db traffic.
  * @param integer $layout_id The ID of the layout to get elements for
  * @return boolean Whether the get succeeded and object is valid
  */
  function get($layout_id="", $instantiated_only=false) {
    if ($layout_id != "") {
      $this->layout_id = $layout_id;
    }
    // Populate ALL metadata elements, as blanks..
    if (!$instantiated_only) {
      $q  = "SELECT * FROM ax_site_meta_element";
      $q .= " WHERE enabled=TRUE";
      $q .= " ORDER BY display_order";
      $allmeta = dbrecordset($q);
      if ($allmeta->hasdata) {
        do {
          $element_id  = $allmeta->field("element_id");
          $schema_name = $allmeta->field("schema_name");
          $meta = new metadata_element($element_id, $schema_name);
          // Add this new element to our array..
          $meta->get_info();
          $this->add_metadata_element($meta);
        } while ($allmeta->get_next());
      }
    }

    // Overlay with metadata which has been defined..
    if ($this->layout_id != "") {
      $q  = "SELECT * FROM ax_layout_metadata";
      $q .= " WHERE layout_id=$this->layout_id";
      $laymeta = dbrecordset($q);
      if ($laymeta->hasdata) {
        do {
          $element_id  = $laymeta->field("element_id");
          $schema_name = $laymeta->field("schema_name");
          $tag_value   = $laymeta->field("meta_tag_value");
          $scheme_id   = $laymeta->field("enc_scheme_id");
          $uri         = $laymeta->istrue("linked_uri");
          $language    = $laymeta->field("language");
          if ($instantiated_only) {
            $meta = new metadata_element($element_id, $schema_name);
            $meta->get_info();
          }
          else {
            $meta = $this->get_metadata_element($element_id);
          }
          if ($meta !== false) {
            $meta->instantiated = true;
            if ($scheme_id != "") {
              $meta->set_scheme($scheme_id);
            }
            if ($uri) {
              $meta->set_uri($tag_value);
            }
            else {
              $meta->set_tag_value($tag_value);
            }
            if ($language != "") {
              $meta->set_language($language);
            }
            // Add this new element to our array..
            $this->add_metadata_element($meta);
          }
        } while ($laymeta->get_next());
        $this->valid = true;
      }
    }
  } // get

  // .....................................................................
  /** Assembles all metadata elements for this layout. */
  function perform_hierarchy_scan() {
    // Determine hierarchical relationships between the
    // metadata elements, ie. parent->child->child.. etc.
    foreach ($this->metadata_elements as $element_id => $element) {
      if (!$element->base_element && $element->parent_element_id != "") {
        $parent_element = $this->get_metadata_element($element->parent_element_id);
        if ($parent_element !== false) {
          $parent_element->child_element_ids[] = $element_id;
          $this->add_metadata_element($parent_element);
        }
      }
    } // foreach
  } // perform_hierarchy_scan

  // .....................................................................
  /** Returns the number of meta elements in this layout.
  * @return integer Number of metadata elements currently in this layout
  */
  function meta_element_count() {
    return count($this->metadata_elements);
  } // meta_element_count

  // .....................................................................
  /**
  * Return meta data tree entry.
  * @return string The table containing all metadata elements so far.
  * @access private
  */
  function metadata_tree_entry($Ttree, $indent, $element_id) {
    global $RESPONSE, $LIBDIR, $cwidth, $ewidth, $cwidth, $pwidth;
    // Write current element..
    $element = $this->get_metadata_element($element_id);
    if ($element !== false) {
      // Do this present element..
      $padding = new img("$LIBDIR/img/_pad.gif", "", "", (18 * $indent), 1);
      $Ttree->tr("axbglite");
      $elem_link = new anchor("#", $element->label);
      if ($element->instantiated) {
        $elem_link->set_onclick("meta_action('edit','$element->element_id','$element->schema_name')");
        $bunset = new form_imagebutton("unset", "", "", "$LIBDIR/img/_redx.gif", "Un-set", 9, 9);
        $bunset->set_onclick("meta_action('unset','$element->element_id','$element->schema_name')");
        $Ttree->td($bunset->render(), "padding-top:5px");
        $Ttree->td_alignment("", "top");
        $Ttree->td($padding->render() . $elem_link->render(), "axfg");
        $Ttree->td_alignment("", "top");
        $Ttree->td(($element->tag_value != "") ? $element->tag_value : "&nbsp;");
        $Ttree->td_alignment("", "top");
        $Ttree->td(($element->language != "") ? $element->language : "&nbsp;");
        $Ttree->td_alignment("center", "top");
        $Ttree->td(($element->enc_scheme_tag != "") ? $element->enc_scheme_tag : "&nbsp;");
        $Ttree->td_alignment("center", "top");
      }
      else {
        $elem_link->set_onclick("meta_action('add','$element->element_id','$element->schema_name')");
        $pad = new form_imagebutton("pad", "", "", "$LIBDIR/img/_pad.gif", "", 9, 1);
        $Ttree->td($pad->render());
        $Ttree->td($padding->render() . $elem_link->render(), "axfg");
        $Ttree->td_alignment("", "top");
        switch ($element->obligation) {
          case "m": $msg = "<span class=\"axhl\">mandatory</span>"; break;
          case "r": $msg = "recommended"; break;
          default: $msg = "&nbsp;";
        }
        $Ttree->td($msg);
        $Ttree->td_alignment("", "top");
        $Ttree->td("&nbsp;");
        $Ttree->td("&nbsp;");
      }

      // And now do any children..
      if (count($element->child_element_ids) > 0) {
        $child_indent = $indent + 1;
        foreach ($element->child_element_ids as $child_element_id) {
          $Ttree = $this->metadata_tree_entry($Ttree, $child_indent, $child_element_id);
        }
      }
    }
    // Return the tree table..
    return $Ttree;
  } // metadata_tree_entry

  // .....................................................................
  /**
  * Returns a string containing a table which contains all of the meta
  * data in a nice tree view.
  * @return string A table containing all metadata elements for viewing
  */
  function metadata_tree() {
    global $RESPONSE;
    $this->perform_hierarchy_scan();
    $Ttree = new table("metadata_tree");
    $Ttree->setpadding(2);
    $Ttree->td("&nbsp;");
    $Ttree->td("<i>Tag</i>", "axfg");
    $Ttree->td("<i>Content</i>", "axfg");
    $Ttree->td("<i>Lang</i>", "axfg");
    $Ttree->td_alignment("center");
    $Ttree->td("<i>Scheme</i>", "axfg");
    $Ttree->td_alignment("center");
    foreach ($this->metadata_elements as $element_id => $element) {
      if ($element->base_element) {
        $indent = 0;
        $Ttree = $this->metadata_tree_entry($Ttree, $indent, $element_id);
      }
    } // foreach
    $Ttree->set_width_profile("1%,25%,49%,5%,20%");
    // Return rendered content..
    return $Ttree->render();
  } // metadata_tree

  // .....................................................................
  /**
  * Add a metadata element to our array of elements. We add it to an
  * associative array keyed on the element_id, ensuring no duplicates.
  * @param object $meta The metadata element object to add
  */
  function add_metadata_element($meta) {
    $this->metadata_elements[$meta->element_id] = $meta;
  } // add_metadata_element

  // .....................................................................
  /**
  * Return a metadata element, given its element_id.
  * @param integer $element_id The element ID of the element to return
  */
  function get_metadata_element($element_id) {
    if (isset($this->metadata_elements[$element_id])) {
      return $this->metadata_elements[$element_id];
    }
    else {
      return false;
    }
  } // get_metadata_element
  // .....................................................................
  /**
  * Saves the layout metadata for a given element. This saves data to
  * the ax_layout_metadata table.
  * @param integer $element_id The element ID of the element to return
  */
  function save_metadata_element($element_id) {
    $element = $this->get_metadata_element($element_id);
    if ($element !== false) {
      if ($element->instantiated) {
        $meup = new dbupdate("ax_layout_metadata");
        $meup->where("layout_id=$this->layout_id");
        $meup->where("AND element_id=$element->element_id");
        $meup->where("AND schema_name='" . escape_string($element->schema_name) . "'");
      }
      else {
        $meup = new dbinsert("ax_layout_metadata");
        $meup->set("layout_id", $this->layout_id);
        $meup->set("element_id", $element->element_id);
        $meup->set("schema_name", $element->schema_name);
      }
      $meup->set("meta_tag_value", $element->tag_value);
      if ($element->enc_scheme_id != "") {
        $meup->set("enc_scheme_id", $element->enc_scheme_id);
      }
      else {
        $meup->set("enc_scheme_id", NULLVALUE);
      }
      $meup->set("linked_uri", $element->linked_uri);
      $meup->set("language", $element->language);
      $ok = $meup->execute();

      // Alter instantiation status if created new..
      if (!$element->instantiated && $ok) {
        $element->instantiated = true;
        $this->add_metadata_element($element);
      }
    }
  } // save_metadata_element
  // .....................................................................
  /**
  * Removes the layout metadata for a given element. This deletes the
  * relevant record from the ax_layout_metadata table.
  * @param integer $element_id The element ID of the element to remove
  */
  function remove_metadata_element($element_id) {
    $element = $this->get_metadata_element($element_id);
    if ($element !== false && $element->instantiated) {
      $medel = new dbdelete("ax_layout_metadata");
      $medel->where("layout_id=$this->layout_id");
      $medel->where("AND element_id=$element->element_id");
      $medel->where("AND schema_name='" . escape_string($element->schema_name) . "'");
      $medel->execute();
    }
  } // remove_metadata_element
  // .....................................................................
  /**
  * Insert the instantiated metatags into the given webpage. This is
  * intended to be used on the $RESPONSE object to actually jam the
  * tags for this layout into the webapage.
  * @param reference Reference to Webpage object to insert metatags into
  */
  function insert_metatags(&$webpage) {
    foreach ($this->metadata_elements as $element_id => $element) {
      if ($element->instantiated) {
        $metatag = $element->metatag();
        $webpage->insert_metatag($element_id, $metatag);
      }
    }
  } // insert_metatags
  // .....................................................................
  /**
  * Render all of the elements as HTML
  * @return string HTML rendering of all contained metadata elements
  */
  function html() {
    $s = "";
    foreach ($this->metadata_elements as $element_id => $element) {
      if ($element->instantiated) {
        $s .= $element->html();
      }
    }
    return $s;
  } // html

} // layout_metadata_elements class

// -----------------------------------------------------------------------
/**
* Function to format a tag name which might have multiple parents.
* The expected format is that the sequence of tag names will each
* be separated by a dot ".". Here we just format it with the first
* descendant lowercased, and all others proper-cased.
* @param string $tag_name The tag name, with descendants separated by dots
*/
function format_tag_name($tag_name) {
  $result = $tag_name;
  $bits = explode(".", $tag_name);
  if (count($bits) > 1) {
    $fmtarr = array();
    $first = true;
    foreach ($bits as $tagbit) {
      if ($first) {
        $fmtarr[] = strtolower($tagbit);
        $first = false;
      }
      else {
        $fmtarr[] = ucfirst($tagbit);
      }
    }
    $result = implode(".", $fmtarr);
  }
  return $result;
} // format_tag_name

// -----------------------------------------------------------------------
/**
* Function to acquire the full tag name including descendants, given
* the tag name, and the parent element ID of that tag.
* @param string $tag_name The starting tag name
* @param integer $parent_elemid The parent element ID of that tag
* @return string The full descendant tag name
*/
function get_full_tag_name($tag_name, $parent_elemid) {
  while ($parent_elemid != "") {
    $parent = dbrecordset("SELECT tag_name FROM ax_meta_element WHERE element_id=$parent_elemid");
    if ($parent->hasdata) {
      $tag_name = $parent->field("tag_name") . "." . $tag_name;
      $parent_elemid = $parent->field("parent_element");
    }
    else {
      $parent_elemid = "";
    }
  }
  return $tag_name;
} // get_full_tag_name

// -----------------------------------------------------------------------
?>