<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    json-rpc-defs.php                                       */
/* Author:      Paul Waite                                              */
/*                                                                      */
/* Description: Definitions for implementing RPC services using the     */
/*              Dojo JSON RPC utilities and tools.                      */
/*                                                                      */
/* ******************************************************************** */
include_once("json-defs.php");

class jsonRPC {
  /**
   * Service definitions, either a string which is a
   * website URL to a defs file, or a JSON literal
   */
  var $service_defs = "";
  /**
   * An array of RPC function calls and parameter names.
   * These are the functions available to user scripting
   * to make RPC calls to the server.
   */
  var $rpcFuncs = array();
  /**
   * User-defined callback function. If left undefined
   * then we simply don't have callback processing. This
   * is for one-way commands requiring no response.
   */
  var $rpcCallbackFuncs = array();
  // .....................................................................
  function jsonRPC($service_defs) {
    $this->service_defs = $service_defs;
  } // jsonRPC
  // .....................................................................
  /**
   * Add an RPC function. These are functions which can be called from
   * user-supplied scripting to perform the RPC with whatever parameters
   * are required in each case (comma-delimited list).
   * eg: $foo->addRPC('myrpcfunc', 'foo,bar');
   * NOTE: the RPC response is stored in two global vars named after the
   * function name '<funcname>_response' and '<funcname>_ioargs'.
   * @param string $rpc_name Name of the RPC function as per service defs
   * @param string $rpc_parms List of parameters to this function
   * @param string $rpc_callback_func Optional callback function for response
   */  
  function addRPC($rpc_name, $rpc_parms, $rpc_callback_func="") {
    $this->rpcFuncs[$rpc_name] = $rpc_parms;
    if ($rpc_callback_func != "") {
      $this->rpcCallbackFuncs[$rpc_name] = $rpc_callback_func;
    }
  } // addRPC
  // .....................................................................
  /**
   * Render the RPC functionality for this page. This is wholly Javascript
   * generation, and it makes sure that the Dojo RPC system is initialised,
   * and the user-specified RPC functions and call-backs are defined.
   */
  function render() {
    global $RESPONSE;
    // If no definitions, then no service..
    if ($this->service_defs != "") {
      
      // Set up javascript for initialisation
      $RESPONSE->head->add_named_script(
            "var jsonRPC=null;\n"
          . "function jsonRPCinit() {\n"
          . " jsonRPC=new dojo.rpc.JsonService('$this->service_defs');\n"
          . "}\n"
          . "dojo.require('dojo.rpc.JsonService');\n"
          ,
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
      // Add the user RPC functions
      foreach ($this->rpcFuncs as $funcname => $parmslist) {
        $js = "function " . $funcname . "(" . $parmslist . ") {\n"
            . " var rpc=jsonRPC." . $funcname . "(" . $parmslist . ");\n"
            ;
        if (isset($this->rpcCallbackFuncs[$funcname])) {
          $js .= " rpc.addCallback(" . $this->rpcCallbackFuncs[$funcname] . ");\n";
        }
        $js .= "}\n";
        $RESPONSE->head->add_named_script($js, "dojo", "javascript", "append", ADD_ONCE_ONLY);
      } // foreach

      // Make sure RPC platform gets initialised onload
      $RESPONSE->head->add_named_script(
            "dojo.addOnLoad(jsonRPCinit);\n",
            "dojo_onloads",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
    }  
  } // render

} // jsonRPC class

// -----------------------------------------------------------------------
/**
 * Function which returns an RPC response as JSON. This is for use with
 * callback functions. The Javascript callback function should look like
 * this example:
 *   mycallback(response, ioargs) {
 *     result = response[0];
 *     error  = response[1];
 *     rpcid  = response[2];
 *     -- your processing --
 *   }
 * Ie. the response is passed as a three-element array of scalars.
 * @param string $result A scalar containing result data
 * @param string $error A scalar containing error info
 * @param integer The original RPC request ID
 */
function jsonRPCresponse($result, $error, $id) {

  $resp = array(
    "result" => $result,
    "error" => ($error != "" ? $error : "null"),
    "id" => $id
    );
  $conv = new JSONconverter();
  return $conv->toJSON($resp);
}

?>