<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    form-defs.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Definitions for data visualization elements by way      */
/*              of HTML forms.                                          */
/*                                                                      */
/* ******************************************************************** */
/** @package form */

/** Include data definitions for key value container */
include_once("data-defs.php");
/** Include date and time functions */
include_once("datetime-defs.php");
/** Include button creation classes */
include_once("button-defs.php");
/** The HTML object classes are required for forms */
include_once("html-defs.php");

// ----------------------------------------------------------------------

/** Standard width for a textual form element */
define("STD_WIDTH", 16);

// Field attributes. We generally implement a display-only
// field as a hidden field on the HTML form, with plain
// text showing the value. This way we get around some
// deficiencies in some browsers which don't implement the
// READONLY attribute properly!

/** Form field is readonly/display only */
define("DISPLAY_ONLY", false);
/** Form element is editable (ie. not readonly) */
define("EDITABLE", true);

// Various form display element types. These define
// how the data looks on the HTML form.

/** General text field */
define("F_TEXT",     1 );
/** Text field containing image URL */
define("F_IMAGE",    2 );
/** Date/time field */
define("F_DATETIME", 3 );
/** Dropdown menu or listbox field */
define("F_COMBO",    4 );
/** Textarea memo field */
define("F_MEMO",     5 );
/** Tickbox or checkbox */
define("F_CHECKBOX", 6 );
/** Radio button */
define("F_RADIO",    7 );

// Case forcing options for text fields
/** Do not force case */
define("NONE",     0 );
/** Force text to uppercase */
define("UPPER",    1 );
/** Force text to lowercase */
define("LOWER",    2 );

// Flag for Multi-select in combo fields..
/** Combo-field is multiple-select */
define("MULTISELECT", true);
/** Combo-field is single-select */
define("SINGLESELECT", false);

// ----------------------------------------------------------------------
/**
* Form class
* A container object for form elements. This is for when we have a
* classical form which is just a set of form elements for filling
* in, rather than a mixture of page layout and elements. It is
* rendered in a vanilla table with each field stacked vertically
* to the right of its label..
* @package form
*/
class form extends HTMLObject {
  // Public
  /** Action attribute */
  var $action = "";
  /** Method attribute */
  var $method = "post";
  /** Encoding type */
  var $enctype = "";
  /** Accept-charset attributes - comma-delimited list */
  var $accept_charsets = "";
  /** Script to call on form submit */
  var $onsubmit;
  /** Whether to render labels for fields */
  var $showlabels = true;
  /** Justification mode for labels "right" or "left" */
  var $labeljustify = "right";
  /** Whether to append colon ":" to labels */
  var $labelcolon = true;
  /** Optional class/style for labels */
  var $labelcss;
  /** Width taken by element (RHS) side of the form in % */
  var $elewidth_pct = 65;
  /** Whether to force forma elements to be read-only */
  var $force_readonly = false;

  // Private
  /** Type of form 'normal' or 'subform' */
  var $type = "normal";
  /** Elements array. Contains form elements
      @access private */
  var $elements;
  // ....................................................................
  /**
  * Constructor
  * Create a form object. Sets basic form attributes.
  * @param string  $name     The name of the form
  * @param string  $title    The title/banner for the form
  * @param string  $action   Action attribute, ie. the script to POST to
  * @param string  $method   Method attribute, POST or GET
  * @param string  $enctype  Encoding type
  */
  function form($name="", $title="", $action="", $method="post", $enctype="", $css="", $charsets="") {
    global $SCRIPT_NAME;
    if ($action == "") {
      $action = $SCRIPT_NAME;
    }
    $this->setname($name);
    $this->settitle($title);
    $this->action = $action;
    $this->method = $method;
    $this->enctype = $enctype;
    // A form can be of two basic types:
    //  'normal'  : we render between <form></form> tags
    //  'subform' : we omit these tags; part of another form
    $this->type = "normal";
    $this->setcss($css);
    $this->accept_charsets = $charsets;
  } // form
  // ....................................................................
  /**
  * Set the proportion of the form taken up by the form field elements
  * as opposed to the labels. This is specified as an integer which
  * is the proportion as a percentage. Eg: 70 would be '70%'.
  * @param integer $pctwidth Percentage width of form for form fields
  */
  function set_fieldwidth_pct($pctwidth=65) {
    $this->elewidth_pct = $pctwidth;
  } // set_fieldwidth_pct
  // ....................................................................
  /** Force all of the contained fields to be rendered read-only */
  function force_readonly($mode=true) {
    $this->force_readonly = $mode;
  } // force_readonly
  // ....................................................................
  /**
  * Adds a form element object to the form. This is usually an object
  * you have previously created eg. with new form_textfield(...) etc.
  * @param object $element The form element object to add
  */
  function add($element) {
    $this->elements[] = $element;
  } // add
  // ....................................................................
  /**
  * Add form separator row
  * Adds a separator row to the form
  * If no heading is given then this produces a ruled line across
  * the form. If a heading is given then this text is put in a row
  * just below the ruled line. Feel free to add in HTML formatting
  * using normal HTML tags like <b></b> etc.
  * @param string $heading Optional separator title/heading text
  * @param string $css Style/class to apply (not rendered at present)
  */
  function add_separator($heading="", $css="") {
    $f = new form_labelfield($heading, "", $css);
    $f->type = "separator";
    $this->add($f);
  } // add_separator
  // ....................................................................
  /**
  * Add text to the form
  * Adds text content to the form spanning both columns. The text is
  * added literally, so you can add in effects like <em>..</em> etc.
  * @param string $text Text to display in the form
  * @param string $css Style/class to apply (NB: not rendered at present)
  */
  function add_text($text, $css="") {
    $f = new form_labelfield($text, "", $css);
    $f->type = "textcontent";
    $this->add($f);
  } // add_text
  // ....................................................................
  /**
  * Add annotation to the form
  * Adds text content to the form in the field column. The text is
  * added literally, so you can add in effects like <em>..</em> etc.
  * @param string $text Text to display in the form
  * @param string $css Style/class to apply (NB: not rendered at present)
  */
  function add_annotation($text, $css="") {
    $f = new form_labelfield($text, "", $css);
    $f->type = "annotation";
    $this->add($f);
  } // add_annotation
  // ....................................................................
  /**
  * Add file upload fields
  * Special function to add repeated fileupload fields to the form.
  * These share the same naming, but with square brackets "[]" appended.
  * @param object  $element  The form element object to add
  * @param integer $count    The number of elements to add
  * @param boolean $labels   Whether we have individual field labels
  */
  function add_fileuploadfields($element, $count=1, $labels=false) {
    // If multiple file upload fields, then make sure that we
    // have the [] postfixed onto the name first..
    if ($count > 1) {
      $name = $element->name;
      $len = strlen($name);
      if ($len <= 2) {
        $name .= "[]";
      }
      else {
        $last2chars = substr($name, $len - 2, 2);
        if ($last2chars != "[]") {
          $name .= "[]";
        }
      }
      $element->name = $name;
    }
    // First element gets the hidden field, and all the
    // rest therefore must have it suppressed..
    $element->include_maxsize = true;
    for ($ix=0; $ix < $count; $ix++) {
      if ($ix == 1) {
        $element->include_maxsize = false;
      }
      if ($ix > 0) {
        if (isset($labels[$ix])) {
          $element->label = $labels[$ix];
        }
        else $element->label = "";
      }
      $this->add($element);
    }
  } // add_fileuploadfields
  // ....................................................................
  /**
  * Add a button
  * Adds a button element to the form.
  * @param object  $button  The button element object to add
  */
  function add_button($button) {
    $this->add($button);
  } // add_button
  // ....................................................................
  /**
  * Set onsubmit script
  * Defines the onsubmit script to call when the user
  * submits the form.
  * @param string $script The script to execute on form submit
  */
  function set_onsubmit($script) {
    $this->onsubmit = $script;
  } // set_onsubmit
  // ....................................................................
  /**
  * Set the form type
  * This can be 'normal' or 'subform'. Use the subform variant when you
  * already have <form> tags being provided by other means. Then when
  * you render the subform only the fields will be returned.
  * @param string $type  The form type 'normal' or 'subform'
  */
  function set_type($type="normal") {
    $this->type = $type;
  } // set_type
  // ....................................................................
  /**
  * Check if form contains file upload field
  * Check all elements for presence of file upload field. This is
  * mainly for internal use.
  * @return boolean True if form contains a file upload field
  */
  function contains_fileuploadfield() {
    $res = false;
    if (isset($this->elements)) {
      foreach ($this->elements as $element) {
        if ($element->type == "file") {
          $res = true;
          break;
        }
      }
    }
    return $res;
  } // contains_fileuploadfield
  // ....................................................................
  /**
  * This renders the form as HTML, including the form tags and every
  * form element in the form.
  * @return string The form as HTML.
  */
  function html() {
    $s = "";
    $formtitle = $this->title;
    $this->title = strip_tags($this->title);
    if (isset($this->elements)) {
      switch ($this->type) {
        case "normal":
          // If we have a file upload field, then set the encoding,
          // unless it has already been manually set..
          if ($this->enctype == "" && $this->contains_fileuploadfield()) {
            $this->enctype = "multipart/form-data";
          }
          $s .= "<form";
          $s .= $this->attributes();
          $s .= " action=\"" . $this->action . "\"";
          $s .= " method=\"" . $this->method . "\"";
          if ($this->enctype != "") {
            $s .= " enctype=\"" . $this->enctype . "\"";
          }
          if ($this->accept_charsets != "") {
            $s .= " accept-charset=\"" . $this->accept_charsets . "\"";
          }
          if (isset($this->onsubmit)) {
            $s .= " onsubmit=\"" . $this->onsubmit . "\"";
          }
          $s .= ">\n";
          break;
        case "subform":
          break;
      } // switch

      $fmtable = new table($this->name);
      $fmtable->setpadding(3);
      $fmtable->inherit_attributes($this);

      // Optional form title area..
      if ($formtitle != "") {
        $fmtable->thead();
        $fmtable->tr();
        $fmtable->td($formtitle);
        if ($this->showlabels) {
          $fmtable->td_colspan(2);
        }
      }

      // Render any elements..
      $fmtable->tbody();
      $rowcnt = 0;
      foreach ($this->elements as $element) {
        if (isset($element->type)) {
          switch ($element->type) {
            // Subform of this form..
            case "subform":
              $element->inherit_attributes($this);
              $element->labeljustify = $this->labeljustify;
              $element->labelcolon = $this->labelcolon;
              $element->force_readonly = $this->force_readonly;
              $fmtable->tr();
              $rowcnt += 1;
              $fmtable->td($element->render());
              $fmtable->td_alignment("", "top");
              if ($this->showlabels) {
                $fmtable->td_colspan(2);
              }
              break;
            // Table separator line..
            case "separator":
              $fmtable->tr();
              $rowcnt += 1;
              $fmtable->td("<hr>");
              if ($this->showlabels) {
                $fmtable->td_colspan(2);
              }
              if ($element->label != "") {
                $fmtable->tr();
                $rowcnt += 1;
                $fmtable->td("<b>" . $element->label . "</b>");
                if (isset($this->labelcss)) {
                  $fmtable->td_css($this->labelcss);
                }
                if ($this->showlabels) {
                  $fmtable->td_colspan(2);
                }
              }
              break;
            // Text done as a labelfield..
            case "textcontent":
              $fmtable->tr();
              $rowcnt += 1;
              $fmtable->td($element->label);
              if (isset($this->labelcss)) {
                $fmtable->td_css($this->labelcss);
              }
              if ($this->showlabels) {
                $fmtable->td_colspan(2);
              }
              break;
            // Annotation done as a labelfield..
            case "annotation":
              $fmtable->tr();
              $rowcnt += 1;
              $fmtable->td();
              $fmtable->td($element->label);
              if (isset($this->labelcss)) {
                $fmtable->td_css($this->labelcss);
              }
              break;
            // Defer hidden fields 'til later
            case "hidden":
              $hiddenfields[] = $element;
              break;
            // Defer image buttons 'til later
            case "image":
              $buttons[] = $element;
              break;
            // Standard form field - label: field
            default:
              if ($this->force_readonly) {
                $element->set_displayonly();
              }
              if ($element->label != "") {
                $label = $element->label;
                if ($this->labelcolon) {
                  $label .= ":";
                }
              }
              else $label = "";
              $fmtable->tr();
              $rowcnt += 1;
              if ($this->showlabels) {
                $fmtable->td($label);
                switch ($element->type) {
                  case "textarea":
                    $fmtable->td_alignment($this->labeljustify, "top");
                    break;
                  case "select":
                    if ($element->multiselect) {
                      $fmtable->td_alignment($this->labeljustify, "top");
                    }
                    else {
                      $fmtable->td_alignment($this->labeljustify);
                    }
                    break;
                  default:
                    $fmtable->td_alignment($this->labeljustify);
                } // switch
                if (isset($this->labelcss)) {
                  $fmtable->td_css($this->labelcss);
                }
              }
              $fmtable->td($element->render());
              if (($element->type == "label" || $element->type == "displayonly")
              && isset($this->labelcss)) {
                $fmtable->td_css($this->labelcss);
              }
              $fmtable->td_alignment("", "top");
          } // switch
        }
        else {
          // Save buttons 'til later..
          if (is_subclass_of($element, "button_object")) {
            $buttons[] = $element;
          }
          // Otherwise render it in the right-hand cell..
          else {
            $fmtable->tr();
            $rowcnt += 1;
            if ($this->showlabels) {
              $fmtable->td("&nbsp;");
            }
            $fmtable->td($element->render());
            $fmtable->td_alignment("", "top");
          }
        }
      } // foreach

      // Render any buttons in one cell..
      if (isset($buttons)) {
        $allbtns = "";
        foreach ($buttons as $button) {
          $allbtns .= $button->render() . "&nbsp;";
        }
        $fmtable->tr();
        $rowcnt += 1;
        if ($this->showlabels) {
          $fmtable->td("&nbsp;");
        }
        $fmtable->td($allbtns);
      }

      // Render the form table content..
      if ($rowcnt == 0) {
        $fmtable->tr();
        $fmtable->td("");
      }

      // Apply width profile..
      $fmtable->set_width_profile((100 - $this->elewidth_pct) . "%," . $this->elewidth_pct . "%");

      // Render it as HTML..
      $fmtable->inherit_attributes($this);
      $s .= $fmtable->html();

      // Render any hidden fields..
      if (isset($hiddenfields)) {
        foreach ($hiddenfields as $hidden) {
          $s .= $hidden->render();
        }
      }
      // End the form if required..
      switch ($this->type) {
        case "normal":
          $s .= "</form>\n";
          break;
        case "subform":
          break;
      } // switch
    }
    // Return the finished form..
    return $s;
  } // html
} // form class

// ----------------------------------------------------------------------
/**
* SubForm class
* The sub-form is a special case of the form class. The difference is
* that it is expected to be part of an existing form, and therefore
* is rendered without the form tags.
* @package form
*/
class subform extends form {
  // ....................................................................
  /**
  * Create a subform object. Sets basic form attributes.
  * @param string  $name     The name of the form
  * @param string  $title    The title/banner for the form
  * @param string  $action   Action attribute, ie. the script to POST to
  * @param string  $method   Method attribute, POST or GET
  * @param string  $enctype  Encoding type
  */
  function subform($name="", $title="", $action="", $method="post", $enctype="") {
    $this->form($name, $title, $action, $method, $enctype);
    $this->type = "subform";
  }
} // subform class

// ----------------------------------------------------------------------
/**
* multipart_form class
* The multipart_form is just a standard form, but with the enctype
* pre-set to the "multipart/form-data" setting required for file
* multiparts of binary form data.
* @package form
*/
class multipart_form extends form {
  /**
  * Create an multipart_form object. Sets basic form attributes.
  * @param string  $name     The name of the form
  * @param string  $title    The title/banner for the form
  * @param string  $action   Action attribute, ie. the script to POST to
  * @param string  $method   Method attribute, POST or GET
  * @param string  $enctype  Encoding type
  */
  function multipart_form($name="", $title="", $action="", $method="post") {
    $this->form($name, $title, $action, $method, "multipart/form-data");
  }
} // multipart_form class

// ----------------------------------------------------------------------
/**
* Form Field class
* Abstract field class, for deriving various form field types.
* This class holds the basic properties and methods which are
* inherited by all form fields.
* @package form
*/
class form_field extends HTMLObject {
  /** Type of field eg: text, hidden, checkbox etc. */
  var $type = "";
  /** Value of the field */
  var $value;
  /** Label to put against the field. NB: this is not displayed unless you make it so. */
  var $label;
  /** True if field is NOT readonly */
  var $editable = EDITABLE;
  /** True if field is disabled */
  var $disabled = false;
  /** If false, disable autocomplete on this field. Mainly used with
      password fields to stop browsers saving passwords insecurely. */
  var $autocomplete = true;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $style    Style to apply to the field
  */
  function form_field($name="", $label="", $value="", $editable=EDITABLE, $css="") {
    $this->setname($name);
    $this->label = $label;
    $this->value = $value;
    $this->editable = $editable;
    $this->setcss($css);
  }
  /** Set the field to be display-only (non-editable) */
  function set_displayonly() {
    $this->editable = false;
  }
  /** Set the label associated with the field
  * @param string $label The label to set for this field
  */
  function setlabel($label)  {
    $this->label = $label;
  }
  /** Set the value of the field
  * @param mixed $value The value to set for this field
  */
  function setvalue($value)  {
    if (!is_array($value)) $value = trim($value);
    $this->value = $value;
  }
  /**
  * Returns current element as a hidden field.
  * @return string HTML for hidden field version of current field.
  */
  function as_hiddenfield() {
    $hid = new form_hiddenfield($this->name, $this->value);
    return $hid->render();
  }
  /**
  * Returns current element as text, with associated hidden field.
  * @return string HTML for hidden field version of current field.
  */
  function as_displayonly() {
    $s = $this->as_hiddenfield();
    if ( isset($this->value) ) $s .= "<strong>$this->value</strong>";
    return $s;
  }
  /**
  * Disable the autocomplete form functionality of the browser for
  * this field. Browsers have, mainly for password fields, been offering
  * auto-completion of contents. This function will stop this from being
  * done, in browsers which obey the "autocomplete='off'" attribute.
  * By default autocomplete is "true" for every field. The attribute is
  * only ever rendered when autocomplete is "false".
  */
  function disable_autocomplete() {
    $this->autocomplete = false;
  }
  /**
  * Render common field properties
  * This method renders the common field properties. Used in the
  * descendant classes to generate the property tags.
  * $return string Common attribute property tags for current field
  * @access private
  */
  function field_attributes() {
    global $RESPONSE;
    $html = "";
    // Add in standard HTML attributes..
    $html .= $this->attributes();
    // Add in our field-specific attributes..
    if ($this->disabled) $html .= " disabled";
    // Readonly behaviour depends on type..
    if (!$this->editable) {
      switch ($this->type) {
        // These elements support the readonly property..
        case "text":
        case "password":
        case "textarea":
          $html .= " readonly";
          break;
        case "hidden":
          break;
        // All the rest have to be disabled. They also
        // render hidden fields, to allow value submit.
        // See each relevant element class.
        default:
          if (!$this->disabled) {
            $html .= " disabled";
          }
      } // switch
    }
    // Disable autocomplete if flagged..
    if (!$this->autocomplete) {
      $html .= " autocomplete=\"off\"";
    }
    return $html;
  }
} // form_field class

// ----------------------------------------------------------------------
/**
* Label Field class
* This is just to display a bare value where the form element would
* normally be, and where we dont want to use a form_textfield in
* non-editable mode because we don't want the hidden field..
* @package form
*/
class form_labelfield extends form_field {
  /**
  * Constructor
  * Create a label field object. Sets basic field attributes.
  * @param string  $label The label which can be displayed alongside the field
  * @param string  $value The value of the field
  * @param string  $css Optional style of class to apply
  */
  function form_labelfield($label="", $value="", $css="") {
    $this->form_field("nullname", $label, $value);
    $this->setcss($css);
    $this->type = "label";
  }
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @see render()
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    return $this->value;
  }
} // form_labelfield class

// ----------------------------------------------------------------------
/**
* Hidden Field class
* This class generates a hidden field.
* @package form
*/
class form_hiddenfield extends form_field {
  /**
  * Constructor
  * Create a hidden field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $value    The value of the field
  */
  function form_hiddenfield($name="", $value="") {
    $this->form_field($name, "", $value);
    $this->type = "hidden";
  }
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @see render()
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"$this->type\"";
    $html .= $this->field_attributes();
    if (isset($this->value)) {
      $html .= " value=" . quoted_valuestring($this->value);
    }
    $html .= ">";
    return $html;
  }
} // form_hiddenfield class

// ----------------------------------------------------------------------
/**
* Button Field class
* This virtual class generates a standard form button field. This is
* a virtual class, used to provide a basis for real buttons. Do not
* instantiate this as an object - use the descendants instead.
* @package form
*/
class form_buttonfield extends form_field {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name  The name of the button
  * @param string  $value The value of the button
  * @param string  $css   CSS class or style to apply to the button
  */
  function form_buttonfield($name="", $value="", $css="") {
    $this->form_field($name, "", $value);
    $this->setcss($css);
  }
  // ....................................................................
  /**
  * Renders the form button as HTML.
  * @see render()
  * @return string  HTML rendering of button
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"$this->type\"";
    $html .= $this->field_attributes();
    if (isset($this->value)) $html .= " value=\"$this->value\"";
    $html .= ">";
    return $html;
  }
} // form_buttonfield class

// ----------------------------------------------------------------------
/**
* Submit Button class
* This class generates a standard form submit button field.
* @package form
*/
class form_submitbutton extends form_buttonfield {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name  The name of the button
  * @param string  $value The value of the button
  * @param string  $css   CSS class or style to apply to the button
  */
  function form_submitbutton($name="", $value="", $css="") {
    $this->form_buttonfield($name, $value, $css);
    $this->type = "submit";
  }
} // form_submitbutton class

// ----------------------------------------------------------------------
/**
* Reset Button class
* This class generates a standard form reset button field.
* @package form
*/
class form_resetbutton extends form_buttonfield {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name  The name of the button
  * @param string  $value The value of the button
  * @param string  $css   CSS class or style to apply to the button
  */
  function form_resetbutton($name="", $value="", $css="") {
    $this->form_buttonfield($name, $value, $css);
    $this->type = "reset";
  }
} // form_resetbutton class

// ----------------------------------------------------------------------
/**
* Image Button class
* This class generates a standard form image button field.
* @package form
*/
class form_imagebutton extends form_buttonfield {
  /** Popup confirmation text to display on button click
      @access private */
  var $confirm_text = "";
  /** Whether to allow form submit on-click
      @access private */
  var $onclick_form_submit = false;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name  The name of the button
  * @param string  $value The value of the button
  * @param string  $css   CSS class or style to apply to the button
  */
  function form_imagebutton($name="", $value="", $css="", $src="", $tooltip="", $width="", $height="", $border=0) {
    $this->form_buttonfield($name, $value, $css);
    $this->type = "image";
    if ($name != "") $this->setalt($name);
    if ($src != "") {
      $this->setimage($src, $tooltip, $width, $height, $border);
    }
  }
  // ....................................................................
  /**
  * Sets the image to display for this button.
  * Usually these details are specified in the initial instantiation
  * @param string  $src     URL or path to image for the button
  * @param string  $tooltip Button image tooltip (title tag) content
  * @param integer $width   Button image width (px)
  * @param integer $height  Button image height (px)
  * @param string  $border  Size of border around image
  */
  function setimage($src, $tooltip="", $width="", $height="", $border=0) {
    $this->setsrc($src);
    $this->settitle($tooltip);
    if ($width != "" || $height != "") {
      $this->setwidth ( (($width  != "") ? $width  : 0) );
      $this->setheight( (($height != "") ? $height : 0) );
    }
    else {
      if (extension_loaded("gd") && file_exists($src)) {
        $szinfo = getimagesize($this->src);
        $this->setwidth ( $szinfo[0] );
        $this->setheight( $szinfo[1] );
      }
    }
    $this->setborder($border);
  }
  // ....................................................................
  /**
  * Set the confirmation text to popup on click using Javascript.
  * @param string  $conf Text to display in the confirmation popup.
  */
  function set_confirm_text($conf) {
    $this->confirm_text = $conf;
  }
  // ....................................................................
  /**
  * This renders the image button as HTML. If we have onclick then we render
  * this as a simple image with a javascript URL rather than a INPUT
  * form element of type "image". This is done to cope with Netscape's
  * lack of an onclick event handler for INPUT TYPE=IMAGE, and we
  * don't use BUTTON since that's only HTML4.
  * @return string HTML rendering of button
  */
  function html($name="") {
    global $RESPONSE;
    if ($name != "") $this->name = $name;
    // Set name to image file if not given..
    if ($this->name == "" && $this->src != "") {
      $fbits = explode(".", basename($this->src));
      $this->name = $fbits[0];
    }
    // Set tooltip to name if not given..
    if ($this->title == "") {
      $this->settitle($this->name);
    }
    // Set alt to name if not given..
    if ($this->alt == "") {
      $this->setalt($this->name);
    }
    // Lame Netscape doesn't support onclick, so use HREF surrogate..
    if ($this->onclick && isset($RESPONSE) && $RESPONSE->browser == BROWSER_NETSCAPE) {
      $html  = "<a href=\"javascript:$this->onclick\"";
      if ($this->tooltip != "") {
        $html .= " onmouseover=\"window.status='$this->tooltip'; return true;\"";
        $html .= " onmouseout=\"window.status=''; return true;\"";
      }
      $html .= ">";
      $myimg = new img($this->src, $this->name, $this->tooltip, $this->width, $this->height);
      $myimg->setborder($this->border);
      if ($this->class != "") $myimg->setclass($this->class);
      if ($this->style != "") $myimg->setstyle($this->style);
      $html .= $myimg->render();
      $html .= "</a>";
    }
    else {
      // This will act as a submit button by default, and
      // will return its value with the form..

      // Onclick/confirmation popup option..
      if (!$this->onclick && $this->confirm_text != "") {
        $this->onclick="return confirm('$this->confirm_text');";
      }
      if ($this->onclick) {
        // Append return false if form submit not allowed. Since the onclick
        // event is defined, we assume that the user's Javascript handler will
        // submit the form if required, unless this flag has been set true..
        if (!$this->onclick_form_submit) {
          if ($this->onclick != "" && substr($this->onclick, -1) != ";") {
            $this->onclick .= ";";
          }
          $this->onclick .= "return false;";
        }
      }
      // Dimensions via style setting..
      if (!isset($RESPONSE) || $RESPONSE->browser != BROWSER_NETSCAPE) {
        if ($this->width)  $this->setstyle("width:$this->width"   . "px;");
        if ($this->height) $this->setstyle("height:$this->height" . "px;");
      }
      // Render the HTML..
      $html  = "<input";
      $html .= " type=\"$this->type\"";
      $html .= $this->field_attributes();
      $html .= ">";
    }
    return $html;
  }

} // form_imagebutton class

// ----------------------------------------------------------------------
/**
* Text Field class. This class generates a text field.
* @package form
*/
class form_textfield extends form_field {
  /** Format specifier (WML)
      @access private */
  var $format;
  /** Maximum user-enterable chars for field
      @access private */
  var $maxlength;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function form_textfield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_field($name, $label, $value, $editable, $css);
    $this->size = $width;
    $this->type = "text";
    $this->format = "*m";
  } // form_textfield
  // ....................................................................
  /**
  * Set the field width (usually in characters).
  * @param integer $width Width of element in characters
  * @param integer $maxlength Maximum no. of characters in field
  */
  function set_width($width, $maxlength=false) {
    $this->size = $width;
    if ($maxlength) {
      $this->maxlength = $maxlength;
    }
  } // set_width
  // ....................................................................
  /**
  * Set field format (WML)
  * @param string $format     WML format specifier.
  */
  function set_format($format) {
    $this->format = $format;
  } // set_format
  // ....................................................................
  /**
  * This renders the field as WML.
  * @return string The field as WML.
  */
  function wml($name="") {
    $wml = "";
    if ($name != "") $this->name = $name;
    if ($this->label != "")      $wml .= $this->label . ": ";
    $wml .= "<input type=\"$this->type\"";
    $wml .= " name=\"$this->name\"";
    if ($this->label != "")      $wml .= " title=\"$this->label\"";
    if ($this->value != "")      $wml .= " value=\"$this->value\"";
    $wml .= " size=\"$this->size\"";
    if ($this->format != "")     $wml .= " format=\"$this->format\"";
    if (isset($this->tabindex))  $wml .= " tabindex=\"$this->tabindex\"";
    if (isset($this->maxlength)) $wml .= " maxlength=\"$this->maxlength\"";
    $wml .= " />";
    return $wml;
  } // wml
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"$this->type\"";
    $html .= $this->field_attributes();
    if (isset($this->value)) $html .= " value=" . quoted_valuestring($this->value);
    if (isset($this->maxlength)) $html .= " maxlength=\"$this->maxlength\"";
    $html .= ">";
    return $html;
  } // html
} // form_textfield class

// ----------------------------------------------------------------------
/**
* Displayonly Field class
* Extends the textfield class. This class renders a textfield as
* text rather than a textbox control. The form value is still
* submitted, but as a hidden field in the form..
* @package form
*/
class form_displayonlyfield extends form_textfield {
  /**
  * Constructor
  * Create a disply-only field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  */
  function form_displayonlyfield($name="", $label="", $value="") {
    $this->form_textfield($name, $label, $value);
    $this->type = "displayonly";
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html() {
    return $this->as_displayonly();
  }
} // form_displayonlyfield

// ----------------------------------------------------------------------
/**
* Password Field class
* Password field element. Same as textfield, but masks user input.
* @package form
*/
class form_passwordfield extends form_textfield {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function form_passwordfield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_textfield($name, $label, $value, $editable, $css, $width);
    $this->type = "password";
  }
} // form_passwordfield class

// ----------------------------------------------------------------------
/**
* File Upload Field class
* A field for uploading files to the webserver. If used with a 'form'
* object, the form will be automatically rendered with the proper
* encoding type by setting the 'enctype' in the form tag.
* Example:
*  $upField = new form_fileuploadfield("userfile", "Upload file", "", 500000);
*  $upform  = new form("upload_frm", "File Upload", "upload.php");
*  $upform->add($upField);
*  $upform->add_button(new submit_button("Upload", "Upload"));
*  echo $upform->render();
* @package form
*/
class form_fileuploadfield extends form_field {
  /** Maximum file suze in bytes
      @access private */
  var $max_file_size = 16384;
  /** Whether to include the max size in the form
      @access private */
  var $include_maxsize = false;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name         The name of the field
  * @param string  $label        The label which can be displayed alongside the field
  * @param string  $value        The value of the field, ie. the filename/path
  * @param integer $maxsize      Maximum size of upload file in bytes
  * @param boolean $incl_maxsize True if we include the MAX_FILE_SIZE hidden field when rendering
  */
  function form_fileuploadfield($name="", $label="", $value="", $maxsize=16384, $incl_maxsize=false) {
    $this->form_field($name, $label, $value);
    $this->type = "file";
    $this->max_file_size = $maxsize;
    $this->include_maxsize = $incl_maxsize;
  }
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"$this->type\"";
    $html .= $this->field_attributes();
    if (isset($this->value)) $html .= " value=\"$this->value\"";
    $html .= ">";
    // Optional hidden field with maximum filesize setting..
    if ($this->include_maxsize) {
      $hid = new form_hiddenfield("MAX_FILE_SIZE", $this->max_file_size);
      $html .= $hid->render();
    }
    return $html;
  } // html
} // form_fileuploadfield class

// ----------------------------------------------------------------------
/**
* Combo Field class
* A field for producing combo boxes (dropdown select menus) or
* multi-line list-boxes, either of which may be single-select or
* multiple select.
* @package form
*/
class form_combofield extends form_textfield {
  // Public
  /** Boolean, multiple item select */
  var $multiselect;

  // Private
  /** Items to display, assoc. array
      @access private */
  var $itemlist;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $lines    Number of lines to show in the drop-down combo box/listbox
  * @param boolean $multi    True if the combo is multiple-select, else false
  */
  function form_combofield($name="", $label="", $value="", $editable=EDITABLE, $css="", $lines=1, $multi=SINGLESELECT) {
    $this->form_textfield($name, $label, $value, $editable, $css, STD_WIDTH);
    $this->size = $lines;
    $this->multiselect = $multi;
    $this->itemlist = new dat_keyvalues();
    $this->type = "select";
  } // form_combofield
  // ....................................................................
  /**
  * Set select field width. Note that selects can only have width set
  * using a style, since they normally auto-set the width to the max
  * length of their options display strings.
  * @param integer $width Width of select element in pixels
  */
  function set_width($widthpx) {
    if (!strstr($widthpx, ":")) $widthpx = "width:" . $widthpx . "px;";
    $this->setstyle($widthpx);
  } // width
  // ....................................................................
  /**
  * Set select field size in lines.
  * @param integer $lines Number of lines to display in the select element.
  */
  function set_size($lines) {
    $this->size = $lines;
  } // set_size
  // ....................................................................
  /**
  * Add a data item
  * Adds a key=>value pair into the combo options collection.
  * @param string $key   The key to use (the field 'value' or ID)
  * @param string $value The value to assign to the key (displayed value)
  */
  function additem($key, $value="???") {
    if ($value == "???") $value = $key;
    $this->itemlist->additem($key, $value);
  } // additem
  // ....................................................................
  /** Clears any existing items */
  function clearitems() {
    $this->itemlist->clearitems();
  } // clearitems
  // ....................................................................
  /**
  * Add ready-made data
  * Use a ready-made, piping hot source of data. This should be a
  * normal Key/Value pair associative array..
  * @param array $data   The array of key=>value pairs to add
  */
  function ovenready_data($data) {
    $this->itemlist->data = $data;
  } // ovenready_data
  // ....................................................................
  /**
  * Add ready-made data. Populates the combo field data from a pre-run query.
  * @param resource $query          An Axyl query object, pre-executed, with data
  * @param string   $keyfield       The name of the keyfield in the data
  * @param string   $displayfields  The names of displayfields, delimited by "|"
  */
  function add_querydata($query, $keyfield, $displayfields) {
    $this->itemlist->add_querydata($query, $keyfield, $displayfields);
  } // add_querydata
  // ....................................................................
  /**
  * This renders the field as WML.
  * @return string The field as WML.
  */
  function wml($name="") {
    $wml = "";
    if ($name != "") $this->name = $name;
    $selix = $this->itemlist->getitemindex($this->value);
    if ($this->editable) {
      $wml .= "<select name='$this->name'";
      if ($selix > 0) $wml .= " ivalue='$selix'";
      $wml .= ">";
      // Options list
      if ($this->itemlist->hasdata()) {
        foreach ($this->itemlist->data as $key => $displayvalue) {
          $wml .= "<option value=" . $key . ">";
          $wml .= $displayvalue . "</option>";
        }
      }
      $wml .= "</select>";
    } else {
      $this->value = $this->itemlist->getitem($this->value);
      $f = new form_textfield($this->name, $this->label, $this->value, DISPLAY_ONLY, $this->style, $this->size);
      $wml = $f->wml();
    }
    return $wml;
  } // wml
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    $html = "";
    if ($name != "") $this->name = $name;

    // Standard SELECT in html..
    $saved_name = $this->name;
    if ($this->multiselect) $this->name .= "[]";
    $html .= "<select";
    $html .= $this->field_attributes();
    if ($this->multiselect) $html .= " multiple";
    $html .= ">";
    $this->name = $saved_name;

    // HTML Options list
    if ($this->itemlist->hasdata()) {
      if (!is_array($this->value) && stristr($this->value, "regex=")) {
        // The value associated with this combo/select is a
        // regular expression to match the relevant key..
        $regbits = explode("=", $this->value);
        $regexp = $regbits[1];
      }
      else $regexp = "";

      $selvalues = "";
      foreach ($this->itemlist->data as $key => $displayvalue) {
        $selected = "";
        if ($regexp != "") {
          // Do regular expression match test..
          if (!empty($regexp) && !empty($key)) {
            if (ereg($regexp, $key)) {
              $selected = "selected ";
              $this->value = $key;
            }
          }
        }
        else {
          // Might be in an array if multiselect..
          if (is_array($this->value)) {
            if (count($this->value) > 0) {
              foreach($this->value as $elem) {
                if ($key == $elem) {
                  $selected = "selected ";
                  break;
                }
              }
            }
            else {
              // An empty array should match a nullstring..
              if ($key == "") {
                $selected = "selected ";
              }
            }
          }
          else {
            // Do normal simple equality test. Take care over the
            // sometime equivalence of 0 and nullstrings..
            if ($this->value === "") {
              if ($key === "") {
                $selected = "selected ";
              }
            }
            else {
              if ($key == $this->value) {
                $selected = "selected ";
              }
            }
          }
        }
        // Render the select option..
        $html .= "<option " . $selected . "value=\"$key\">$displayvalue";

        // Remember selected values..
        if ($selected != "") {
          if ($selvalues != "") {
            $selvalues .= ",";
          }
          $selvalues .= $key;
        }
      }
    }
    $html .= "</select>";

    if (!$this->editable && !$this->disabled) {
      // Not editable, so render it as a hidden field
      // for the data, and plain text for the user..
      // Implement as hidden field value, plus visible
      // normal text display-value..
      $name = $this->name;
      if ($this->multiselect) $name .= "[]";
      $hidf = new form_hiddenfield($name, $selvalues);
      $html .= $hidf->render();
    }
    return $html;
  } // html
} // form_combofield class

// ----------------------------------------------------------------------
/**
* Jumpmenu Field class
* A special case of combofield where we define key/value data made up
* of display-value and URL pairs.
* @package form
*/
class form_jumpmenu extends form_combofield {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $lines    Number of lines to show in the drop-down combo box/listbox
  */
  function form_jumpmenu($name="", $label="", $value="", $editable=EDITABLE, $css="", $lines=1) {
    $this->form_combofield($name, $label, $value, $editable, $css, $lines);
  } // form_jumpmenu
  // ....................................................................
  /**
  * This renders the field as WML.
  * @return string The field as WML.
  */
  function wml($name="") {
    $wml = "";
    if ($name != "") $this->name = $name;
    if ($this->editable) {
      $wml .= "<select name=\"$this->name\"";
      if ($this->label != "") $wml .= " title=\"" . $this->label . "\"";
      if ($this->value != "") $wml .= " ivalue=\"" . $this->value . "\"";
      $wml .= ">";
      // Options list
      if ($this->itemlist->hasdata()) {
        $jix = 1;
        foreach ($this->itemlist->data as $displayvalue => $jumpurl) {
          if (function_exists("href_rewrite")) {
            $jumpurl = href_rewrite($jumpurl);
          }
          $wml .= "<option value=\"$jix\"";
          $wml .= " onpick=\"$jumpurl\"";
          $wml .= ">";
          $wml .= $displayvalue . "</option>";
          $jix += 1;
        }
      }
      $wml .= "</select>";
    }
    else {
      $this->value = $this->itemlist->getitem($this->value);
      $f = new form_textfield(
                  $this->name,
                  $this->label,
                  $this->value,
                  DISPLAY_ONLY,
                  $this->style,
                  $this->width
                  );
      if (isset($this->class) && $this->class != "") {
        $f->setcss($this->class);
      }
      $wml = $f->wml();
    }
    return $wml;
  } // wml
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    $html = "";
    if ($name != "") $this->name = $name;
    // Standard jumpmenu in html..
    $html .= "<select";
    $html .= $this->field_attributes();
    $html .= ">";

    // HTML Options list
    if ($this->itemlist->hasdata()) {
      if (stristr($this->value, "regex=")) {
        // The value associated with this combo/select is a
        // regular expression to match the relevant key..
        $regbits = explode("=", $this->value);
        $regexp = $regbits[1];
      }
      else $regexp = "";

      foreach ($this->itemlist->data as $displayvalue => $jumpurl) {
        $html .= "<option";
        if ($regexp != "") {
          // Do regular expression match test..
          if (!empty($regexp) && !empty($displayvalue)) {
            if (ereg($regexp, $displayvalue)) {
              $html .= " selected";
              $this->value = $displayvalue;
            }
          }
        }
        else {
          // Do normal equality test..
          if ($displayvalue == $this->value) $html .= " selected";
        }
        $html .= " value=\"$displayvalue\"";
        $html .= " onclick=\"javascript:document.location.href='$jumpurl'\"";
        $html .= ">";
        $html .= $displayvalue;
      }
    }
    $html .= "</select>";

    if (!$this->editable) {
      // Not editable, so render it as a hidden field
      // for the data, and plain text for the user..
      // Implement as hidden field value, plus visible
      // normal text display-value..
      $hidf = new form_hiddenfield($this->name, $this->value);
      $html .= $hidf->render();
      $html .= "<strong>$this->itemlist->getitem($this->value)</strong>";
    }
    return $html;
  } // html
} // form_jumpmenu class

// ----------------------------------------------------------------------
/**
* Memo Field class
* A field which renders a textarea form element. These are used to
* allow people to input large tracts of text by typing or copy/paste.
* @package form
*/
class form_memofield extends form_textfield {
  /** Number of rows in the text box */
  var $rows;
  /** Wrapping mode: 'virtual', 'physical', or 'off'
      @access private */
  var $wrapmode;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of the memo field in characters
  * @param integer $rows     Height of the memo field in lines
  * @param string  $wrapmode Wrapping mode: 'virtual', 'physical', or 'off'
  */
  function form_memofield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH, $rows=5, $wrapmode="virtual") {
    $this->form_textfield($name, $label, $value, $editable, $css, $width);
    $this->rows = $rows;
    $this->type = "textarea";
    $this->wrapmode = $wrapmode;
  } // form_memofield
  // ....................................................................
  /**
  * Set the wrap mode for the textarea.
  * @param string $mode Wrapping mode: 'virtual', 'physical', or 'off'
  */
  function set_wrapmode($mode="virtual") {
    $this->wrapmode = $mode;
  } // set_wrapmode
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @see render()
  * @return string The field as HTML.
  */
  function html($name="") {
    global $RESPONSE;
    // Figure out a legal wrapmode..
    if (isset($RESPONSE)) {
      if ($RESPONSE->browser == BROWSER_MOZILLA || $RESPONSE->browser == BROWSER_NETSCAPE) {
        if ($this->wrapmode == "physical") $this->wrapmode = "hard";
        elseif ($this->wrapmode == "virtual") $this->wrapmode = "soft";
      }
      else {
        if ($this->wrapmode == "hard") $this->wrapmode = "physical";
        elseif ($this->wrapmode == "soft") $this->wrapmode = "virtual";
      }
    }

    $html = "";
    if ($name != "") $this->name = $name;
    $html .= "<textarea";
    $html .= $this->field_attributes();
    if (isset($this->size))     $html .= " cols=\"$this->size\"";
    if (isset($this->rows))     $html .= " rows=\"$this->rows\"";
    if (isset($this->wrapmode)) $html .= " wrap=\"$this->wrapmode\"";
    $html .= ">";
    if (isset($this->value)) $html .= $this->value;
    $html .= "</textarea>";
    return $html;
  } // html
  // ....................................................................
  /**
  * Set the number of rows for this memo field.
  * @param integer $rows The number of rows for this memo field.
  */
  function set_rows($rows) {
    $this->rows = $rows;
  } // set_rows
} // form_memofield class

// ----------------------------------------------------------------------
/**
* Checkbox Field class
* A field which renders a checkbox form element.
* @package form
*/
class form_checkbox extends form_field {
  /** True if checkbox is ticked/checked */
  var $checked;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param boolean $checked  True if element should be checked/ticked
  */
  function form_checkbox($name="", $label="", $value="", $editable=EDITABLE, $css="", $checked=FALSE) {
    $this->form_field($name, $label, $value, $editable, $css="");
    $this->checked = $checked;
    if ($value != "") $this->value = $value;
    else $this->value = $name;
    $this->type = "checkbox";
  } // form_checkbox
  // ....................................................................
  /** Sets the checkbox to the checked state. */
  function check() {
    $this->checked = true;
  } // check
  // ....................................................................
  /** Sets the checkbox to the unchecked state. */
  function uncheck() {
    $this->checked = false;
  }  // uncheck
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"checkbox\"";
    $html .= $this->field_attributes();
    if ($this->checked) $html .= " checked";
    $html .= " value=\"$this->value\"";
    $html .= ">";
    // Only send the readonly field if its checked..
    if (!$this->editable && $this->checked) {
      $hidf = new form_hiddenfield($this->name, $this->value);
      $html .= $hidf->render();
    }
    return $html;
  } // html
} // form_checkbox class

// ----------------------------------------------------------------------
/**
* Radio Button Field class
* A field which renders a radio form element.
* @package form
*/
class form_radiobutton extends form_field {
  /** True if this element should be selected */
  var $checked;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param boolean $checked  True if element should be checked/ticked
  */
  function form_radiobutton($name="", $label="", $value="", $editable=EDITABLE, $css="", $checked=FALSE) {
    $this->form_field($name, $label, $value, $editable, $css="");
    $this->checked = $checked;
    if ($value != "") $this->value = $value;
    else $this->value = $name;
    $this->type = "radio";
  } // form_radiobutton
  // ....................................................................
  /** Sets the checkbox to the checked state. */
  function check() {
    $this->checked = true;
  } // check
  // ....................................................................
  /** Sets the checkbox to the unchecked state. */
  function uncheck() {
    $this->checked = false;
  } // uncheck
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") $this->name = $name;
    $html  = "<input";
    $html .= " type=\"radio\"";
    $html .= $this->field_attributes();
    if ($this->checked) $html .= " checked";
    $html .= " value=\"$this->value\"";
    $html .= ">";
    if (!$this->editable) {
      $hidf = new form_hiddenfield($this->name, $this->value);
      $html .= $hidf->render();
    }
    return $html;
  }
} // form_radiobutton class

// ----------------------------------------------------------------------
/**
* Radio Group Field class
* A field which groups a set of radio form elements together.
* This element actually extends the form_combofield, since it is basically
* a list of selectable options, like a single-select dropdown menu is.
* As such, you can add key->value options using the 'additem' method, just
* like for combo fields.
* @package form
*/
class form_radiogroup extends form_combofield {
  /** Group orientation: 'horizontal' or 'vertical' */
  var $orientation = "horizontal";
  /** Side to place label: 'right' or 'left' */
  var $labelside = "right";
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $style    Style to apply to the field
  * @param boolean $orientation Group orientation: 'horizontal' (default) or 'vertical'
  */
  function form_radiogroup($name="", $label="", $value="", $editable=EDITABLE, $css="", $orientation="horizontal") {
    $this->form_combofield($name, $label, $value, $editable, $css);
    $this->orientation = $orientation;
    if ($value != "") $this->value = $value;
    else $this->value = $name;
    $this->type = "radiogroup";
  } // form_radiogroup
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    $html = "";
    if ($name != "") $this->name = $name;
    if ($this->itemlist->hasdata()) {
      $Trg = new table("radiogroup");
      $Trg->setwidth("");
      $first = true;
      while ( list($key, $displayvalue) = each($this->itemlist->data) ) {
        $rad = new form_radiobutton($this->name, $displayvalue, $key);
        if ($this->value == $key) $rad->check();
        if ($this->labelside == "left") $radbtn = "$displayvalue&nbsp;" . $rad->render();
        else $radbtn = $rad->render() . "&nbsp;$displayvalue";
        if ($first) {
          if (isset($this->style) && $this->style != "") $Trg->tbody($this->style);
          elseif (isset($this->class) && $this->class != "") $Trg->tbody($this->class);
          $Trg->tr();
          $first = false;
        }
        elseif ($this->orientation == "vertical") {
          $Trg->tr();
        }
        $Trg->td( $radbtn );
      }
      $html .= $Trg->render();
    }
    return $html;
  } // html
} // form_radiogroup class

// ----------------------------------------------------------------------
/**
* Image Field class
* This is a hybrid field. It is basically a text field where the
* value is expected to be the location of an image, either a real path
* to a file or a URI.
* @package form
*/
class form_imagefield extends form_textfield {
  /** Prefix to image URL/path. This will be prefixed to the value. */
  var $prefix;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  * @param string  $prefix   Directory path prefix to apply to image file path
  */
  function form_imagefield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH, $prefix="") {
    $this->form_textfield($name, $label, $value, $editable, $css, $width);
    if ($prefix != "" && substr($prefix, -1) != "/") {
      $prefix .= "/";
    }
    $this->prefix = $prefix;
  } // form_imagefield
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    $s  = "<table border=0 cellpadding=1 cellspacing=0>";
    $s .= "<tr><td colspan=2 valign=top>" . form_textfield::html() . "</td></tr>";
    $s .= "<tr><td valign=top>";
    if ($this->value != "") {
      $imgurl = $this->prefix . $this->value;
      $img = new image("img_" . $this->name, $imgurl, "Image at $imgurl");
      $s .= $img->html();
    }
    $s .= "</td>";
    $s .= "<td>";
    if ($imgurl != "") {
      if ($this->prefix != "") {
        $s .= "Path: $this->prefix<br>";
      }
      $siz = getimagesize($imgurl);
      if (isset($siz[0])) {
        $s .= "Size: " . $siz[0] . " x " . $siz[1] . "<br>";
        switch ($siz[2]) {
          case 1: $type = "GIF"; break;
          case 2: $type = "JPG"; break;
          case 3: $type = "PNG"; break;
          case 4: $type = "SWF"; break;
          case 5: $type = "PSD"; break;
          case 6: $type = "BMP"; break;
          default: "??";
        }
        $s .= "Type: $type";
      }
    }
    $s .= "</td></tr>";
    $s .= "</table>\n";
    return $s;
  } // html
} // form_imagefield class

// ----------------------------------------------------------------------
/**
* Tandem Field class
* This is a rather bizarre hybrid field. It is basically a pair of text
* fields. The second field is initialised with the same properties as the
* first but can be individually styled etc. by accessing the relevant
* property in this class. The name given to the tandem field is the
* same as the parent field, but with the prefix "tandem_" applied.
* @package form
*/
class form_tandemfield extends form_textfield {
  /** The second field object in tandem with this one */
  var $tandemfield;
  /** Overall width in pixels */
  var $tandemwidth;
  /** True if main field should be suppressed */
  var $summaryfield = false;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function form_tandemfield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_textfield($name, $label, $value, $editable, $css, $width);
    $this->tandemfield = new form_textfield("tandem_$name", $label, $value, $editable, $css, $width);
  } // form_tandemfield
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * This renders the field as HTML.
  * @see render()
  * @return string The field as HTML.
  */
  function html() {
    $Tt = new table();
    if (isset($this->tandemwidth)) {
      $Tt->setwidth($this->tandemwidth);
    }
    $Tt->tr();
    if ($this->summaryfield) {
      $Tt->td( "&nbsp;" );
      $Tt->td( $this->tandemfield->html(), "padding-left:5px" );
    }
    else {
      $Tt->td( form_textfield::html(), "padding-right:5px" );
      $Tt->td( $this->tandemfield->html(), "padding-left:5px" );
    }
    $Tt->set_width_profile("50%,50%");
    return $Tt->render();
  } // html
} // form_tandemfield class

// ----------------------------------------------------------------------
?>