<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    database-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing DATABASES                      */
/*                                                                      */
/* ******************************************************************** */
/** @package database */

/** Stopwatch microtimer */
include_once("timer-defs.php");

/** Connect persistent to DB */
define("PERSISTENT", true);

/** Connect non-persistent to DB */
define("NOT_PERSISTENT", false);

/** Default datasource for queries @see add_database() */
define("DEFAULT_DATASOURCE", true);

// ----------------------------------------------------------------------
/**
* Datasources
* A datasources class is just a bunch of databases. If you want
* to access a database, register it in here first, then you
* can select it to perform queries on later.
* @package database
*/
class datasources {
  /** An array of database objects. All databases we can use as datasources */
  var $database;
  /** Default database name */
  var $db_name_default = "";
  /** Name of currently selected database */
  var $db_name_selected = "";
  // ....................................................................
  /** Constructor */
  function datasources() { }
  // ....................................................................
  /**
  * Constructor
  * Add a new base to our list of datasources. The dbtype and the name
  * are the only mandatory parameters.
  * @param string  $dbtype    The type of database eg: 'postgres', 'mssql' etc.
  * @param string  $name      The name of the database
  * @param string  $user      Name of a user who can access the database
  * @param string  $passwd    The password the user can access the database with
  * @param string  $host      The hostname of the machine running the database (TCP/IP)
  * @param integer $port      The port number of the database server
  * @param boolean $default   True if the database is the default database
  */
  function add_database($dbtype, $name, $user="", $passwd="", $host="", $port=0, $default=false) {
    switch ($dbtype) {
      case "postgres":
        include_once("db-postgres.php");
        $this->database[$name] = new db_postgres($name, $user, $passwd, $host, $port);
        break;
      case "odbc":
        include_once("db-odbc.php");
        $this->database[$name] = new db_odbc($name, $user, $passwd, $host, $port);
        break;
      case "mssql_server":
        include_once("db-mssql-server.php");
        $this->database[$name] = new db_mssql_server($name, $user, $passwd, $host, $port);
        break;
      case "mysql":
        include_once("db-mysql.php");
        $this->database[$name] = new db_mysql($name, $user, $passwd, $host, $port);
        break;
      case "oracle":
        include_once("db-oracle.php");
        $this->database[$name] = new db_oracle($name, $user, $passwd, $host, $port);
        break;
    }
    // Make sure the default database is selected..
    if ($default) {
      // Select the default DB. This tries to
      // connect to it..
      $this->set_default($name);
      $this->select($name);

      // It is a fatal application error if the default
      // database cannot be connected..
      if (!$this->connected($name)) {
        $errmsg = "APPFATAL: " . APP_NAME . ": Default database not connected. Exit stage left.";
        error_log($errmsg, 0);
        die($errmsg);
      }
    }
    return $this;
  } // add_database
  // ....................................................................
  /**
  * This will connect it if it isn't already connected. Calling this
  * with no database name will select the default one. Returns the
  * database unique identifier, or false if none was selected.
  * @param string $db_name  The name of the database to select
  * @return resource The database resource ID
  */
  function select($db_name="") {
    global $RESPONSE;
    global $HTTP_HOST;
    $dbid = false;
    if ($db_name == "") {
      $db_name = $this->db_name_default;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $this->db_name_selected = $db_name;
      if (!$db->connected) {
        // Check if we should connect persistently..
        if (isset($RESPONSE) && $RESPONSE->InPersistentHostsList($HTTP_HOST)) {
          $connmode = PERSISTENT;
        }
        else {
          $connmode = NOT_PERSISTENT;
        }
        $db->connect($connmode);
        $this->database[$db_name] = $db;
      }
      if ($db->connected) {
        $dbid = $db->dbid;
      }
    }
    return $dbid;
  } // select
  // ....................................................................
  /**
  * Internal function to set the name of the default database.
  * The database must exist as a defined database already.
  * @param string $db_name  The name of the database
  */
  function set_default($db_name) {
    if (isset($this->database[$db_name])) {
      $this->db_name_default = $db_name;
      return $this;
    }
  } // set_default
  // ....................................................................
  /**
  * Returns the database resource ID of the given database name.
  * If dbname is not given, returns ID of currently selected DB.
  * @param string $db_name  The name of the database
  * @return resource Database resource ID
  */
  function dbid($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $res = $this->database[$db_name]->dbid;
    }
    return $res;
  } // dbid
  // ....................................................................
  /**
  * Returns the database type of the given database name.
  * If dbname is not given, returns type of DB currently selected.
  * @param string $db_name The name of the database
  * @return string Database type string
  */
  // Returns the database type of the selected database.
  function dbtype($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $res = $this->database[$db_name]->type;
    }
    return $res;
  } // dbtype
  // ....................................................................
  /**
  * Returns connected status of named database, or the currently
  * selected one if no name given.
  * @param string $db_name  The name of the database
  * @return boolean Database connection status true or false
  */
  function connected($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $res = $this->database[$db_name]->connected;
    }
    return $res;
  } // connected
  // ....................................................................
  /**
  * Connects to the database which has been selected in the mode
  * specified, or non-peristent otherwise.
  * @param boolean $persistent Whether to connect persistently or not
  * @return boolean Whether database connection was successful
  */
  function connect($persistent=NOT_PERSISTENT) {
    $connected = false;
    if (isset($this->database[$this->db_name_selected])) {
      $this->database[$this->db_name_selected]->connect($persistent);
      if ($this->database[$this->db_name_selected]->connected) {
        $connected = true;
      }
      else {
        $errmsg  = "Failed to connect to database '" . $this->name . "' ";
        $errmsg .= "type='"   . $this->type . "' ";
        $errmsg .= "host='"   . $this->host . "' ";
        $errmsg .= "port='"   . $this->port . "' ";
        $errmsg .= "user='"   . $this->user . "' ";
        $errmsg .= "passwd=" . ($this->passwd != "") ? "xxxx" : "(none) ";
        if ($persistent) $errmsg .= "persistent=yes";
        else $errmsg .= "persistent=no";
        error_log("CONNFAIL: $errmsg");
      }
    }
    return $connected;
  } // connect
  // ....................................................................
  /**
  * Disconnect the currently selected database.
  */
  function disconnect() {
    if (isset($this->database[$this->db_name_selected])) {
      $this->database[$this->db_name_selected]->disconnect();
    }
  } // disconnect
  // ....................................................................
  /**
  * Execute a query on the connected database.
  * @param string $sql The SQL query to execute on the database
  * @return resource A database query resource ID, or false if query failed
  */
  function query($sql) {
    $rid = false;
    if (isset($this->database[$this->db_name_selected])) {
      $rid = $this->database[$this->db_name_selected]->query($sql);
    }
    return $rid;
  } // query
  // ....................................................................
  /**
  * Returns SQL statement most recently executed on the current DB.
  * NB: the format and/or content of this SQL may differ from the SQL
  * originally submitted, due to database-dependent transformations,
  * hence the usefulness of this method.
  * @return string The SQL statement last executed on current database.
  */
  function get_last_sql() {
    $sql = "";
    if (isset($this->database[$this->db_name_selected])) {
      $sql = $this->database[$this->db_name_selected]->last_sql;
    }
    return $sql;
  } // get_last_sql
  // ....................................................................
  /**
  * Return the number of rows returned by a SELECT query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows returned by the query
  */
  function numrows($rid) {
    $rows = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = ($rid !== false) ? $db->numrows($rid) : 0;
    }
    return $rows;
  } // numrows
  // ....................................................................
  /**
  * Return the number of rows affected by a query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows affected by the query
  */
  function affectedrows($rid) {
    $rows = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = ($rid !== false) ? $db->affectedrows($rid) : 0;
    }
    return $rows;
  } // affectedrows
  // ....................................................................
  /**
  * Free the result of a query
  * @param resource $rid The query resource ID
  */
  function freeresult($rid) {
    if (isset($this->database[$this->db_name_selected]) && $rid !== false) {
      $this->database[$this->db_name_selected]->freeresult($rid);
    }
  } // freeresult
  // ....................................................................
  /**
  * Return the last error message.
  * @return string The last error message which was generated
  */
  function errormessage() {
    $errmsg = "";
    if (isset($this->database[$this->db_name_selected])) {
      $errmsg = $this->database[$this->db_name_selected]->errormessage();
    }
    return $errmsg;
  } // errormessage
  // ....................................................................
  /**
  * Return the specified row, as a standard (enumerated) array of
  * field values.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Enumerated array of field values
  */
  function fetch_row($rid, $rowno) {
    $rows = false;
    if (isset($this->database[$this->db_name_selected]) && $rid !== false) {
      $rows = $this->database[$this->db_name_selected]->fetch_row($rid, $rowno);
    }
    return $rows;
  } // fetch_row
  // ....................................................................
  /**
  * Return the specified row, as an associative array of fields
  * in a fieldname => value format.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Associative array of field values
  */
  function fetch_array($rid, $rowno) {
    $arr = false;
    if (isset($this->database[$this->db_name_selected]) && $rid !== false) {
      $arr = $this->database[$this->db_name_selected]->fetch_array($rid, $rowno);
    }
    return $arr;
  } // fetch_array
  // ....................................................................
  /**
  * Start a database transaction
  * @return boolean Flag indicating successful start of transaction
  */
  function begin_transaction() {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->begin_transaction();
    }
    return $res;
  } // begin_transaction
  // ....................................................................
  /**
  * Commit open database transaction
  * @return boolean Flag indicating successful commit of transaction
  */
  function commit() {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->commit();
    }
    return $res;
  } // commit
  // ....................................................................
  /**
  * Roll back the current database transaction. All queries executed
  * as part of the open transaction will be rolled back.
  * @return boolean Flag indicating successful rollback of transaction
  */
  function rollback() {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->rollback();
    }
    return $res;
  } // rollback
  // ....................................................................
  /**
  * Return a Php boolean from a database field value. The database field
  * is expected to be a container of some form of logical value. Here
  * is where we convert it according to the current database.
  * @param mixed $dbvalue The value from the database field to convert
  * @return boolean The boolean value derived from the field value
  */
  function bool_from_db_value($dbvalue) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->bool_from_db_value($dbvalue);
    }
    return $res;
  } // bool_from_db_value
  // ....................................................................
  /**
  * Return a suitable database field value to contain the value for
  * the given boolean.
  * @param boolean $boolvalue The boolean value to convert
  * @return mixed The value suitable for the database field
  */
  function db_value_from_bool($boolvalue) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->db_value_from_bool($boolvalue);
    }
    return $res;
  } // db_value_from_bool
  // ....................................................................
  /**
  * Return the current sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The current sequence value
  */
  function current_sequencevalue($sequencename, $table, $column) {
    $res = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->current_sequencevalue($sequencename, $table, $column);
    }
    return $res;
  } // current_sequencevalue
  // ....................................................................
  /**
  * Return the next sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The next sequence value
  */
  function next_sequencevalue($sequencename, $table, $column) {
    $res = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->next_sequencevalue($sequencename, $table, $column);
    }
    return $res;
  } // next_sequencevalue
  // ....................................................................
  /**
  * Set the sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param integer $newval The sequence value to set
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return boolean Whether the assignment succeeded or not
  */
  function set_sequencevalue($newval, $sequencename, $table, $column) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->set_sequencevalue($newval, $sequencename, $table, $column);
    }
    return $res;
  } // set_sequencevalue
  // ....................................................................
  /**
  * Set the database date style. This affect the format that dates will
  * be displayed in, and the format they are submitted in.
  * @param string $datestyle The date style code to set
  * @return boolean Whether the setting succeeded or not
  */
  function set_datestyle($datestyle) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->set_datestyle($datestyle);
    }
    return $res;
  } // set_datestyle
  // ....................................................................
  /**
  * Set the database character encoding. This affects the encoding of
  * characters in the database.
  * @param string $encoding The character encoding to set
  * @return boolean Whether the setting succeeded or not
  */
  function set_char_encoding($encoding) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->set_char_encoding($encoding);
    }
    return $res;
  } // set_char_encoding
  // ....................................................................
  /**
  * General-purpose lock method. We pass the elements of the lock, which
  * is the list of tables to lock, and the lock-mode. The latter mode is
  * database-specific, and therefore flexible.
  * @param string $tablelist List of tables to lock, comma-delimited
  * @param string $mode Databes-specific locking-mode or type
  * @return boolean Whether the setting succeeded or not
  */
  function lock($tablelist, $mode) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->lock($tablelist, $mode);
    }
    return $res;
  } // lock
  // ....................................................................
  /**
  * Given an Axyl SQL query object, build the SQL string from it
  * in suitable format for the currently connected database server.
  * @param object $sqlquery An Axyl query object
  * @return string The SQL string built from the query object
  */
  function SQL($sqlquery) {
    $res = false;
    if (isset($this->database[$this->db_name_selected])) {
      $res = $this->database[$this->db_name_selected]->SQL($sqlquery);
    }
    return $res;
  } // SQL

} // datasources class

// ----------------------------------------------------------------------
/**
* Define a database. This is a parent class to all of the supported
* database flavours. It holds the main data describing a database
* and it's connection. The actual functionality to connect to a
* physical database and access its data is defined in the child
* classes of this one. For example, see file 'db-postgres.php'.
*
* Normal users of the system should not have to deal with this
* class directly.
*
* The datasources class is a container for multiple 'databases' or
* instances of this class.
* @package database
*/
class database {
  // Public
  /** Type of database eg: "postgres", "mssql_server".. */
  var $type = "";
  /** Name of this database */
  var $name = "";
  /** Host server of this database */
  var $host = "";
  /** Port to access it via TCP */
  var $port = 0;
  /** Default user to connect as */
  var $user = "";
  /** Default password to connect as */
  var $passwd = "";
  /** Flag true if database was connected ok */
  var $connected = false;

  // Private
  /** True if we want a persistent connection
      @access private */
  var $persistent = false;
  /** Unique identifier for database access
      @access private */
  var $dbid = false;
  /** The SQL statement last executed on this database. This
      value is set in the underlying DB module, query() method.
      @access private */
  var $executable_sql = "";
  /** The result ID last returned by a query on this DB. Also
      set in the underlying DB module query()
      @access private */
  var $rid;
  /** Microtimer for query execute timing
      @access private */
  var $timer;
  // ....................................................................
  /**
  * Constructor
  * @param string  $name    The database name
  * @param string  $user    The username of user who can access the database
  * @param string  $passwd  The user password which can access the database
  * @param string  $host    The hostname of the machine running the database
  * @param integer $port    The port number of the database server
  */
  function database($name="", $user="", $passwd="", $host="", $port=0) {
    $this->name = $name;

    // If host and port ar not specified, then
    // we assume the database is local..
    $this->host = $host;
    $this->port = $port;

    // These can be used as defaults..
    $this->user = $user;
    $this->passwd = $passwd;

    // Timer..
    $this->timer = new microtimer();
  } // database

  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function connect($persistent) {
    return false;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function disconnect() {
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function query($sql) {
    return false;
  }
  // ....................................................................
  /**
  * This method is usually called after the underlying DB module query()
  * method has executed the query. It examines the returned code and
  * if debugging is enabled it reports the query & stats accordingly.
  * Usually an internally executed method only.
  * @access private
  */
  function query_report() {
    global $SQL_EXEC_THRESHOLD, $RESPONSE;
    // Now examine the result..
    if ($this->rid != false) {
      if (debugging()) {
        $errstr  = "QOK: $this->executable_sql";
        $errstr .= " (Time: " . $this->timer->formatted_millisecs() . "mS)";
        debugbr($errstr, DBG_SQL);
      }
      // Log excessive query execution times to syslog..
      if (isset($SQL_EXEC_THRESHOLD) && $SQL_EXEC_THRESHOLD > 0) {
        if ($this->timer->millisecs() > $SQL_EXEC_THRESHOLD) {
          $errstr  = APP_NAME . ": " . $timer->formatted_millisecs() . "mS ";
          $errstr .= "Exceeds Threshold ($SQL_EXEC_THRESHOLD): $executed_sql";
          error_log($errstr);
        }
      }
    }
    else {
      // Log the failed query..
      $errstr = "QFAIL: " . APP_NAME . ": $this->executable_sql";
      error_log($errstr, 0);
      $db_err = $RESPONSE->datasource->errormessage();
      if ($db_err) $errstr .= " DBSERVER: $db_err";
      $this->last_errormsg = $errstr;
      if (debugging()) {
        debugbr($errstr, DBG_SQL);
      }
      // Set failed status for any open transaction..
      if ($global_tran->open) {
        $global_tran->failed = true;
        $global_tran->failed_msg = $errstr;
      }
    }
  } // query_report
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function numrows($rid) {
    return 0;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function affectedrows($rid) {
    return 0;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function freeresult($rid) {
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function errormessage($rid) {
    return "";
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function fetch_row($rid, $rowno) {
    return false;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function fetch_array($rid, $rowno) {
    return false;
  }
  // ....................................................................
  /**
  * Start a new database transaction.
  * @return boolean Whether transaction was started or not
  */
  function begin_transaction() {
    return $this->query("BEGIN");
  }
  // ....................................................................
  /**
  * Commit the currently open database transaction.
  * @return boolean Whether the commit succeeded or not
  */
  function commit() {
    return $this->query("COMMIT");
  }
  // ....................................................................
  /**
  * Rollback the currently open database transaction.
  * @return boolean Whether the rollback succeeded or not
  */
  function rollback() {
    return $this->query("ROLLBACK");
  }
  // ....................................................................
  /**
  * Return a Php boolean from a database field value. The database field
  * is expected to be a container of some form of logical value. Here
  * is where we convert it according to the current database.
  * @param mixed $dbvalue The value from the database field to convert
  * @return boolean The boolean value derived from the field value
  */
  function bool_from_db_value($dbvalue) {
    return ($dbvalue == 1);
  }
  // ....................................................................
  /**
  * Return a suitable database field value to contain the value for
  * the given boolean.
  * @param boolean $boolvalue The boolean value to convert
  * @return mixed The value suitable for the database field
  */
  function db_value_from_bool($boolvalue) {
    return $boolvalue ? 1 : 0;
  }
  // ....................................................................
  /**
  * Return the current sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The current sequence value
  */
  function current_sequencevalue($sequencename, $table, $column) {
    $seq = 0;
    $rid = $this->query("SELECT MAX($column) FROM $table" );
    if ($rid !== false) {
      $row = $this->fetch_row($rid, 0);
      $seq = $row[0];
    }
    return $seq;
  }
  // ....................................................................
  /**
  * Return the next sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The next sequence value
  */
  function next_sequencevalue($sequencename, $table, $column) {
    return (1 + $this->current_sequencevalue($sequencename, $table, $column));
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function set_sequencevalue($newval, $sequencename, $table, $column) {
    return true;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function set_datestyle($datestyle) {
    return true;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function set_char_encoding($encoding) {
    return true;
  }
  // ....................................................................
  /** This method must be defined in the child class.
  * @abstract
  */
  function lock($tablelist, $mode) {
    return true;
  }
  // ....................................................................
  /**
  * Given an Axyl SQL query object, build the SQL string from it
  * in suitable format for the currently connected database server.
  * @param pointer $sqlquery Pointer to an Axyl query object
  * @return string The SQL string built from the query object
  */
  function SQL($sqlquery) {
    $sql = "";
    switch (strtoupper($sqlquery->type)) {
      case "SELECT":
        $sql .= "SELECT ";
        if ($sqlquery->fields->total == 0) $sql .= "*";
        else $sql .= $sqlquery->fields->listed();
        $sql .= " FROM ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        if ($sqlquery->groupby->total > 0) {
          $sql .= " GROUP BY ";
          $sql .= $sqlquery->groupby->listed();
        }
        if ($sqlquery->orderby->total > 0) {
          $sql .= " ORDER BY ";
          $sql .= $sqlquery->orderby->listed();
        }
        break;

      case "INSERT":
        $sql .= "INSERT INTO ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->fields->total > 0) {
          $sql .= " (" . $sqlquery->fields->listed() . ")";
        }
        $sql .= " VALUES ";
        $sql .= "(" . $sqlquery->fields->values() . ")";
        break;

      case "DELETE":
        $sql .= "DELETE FROM ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        break;

      case "UPDATE":
        $sql .= "UPDATE ";
        $sql .= $sqlquery->tables->listed();
        $sql .= " SET ";
        $sql .= $sqlquery->fields->equated();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        break;
    }
    // Render any NULL values..
    $sql = str_replace("'".NULLVALUE."'", "NULL", $sql);

    // Return SQL we have built..
    return $sql;
  }
  // ....................................................................
  /**
  * Make conversions of boolean syntax found in the SQL string and
  * return the 'standardised' SQL. This assumes that Axyl SQL will
  * be written in the form 'WHERE foo=TRUE'.
  * @param string $sql SQL string to make conversions in
  * @return string The converted SQL string
  */
  function convert_boolean_syntax($sql) {
    $fixsql = $sql;
    // Quick check is more efficient then regexes..
    if (stristr($sql, "TRUE") || stristr($sql, "FALSE")) {
      $fixsql = preg_replace("/( WHERE.*?[\S]+=)TRUE/ie", "'\\1'.'1'", $sql);
      $fixsql = preg_replace("/( WHERE.*?[\S]+=)FALSE/ie", "'\\1'.'0'", $fixsql);
    }
    return $fixsql;
  }
} // database class

// ----------------------------------------------------------------------
?>