#! /bin/sh
# Set Axyl website permissions
# Execute this script from inside an Axyl website root directory to
# ensure that permissions are set correctly.
#
# With no parameters passed on the command-line, this script will
# try to find out the Webserver user and group, and assume the
# Web Administrator is 'wwwadmin'. Optionally you may pass these
# in as follows:
#      $1  -  Webserver Administrator username
#      $2  -  Webserver username
#      $3  -  Webserver group (if not supplied, assumed same as user)
#
# Ownership and group settings will then be based on these values, with
# most of the website owned by the Axyl User, and a group of the
# Webserver Administrator. Only the 'var' sub-directory differs from
# this, with a group of Webserver Group.
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
set -e

# Gotta have the power..
if [ $(id -u) != 0 ]
then
  echo "You must be root to run this script."
  exit 1
fi

# Sanity checks on where we are..
if [ ! -f "./application.xml" -o ! -d "./lib" ]
then
  echo "you must be in the root directory of an Axyl website to run"
  echo "this script. Change directory first."
  exit 1
fi

if [ `pwd` = "/" ]
then
  echo "you cannot execute this script from the root directory!"
  exit 1
fi

# Acquire the local configuration..
CONF=/etc/axyl/axyl.conf
if [ ! -f $CONF ]
then
  echo "Axyl configuration file $CONF not found!"
  exit 1
else
  . $CONF
fi

# Get access to some common stuff..
. ${AXYL_HOME}/install/axyl-common.sh

# Defaults for webserver users and group..
WWWADMIN=wwwadmin
WWWUSER=www-data
WWWGROUP=$WWWUSER

# Deal with passed-in parameters..
if [ "$1" != "" ]
then
  WWWADMIN=$1
  if [ "$2" != "" ]
  then
    WWWUSER=$2
    if [ "$3" != "" ]
    then
      WWWGROUP=$3
    else
      WWWGROUP=$WWWUSER
    fi
  fi
else
  # Try to ascertain the settings from Apache config..
  if [ -r ${APACHE_CONFDIR}/httpd.conf ]
  then
    G=`grep "^Group " ${APACHE_CONFDIR}/httpd.conf | cut -d" " -f2`
    U=`grep "^User " ${APACHE_CONFDIR}/httpd.conf | cut -d" " -f2`
    [ "$G" != "" ] && WWWGROUP=$G
    [ "$U" != "" ] && WWWUSER=$U
  fi
  # Ensure they exist..
  mkuser $WWWUSER $WWWGROUP /var/www /bin/sh WWW User
  mkgroup $WWWADMIN
fi

# Generally writeable only for axyl and WWW administrators..
chown -R ${AXUSER}:${WWWADMIN} .
chmod -R 0664 .
chmod -R g+sw .
find . -type d -exec chmod a+x {} \;
# Executable Perl scripts..
find . -type f -name "*.pl" -exec chmod ug+x {} \;
# Php executables in the scripts directory..
if [ -d scripts ]
then
  find scripts -type f -name "*.php" -exec chmod ug+x {} \;
fi
# Make all shell scripts executable..
scripts=`find . -type f -exec grep -l -E -e "(\#\![[:space:]]*\/bin\/bash)|(\#\![[:space:]]*\/bin\/sh)" {} \;`
if [ "$scripts" != "" ]
then
  for script in $scripts
  do
    chmod ug+x $script
  done
fi

# Except var directory is writeable to webserver too..
chgrp -R ${WWWUSER} ./var

echo "permissions have been set."

exit 0