#!/bin/sh
# Install a database for an Axyl website. Note this is only expected
# to be run if you have previously installed Axyl and a valid
# /etc/axyl/axyl.conf exists.

# NOTE: This is not normally run standalone. The create-axyl-website.sh
# script usually runs this as part of what it does to set up a brand
# new Axyl website.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# Get common funcs and config vars etc..
if [ -f ./axyl-common.sh ]
then
  . ./axyl-common.sh
else
  if [ "$AXYL_HOME" != "" ]
  then
    . ${AXYL_HOME}/install/axyl-common.sh
  else
    echo "ERROR: axyl-common.sh not found. You should execute this"
    echo "script from within the Axyl 'install' directory."
    exit 1
  fi
fi

# This script requires that Axyl is properly installed. If it isn't
# then the following vars will not be defined, which is an error.
if [ "$AXUSER" = "" -o "$AXYL_HOME" = "" -o "$AXYL_DATA" = "" ]
then
  echo "ERROR: Axyl is not configured properly and AXUSER, AXYL_HOME"
  echo "and AXYL_DATA are undefined. Check /etc/axyl/axyl.conf exists"
  echo "and contains these settings. Otherwise, do you need to run"
  echo "./setup-axyl.sh first?"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INCOMING PARAMETERS
#  $1 - INSTALLTYPE
#  $2 - DBTYPE
#  $3 - DBNAME
#  $4 - DBUSER
#  $5 - DBPASSWD ('none' means a blank password)
#  $6 - DBHOST ('direct' means a local database server)
#  $7 - DBPORT
INSTALLTYPE=
DBTYPE=
DBNAME=
DBUSER=
DBPASSWD=
DBHOST=
DBPORT=5432
[ ! -z $1 ] && INSTALLTYPE=$1
[ ! -z $2 ] && DBTYPE=$2
[ ! -z $3 ] && DBNAME=$3
[ ! -z $4 ] && DBUSER=$4
[ ! -z $5 ] && DBPASSWD=$5
[ ! -z $6 ] && DBHOST=$6
[ ! -z $7 ] && DBPORT=$7
if [ "$INSTALLTYPE" = "" -o "$AXUSER" = "" -o "$AXYL_HOME" = "" -o "$AXYL_DATA" = "" ]
then
  echo "ERROR: install-db.sh: parameters."
  echo "usage: install-db.sh empty|demo dbtype dbname dbuser [dbpasswd] [dbhost] [dbport]"
  exit 1
fi

META=n
if [ "$INSTALLTYPE" = "empty" ]
then
  echo ""
  echo "Axyl can be created with fully-defined Meta Data Schemas, namely"
  echo "the Dublin Core, AGLS, and NZGLS definitions. This is appropriate if"
  echo "you are going to be creating a website which requires meta data"
  echo "content categorisation to any of these standards (eg. a Government"
  echo "website). If you want this to be installed, say so now."
  echo -n "Install Axyl meta-data definitions (y/n) [${META}] :"
  read ANS
  [ "$ANS" = "y" -o "$ANS" = "Y" ] && META=y
fi


# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# CALL APPROPRIATE DATABASE INSTALLER

# DBTYPE values:
#     1 - Postgresql
#     2 - MySQL
#     3 - ODBC
#     4 - Oracle
#     5 - MS SQL Server
#     6 - Other
#
echo ""
echo "Axyl Database Installation"
case $DBTYPE in
  # Postgresql
  1)
    . ${AXYL_HOME}/db/postgres/install-db.sh $INSTALLTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META
    ;;
  # MySQL
  2)
    . ${AXYL_HOME}/db/mysql/install-db.sh $INSTALLTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META
    ;;
  # ODBC
  3)
    . ${AXYL_HOME}/db/odbc/install-db.sh $INSTALLTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META
    ;;
  # Oracle
  4)
    . ${AXYL_HOME}/db/oracle/install-db.sh $INSTALLTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META
    ;;
  # SQL Server
  5)
    . ${AXYL_HOME}/db/mssql_server/install-db.sh $INSTALLTYPE $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT $META
    ;;
  # Something else
  *)
    echo ""
    echo "*********************************************************************"
    echo "IMPORTANT NOTICE!"
    echo "You have chosen a database type which does not currently have an"
    echo "Axyl installer. At the present time we only provide an automatic"
    echo "installation process for Postgresql."
    echo ""
    echo "In the directory $AXYL_HOME/db/postgres you will find the Axyl core"
    echo "schema in 'axyl_core.sql'. This SQL definition may, with some editing,"
    echo "suit your database creation needs. Failing that, if you have the"
    echo "Sybase Powerdesigner package there is a PDM file which can be found"
    echo "in $AXYL_HOME/db, called 'axyl_core.pdm' which you could be used to"
    echo "produce a new definition."
    echo ""
    echo "Note also the file 'axyl_core_data.sql', is an SQL definition, which"
    echo "has the mandatory initial dataset for Axyl. This must be loaded."
    echo "*********************************************************************"
    echo -n "[ENTER] to continue.."
    read ANS
    ;;
esac

# END