#!/bin/bash
# Remove a database for an Axyl website. Note this is only expected
# to be run if you have previously installed Axyl and a valid
# /etc/axyl/axyl.conf exists.

# NOTE: This is not normally run standalone. The remove-axyl-website.sh
# script usually runs this as part of what it does to remove an
# existing Axyl website.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# Get common funcs and config vars etc.
# Acquire the local configuration..
CONFDIR=/etc/axyl
CONF=${CONFDIR}/axyl.conf
if [ ! -f $CONF ]
then
  echo "Axyl configuration file $CONF not found!"
  exit 2
else
  . $CONF
  if [ ! -d $AXYL_HOME ] ; then
    echo "FATAL: the Axyl root directory '$AXYL_HOME' does not exist."
    exit 6
  fi
  . ${AXYL_HOME}/install/axyl-common.sh
fi

# This script requires that Axyl is properly installed. If it isn't
# then the following vars will not be defined, which is an error.
if [ "$AXUSER" = "" -o "$AXYL_DATA" = "" ]
then
  echo "ERROR: Axyl is not configured properly and one of AXUSER, and/or"
  echo "AXYL_DATA are undefined. Check /etc/axyl/axyl.conf exists and"
  echo "contains these settings. Otherwise, do you need to run script"
  echo "./setup-axyl.sh first?"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# INCOMING PARAMETERS
#  $1  - DBTYPE
#  $2  - DBNAME
#  $3  - DBUSER
#  $4  - DBPASSWD ('none' means a blank password)
#  $5  - DBHOST ('direct' means a local database server)
#  $6  - DBPORT
DBTYPE=""
DBNAME=""
DBUSER=""
DBPASSWD=""
DBHOST=""
DBPORT=5432

POS=0
while [ $# -gt 0 ] ; do
  POS=`expr $POS + 1`
  case $POS in
    1)  DBTYPE=$1
        ;;
    2)  DBNAME=$1
        ;;
    3)  DBUSER=$1
        ;;
    4)  DBPASSWD=$1
        ;;
    5)  DBHOST=$1
        ;;
    6)  DBPORT=$1
        ;;
  esac
  shift
done

if [ "$DBTYPE" = "" -o "$DBNAME" = "" -o "$DBUSER" = "" ]
then
  echo "ERROR: remove-db.sh: parameters."
  echo "usage: remove-db.sh dbtype dbname dbuser [dbpasswd] [dbhost] [dbport]"
  exit 1
fi

# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# CALL APPROPRIATE DATABASE REMOVER

# DBTYPE values:
#     1 - Postgresql
#     2 - MySQL
#     3 - ODBC
#     4 - Oracle
#     5 - MS SQL Server
#     6 - Other
#
echo ""
echo "Axyl Database Removal"

# Assert host and port to sensible values
if [ "$DBHOST" = "" ] ; then
  DBHOST=direct
  DBPORT=
fi

# Assert password sensible value
if [ "$DBPASSWD" = "" ] ; then
  DBPASSWD=none
fi

case $DBTYPE in
  # Postgresql
  1)
    . ${AXYL_HOME}/db/postgres/remove-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT
    ;;
  # MySQL
  2)
    . ${AXYL_HOME}/db/mysql/remove-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT
    ;;
  # ODBC
  3)
    . ${AXYL_HOME}/db/odbc/remove-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT
    ;;
  # Oracle
  4)
    . ${AXYL_HOME}/db/oracle/remove-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT
    ;;
  # SQL Server
  5)
    . ${AXYL_HOME}/db/mssql_server/remove-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT
    ;;
  # Something else
  *)
    echo ""
    echo "*********************************************************************"
    echo "IMPORTANT NOTICE!"
    echo "You have chosen a database type (code=$DBTYPE) which does not currently"
    echo "have an Axyl remover. At the present time we only provide a removal process"
    echo "for Postgresql."
    echo ""
    echo "You must remove this database using its administrative commands."
    echo ""
    echo "*********************************************************************"
    echo -n "[ENTER] to continue.."
    read ANS
    ;;
esac

# END