<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    button-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing and using BUTTONS.             */
/*   USING BUTTONS:                                                     */
/*   With these buttons, the 'onclick' parameter can be set             */
/*   to STD_ONCLICK to use the buttons with the standard framework      */
/*   of navigation provided with "nav-action.php". In this case         */
/*   the "$action" parameter of the button is used to set "$nav_action" */
/*   and the "$parms" parameter is used to set "$nav_parms".            */
/*   But mostly this feature will not be of interest as it was          */
/*   developed for rugbylive.                                           */
/*                                                                      */
/* ******************************************************************** */
/** @package buttons */

/** Image classes */
include_once("image-defs.php");

// ----------------------------------------------------------------------
/** This value is used with buttons. If specified in place of
* the $onclick parameter when making a button, it will
* format the default onclick call, which dovetails in
* with our framework for navigating the site with calls
* to "nav-action.php".
*/
define("STD_ONCLICK", "STD_ONCLICK" );

/** Same as above, execpt we are indicating that we want the
* action to be directed to a new window.
*/
define("STD_ONCLICK_NEWWIN", "STD_ONCLICK_NEWWIN" );

// ----------------------------------------------------------------------
/**
* Virtual button_object
* A standard button object for a form. This is a virtual class so we can
* define a set of common functionality for descendants. Never
* instantiate this class.
* @access private
* @package buttons
*/
class button_object extends RenderableObject {
  /** Name of this button */
  var $action;
  /** Label associated with the button */
  var $label = "";
  /** Miscellaneous optional parameters */
  var $parms = "";
  /** Button style string */
  var $style = "";
  // ....................................................................
  /**
  * Constructor
  * Create a button object. Sets basic button attributes.
  * @param string  $action  The name of the form control
  * @param string  $label   The text label for std HTML buttons
  * @param string  $parms   Special parameter string (RugbyLive)
  * @param string  $style   Style setting to apply to the button
  */
  function button_object($action, $label="", $parms="", $style="") {
    $this->action = $action;
    $this->label = $label;
    $this->parms = $parms;
    $this->style = $style;
  }
  // ....................................................................
  /**
  * Sets the label for the button.
  * On a standard HTML button this will be displayed on the
  * top of the button. For buttons which use custom images, setting
  * this property has no effect.
  * @param string $label  The text label to display on the button
  */
  function set_label($label) {
    $this->label = $label;
  }
  // ....................................................................
  /**
  * Sets parameter string associated with the button.
  *
  * This is useful only for the RugbyLive website, where a particular
  * button-clicking framework was implemented.
  * Sets the parameters associated with the button. This is only
  * useful if you are using the STD_ONCLICK behaviour in the
  * way used by the rugbyvu website.
  * @param string $parms  The paramters to associate with the button
  * @access private
  */
  function setparms($parms) {
    $this->parms = $parms;
  }
  // ....................................................................
  /**
  * Sets button style
  *
  * @param string $style  The style setting to apply to this button
  */
  function setstyle($style) {
    $this->style = $style;
  }
} // button_object class

// ----------------------------------------------------------------------
/**
* A standard button for a form. Does nothing unless you define an
* onclick event handler for it.
* @package buttons
*/
class button extends button_object {
  /** Script to be executed onclick */
  var $onclick = "";
  /** Speed-key associated with button (ALT-<key>) */
  var $accesskey = "";
  /** Confirmation text for onclick */
  var $confirmation = "";
  // ....................................................................
  /**
  * Constructor
  * Create a button object. Sets basic button attributes.
  * @param string  $action    The name of the form control
  * @param string  $label     The text label for std HTML buttons
  * @param string  $parms     Special parameter string (RugbyLive)
  * @param string  $onclick   Javascript code to execute onclick
  * @param string  $accesskey Speed-key to assoicate with button
  * @see button_object
  */
  function button($action, $label="", $parms="", $onclick="", $accesskey="") {
    $this->button_object($action, $label, $parms);
    $this->accesskey = $accesskey;
    $this->confirmation = "";
    $this->set_onclick($onclick);
  }
  // ....................................................................
  /**
  * Sets parameter string associated with the button.
  * Sets the onclick script to execute when the button is clicked.
  * @param string  $onclick   Javascript code to execute onclick
  */
  function set_onclick($onclick) {
    if ($onclick != "") {
      // Check if they want the standard onclick handler to be
      // invoked by this button. This assumes the relevant handler
      // is defined in the header. See "gen-header.php" (rugbyvu).
      if ($onclick == STD_ONCLICK) {
        $this->onclick = "";
        if ($this->confirmation != "") $this->onclick .= "btn_confirm('$this->confirmation'); ";
        $this->onclick .= "set_nav_parms('$this->parms'); btn_press('$this->action');";
      } else $this->onclick = $onclick;
    }
    else $this->onclick = "";
    return $this;
  }
  // ....................................................................
  /**
  * Sets parameter string associated with the button.
  * Sets the confirm text to use when this button is clicked. This is
  * a way of, for example, dealing with warnings for record delete.
  * NOTE: This is only applicable with the rugbyvu STD framework.
  * @param string  $conf  Confirmation text for Javascript popup
  */
  function set_confirmation($conf) {
    $this->confirmation = $conf;
    $this->set_onclick(STD_ONCLICK);
    return $this;
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * Renders the button as HTML.
  * @see render()
  * @return string  HTML rendering of button
  */
  function html() {
    $html = "<input name=\"$this->action\"";
    $html .= " type=button";
    if ($this->label != "") $html .= " value=\"$this->label\"";
    if ($this->onclick != "") $html .= " onclick=\"$this->onclick\"";
    if ($this->style != "") $html .= " style=\"$this->style\"";
    if ($this->accesskey != "") $html .= " accesskey=\"" . $this->accesskey . "\"";
    $html .= ">";
    return $html;
  }
} // button class

// ----------------------------------------------------------------------
/**
* Image button.
* A form elelement which behaves like a submit
* button, but takes the form of the given image
* @package buttons
*/
class image_button extends button {
  /** Image object for this button */
  var $img;
  /** Target URL for a window to open when button clicked */
  var $target = "";
  /** Form element type ID */
  var $type = "image";
  /** Title/tooltip */
  var $title = "";
  /** Border size, pixels */
  var $border = 0;
  /** Popup confirmation text to display on button click */
  var $confirm_text = "";
  /** Whether to allow form submit on-click */
  var $onclick_form_submit = false;
  // ....................................................................
  /**
  * Constructor
  * @param string  $action  The name of the button control in the form
  * @param string  $label   The label which might be attached to the button
  * @param string  $parms   Optional parameters
  * @param string  $onclick Optional javascript for onclick event
  * @param string  $src     URL or path to image for the button
  * @param integer $width   Button image width (px)
  * @param integer $height  Button image height (px)
  * @param string  $alt     Button image alt tag content
  * @param string  $border  Size of border around image
  * @see button()
  */
  function image_button($action, $label="", $parms="", $onclick="", $src="", $width=0, $height=0, $alt="", $border=0) {
    // This is traditionally used as a title/tooltip string..
    $this->title = $alt;
    // Whereas really, alts should just show the button action..
    $alt = $action;
    $this->button($action, $label, $parms, $onclick);
    $this->img = new image($action, $src, $alt, $width, $height);
    $this->border = $border;
  }
  // ....................................................................
  /**
  * Sets the image to display for this button.
  * Usually these details are specified in the initial instantiation
  * @see image_button()
  * @param string  $src     URL or path to image for the button
  * @param integer $width   Button image width (px)
  * @param integer $height  Button image height (px)
  * @param string  $alt     Button image alt tag content
  * @param string  $border  Size of border around image
  */
  function set_image($src, $width=0, $height=0, $alt="", $border=0) {
    $this->img->src = $src;
    if ($width != 0) $this->img->width = $width;
    if ($height != 0) $this->img->height = $height;
    if ($alt != "") $this->img->alt = $alt;
    if ($alt != "") $this->img->alt = $alt;
    $this->border = $border;
    return $this;
  }
  // ....................................................................
  /**
  * Set the confirmation text to popup on click.
  * @param string  $conf Tetx to display in the confirmation popup.
  */
  function set_confirm_text($conf) {
    $this->confirm_text = $conf;
  }
  // ....................................................................
  /**
  * Specific function for use with the rugbyvu framework. This causes
  * a button click to open another browser window as a popup.
  * @param string  $target     URL to fill window with
  * @param integer $win_width  Width of new window (px)
  * @param integer $win_height Height of new window (px)
  */
  function set_window_target($target, $win_width=200, $win_height=200) {
    $this->target = $target;
    $this->onclick = " make_dialog('$this->target','$win_width','$win_height','$this->action','$this->parms');";
    return $this;
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  *
  * This renders the image button as HTML. If we have onclick then we render
  * this as a simple image with a javascript URL rather than a INPUT
  * form element of type "image". This is done to cope with Netscape's
  * lack of an onclick event handler for INPUT TYPE=IMAGE, and we
  * don't use BUTTON since that's only HTML4.
  * @see render()
  * @return string  HTML rendering of button
  */
  function html($action="") {
    global $RESPONSE;
    $img = $this->img;
    // Lame Netscape doesn't support onclick, so use HREF surrogate..
    if ($this->onclick && isset($RESPONSE) && $RESPONSE->browser == BROWSER_NETSCAPE) {
      $html  = "<a href=\"javascript:$this->onclick\"";
      if ($this->label != "") {
        $html .= " onmouseover=\"window.status='$this->label'; return true;\"";
        $html .= " onmouseout=\"window.status=''; return true;\"";
      }
      $html .= ">";
      $html .= "<img src=\"$img->src\"";
      if ($img->width) $html .= " width=\"$img->width\"";
      if ($img->height) $html .= " height=\"$img->height\"";
      $html .= " border=\"0\"";
      if ($img->alt) $html .= " alt=\"$img->alt\"";
      if ($this->title != "") $html .= " title=\"$this->title\"";
      $html .= "></a>";
    }
    else {
      // This will act as a submit button by default, and
      // will return its value with the form..
      if ($action != "") $this->action = $action;
      $html = "<input name=\"$this->action\"";
      $html .= " type=\"image\"";
      $html .= " src=\"$img->src\"";
      $html .= " border=\"$this->border\"";
      if ($img->alt) $html .= " alt=\"$img->alt\"";
      if ($this->title != "") $html .= " title=\"$this->title\"";
      if (!$this->onclick && $this->confirm_text != "") {
        $this->onclick="return confirm('$this->confirm_text');";
      }
      if ($this->onclick) {
        // Append return false if form submit not allowed..
        if (!$this->onclick_form_submit) {
          if ($this->onclick != "" && substr($this->onclick, -1) != ";") {
            $this->onclick .= ";";
          }
          $this->onclick .= "return false;";
        }
        $html .= " onclick=\"$this->onclick\"";
      }
      if ($this->label) $html .= " value=\"$this->label\"";
      if (!isset($RESPONSE) || $RESPONSE->browser != BROWSER_NETSCAPE) {
        if ($img->width) $this->style .= " width:" . $img->width . "px;";
        if ($img->height) $this->style .= " height:" . $img->height . "px;";
      }
      if ($this->style != "") $html .= " style=\"" . trim($this->style) . "\"";
      $html .= ">";
    }
    return $html;
  }
} // image_button class

// ----------------------------------------------------------------------
/**
* Hover button.
* Hover button. Uses the 'hover' class to implement a button which has a
* an alternate rollover image, and will also implement our forms submission
* action. Uses the "hover" class from the "image" module.
* Here's how we might implement a hover button..
*  $hb = new hover_button("commentary", "Do commentary", "commentary", $onclick=STD_ONCLICK);
*  $hb->set_image("img/bCommentate.gif", 99, 23);
*  $hb->set_image_over("img/bCommentate_over.gif", 99, 23);
*  Then to display it: echo $hb->render();
* @package buttons
* @see button
*/
class hover_button extends button {
  /** The hover image associated with the button */
  var $hover;
  // ....................................................................
  /**
  * Constructor
  * @param string  $action  The name of the button control in the form
  * @param string  $label   The label which might be attached to the button
  * @param string  $parms   Optional parameters
  * @param string  $onclick Optional javascript for onclick event
  * @see button()
  */
  function hover_button($action, $label="", $parms="", $onclick="") {
    $this->button($action, $label, $parms, $onclick);
    if ($this->onclick != "")
      $hover_url = "javascript:" . $this->onclick;
    else $hover_url = "";
    $this->hover = new hover($action, $hover_url, $label);
  }
  // ....................................................................
  /**
  * Sets button mouse-out image
  * Sets the standard image for this button. This is the image which
  * is shown with the mouse NOT over the top of it.
  * @see set_image_over()
  */
  function set_image($src, $width, $height, $border=0) {
    $this->hover = $this->hover->set_image($src, $width, $height, $border);
  }
  // ....................................................................
  /**
  * Sets button mouse-over image
  * Sets the mouse-over image for this button. This is the image which
  * is shown with the mouse over the top of it.
  * @see set_image()
  */
  function set_image_over($src, $width, $height, $border=0) {
    $this->hover = $this->hover->set_image_over($src, $width, $height, $border);
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * Renders the hover button as HTML. In this case we simply render the
  * hover image which is associated with this button.
  * @see render(), hover_image()
  * @return string  HTML rendering of button
  */
  function html() {
    return $this->hover->html();
  }
} // hover_button class

// ----------------------------------------------------------------------
/**
* Submit button
*
* A standard form submit button appearing as the standard grey widget.
* @package buttons
* @see button
*/
class submit_button extends button {
  /**
  * Constructor
  * @param string  $action  The name of the button control in the form
  * @param string  $label   The label which might be attached to the button
  * @param string  $parms   Optional parameters
  * @param string  $onclick Optional javascript for onclick event
  * @see button()
  */
  function submit_button($action, $label="", $parms="", $onclick="") {
    $this->button($action, $label, $parms, $onclick);
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  * Renders the submit button as HTML.
  * @see render()
  * @return string  HTML rendering of button
  */
  function html() {
    $html = "<input name=\"$this->action\" type=submit";
    if ($this->label) $html .= " value=\"$this->label\"";
    if ($this->onclick) $html .= " onclick=\"$this->onclick\"";
    if ($this->style != "") $html .= " style=\"$this->style\"";
    $html .= ">";
    return $html;
  }
} // submit_button class

// ----------------------------------------------------------------------
/**
* Reset button
* A standard form reset button appearing as the standard grey widget.
* @package buttons
* @see button
*/
class reset_button extends button {
  /**
  * Constructor
  * @param string  $action  The name of the button control in the form
  * @param string  $label   The label which might be attached to the button
  * @param string  $parms   Optional parameters
  * @param string  $onclick Optional javascript for onclick event
  * @see button()
  */
  function reset_button($action, $label="", $parms="", $onclick="") {
    $this->button($action, $label, $parms, $onclick);
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  *
  * Renders the reset button as HTML.
  * @see render()
  * @return string  HTML rendering of button
  */
  function html() {
    $html = "<input name=\"$this->action\" type=reset>";
    if ($this->label) $html .= " value=\"$this->label\"";
    if ($this->onclick) $html .= " onclick=\"$this->onclick\"";
    if ($this->style != "") $html .= " style=\"$this->style\"";
    return $html;
  }
} // reset_button class

// ----------------------------------------------------------------------
/**
* Clickable link
* A standard clickable link. This class incorporates a few useful
* features over and above the standard <A> tag functionality. You can
* define linkover text to appear in the browser status bar when the
* mouse moves over the link, and the object can be rendered as
* either HTML or WML.
* @package buttons
*/
class Link extends RenderableObject {
  /** The URL to go to when link is clicked */
  var $href = "";
  /** The label to display for the link */
  var $label = "";
  /** Target frame for the URL */
  var $target = "";
  /** Script to execute on mouse over */
  var $onmouseover = "";
  /** Script to execute on mouse out */
  var $onmouseout = "";
  /** Font settings for link text */
  var $font = "";
  /** Status bar text when link moused over */
  var $linkover_text = "";
  /** Stylesheet class name for highlighted link */
  var $highlightclass = false;
  /** Style to apply to the link text */
  var $style = "";
  /**
  * Constructor
  * @param string  $href        The URL to go to when clicked
  * @param string  $label       The text to display for the link
  * @param string  $onmouseover Javascript to execute on mouse-over
  * @param string  $onmouseout  Javascript to execute on mouse-out
  * @param string  $font        Font settings
  * @param string  $target      Target frame eg: '_blank', 'myframe' etc.
  */
  function Link($href, $label="", $onmouseover="", $onmouseout="", $font="", $target="") {
    $this->href = $href;
    $this->label = $label;
    $this->target = $target;
    $this->onmouseover = $onmouseover;
    $this->onmouseout = $onmouseout;
    $this->font = $font;
  }
  // ....................................................................
  /**
  * Sets the font
  * Deprecated. Sets the font to use with the link text. Better
  * to use a stylesheet these days.
  */
  function set_font($font) {
    $this->font = $font;
  }
  // ....................................................................
  /**
  * Sets the URL
  * Sets the URL to go to when the link is clicked. usually this
  * is specified on instantiation.
  * @see Link()
  */
  function set_href($href) {
    $this->href = $href;
  }
  // ....................................................................
  /**
  * Sets the style
  * Sets the style for the href tag.
  */
  function setstyle($style) {
    $this->style = $style;
  }
  // ....................................................................
  /**
  * Defines special highlight class name
  * Defines the name of a class in your stylesheet to use for the
  * purpose of highlighting. This property is initialised to be
  * 'false', but if defined, then the link is spanned by a <span>
  * tag with the given class name to highlight it accordingly.
  */
  function highlight($highlightclass) {
    $this->highlightclass = $highlightclass;
  }
  // ....................................................................
  /**
  * Set status text to show on mouseover
  * Will do a mouseover status message, if onmouseover is not already
  * defined, otherwise it will be ignored. This is useful to avoid
  * long and ugly URL's appearing in the status bar.
  */
  function set_linkover_text($txt) {
    $this->linkover_text = $txt;
  }
  // ....................................................................
  /**
  * Renders the reset button as WML.
  * @see render()
  * @return string WML rendering of button
  */
  function wml() {
    if (function_exists("href_rewrite")) {
      $this->href = href_rewrite($this->href);
    }
    return "<a href=\"" . $this->href . "\">" . $this->label . "</a>";
  }
  // ....................................................................
  /**
  * Use render() to render this element in your page.
  *
  * Renders the reset button as HTML.
  * @see render()
  * @return string HTML rendering of button
  */
  function html() {
    $html = "";
    if ($this->highlightclass != "") $html .= "<span class=$this->highlightclass>";
    $html  .= "<a href=\"" . $this->href . "\"";
    if (isset($this->style) && $this->style != "") $html .= " style=\"$this->style\"";
    if ($this->target != "") $html .= " target=\"$this->target\"";
    if ($this->linkover_text != "" && $this->onmouseover == "") {
      $this->onmouseover = "status='$this->linkover_text';return true;";
      $this->onmouseout  = "status='';return true;";
    }
    if ($this->onmouseover != "") $html .= " onmouseover=\"" . $this->onmouseover . "\"";
    if ($this->onmouseout != "")  $html .= " onmouseout=\""  . $this->onmouseout  . "\"";
    $html .= ">";
    if ($this->font != "") $html .= "<font " . $this->font . ">";
    $html .= $this->label;
    if ($this->font != "") $html .= "</font>";
    $html .= "</a>";
    if ($this->highlightclass != "") $html .= "</span>";
    return $html;
  }
} // Link

// ----------------------------------------------------------------------
?>