#!/bin/sh
# Install the Lucene Server if not already installed. This script sets
# up the Axyl Lucene data areas, the Server configuration and logging
# configuration files.
#
# Although this installer is mainly called from either 'setup-axyl.sh'
# in the main Axyl install directory, or the Debian postinst script
# for the axyl-lucene package, it can also be run from the command-line
# by a hopeful user, and will install Lucene, given a port-number.
#
# Command-line arguments
#   $1  - LUCENE_PORT
#   $2  - OVERWITE MODE (y or n, or undefined = n)
#   $3  - DEBCONF MODE (y or n, or undefined = n)
#
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
# GET EXISTING AXYL CONFIG, AND COMMON VARS & FUNCTIONS..
CONF=/etc/axyl/axyl.conf
if [ -f $CONF ]; then
  . $CONF
  if [ -x ${AXYL_HOME}/install/axyl-common.sh ] ; then
    . ${AXYL_HOME}/install/axyl-common.sh
  fi
fi

# SET THE LUCENE PORT..
if [ "$1" != "" ]; then
  LUCENE_PORT=$1
else
  echo "usage: install-lucene.sh PORTNUMBER" >/dev/tty
  exit 1
fi

# SET OUR FLAGS..
LUCENE_OVERWRITE=n
LUCENE_DEBCONF=n
[ "$2" != "" ] && LUCENE_OVERWRITE=$2
[ "$3" != "" ] && LUCENE_DEBCONF=$3

# This script requires that Axyl is properly installed. If it isn't
# then the following vars will not be defined, which is an error.
if [ "$AXUSER" = "" -o "$AXYL_HOME" = "" -o "$AXYL_DATA" = "" ]; then
  echo "ERROR: Axyl is not configured properly and AXUSER, AXYL_HOME" >/dev/tty
  echo "and AXYL_DATA are undefined. Check $CONF exists" >/dev/tty
  echo "and contains these settings. Otherwise, do you need to run" >/dev/tty
  echo "setup-axyl.sh first?" >/dev/tty
  exit 1
fi

# We can only proceed if axyl-lucene is actually installed in the
# local Axyl installation..
if [ ! -e ${AXYL_HOME}/lucene ] ; then
  echo "ERROR: the axyl-lucene package is not installed!" >/dev/tty
  echo "axyl-lucene must be already unpacked and installed in the 'lucene'" >/dev/tty
  echo "sub-directory of the axyl package." >/dev/tty
  exit 1
fi

# Makes the Axyl Lucene data area. If the area already exists then
# existing index and log data will be preserved, but the configuration
# files will be re-asserted. The intent is to establish correctness..
make_lucene_area() {
  # Create dirs and initialise contents..
  mkthisdir ${AXYL_DATA}/lucene
  mkthisdir ${AXYL_DATA}/lucene/servers
  dupedir ${AXYL_HOME}/lucene/servers ${AXYL_DATA}/lucene/servers
  mkthisdir ${AXYL_LOGS}/lucene

  # Substitute variable values..
  sub_vars $AXYL_DATA "_LUCENE_DATA_" "$AXYL_DATA/lucene"
  sub_vars $AXYL_DATA "_LUCENE_HOME_" "$AXYL_HOME/lucene"
  sub_vars $AXYL_DATA "_LU_DOMAIN_" "axyl"
  sub_vars $AXYL_DATA "_LUCENE_PORT_" "$LUCENE_PORT"
  sub_vars $AXYL_DATA "_LUCENE_LOGPATH_" "${AXYL_LOGS}/lucene"

  # Set permissions..
  chown -R ${AXUSER}:${AXUSER} ${AXYL_DATA}/lucene
  chmod -R 0644 ${AXYL_DATA}/lucene
  find ${AXYL_DATA}/lucene -type d -exec chmod 6755 {} \;
  find ${AXYL_LOGS} -type d -exec chmod 0755 {} \;
}

# Whether to do the Lucene install or not. We always do it if it has
# been forced with $LUCENE_OVERWRITE=y. Otherwise we will only do it if
# the Lucene data structures are missing.
DOINST=1
[ -e ${AXYL_DATA}/lucene/servers ] && DOINST=0
[ "$LUCENE_OVERWRITE" = "y" ] && DOINST=1
if [ $DOINST -eq 1 ]; then
  # A Java runtime is required. Try JAVA_HOME first, and
  # if not then try /usr/bin/java..
  if [ -z $JAVA_HOME ]; then
    if [ -x /usr/bin/java ]; then
      JAVA_BIN=/usr/bin
    fi
  else
    JAVA_BIN=${JAVA_HOME}/bin
  fi
  # Require Java executable..
  JAVA=${JAVA_BIN}/java
  if [ ! -x $JAVA ]; then
    echo "" > /dev/tty
    echo "A pre-requisite for Axyl Lucene is that you have a Java Runtime" > /dev/tty
    echo "Environment (JRE) installed, which doesn't seem to be the case currently." > /dev/tty
    echo "You need to install a JRE and and set environment var 'JAVA_HOME' to" > /dev/tty
    echo "point at it, or maybe install a Java runtime package which provides" > /dev/tty
    echo "a /usr/bin/java executable." > /dev/tty
  fi
  # Make sure port is specified..
  if [ "$LUCENE_PORT" = "null" ]; then
    LUCENE_PORT=22222
    echo -n "Enter the port number Lucene will listen on [$LUCENE_PORT] :" > /dev/tty
    read ANS < /dev/tty
    [ "x$ANS" != "x" ] && LUCENE_PORT=$ANS
  fi

  # Create dirs and initialise contents..
  make_lucene_area

  # Sort out the init.d script too. We only do this if we are
  # doing a non-Debian installation..
  if [ "$LUCENE_DEBCONF" = "n" ] ; then
    if [ -d /etc/init.d ]; then
      if [ ! -f /etc/init.d/axyl-lucene ]; then
        if [ -f ${AXYL_HOME}/lucene/etc/init.d/axyl-lucene ]; then
          cp ${AXYL_HOME}/lucene/etc/init.d/axyl-lucene /etc/init.d/axyl-lucene
          chmod u+x /etc/init.d/axyl-lucene
          if [ -x /usr/sbin/update-rc.d ]; then
            update-rc.d axyl-lucene defaults 99 10 >/dev/null 2>&1
          else
            echo "Your Axyl Lucene start-up script is has been installed here:" > /dev/tty
            echo "   /etc/init.d/axyl-lucene" > /dev/tty
            echo "You might want to symlink this into your rcN.d directories, so" > /dev/tty
            echo "as to start/stop Lucene according to run-level. We will start" > /dev/tty
            echo "it here for you, just now.." > /dev/tty
          fi
          # Start Lucene now..
          if [ -x /usr/sbin/invoke-rc.d ]; then
            invoke-rc.d axyl-lucene start
          else
            /etc/init.d/axyl-lucene start
          fi
        fi
      fi
    else
      echo "Your Axyl Lucene start-up script is currently here:" > /dev/tty
      echo "   ${AXYL_HOME}/etc/init.d/axyl-lucene" > /dev/tty
      echo "You might want to copy this to a relevant system location, which" > /dev/tty
      echo "will allow Lucene to be started automatically on system boot." > /dev/tty
    fi
  fi

  # Re-write the Axyl config, to reflect any Lucene port changes..
  rewrite_axylconf $AXUSER $AXYL_HOME $AXYL_DATA $AXYL_LOGS $LUCENE_PORT

  # Confirm it is done..
  echo "Axyl Luceneserver has been installed." > /dev/tty
else
  # We re-assert the Lucene configuration which governs the Lucene
  # operation, since these things may have changed..
  make_lucene_area

  # Even though Lucene data already exists, we may be re-installing
  # Axyl Lucene, so we have to make sure the port is set in config..
  rewrite_axylconf $AXUSER $AXYL_HOME $AXYL_DATA $AXYL_LOGS $LUCENE_PORT

  # Confirm it..
  if [ "$LUCENE_DEBCONF" = "n" ] ; then
    echo "Axyl Lucene re-configured. Running on port $LUCENE_PORT"
  fi
fi

# END