#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <signal.h>
#include <ctype.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif
#include <unistd.h>

#include "../include/fio.h"
#include "../include/disk.h"
#include "../include/prochandle.h"
#include "guiutils.h"
#include "cdialog.h"
#include "cfg.h"
#include "animicon.h"
#include "obj.h"
#include "objfio.h"
#include "win.h"
#include "winanimicon.h"
#include "wincb.h"
#include "windnd.h"
#include "winlist.h"
#include "winopcb.h"
#include "core.h"
#include "cfglist.h"
#include "config.h"

#include "images/icon_avscan_48x48.xpm"

#include "images/icon_avscan_20x20.xpm"
#include "images/icon_biohazard_20x20.xpm"
#include "images/icon_virus_db_20x20.xpm"
#include "images/icon_virus_db_32x32.xpm"
#include "images/icon_close_20x20.xpm"
#include "images/icon_exit_20x20.xpm"
#include "images/icon_add_20x20.xpm"
#include "images/icon_edit_20x20.xpm"
#include "images/icon_remove_20x20.xpm"
#include "images/icon_new_20x20.xpm"
#include "images/icon_cut_20x20.xpm"
#include "images/icon_copy_20x20.xpm"
#include "images/icon_paste_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"
#include "images/icon_select_20x20.xpm"
#include "images/icon_properties2_20x20.xpm"
#include "images/icon_run_20x20.xpm"
#include "images/icon_stop_20x20.xpm"
#include "images/icon_folder_opened_20x20.xpm"
#include "images/icon_planet_20x20.xpm"
#include "images/icon_trash_20x20.xpm"
#include "images/icon_move_file_20x20.xpm"
#include "images/icon_open_20x20.xpm"
#include "images/icon_save_as_20x20.xpm"
#include "images/icon_clear_20x20.xpm"
#include "images/icon_reload_20x20.xpm"
#include "images/icon_search_20x20.xpm"
#include "images/icon_tool_bar_20x20.xpm"
#include "images/icon_status_bar_20x20.xpm"
#include "images/icon_help_20x20.xpm"
#include "images/icon_about_20x20.xpm"


/* Process */
void WinProcessOutputDelete(win_struct *win);
void WinScanProcessStart(
	win_struct *win,
	const gchar *db_location,	/* Virus Scanner Database Location */
	const gchar *location,		/* Location of object to scan */
	gboolean recursive,		/* Recursive? */
	gboolean executables_only	/* Executables Only? */
);
void WinScanProcessStop(win_struct *win);
gboolean WinScanProcessIsRunning(win_struct *win);
void WinUpdateProcessStart(win_struct *win);
void WinUpdateProcessStop(win_struct *win);
gboolean WinUpdateProcessIsRunning(win_struct *win);

/* Scan List & Settings Update */
void WinScanListUpdate(win_struct *win);
void WinScanSettingsUpdate(win_struct *win, const obj_struct *obj);
void WinResultsLocationUpdate(win_struct *win, const gchar *location);

/* Operation IDs */
win_opid_struct *WinOPIDNew(
	win_opid id, win_struct *win
);
void WinOPIDDelete(win_opid_struct *opid);
GList *WinOPIDListNew(win_struct *win);
void WinOPIDListDelete(GList *glist);
win_opid_struct *WinOPIDListFind(GList *glist, win_opid id);

/* Win */ 
static void WinBuildMenuBar(win_struct *win);
static void WinBuildToolBar(win_struct *win);
static void WinBuildLists(win_struct *win);
static void WinBuildStatusBar(win_struct *win);
static void WinBuildListsMenu(win_struct *win);

win_struct *WinNew(gpointer core_ptr);
void WinSavePositions(win_struct *win);
void WinUpdate(win_struct *win);
void WinSetBusy(win_struct *win, gboolean busy);
void WinSetPassiveBusy(win_struct *win, gboolean busy);
gboolean WinToolBarIsShown(win_struct *win);
void WinToolBarSetShow(win_struct *win, gboolean show);
void WinToolBarUpdateItems(win_struct *win);
gboolean WinStatusBarIsShown(win_struct *win);
void WinStatusBarSetShow(win_struct *win, gboolean show);
void WinStatusProgress(
	win_struct *win, gfloat val, gboolean allow_gtk_iteration
);
void WinStatusMessage(
	win_struct *win, const gchar *mesg, gboolean allow_gtk_iteration
);
gboolean WinIsMapped(win_struct *win);          
void WinMap(win_struct *win);         
void WinUnmap(win_struct *win);
void WinDelete(win_struct *win);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


#define WIN_WIDTH			600
#define WIN_HEIGHT			400
#define WIN_STATUS_BAR_HEIGHT		26


/*
 *	Closes and deletes the Win's process output files.
 */
void WinProcessOutputDelete(win_struct *win)
{
	if(win == NULL)
	    return;

	/* Close stdout & stderr files */
	if(win->stdout_fp != NULL)
	{
	    FClose(win->stdout_fp);
	    win->stdout_fp = NULL;
	}
	if(win->stderr_fp != NULL)
	{
	    FClose(win->stderr_fp);
	    win->stderr_fp = NULL;
	}

	/* Delete stdout & stderr files */
	if(win->stdout_path != NULL)      
	{
	    unlink(win->stdout_path);  
	    g_free(win->stdout_path);
	    win->stdout_path = NULL;
	}
	if(win->stderr_path != NULL)
	{
	    unlink(win->stderr_path);
	    g_free(win->stderr_path);
	    win->stderr_path = NULL;
	}
}

/*
 *	Starts the scan process.
 *
 *	Any current scan process will be stopped first.
 */
void WinScanProcessStart(
	win_struct *win,
	const gchar *db_location,	/* Virus Scanner Database Location */
	const gchar *location,		/* Location of object to scan */
	gboolean recursive,		/* Recursive? */
	gboolean executables_only	/* Executables Only? */
)
{
	const gchar *tmp_dir;
	gchar *cmd;
	const cfg_item_struct *cfg_list;
	GtkWidget *toplevel;
	core_struct *core;

	if(win == NULL)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	/* Get database path from configuration if it is not
	 * specified
	 */
	if(db_location == NULL)
	    db_location = CFGItemListGetValueS(
		cfg_list, CFG_PARM_DB_LOCATION
	    );
	if(STRISEMPTY(db_location))
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundWarning(core->edv_ctx); 
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
"The virus database location has not been specified.\n\
\n\
To set the virus database location go to Database->Location...",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}


	/* Interrupt the scan process and delete stdout and stderr
	 * files
	 */
	WinScanProcessStop(win);

	/* Nothing to scan? */
	if(STRISEMPTY(location))
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundWarning(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
"The scan location has not been specified.",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    return;
	}

	/* Set Results Location */
	WinResultsLocationUpdate(win, location);

	/* Clear the Results List */
	WinResultsListClear(win);

	/* Start playing animated icon */
	AnimIconPlay(win->anim_icon);

	/* Set scan monitor timeout callback */
	win->scan_toid = gtk_timeout_add(
	    250l,
	    WinScanMonitorTOCB, win
	);

	/* Get tempory files directory */
	tmp_dir = g_getenv(ENV_VAR_NAME_TMPDIR);
	if(tmp_dir == NULL)
#if defined(P_tmpdir)
	    tmp_dir = P_tmpdir;
#elif defined(__MSW__)
	    tmp_dir = "C:\\TEMP";
#else
	    tmp_dir = "/tmp";
#endif

	/* Create output file paths */
	win->stdout_path = STRDUP(PrefixPaths(
	    tmp_dir,
	    PROG_NAME "XXXXXX"
	));
	mkstemp(win->stdout_path);
	win->stderr_path = STRDUP(PrefixPaths(
	    tmp_dir,
	    PROG_NAME "XXXXXX"
	));
	mkstemp(win->stderr_path);

	/* Format scan command */
	cmd = g_strdup_printf(
	    "\"%s\" --scan \"%s\"%s%s --database \"%s\"",
	    core->prog_path, location,
	    recursive ? " --recursive" : "",
	    executables_only ? " --executables-only" : "",
	    db_location
	);

	/* Run scan command */
	win->scan_pid = (gint)ExecOE(
	    cmd, win->stdout_path, win->stderr_path
	);

	/* Delete command */
	g_free(cmd);

	/* Failed to run scan command? */
	if(win->scan_pid <= 0)
	{
	    char *s = g_strdup_printf(
"Unable to execute a scanning process of this program:\n\
\n\
    %s",
		core->prog_path
	    );
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
		s,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    free(s);
	    CDialogSetTransientFor(NULL);

	    WinScanProcessStop(win);
	    WinUpdate(win);
	    return;
	}

	/* Open output files for reading */
	win->stdout_fp = FOpen(win->stdout_path, "rb");
	win->stderr_fp = FOpen(win->stderr_path, "rb");

	WinSetPassiveBusy(win, TRUE);

	WinUpdate(win);
}

/*
 *	Interrupts the Win's scan process and deletes stdout and stderr
 *	files.
 */
void WinScanProcessStop(win_struct *win)
{
	if(win == NULL)
	    return;

	/* Stop animated icon from playing */
	AnimIconPause(win->anim_icon);
	AnimIconSeek(win->anim_icon, 0);

	/* Remove scan timeout callback */
	if(win->scan_toid > 0)
	{
	    gtk_timeout_remove(win->scan_toid);
	    win->scan_toid = 0;
	}

	/* Interrupt the scan child process */
	if(win->scan_pid > 0)
	{
	    kill(win->scan_pid, SIGINT);
	    win->scan_pid = 0;
	}

	/* Close and delete the stdout & stderr files */
	WinProcessOutputDelete(win);

	/* Reset stop count */
	win->stop_count = 0;

	/* Reset progress */
	WinStatusProgress(win, 0.0f, FALSE);

	WinSetPassiveBusy(win, FALSE);

	WinUpdate(win);
}

/*
 *	Checks if the Win's scan process is running.
 */
gboolean WinScanProcessIsRunning(win_struct *win)
{
	gint p = (win != NULL) ? win->scan_pid : 0;
	return((p > 0) ? ExecProcessExists((pid_t)p) : FALSE);
}


/*
 *	Starts the virus database update process.
 *
 *	Any current virus database update process will be stopped first.
 */
void WinUpdateProcessStart(win_struct *win)
{
	gint effective_uid = (gint)geteuid();
#if defined(HAVE_CLAMAV)
	const gchar *prog = PREFIX "/bin/freshclam";
#else
	const gchar *prog = PREFIX "/bin/freshclam";
#warning WinUpdateProcessStart(): No antivirus library defined.
#endif
	const gchar *tmp_dir;
	gchar *cmd;
	struct stat stat_buf;
	GtkWidget *toplevel;
	const cfg_item_struct *cfg_list;
	core_struct *core;

	if(win == NULL)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);
	cfg_list = core->cfg_list;             

	/* Interrupt virus database update process and delete stdout
	 * and stderr files
	 */
	WinUpdateProcessStop(win);

	/* Check if the virus update program exists */
	if(stat(prog, &stat_buf))
	{
	    const gint errnum = (gint)errno;
	    gchar *s, *error_msg = STRDUP(g_strerror(errnum));
	    if(error_msg != NULL)
	    {
		*error_msg = toupper(*error_msg);
		s = g_strdup_printf(
"Unable to find the virus database update program:\n\
\n\
    %s\n\
\n\
%s.",
		    prog, error_msg
		);
#ifdef HAVE_EDV2
		EDVPlaySoundWarning(core->edv_ctx);
#endif
		CDialogSetTransientFor(toplevel);
		CDialogGetResponse(
		    "Virus Database Update Failed",
		    s,
		    NULL,
		    CDIALOG_ICON_WARNING,
		    CDIALOG_BTNFLAG_OK,
		    CDIALOG_BTNFLAG_OK
		);
		free(s);
		CDialogSetTransientFor(NULL);
		g_free(error_msg);
	    }
	    return;
	}
	/* Check if its permissions are set up properly */
	if(!(stat_buf.st_mode & S_ISUID) &&
	   (effective_uid != stat_buf.st_uid) &&
	   (effective_uid != 0)
	)
	{
	    gint response;
	    gchar *s = g_strdup_printf(
"You may not be able to update the virus database because\n\
the virus database update program:\n\
\n\
    %s\n\
\n\
Is not set SETUID and you are not the owner of it.\n\
\n\
Do you want to try and update the virus database anyway?",
		prog
	    );
#ifdef HAVE_EDV2
	    EDVPlaySoundWarning(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Virus Database Update Warning",
		s,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    free(s);
	    CDialogSetTransientFor(NULL);
	    if(response != CDIALOG_RESPONSE_YES)
		return;
	}

	/* Set virus database update monitor timeout callback */
	win->update_toid = gtk_timeout_add(
	    250l,
	    WinUpdateMonitorTOCB, win
	);

	/* Get tempory files directory */
	tmp_dir = g_getenv(ENV_VAR_NAME_TMPDIR);
	if(STRISEMPTY(tmp_dir))
#if defined(P_tmpdir)
	    tmp_dir = P_tmpdir;
#elif defined(__MSW__)
	    tmp_dir = "C:\\TEMP";
#else
	    tmp_dir = "/tmp";
#endif

	/* Create output file paths */
	win->stdout_path = STRDUP(PrefixPaths(
	    tmp_dir,
	    PROG_NAME "XXXXXX"
	));
	mkstemp(win->stdout_path);
	win->stderr_path = STRDUP(PrefixPaths(
	    tmp_dir,
	    PROG_NAME "XXXXXX"
	));
	mkstemp(win->stderr_path);

	/* Format virus database update command */
	cmd = g_strdup_printf(
	    "\"%s\" --verbose",
	    prog
	);

	/* Run virus database update command */
	win->update_pid = (gint)ExecOE(
	    cmd, win->stdout_path, win->stderr_path
	);

	/* Delete command */
	g_free(cmd);

	/* Failed to run virus database update command? */
	if(win->update_pid <= 0) 
	{
	    char *s = g_strdup_printf(
"Unable to execute the virus database update program:\n\
\n\
    %s",
		prog
	    );
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Virus Database Update Failed",
		s,
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    free(s);
	    CDialogSetTransientFor(NULL);

	    WinUpdateProcessStop(win);
	    WinUpdate(win);
	    return;
	}

	/* Open output files for reading */
	win->stdout_fp = FOpen(win->stdout_path, "rb");
	win->stderr_fp = FOpen(win->stderr_path, "rb");

	WinSetPassiveBusy(win, TRUE);

	WinUpdate(win);
}

/*
 *	Interrupts the Win's virus database update process and deletes
 *	stdout & stderr files.
 */
void WinUpdateProcessStop(win_struct *win)
{
	if(win == NULL)
	    return;

	/* Remove the virus database update timeout callback */
	if(win->update_toid > 0)
	{
	    gtk_timeout_remove(win->update_toid);
	    win->update_toid = 0;
	}

	/* Interrupt the virus database update child process */
	if(win->update_pid > 0)       
	{
	    kill(win->update_pid, SIGINT);
	    win->update_pid = 0;
	}

	/* Close and delete the stdout & stderr files */
	WinProcessOutputDelete(win);

	/* Reset stop count */
	win->stop_count = 0;

	/* Reset progress */
	WinStatusProgress(win, 0.0f, FALSE);

	WinSetPassiveBusy(win, FALSE);

	WinUpdate(win);
}

/*
 *	Checks if the Win's virus update update process is running.
 */
gboolean WinUpdateProcessIsRunning(win_struct *win)
{
	gint p = (win != NULL) ? win->update_pid : 0;
	return((p > 0) ? ExecProcessExists((pid_t)p) : FALSE);
}


/*
 *	Updates the scan list.
 */
void WinScanListUpdate(win_struct *win)
{
	gchar *path;
	GList *glist, *obj_list;
	GtkCList *clist;
#ifdef HAVE_EDV2
	edv_context_struct *ctx;
#endif
	core_struct *core;

	if(win == NULL)
	    return;

	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);
#ifdef HAVE_EDV2
	ctx = core->edv_ctx;
#endif

	/* Begin updating the scan list */

	gtk_clist_freeze(clist);

	/* Clear the scan list */
	gtk_clist_clear(clist);

	/* Load scan list from file */
	path = STRDUP(PrefixPaths(
	    core->prog_data_dir, AVSCAN_SCAN_LIST_FILE
	));
	obj_list = ObjListOpen(path, NULL, NULL);
	g_free(path);

	/* Transfer each object to the Win's scan list */
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	    WinScanListAppend(win, OBJ(glist->data));

	/* Delete scan list */
	g_list_free(obj_list);                         

	gtk_clist_thaw(clist);

	WinUpdate(win);
}

/*
 *	Updates the Win's scan settings with the values from the
 *	specified object.
 */
void WinScanSettingsUpdate(win_struct *win, const obj_struct *obj)
{
	GtkWidget *w;

	if(win == NULL)
	    return;

	win->freeze_count++;

	w = win->location_entry;
	if(w != NULL)
	{
	    const gchar *s = (obj != NULL) ? obj->path : NULL;
	    gtk_entry_set_text(
		GTK_ENTRY(w), (s != NULL) ? s : ""
	    );
	}

	GUIMenuItemSetCheck(
	    win->recursive_micheck,
	    (obj != NULL) ? obj->recursive : FALSE,
	    TRUE
	);
	w = win->recursive_check;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		(obj != NULL) ? obj->recursive : FALSE
	    );

	GUIMenuItemSetCheck(
	    win->executables_only_micheck,
	    (obj != NULL) ? obj->executables_only : FALSE,
	    TRUE
	);
	w = win->executables_only_check;
	if(w != NULL)
	    gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(w),
		(obj != NULL) ? obj->executables_only : FALSE
	    );

	win->freeze_count--;

	WinUpdate(win);
}

/*
 *	Updates the Win's results location.
 */
void WinResultsLocationUpdate(win_struct *win, const gchar *location)
{
	GtkWidget *w;

	if(win == NULL)
	    return;  

	win->freeze_count++;

	w = win->results_location_label;
	if(w != NULL)
	    gtk_label_set_text(
		GTK_LABEL(w), (location != NULL) ? location : ""
	    );

	win->freeze_count--;

/*	WinUpdate(win); */
}


/*
 *	Creates a new Win OPID.
 */
win_opid_struct *WinOPIDNew(
	win_opid id, win_struct *win
)
{
	win_opid_struct *opid = WIN_OPID(
	    g_malloc0(sizeof(win_opid_struct))
	);
	opid->id = id;
	opid->win = win;
	return(opid);
}

/*
 *	Deletes the Win OPID.
 */
void WinOPIDDelete(win_opid_struct *opid)
{
	if(opid == NULL)
	    return;

	g_free(opid->button_name);
	g_free(opid->menu_name);
	g_free(opid->tooltip);
	g_free(opid);
}

/*
 *	Creates a new Win OPID list.
 */
GList *WinOPIDListNew(win_struct *win)
{
	gboolean allow_multiple;
	toolbar_item_type item_type;
	const gchar	*button_name,
			*menu_name,
			*tooltip;
	guint	accel_key,
		accel_mods;
	guint8	**button_icon_data,
		**menu_icon_data;
	void (*func_cb)(GtkWidget *, gpointer);
	GList *glist = NULL;

#define DO_APPEND(_id_)	{			\
 win_opid_struct *opid = WinOPIDNew(		\
  (_id_), win					\
 );						\
 opid->allow_multiple = allow_multiple;		\
 opid->item_type = item_type;			\
 opid->button_name = STRDUP(button_name);	\
 opid->menu_name = STRDUP(menu_name);		\
 opid->tooltip = STRDUP(tooltip);		\
 opid->accel_key = accel_key;			\
 opid->accel_mods = accel_mods;			\
 opid->button_icon_data = button_icon_data;	\
 opid->menu_icon_data = menu_icon_data;		\
 opid->func_cb = func_cb;			\
 opid->enter_func_cb = WinOPIDEnterEventCB;	\
 opid->leave_func_cb = WinOPIDLeaveEventCB;	\
 glist = g_list_append(glist, opid);		\
}

	allow_multiple = TRUE;
	item_type = TOOLBAR_ITEM_SEPARATOR;
	button_name = NULL;
	menu_name = button_name;
	tooltip = NULL;
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = button_icon_data;
	func_cb = NULL;
	DO_APPEND(WIN_OPID_SEPARATOR)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Close";
	menu_name = button_name;
	tooltip = "Close this window";
	accel_key = GDK_w;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_close_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinCloseCB;
	DO_APPEND(WIN_OPID_CLOSE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Exit";
	menu_name = "Exit";
	tooltip = "Close all of this application's windows and exit";
	accel_key = GDK_q;
	accel_mods = GDK_CONTROL_MASK; 
	button_icon_data = (guint8 **)icon_exit_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinExitCB;
	DO_APPEND(WIN_OPID_EXIT)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Add...";
	menu_name = "Add...";
	tooltip = "Add a new scan item";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_add_20x20_xpm;
	menu_icon_data = button_icon_data;                 
	func_cb = WinAddCB;
	DO_APPEND(WIN_OPID_ADD)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Edit...";
	menu_name = "Edit...";
	tooltip = "Edit the selected scan item";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_edit_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinEditCB;
	DO_APPEND(WIN_OPID_EDIT)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Remove";
	menu_name = "Remove";
	tooltip = "Remove the selected scan item(s)";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_remove_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinRemoveCB;
	DO_APPEND(WIN_OPID_REMOVE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "New...";
	menu_name = "New...";
	tooltip = "Create a new scan item";
	accel_key = GDK_n;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_new_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinAddCB;
	DO_APPEND(WIN_OPID_NEW)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Cut";
	menu_name = "Cut";
	tooltip = "Cut the selected scan item(s) to the clipboard";
	accel_key = GDK_x;
	accel_mods = GDK_CONTROL_MASK; 
	button_icon_data = (guint8 **)icon_cut_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinCutCB;
	DO_APPEND(WIN_OPID_CUT)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Copy";
	menu_name = "Copy";
	tooltip = "Copy the selected scan item(s) to the clipboard";
	accel_key = GDK_c;
	accel_mods = GDK_CONTROL_MASK; 
	button_icon_data = (guint8 **)icon_copy_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinCopyCB;
	DO_APPEND(WIN_OPID_COPY)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Paste";
	menu_name = "Paste";
	tooltip = "Paste scan item(s) from the clipboard";
	accel_key = GDK_v;
	accel_mods = GDK_CONTROL_MASK; 
	button_icon_data = (guint8 **)icon_paste_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinPasteCB;
	DO_APPEND(WIN_OPID_PASTE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Delete";
	menu_name = "Delete";
	tooltip = "Delete the selected scan item(s)";
	accel_key = GDK_Delete;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_cancel_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinRemoveCB;
	DO_APPEND(WIN_OPID_DELETE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all scan items";
	accel_key = GDK_a;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_select_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinSelectAllCB;
	DO_APPEND(WIN_OPID_SELECT_ALL)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "UnselAll";
	menu_name = "Unselect All";
	tooltip = "Unselect all scan items";
	accel_key = GDK_u;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_select_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinUnselectAllCB; 
	DO_APPEND(WIN_OPID_UNSELECT_ALL)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip = "Invert selection of scan items";
	accel_key = GDK_i;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_select_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinInvertSelectionCB; 
	DO_APPEND(WIN_OPID_INVERT_SELECTION)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Properties";
	menu_name = "Properties...";
	tooltip = "Edit the selected scan item's properties";
	accel_key = GDK_Return;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_properties2_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinEditCB;
	DO_APPEND(WIN_OPID_PROPERTIES)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;  
	button_name = "Start";
	menu_name = button_name;
	tooltip = "Start scanning";
	accel_key = GDK_Return;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_run_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinStartCB;
	DO_APPEND(WIN_OPID_START)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;  
	button_name = "Stop";
	menu_name = button_name;
	tooltip = "Stop the current operation";
	accel_key = GDK_Escape;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_stop_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinStopCB;
	DO_APPEND(WIN_OPID_STOP)

	allow_multiple = FALSE;   
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Location";
	menu_name = "Location...";
	tooltip = "Set scan location";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_folder_opened_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinLocationCB;
	DO_APPEND(WIN_OPID_LOCATION)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "Recurse";
	menu_name = "Recursive";
	tooltip = "Toggle recursive scanning";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = button_icon_data;
	func_cb = WinRecursiveCB;
	DO_APPEND(WIN_OPID_RECURSIVE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "ExecOnly";
	menu_name = "Executables Only";
	tooltip = "Toggle only scanning objects that are executable";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = button_icon_data;                         
	func_cb = WinExecutablesOnlyCB;
	DO_APPEND(WIN_OPID_EXECUTABLES_ONLY)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Recycle";
	menu_name = "Recycle";
	tooltip = "Recycle the selected object(s)";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_trash_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinResultsRecycleCB;
	DO_APPEND(WIN_OPID_RESULTS_RECYCLE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Move";
	menu_name = "Move...";
	tooltip = "Move the selected object(s)";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_move_file_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinResultsMoveCB;
	DO_APPEND(WIN_OPID_RESULTS_MOVE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "SelAll";
	menu_name = "Select All";
	tooltip = "Select all objects";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_select_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinResultsSelectAllCB;
	DO_APPEND(WIN_OPID_RESULTS_SELECT_ALL)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "UnselAll";       
	menu_name = "Unselect All";  
	tooltip = "Unselect all objects";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_select_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinResultsUnselectAllCB;
	DO_APPEND(WIN_OPID_RESULTS_UNSELECT_ALL)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "InvSel";
	menu_name = "Invert Selection";
	tooltip = "Invert selection of objects";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_select_20x20_xpm; 
	menu_icon_data = NULL;
	func_cb = WinResultsInvertSelectionCB;  
	DO_APPEND(WIN_OPID_RESULTS_INVERT_SELECTION)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Open";
	menu_name = "Open...";
	tooltip = "Open scan results log";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_open_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinResultsOpenCB;
	DO_APPEND(WIN_OPID_RESULTS_OPEN)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;       
	button_name = "Save As";
	menu_name = "Save As...";
	tooltip = "Save scan results log as";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_save_as_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinResultsSaveAsCB;
	DO_APPEND(WIN_OPID_RESULTS_SAVE_AS)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Clear";
	menu_name = "Clear";
	tooltip = "Clear scan results log";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_clear_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinResultsClearCB;
	DO_APPEND(WIN_OPID_RESULTS_CLEAR)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Refresh";
	menu_name = "Refresh";
	tooltip = "Refresh the virus database";
	accel_key = GDK_F5;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_reload_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinDBRefreshCB;
	DO_APPEND(WIN_OPID_DB_REFRESH)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Location";
	menu_name = "Location...";
	tooltip = "Set the virus scanner's database location";
	accel_key = GDK_d;
	accel_mods = GDK_CONTROL_MASK;
	button_icon_data = (guint8 **)icon_virus_db_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinDBLocationCB;
	DO_APPEND(WIN_OPID_DB_LOCATION)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Update";
	menu_name = "Update From Internet";
	tooltip = "Get the latest virus database from the Internet";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_planet_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinDBUpdateNetCB;
	DO_APPEND(WIN_OPID_DB_UPDATE_NET)


	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "Scan";
	menu_name = "Scan Page";
	tooltip = "Go to the scan page";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = NULL;
	func_cb = WinViewScanPageCB;
	DO_APPEND(WIN_OPID_VIEW_SCAN_PAGE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "Results";
	menu_name = "Results Page"; 
	tooltip = "Go to the results page";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = NULL;
	func_cb = WinViewResultsPageCB;
	DO_APPEND(WIN_OPID_VIEW_RESULTS_PAGE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "Database";
	menu_name = "Database Page";
	tooltip = "Go to the virus database page";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = NULL;
	menu_icon_data = NULL;
	func_cb = WinViewDBPageCB;
	DO_APPEND(WIN_OPID_VIEW_DB_PAGE)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "ToolBar";
	menu_name = "Tool Bar";
	tooltip = "Show/hide the tool bar";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_tool_bar_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinViewToolBarCB;
	DO_APPEND(WIN_OPID_VIEW_TOOLBAR)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_TOGGLE_BUTTON;
	button_name = "StatusBar";
	menu_name = "Status Bar";
	tooltip = "Show/hide the status bar";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_status_bar_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinViewStatusBarCB;
	DO_APPEND(WIN_OPID_VIEW_STATUSBAR)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Help";
	menu_name = "Contents";
	tooltip = "Index of all help pages";
	accel_key = GDK_F1;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinHelpContentsCB;
	DO_APPEND(WIN_OPID_HELP_CONTENTS)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Scan?";
	menu_name = "How To Scan";
	tooltip = "Help on scanning objects for viruses";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinHelpScanningCB;
	DO_APPEND(WIN_OPID_HELP_SCANNING)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "Update?";
	menu_name = "How To Update";
	tooltip = "Help on updating the virus database";
	accel_key = 0;                      
	accel_mods = 0;    
	button_icon_data = (guint8 **)icon_help_20x20_xpm;
	menu_icon_data = NULL;
	func_cb = WinHelpUpdatingCB;
	DO_APPEND(WIN_OPID_HELP_UPDATING)

	allow_multiple = FALSE;
	item_type = TOOLBAR_ITEM_BUTTON;
	button_name = "About";
	menu_name = "About...";
	tooltip = "About this application";
	accel_key = 0;
	accel_mods = 0;
	button_icon_data = (guint8 **)icon_about_20x20_xpm;
	menu_icon_data = button_icon_data;
	func_cb = WinHelpAboutCB;
	DO_APPEND(WIN_OPID_ABOUT)


#undef DO_APPEND

	return(glist);
}

/*
 *	Deletes the Win OPID list.
 */
void WinOPIDListDelete(GList *glist)
{
	if(glist == NULL)
	    return;

	g_list_foreach(glist, (GFunc)WinOPIDDelete, NULL);
	g_list_free(glist);
}

/*
 *	Returns the Win OPID that matches the specified id in the
 *	list.
 */
win_opid_struct *WinOPIDListFind(GList *glist, win_opid id)
{
	win_opid_struct *opid;

	for(;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    opid = WIN_OPID(glist->data);
	    if((opid != NULL) ? (opid->id == id) : FALSE)
		return(opid);
	}

	return(NULL);
}



/*
 *	Builds the Win's Menu Bar.
 */
static void WinBuildMenuBar(win_struct *win)
{
	guint8 **icon_data;
	GtkAccelGroup *accelgrp = win->accelgrp;
	GtkWidget *w, *parent, *menubar, *menu, *menuitem_func_w;
/*	core_struct *core = win->core; */

	parent = win->menubar_handle;


	/* Recreate Menu Bar */
	GTK_WIDGET_DESTROY(win->menubar)
	win->menubar = menubar = w = GUIMenuBarCreate(NULL);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);

#define DO_ADD_MENU_ITEM_LABEL(_id_)	{	\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)				\
 );						\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(                        \
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   (gpointer *)&menuitem_func_w,		\
   opid->win, opid->func_cb			\
  );                                            \
  GUISetMenuItemCrossingCB(                     \
   w,                                           \
   (gpointer)opid->enter_func_cb, opid,		\
   (gpointer)opid->leave_func_cb, opid		\
  );                                            \
 }						\
 else						\
  w = NULL;					\
}

#define DO_ADD_MENU_ITEM_CHECK(_id_)	{	\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)				\
 );						\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   (gpointer *)&menuitem_func_w,		\
   opid->win, opid->func_cb			\
  );						\
  GUISetMenuItemCrossingCB(			\
   w,						\
   (gpointer)opid->enter_func_cb, opid,		\
   (gpointer)opid->leave_func_cb, opid		\
  );						\
 }						\
 else                                           \
  w = NULL;                                     \
}

#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,     \
  NULL, NULL, 0, 0, NULL,                       \
  NULL, NULL                                    \
 );                                             \
}

	/* Create the File menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_NEW)
	    win->new_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_CLOSE)
	    win->close_mi = w;
	}
	win->file_mlabel = GUIMenuAddToMenuBar(
	    menubar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Archivo"
#elif defined(PROG_LANGUAGE_FRENCH)
"Fichier"
#elif defined(PROG_LANGUAGE_GERMAN)
"Akte"
#elif defined(PROG_LANGUAGE_ITALIAN)
"File"
#elif defined(PROG_LANGUAGE_DUTCH)
"Dossier"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Arquivo"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Arkiv"
#else
"File"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Create the Edit menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_CUT)
	    win->cut_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_COPY)
	    win->copy_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_PASTE)
	    win->paste_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_DELETE)
	    win->delete_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_SELECT_ALL)
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_UNSELECT_ALL)
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_INVERT_SELECTION)

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_PROPERTIES)
	    win->properties_mi = w;
	}
	win->edit_mlabel = GUIMenuAddToMenuBar(
	    menubar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Redacte"
#elif defined(PROG_LANGUAGE_FRENCH)   
"Editer"
#elif defined(PROG_LANGUAGE_GERMAN)
"Redigieren"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Redigere"
#elif defined(PROG_LANGUAGE_DUTCH)
"Bewerking"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Edite"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Rediger"
#else
"Edit"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);

	/* Create the Scan menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_START)
	    win->start_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_STOP)
	    win->stop_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_LOCATION)
	    win->location_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_RECURSIVE)
	    win->recursive_micheck = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_EXECUTABLES_ONLY)
	    win->executables_only_micheck = w;
	}
	win->scan_mlabel = GUIMenuAddToMenuBar(
	    menubar, menu, "Scan", GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Create the Results menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_RECYCLE)
	    win->results_recycle_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_MOVE)
	    win->results_move_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_SELECT_ALL)
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_UNSELECT_ALL)
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_INVERT_SELECTION)

	    DO_ADD_MENU_SEP   

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_OPEN)
	    win->results_open_mi = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_SAVE_AS)
	    win->results_save_as_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_CLEAR)
	    win->results_clear_mi = w;
	}
	win->results_mlabel = GUIMenuAddToMenuBar(
	    menubar, menu, "Results", GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Create the Virus Database menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_DB_REFRESH)
	    win->db_refresh_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_DB_LOCATION)
	    win->db_location_mi = w;
 
	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_DB_UPDATE_NET)
	    win->db_update_net_mi = w;
	}
	win->db_mlabel = GUIMenuAddToMenuBar(
	    menubar, menu, "Database", GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Create the View menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_VIEW_SCAN_PAGE)
	    win->view_scan_page_micheck = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_VIEW_RESULTS_PAGE)
	    win->view_results_page_micheck = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_VIEW_DB_PAGE)
	    win->view_db_page_micheck = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_VIEW_TOOLBAR)
	    win->view_toolbar_micheck = w;

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_CHECK(WIN_OPID_VIEW_STATUSBAR)
	    win->view_statusbar_micheck = w;
	}
	win->view_mlabel = GUIMenuAddToMenuBar( 
	    menubar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Vista"
#elif defined(PROG_LANGUAGE_FRENCH)
"Vue"
#elif defined(PROG_LANGUAGE_GERMAN)
"Blick"
#elif defined(PROG_LANGUAGE_ITALIAN)
"Veduta"
#elif defined(PROG_LANGUAGE_DUTCH)
"Overzicht"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Vista"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Sikt"
#else
"View"
#endif
	    , GUI_MENU_BAR_ALIGN_LEFT
	);


	/* Create the Help menu */
	menu = GUIMenuCreateTearOff();
	if(menu != NULL)
	{
	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_HELP_CONTENTS)
	    win->help_contents_mi = w;

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_HELP_SCANNING)
/*	    win->help_scanning_mi = w; */

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_HELP_UPDATING)
/*	    win->help_scanning_mi = w; */

	    DO_ADD_MENU_SEP

	    icon_data = NULL;
	    DO_ADD_MENU_ITEM_LABEL(WIN_OPID_ABOUT)
	    win->about_mi = w;
	}
	win->help_mlabel = GUIMenuAddToMenuBar( 
	    menubar, menu,
#if defined(PROG_LANGUAGE_SPANISH)
"Ayuda"
#elif defined(PROG_LANGUAGE_FRENCH)
"Aide"
#elif defined(PROG_LANGUAGE_GERMAN)
"Hilfe"
#elif defined(PROG_LANGUAGE_ITALIAN)
"L'Aiuto"
#elif defined(PROG_LANGUAGE_DUTCH)
"Hulp"
#elif defined(PROG_LANGUAGE_PORTUGUESE)
"Ajuda"
#elif defined(PROG_LANGUAGE_NORWEGIAN)
"Hjelp"
#else
"Help"
#endif
	    , GUI_MENU_BAR_ALIGN_RIGHT
	);

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SEP
}

/*
 *	Builds the Win's Tool Bar.
 */
static void WinBuildToolBar(win_struct *win)
{
	GList *glist, *tb_cfg;
	toolbar_item_struct **tb_item = NULL;
	gint tb_total_items = 0;
	toolbar_display tb_display = TOOLBAR_DISPLAY_PICTURES;
	toolbar_relief tb_relief = TOOLBAR_RELIEF_NONE;
	GtkWidget *parent;
	toolbar_struct *tb;
	win_opid_struct *opid;
/*	core_struct *core = win->core; */

	parent = win->toolbar_handle;


	/* Create Tool Bar configuration */
	glist = NULL;
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_START
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_STOP
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_SEPARATOR
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_NEW
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_CUT
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_COPY
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_PASTE
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_DELETE
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_PROPERTIES
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_SEPARATOR
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_RESULTS_RECYCLE
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_RESULTS_MOVE
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_RESULTS_OPEN
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_RESULTS_SAVE_AS
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_RESULTS_CLEAR
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_SEPARATOR
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_DB_LOCATION
	);
	glist = g_list_append(
	    glist, (gpointer)WIN_OPID_DB_UPDATE_NET
	);

	tb_cfg = glist;

	/* Create Tool Bar Items list from the Tool Bar configuration */
	for(glist = tb_cfg;
	    glist != NULL;
	    glist = g_list_next(glist)
	)
	{
	    /* Get Operation ID from the Tool Bar configuration which
	     * specifies the ID
	     */
	    opid = WinOPIDListFind(win->opid, (win_opid)glist->data);
	    if(opid == NULL)
		continue;

	    ToolBarItemListAppend(
		&tb_item, &tb_total_items,
		ToolBarItemNew(
		    opid->item_type,		/* Item type */
		    opid->button_name,		/* Label text */
		    opid->button_icon_data,	/* Icon data */
		    opid->tooltip,		/* Tooltip text */
		    opid->id,			/* ID */
		    WinToolBarItemCB, opid,
		    WinToolBarItemEnterCB, opid,
		    WinToolBarItemLeaveCB, opid
		)
	    );
	}


	/* Recreate Tool Bar */
	ToolBarDelete(win->toolbar);
	win->toolbar = tb = ToolBarNew(
	    tb_item, tb_total_items, parent,
	    tb_display, tb_relief, FALSE
	);
	ToolBarMap(tb);

	/* Delete Tool Bar items list */
	ToolBarItemListDeleteAll(&tb_item, &tb_total_items);

	/* Delete Tool Bar configuration */
	g_list_free(tb_cfg);

	/* Need to queue resize since Tool Bar size may have changed */
	gtk_widget_queue_resize(win->toplevel);
}

/*
 *	Builds the Win's Lists.
 */
static void WinBuildLists(win_struct *win)
{
	const gint	border_major = 5,
			border_minor = 2;
	const GtkTargetEntry dnd_tar_types[] = {
		{ "text/plain",		0,	DND_INFO_TEXT_PLAIN },
		{ "text/uri-list",	0,	DND_INFO_TEXT_URI_LIST },
		{ "STRING",		0,	DND_INFO_STRING }
	};
	const GtkTargetEntry dnd_src_types[] = {
		{ "text/plain",		0,	DND_INFO_TEXT_PLAIN },
		{ "text/uri-list",	0,	DND_INFO_TEXT_URI_LIST },
		{ "STRING",		0,	DND_INFO_STRING }
	};
	gint columns;
	gchar **heading;
	gpointer entry_rtn, browse_rtn;
	GtkWidget *w, *parent, *parent2, *parent3, *parent4, *parent5, *tablabel;
	GtkNotebook *notebook;
	GtkCList *clist;
	GtkCTree *ctree;

#define NEW_BUTTON(_id_,_parent_,_defaultable_){\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)				\
 );						\
 if(opid != NULL) {				\
  w = GUIButtonPixmapLabelH(			\
   opid->button_icon_data, opid->button_name,	\
   NULL						\
  );						\
  if(_defaultable_) {				\
   GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);	\
   gtk_widget_set_usize(			\
    w,						\
    GUI_BUTTON_HLABEL_WIDTH_DEF,		\
    GUI_BUTTON_HLABEL_HEIGHT_DEF		\
   );						\
  } else {					\
   gtk_widget_set_usize(			\
    w,						\
    GUI_BUTTON_HLABEL_WIDTH,			\
    GUI_BUTTON_HLABEL_HEIGHT			\
   );						\
  }						\
  gtk_box_pack_start(				\
   GTK_BOX(_parent_), w, FALSE, FALSE, 0	\
  );						\
  gtk_signal_connect(				\
   GTK_OBJECT(w), "enter_notify_event",		\
   GTK_SIGNAL_FUNC(opid->enter_func_cb), opid	\
  );						\
  gtk_signal_connect(				\
   GTK_OBJECT(w), "leave_notify_event",		\
   GTK_SIGNAL_FUNC(opid->leave_func_cb), opid	\
  );						\
  gtk_signal_connect(				\
   GTK_OBJECT(w), "clicked",			\
   GTK_SIGNAL_FUNC(opid->func_cb), opid->win	\
  );						\
  GUISetWidgetTip(w, opid->tooltip);		\
  gtk_widget_show(w);				\
 }						\
 else						\
  w = NULL;					\
}
#define SET_CROSSING(_id_,_w_)	{		\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)				\
 );						\
 gtk_signal_connect(				\
  GTK_OBJECT(_w_), "enter_notify_event",	\
  GTK_SIGNAL_FUNC(opid->enter_func_cb), opid	\
 );						\
 gtk_signal_connect(				\
  GTK_OBJECT(_w_), "leave_notify_event",	\
  GTK_SIGNAL_FUNC(opid->leave_func_cb), opid	\
 );						\
}

	parent = win->lists_vbox;

	/* Main hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent = w;
#if 0
	/* Left column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);                                   
	parent2 = w;
#endif
	/* Left column notebook */
	win->notebook = w = gtk_notebook_new();
	notebook = GTK_NOTEBOOK(w);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0); 
	gtk_notebook_set_tab_pos(notebook, GTK_POS_TOP);
	gtk_notebook_set_scrollable(notebook, TRUE);   
	gtk_notebook_set_show_tabs(notebook, TRUE); 
	gtk_notebook_set_show_border(notebook, TRUE);
	gtk_signal_connect(
	    GTK_OBJECT(w), "switch_page",
	    GTK_SIGNAL_FUNC(WinNotebookSwitchPageCB), win
	);
	gtk_widget_show(w);
	parent2 = w;


	/* Scan page tab */
	tablabel = w = gtk_hbox_new(FALSE, border_minor);
	w = gtk_pixmap_new_from_xpm_d(
	    NULL, NULL,
	    (guint8 **)icon_avscan_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Scan");
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Scan page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
	    GTK_NOTEBOOK(parent2), w, tablabel
	);
	gtk_widget_show(w);
	parent3 = w;

	/* Scan list frame */
	w = gtk_frame_new("Scan List");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Scan list scrolled window */
	w = gtk_scrolled_window_new(NULL, NULL); 
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Scan list */
	columns = 3;
	heading = (gchar **)g_malloc(columns * sizeof(gchar *));
	heading[0] = "Name";
	heading[1] = "Location";
	heading[2] = "Last Scanned";
	win->scan_clist = w = gtk_clist_new_with_titles(columns, heading);
	g_free(heading);
	clist = GTK_CLIST(w);
/*	gtk_widget_set_usize(w, -1, 115); */
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_passive(clist); 
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_width(clist, 0, 150);
	gtk_clist_set_column_resizeable(clist, 1, FALSE);
	gtk_clist_set_column_width(clist, 1, 210);
	gtk_clist_set_column_resizeable(clist, 2, FALSE);
	gtk_clist_set_column_width(clist, 2, 10);
	gtk_clist_set_row_height(clist, WIN_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "resize_column",
	    GTK_SIGNAL_FUNC(WinListResizeColumnCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "click_column",
	    GTK_SIGNAL_FUNC(WinListClickColumnCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(WinListSelectRowCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(WinListUnselectRowCB), win
	);
	GUIDNDSetSrc(
	    w,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    WinListDragDataGetCB,
	    WinListDragDataDeleteCB,
	    NULL,
	    win
	); 
	GUIDNDSetTar(
	    w,
	    dnd_tar_types,
	    sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE,	/* Actions */ 
	    GDK_ACTION_MOVE,			/* Default action if same */
	    GDK_ACTION_COPY,			/* Default action */
	    WinListDragDataReceivedCB,
	    win
	); 
	gtk_widget_show(w);

	/* Scan list buttons hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	NEW_BUTTON(WIN_OPID_ADD, parent5, FALSE);
	win->add_btn = w;
	NEW_BUTTON(WIN_OPID_EDIT, parent5, FALSE);
	win->edit_btn = w;
	NEW_BUTTON(WIN_OPID_REMOVE, parent5, FALSE);   
	win->remove_btn = w;


	/* Scan frame */
	w = gtk_frame_new("Scan");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Location */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	w = GUIPromptBarWithBrowse(
	    NULL, "Location:",
	    NULL, &entry_rtn, &browse_rtn,
	    win, WinLocationCB
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, TRUE, TRUE, 0);
	gtk_widget_show(w);

	w = (GtkWidget *)entry_rtn;
	win->location_entry = w;
	GUIDNDSetTar(
	    w,
	    dnd_tar_types,
	    sizeof(dnd_tar_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY,			/* Actions */
	    GDK_ACTION_COPY,			/* Default action if same */
	    GDK_ACTION_COPY,			/* Default action */
	    WinEntryDragDataReceivedCB,
	    win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(WinStartCB), win
	);
	SET_CROSSING(WIN_OPID_LOCATION, w);
	GUISetWidgetTip(w,
"Enter the location of the file or directory to scan"
	);

	/* Scan settings hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	win->recursive_check = w = gtk_check_button_new_with_label("Recursive");
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	SET_CROSSING(WIN_OPID_RECURSIVE, w);
	GUISetWidgetTip(w,
"Check this to scan recursively into each sub directory (if the\
 specified location is a directory)"
	);
	gtk_widget_show(w);

	win->executables_only_check = w = gtk_check_button_new_with_label(
	    "Executables Only"
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	SET_CROSSING(WIN_OPID_EXECUTABLES_ONLY, w);
	GUISetWidgetTip(w,
"Check this to only scan objects who's permissions are set executable"
	);
	gtk_widget_show(w);


	/* Results page tab */
	tablabel = w = gtk_hbox_new(FALSE, border_minor);
	w = gtk_pixmap_new_from_xpm_d(
	    NULL, NULL,
	    (guint8 **)icon_biohazard_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Results");
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Results page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(
	    GTK_NOTEBOOK(parent2), w, tablabel
	);
	gtk_widget_show(w);
	parent3 = w;

	/* Results list frame */
	w = gtk_frame_new("Results");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Results location hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Prefix icon & label */
	w = gtk_pixmap_new_from_xpm_d(
	    NULL, NULL,
	    (guint8 **)icon_folder_opened_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Location:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	/* Label */
	win->results_location_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Results list scrolled window */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Results list */
	columns = 2;
	heading = (gchar **)g_malloc(columns * sizeof(gchar *));
	heading[0] = "Object";
	heading[1] = "Virus/Problem";
	win->results_clist = w = gtk_clist_new_with_titles(columns, heading);
	g_free(heading);
	clist = GTK_CLIST(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win              
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_realize(w);
	gtk_clist_column_titles_active(clist);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_EXTENDED);
	gtk_clist_set_column_resizeable(clist, 0, FALSE);
	gtk_clist_set_column_width(clist, 0, 220);
	gtk_clist_set_column_resizeable(clist, 1, FALSE);
	gtk_clist_set_column_width(clist, 1, 10);
	gtk_clist_set_row_height(clist, WIN_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);   
	gtk_signal_connect(
	    GTK_OBJECT(w), "resize_column",
	    GTK_SIGNAL_FUNC(WinListResizeColumnCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "click_column",
	    GTK_SIGNAL_FUNC(WinListClickColumnCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "select_row",
	    GTK_SIGNAL_FUNC(WinListSelectRowCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "unselect_row",
	    GTK_SIGNAL_FUNC(WinListUnselectRowCB), win  
	);
	GUIDNDSetSrc(
	    w,
	    dnd_src_types,
	    sizeof(dnd_src_types) / sizeof(GtkTargetEntry),
	    GDK_ACTION_COPY | GDK_ACTION_MOVE |
		GDK_ACTION_LINK,		/* Actions */
	    GDK_BUTTON1_MASK,			/* Buttons */
	    NULL,
	    WinListDragDataGetCB,
	    WinListDragDataDeleteCB,
	    NULL,
	    win
	);
	gtk_widget_show(w);

	/* Results list buttons hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	NEW_BUTTON(WIN_OPID_RESULTS_RECYCLE, parent5, FALSE);
	win->results_recycle_btn = w;
	NEW_BUTTON(WIN_OPID_RESULTS_MOVE, parent5, FALSE);
	win->results_move_btn = w;

	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);   
	parent5 = w;
	NEW_BUTTON(WIN_OPID_RESULTS_CLEAR, parent5, FALSE);
	win->results_clear_btn = w;


	/* Virus Database page tab */
	tablabel = w = gtk_hbox_new(FALSE, border_minor);
	w = gtk_pixmap_new_from_xpm_d(
	    NULL, NULL,
	    (guint8 **)icon_virus_db_20x20_xpm
	);
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Database");
	gtk_box_pack_start(GTK_BOX(tablabel), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Virus Database page */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_notebook_append_page(                                 
	    GTK_NOTEBOOK(parent2), w, tablabel
	);
	gtk_widget_show(w);
	parent3 = w;

	/* Virus Database tree frame */
	w = gtk_frame_new("Virus Database");
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_major);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_widget_show(w);
	parent4 = w;

	/* Virus Database Stats & Buttons hbox */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);                                      
	parent5 = w;       


	/* Virus Database Stats Prefixes icon & label */
	w = gtk_pixmap_new_from_xpm_d(
	    NULL, NULL,
	    (guint8 **)icon_virus_db_32x32_xpm
	);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = gtk_label_new("Location:\nLast Updated:\nPatterns:");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_RIGHT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);                                      

	/* Virus Database Stats label */
	win->db_stats_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);

	/* Vbox for Virus Database Buttons */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_end(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

#if 0
	NEW_BUTTON(WIN_OPID_DB_REFRESH, parent5, FALSE);
	win->db_refresh_btn = w;
#endif
	NEW_BUTTON(WIN_OPID_DB_LOCATION, parent5, FALSE);
	win->db_location_btn = w;

	NEW_BUTTON(WIN_OPID_DB_UPDATE_NET, parent5, FALSE);
	win->db_update_net_btn = w;

	/* Find entry */
	w = GUIPromptBar(
	    (guint8 **)icon_search_20x20_xpm, "Find:",
	    NULL, &entry_rtn
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	w = (GtkWidget *)entry_rtn;
	win->db_find_entry = w;
	gtk_signal_connect(
	    GTK_OBJECT(w), "activate",
	    GTK_SIGNAL_FUNC(WinDBFindEntryCB), win
	);

	/* Virus Database tree scrolled window */
	w = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(
	    GTK_SCROLLED_WINDOW(w),
	    GTK_POLICY_AUTOMATIC,
	    GTK_POLICY_AUTOMATIC
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent5 = w;
	/* Virus Database tree */
	columns = 1;
	win->db_ctree = w = gtk_ctree_new(columns, 0);
	clist = GTK_CLIST(w);
	ctree = GTK_CTREE(w);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK |
	    GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK |
	    GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
	    GDK_POINTER_MOTION_MASK | GDK_POINTER_MOTION_HINT_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_press_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "button_release_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_signal_connect_after(
	    GTK_OBJECT(w), "motion_notify_event",
	    GTK_SIGNAL_FUNC(WinListEventCB), win
	);
	gtk_container_add(GTK_CONTAINER(parent5), w);
	gtk_widget_realize(w);
	gtk_clist_set_selection_mode(clist, GTK_SELECTION_SINGLE);
	gtk_ctree_set_line_style(ctree, GTK_CTREE_LINES_DOTTED);
	gtk_clist_set_column_auto_resize(clist, 0, TRUE);
	gtk_clist_set_column_justification(
	    clist, 0, GTK_JUSTIFY_LEFT
	);
	gtk_clist_set_row_height(clist, WIN_LIST_ROW_SPACING);
	gtk_clist_set_shadow_type(clist, GTK_SHADOW_IN);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_select_row",
	    GTK_SIGNAL_FUNC(WinTreeSelectRowCB), win   
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_unselect_row",
	    GTK_SIGNAL_FUNC(WinTreeUnselectRowCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_expand",
	    GTK_SIGNAL_FUNC(WinTreeExpandCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "tree_collapse",
	    GTK_SIGNAL_FUNC(WinTreeCollapseCB), win
	);
	gtk_widget_show(w);


	/* Right column vbox */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Animated Icon hbox */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;
	/* Animated Icon */
	win->anim_icon = w = WinAnimIconNew();
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Buttons vbox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	NEW_BUTTON(WIN_OPID_START, parent3, TRUE);
	win->start_btn = w;
	NEW_BUTTON(WIN_OPID_STOP, parent3, TRUE);
	win->stop_btn = w;

	/* Buttons vbox */
	w = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_end(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	NEW_BUTTON(WIN_OPID_CLOSE, parent3, TRUE);
	win->close_btn = w;



#undef NEW_BUTTON
#undef SET_CROSSING
}

/*
 *	Builds the Win's Status Bar.
 */
static void WinBuildStatusBar(win_struct *win)
{
	gint attach_x = 0;
	GtkAdjustment *adj;
	GtkWidget *w, *parent, *parent2, *parent3;

	parent = win->main_vbox;


	/* Begin recreating status bar */
	GTK_WIDGET_DESTROY(win->statusbar)

	/* Main outer frame */
	win->statusbar = w = gtk_frame_new(NULL);
	gtk_widget_set_usize(w, -1, WIN_STATUS_BAR_HEIGHT);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_OUT);
	gtk_box_pack_end(GTK_BOX(parent), w, FALSE, FALSE, 0);
	parent = w;

	/* Main table */
	w = gtk_table_new(1, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Progress bar */
	adj = (GtkAdjustment *)gtk_adjustment_new(
	    0.0f, 1.0f, 100.0f,
	    0.0f, 0.0f, 0.0f
	);
	win->statusbar_progress = w = gtk_progress_bar_new_with_adjustment(adj);
	gtk_widget_set_usize(w, 100, 20);
	gtk_progress_bar_set_orientation(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_LEFT_TO_RIGHT
	);
	gtk_progress_bar_set_bar_style(
	    GTK_PROGRESS_BAR(w), GTK_PROGRESS_CONTINUOUS
	);
	gtk_progress_set_activity_mode(
	    GTK_PROGRESS(w), FALSE
	);
	gtk_table_attach(
	    GTK_TABLE(parent), w,
	    attach_x, attach_x + 1,
	    0, 1,
	    0,
	    GTK_SHRINK,
	    0, 0
	);
	attach_x++;
	gtk_widget_show(w);


	/* Label */
	w = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_container_border_width(GTK_CONTAINER(w), 1);
	gtk_table_attach(
	    GTK_TABLE(parent), w,
	    attach_x, attach_x + 1,
	    0, 1,
	    GTK_SHRINK | GTK_EXPAND | GTK_FILL,
	    GTK_FILL,
	    0, 0
	);
	attach_x++;
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_hbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_widget_show(w);
	parent3 = w;

	win->statusbar_label = w = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(w), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 2);
	gtk_widget_show(w);
}


/*
 *	Builds the Win's right-click menus.
 */
static void WinBuildListsMenu(win_struct *win)
{
	guint8 **icon_data;
	GtkAccelGroup *accelgrp = win->accelgrp;
	GtkWidget *w, *menu, *menuitem_func_w;

#define DO_ADD_MENU_ITEM_LABEL(_id_)	{	\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)				\
 );						\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_LABEL, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   (gpointer *)&menuitem_func_w,		\
   opid->win, opid->func_cb			\
  );                                            \
  GUISetMenuItemCrossingCB(                     \
   w,                                           \
   (gpointer)opid->enter_func_cb, opid,         \
   (gpointer)opid->leave_func_cb, opid          \
  );                                            \
 }						\
 else                                           \
  w = NULL;                                     \
}

#define DO_ADD_MENU_ITEM_CHECK(_id_)	{	\
 win_opid_struct *opid = WinOPIDListFind(	\
  win->opid, (_id_)                             \
 );						\
 if(opid != NULL) {				\
  w = GUIMenuItemCreate(			\
   menu, GUI_MENU_ITEM_TYPE_CHECK, accelgrp,	\
   (icon_data != NULL) ?			\
    icon_data : opid->menu_icon_data,		\
   opid->menu_name,				\
   opid->accel_key, opid->accel_mods,		\
   (gpointer *)&menuitem_func_w,		\
   opid->win, opid->func_cb			\
  );                                            \
  GUISetMenuItemCrossingCB(                     \
   w,                                           \
   (gpointer)opid->enter_func_cb, opid,         \
   (gpointer)opid->leave_func_cb, opid          \
  );                                            \
 }						\
 else                                           \
  w = NULL;                                     \
}

#define DO_ADD_MENU_SEP		{		\
 w = GUIMenuItemCreate(				\
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL,	\
  NULL, NULL, 0, 0, NULL,			\
  NULL, NULL					\
 );						\
}
	/* Scan List right-click menu */
	GTK_WIDGET_DESTROY(win->scan_list_menu)
	win->scan_list_menu = menu = GUIMenuCreate();

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_START)
	win->scan_list_menu_start_mi = w;

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_STOP)
	win->scan_list_menu_stop_mi = w;

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_NEW)
	win->scan_list_menu_new_mi = w;

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_CUT)
	win->scan_list_menu_cut_mi = w;

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_COPY)
	win->scan_list_menu_copy_mi = w;

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_PASTE)
	win->scan_list_menu_paste_mi = w;

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_DELETE)
	win->scan_list_menu_delete_mi = w;

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_SELECT_ALL)
	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_UNSELECT_ALL)
	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_INVERT_SELECTION)

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_PROPERTIES)
	win->scan_list_menu_properties_mi = w;


	/* Results List right-click menu */
	GTK_WIDGET_DESTROY(win->results_list_menu)
	win->results_list_menu = menu = GUIMenuCreate();

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_RECYCLE)
	win->results_list_menu_recycle_mi = w;

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_MOVE)
	win->results_list_menu_move_mi = w;

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_SELECT_ALL)
	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_UNSELECT_ALL)
	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_INVERT_SELECTION)

	DO_ADD_MENU_SEP

	icon_data = NULL;
	DO_ADD_MENU_ITEM_LABEL(WIN_OPID_RESULTS_CLEAR)
	win->results_list_menu_clear_mi = w;


#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_ITEM_CHECK
#undef DO_ADD_MENU_SEP
}



/*
 *	Creates a new Win.
 */
win_struct *WinNew(gpointer core_ptr)
{
	GdkWindow *window;
	GtkAccelGroup *accelgrp;
	GtkWidget *w, *parent, *parent2;
	GtkWidget *toplevel, *main_vbox;
	const cfg_item_struct *cfg_list;
	core_struct *core = CORE(core_ptr);
	win_struct *win = WIN(
	    g_malloc0(sizeof(win_struct))
	);

	cfg_list = core->cfg_list;

	win->toplevel = toplevel = gtk_window_new(GTK_WINDOW_TOPLEVEL);  
	win->accelgrp = accelgrp = gtk_accel_group_new();
	win->processing = FALSE;
	win->busy_count = 0;
	win->passive_busy_count = 0;
	win->freeze_count = 0;
	win->stop_count = 0;
	win->page_num = WIN_PAGE_NUM_SCAN;
	win->core = core_ptr;
	win->opid = WinOPIDListNew(win);
	win->menubar_map_state = TRUE;
	win->toolbar_map_state = CFGItemListGetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_TOOL_BAR
	);
	win->statusbar_map_state = CFGItemListGetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_STATUS_BAR
	);
	win->statusbar_progress_last = 0.0f;
	win->db_root_node = NULL;
	win->db_ctree_load_idleid = 0;
	win->scan_toid = 0;
	win->scan_pid = 0;
	win->update_toid = 0;
	win->update_pid = 0;
	win->stdout_fp = NULL;
	win->stderr_fp = NULL;
	win->stdout_path = NULL;
	win->stderr_path = NULL;

	w = toplevel;
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(WinEventCB), win
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(WinEventCB), win
	);
	if(cfg_list != NULL)
	{
	    const gint	x = CFGItemListGetValueI(cfg_list, CFG_PARM_WIN_X),
			y = CFGItemListGetValueI(cfg_list, CFG_PARM_WIN_Y);
	    gtk_widget_set_uposition(w, x, y);
	    gtk_widget_set_usize(w, WIN_WIDTH, WIN_HEIGHT);
	}
	gtk_widget_realize(w);
	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_decorations(
		window,
		GDK_DECOR_BORDER | GDK_DECOR_TITLE | GDK_DECOR_MENU |
		GDK_DECOR_MINIMIZE
	    );
	    gdk_window_set_functions(
		window,
		GDK_FUNC_MOVE | GDK_FUNC_MINIMIZE | GDK_FUNC_CLOSE
	    );
	    GUISetWMIcon(window, (guint8 **)icon_avscan_48x48_xpm);
	    gtk_window_set_title(GTK_WINDOW(w), PROG_NAME_FULL);
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(WinEventCB), win
	);
	gtk_container_border_width(GTK_CONTAINER(w), 0);
	gtk_window_add_accel_group(GTK_WINDOW(w), accelgrp);
	parent = w;


	/* Main VBox */
	win->main_vbox = main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), 0);                        
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;


	/* Build Menu Bar */
	win->menubar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	win->menubar = NULL;
	WinBuildMenuBar(win);

	/* Build Tool Bar */
	win->toolbar_handle = w = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	win->toolbar = NULL;
	WinBuildToolBar(win);


	/* Vbox for the lists */
	win->lists_vbox = w = gtk_vbox_new(FALSE, 0);
/*	gtk_container_border_width(GTK_CONTAINER(w), 0); */
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Build lists */
	WinBuildLists(win);


	/* Build status bar */
	win->statusbar = NULL;
	WinBuildStatusBar(win);


	/* Build right click menu */
	win->scan_list_menu = NULL;
	WinBuildListsMenu(win);


	/* Show/hide menu bar */
	w = win->menubar_handle;
	if(win->menubar_map_state)
	    gtk_widget_show(w);
	else
	    gtk_widget_hide(w);

	/* Show/hide tool bar */
	w = win->toolbar_handle;
	if(win->toolbar_map_state)
	    gtk_widget_show(w);
	else
	    gtk_widget_hide(w);

	/* Show/hide status bar */
	w = win->statusbar;
	if(win->statusbar_map_state)
	    gtk_widget_show(w);
	else
	    gtk_widget_hide(w);


	/* Set initial notebook page */
	gtk_notebook_set_page(
	    GTK_NOTEBOOK(win->notebook), (gint)win->page_num
	);


	WinUpdate(win);


	return(win);
}

/*
 *	Records the Win's widget positions to the configuration.
 */
void WinSavePositions(win_struct *win)
{
	GtkWidget *w;
	core_struct *core = CORE((win != NULL) ? win->core : NULL);
	cfg_item_struct *cfg_list = (core != NULL) ? core->cfg_list : NULL;

	if(cfg_list == NULL)
	    return;

	/* Toplevel Position */
	w = win->toplevel;
	if(w != NULL)
	{
	    GdkWindow *window = w->window;
	    gint x = 0, y = 0;

	    if(window != NULL)
		gdk_window_get_root_origin(window, &x, &y);

	    CFGItemListSetValueI(
		cfg_list, CFG_PARM_WIN_X,
		x, FALSE
	    );          
	    CFGItemListSetValueI(
		cfg_list, CFG_PARM_WIN_Y,
		y, FALSE
	    );
#if 0
	    width = w->allocation.width;
	    height = w->allocation.height;
	    CFGItemListSetValueI(
		cfg_list, CFG_PARM_WIN_WIDTH,
		width, FALSE
	    );
	    CFGItemListSetValueI(
		cfg_list, CFG_PARM_WIN_HEIGHT,
		height, FALSE
	    );
#endif
	}

	/* Show Tool Bar */
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_TOOL_BAR,
	    win->toolbar_map_state, FALSE
	);

	/* Show Status Bar */
	CFGItemListSetValueI(
	    cfg_list, CFG_PARM_WIN_SHOW_STATUS_BAR,
	    win->statusbar_map_state, FALSE
	);

	/* Scan List */
	w = win->scan_clist;
	if(w != NULL)
	{
	    gint i;
	    gchar *path = STRDUP(PrefixPaths(
		core->prog_data_dir, AVSCAN_SCAN_LIST_FILE
	    ));
	    GList *obj_list = NULL;
	    GtkCList *clist = GTK_CLIST(w);

	    /* Get scan list */
	    for(i = 0; i < clist->rows; i++)
		obj_list = g_list_append(
		    obj_list,
		    gtk_clist_get_row_data(clist, i)
		);

	    /* Save scan list */
	    ObjListSave(path, obj_list, NULL, NULL);

	    g_list_free(obj_list);
	}

}

/*
 *	Updates the Win's widgets to reflect current values.
 */
void WinUpdate(win_struct *win)
{
	gboolean sensitive, toggled, scanning, db_loading, db_updating;
	gint last_selected_scan_item, last_selected_results_item;
	win_page_num page_num;
	GList *glist;
	GtkCList *clist;
	toolbar_struct *toolbar;

	if(win == NULL)
	    return;

	win->freeze_count++;

	toolbar = win->toolbar;

	page_num = win->page_num;
	scanning = WinScanProcessIsRunning(win);
	db_loading = (win->db_ctree_load_idleid > 0) ? TRUE : FALSE;
	db_updating = WinUpdateProcessIsRunning(win);

	clist = (GtkCList *)win->scan_clist;
	glist = (clist != NULL) ? clist->selection_end : NULL;
	last_selected_scan_item = (glist != NULL) ?
	    (gint)glist->data : -1;

	clist = (GtkCList *)win->results_clist;
	glist = (clist != NULL) ? clist->selection_end : NULL;
	last_selected_results_item = (glist != NULL) ?           
	    (gint)glist->data : -1;


	/* Menu Bar Labels */
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->file_mlabel, sensitive)
	sensitive = (page_num == WIN_PAGE_NUM_SCAN) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->edit_mlabel, sensitive)
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->scan_mlabel, sensitive)  
	sensitive = (page_num == WIN_PAGE_NUM_RESULTS) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->results_mlabel, sensitive)  
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->view_mlabel, sensitive)  
	sensitive = TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->help_mlabel, sensitive)  

	/* New and Add */
	sensitive = (page_num == WIN_PAGE_NUM_SCAN) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->new_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->add_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_NEW, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_ADD, sensitive
	);      

	/* Close */
	sensitive = (scanning || db_loading || db_updating) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->close_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->close_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_CLOSE, sensitive
	);

	/* Cut, Copy, Delete, Edit, Remove, and Properties */
	sensitive = ((page_num == WIN_PAGE_NUM_SCAN) &&
	    (last_selected_scan_item > -1)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->cut_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->copy_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->delete_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->properties_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->edit_btn, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->remove_btn, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_cut_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_copy_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_delete_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_properties_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_EDIT, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_REMOVE, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_CUT, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_COPY, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_DELETE, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_PROPERTIES, sensitive
	);

	/* Paste */
	sensitive = (page_num == WIN_PAGE_NUM_SCAN) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->paste_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_paste_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_PASTE, sensitive
	);

	/* Start */
	sensitive = (scanning || db_updating) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->start_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->start_btn, sensitive) 
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_start_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_START, sensitive
	);

	/* Stop */
	sensitive = (scanning || db_loading || db_updating) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->stop_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->stop_btn, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->scan_list_menu_stop_mi, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_STOP, sensitive
	);


	/* Results Recycle & Move */
	sensitive = ((page_num == WIN_PAGE_NUM_RESULTS) &&
	    (last_selected_results_item > -1)) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->results_recycle_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_list_menu_recycle_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_recycle_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_RESULTS_RECYCLE, sensitive
	);
	GTK_WIDGET_SET_SENSITIVE(win->results_move_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_list_menu_move_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_move_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_RESULTS_MOVE, sensitive
	);

	/* Results Open, Save As, and Clear */ 
	sensitive = ((page_num == WIN_PAGE_NUM_RESULTS) &&
	    !scanning) ? TRUE : FALSE;
	GTK_WIDGET_SET_SENSITIVE(win->results_open_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_save_as_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_clear_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_list_menu_clear_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->results_clear_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_RESULTS_OPEN, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_RESULTS_SAVE_AS, sensitive
	);
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_RESULTS_CLEAR, sensitive
	);


	/* Virus Database Refresh */
	sensitive = ((page_num != WIN_PAGE_NUM_DB) ||
	    scanning || db_loading || db_updating) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->db_refresh_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->db_refresh_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_DB_REFRESH, sensitive
	);

	/* Virus Database Location */
	sensitive = (scanning || db_loading || db_updating) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->db_location_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->db_location_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_DB_LOCATION, sensitive
	);

	/* Virus Database Update Net */
	sensitive = (scanning || db_loading || db_updating) ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->db_update_net_mi, sensitive)
	GTK_WIDGET_SET_SENSITIVE(win->db_update_net_btn, sensitive)
	ToolBarItemSetSensitiveID(
	    toolbar, WIN_OPID_DB_UPDATE_NET, sensitive
	);


	/* View Scan Page */
	toggled = (page_num == WIN_PAGE_NUM_SCAN) ? TRUE : FALSE;
	GUIMenuItemSetCheck(
	    win->view_scan_page_micheck,
	    toggled, TRUE
	);
	ToolBarItemSetToggleID(
	    toolbar, WIN_OPID_VIEW_SCAN_PAGE, toggled
	);

	/* View Results Page */
	toggled = (page_num == WIN_PAGE_NUM_RESULTS) ? TRUE : FALSE;
	GUIMenuItemSetCheck(
	    win->view_results_page_micheck,
	    toggled, TRUE
	);
	ToolBarItemSetToggleID(
	    toolbar, WIN_OPID_VIEW_RESULTS_PAGE, toggled
	);

	/* View Virus Database Page */
	toggled = (page_num == WIN_PAGE_NUM_DB) ? TRUE : FALSE;
	GUIMenuItemSetCheck(
	    win->view_db_page_micheck,
	    toggled, TRUE
	);
	ToolBarItemSetToggleID(
	    toolbar, WIN_OPID_VIEW_DB_PAGE, toggled
	);

	/* View Tool Bar */
	toggled = win->toolbar_map_state;
	GUIMenuItemSetCheck(
	    win->view_toolbar_micheck,
	    toggled, TRUE
	);
	ToolBarItemSetToggleID(
	    toolbar, WIN_OPID_VIEW_TOOLBAR, toggled
	);

	/* View Status Bar */
	toggled = win->statusbar_map_state;
	GUIMenuItemSetCheck(
	    win->view_statusbar_micheck,
	    toggled, TRUE
	);
	ToolBarItemSetToggleID(
	    toolbar, WIN_OPID_VIEW_STATUSBAR, toggled
	);


	/* Virus Database Find Entry */
	sensitive = db_loading ? FALSE : TRUE;
	GTK_WIDGET_SET_SENSITIVE(win->db_find_entry, sensitive)


	win->freeze_count--;
}

/*
 *	Sets the Win as busy or ready.
 */
void WinSetBusy(win_struct *win, gboolean busy)
{
	GdkCursor *cur;
	GdkWindow *window;
	GtkWidget *w;
	core_struct *core;
	if(win == NULL)
	    return;

	w = win->toplevel;
	core = win->core;
	if(core == NULL)
	    return;

	if(busy)
	{
	    win->busy_count++;
	    if(win->busy_count > 1)
		return;
	    cur = core->busy_cur;
	}
	else
	{
	    win->busy_count--;
	    if(win->busy_count < 0)
		win->busy_count = 0;
	    if(win->busy_count > 0)
		return;
	    cur = (win->passive_busy_count > 0) ?
		core->passive_busy_cur : NULL;
	}

	window = w->window;
	if(window != NULL)
	{
	    gdk_window_set_cursor(window, cur);
	    gdk_flush();
	}
}

/*
 *      Sets the Win as passive busy or ready.
 */
void WinSetPassiveBusy(win_struct *win, gboolean busy)
{
	GdkCursor *cur;
	GtkWidget *w;
	core_struct *core;

	if(win == NULL)
	    return;

	w = win->toplevel;
	core = CORE(win->core);
	if((w == NULL) || (core == NULL))
	    return;

	if(busy)
	{
	    win->passive_busy_count++;
	    if((win->passive_busy_count > 1) || (win->busy_count > 0))
		return;
	    cur = core->passive_busy_cur;
	}
	else
	{
	    win->passive_busy_count--;
	    if(win->passive_busy_count < 0)
		win->passive_busy_count = 0;
	    if((win->passive_busy_count > 0) || (win->busy_count > 0))
		return;
	    cur = NULL;
	}

	if(w->window != NULL)
	{
	    gdk_window_set_cursor(w->window, cur);
	    gdk_flush();
	}
}

/*                          
 *	Checks if the Win's Tool Bar is shown.      
 */
gboolean WinToolBarIsShown(win_struct *win)
{
	return((win != NULL) ? win->toolbar_map_state : FALSE);
}

/*
 *	Show/hide the Win's Tool Bar.
 */
void WinToolBarSetShow(win_struct *win, gboolean show)
{
	GtkWidget *w = (win != NULL) ? win->toolbar_handle : NULL;
	if(w == NULL)
	    return;

	win->freeze_count++;

	if(win->toolbar_map_state && !show)
	{
	    gtk_widget_hide(w);
	    win->toolbar_map_state = FALSE;
	}
	else if(!win->toolbar_map_state && show)
	{
	    gtk_widget_show(w);
	    win->toolbar_map_state = TRUE;
	}

	WinUpdate(win);

	win->freeze_count--;
}

/*
 *	Recreates the Win's Tool Bar Items.
 */
void WinToolBarUpdateItems(win_struct *win)
{
	if(win == NULL)
	    return;

	/* Recreate Tool Bar and the Tool Bar Items */
	WinBuildToolBar(win);
}


/*                          
 *	Checks if the Win's Status Bar is shown.
 */
gboolean WinStatusBarIsShown(win_struct *win)
{
	return((win != NULL) ? win->statusbar_map_state : FALSE);
}

/*
 *	Show/hide the Win's Status Bar.
 */
void WinStatusBarSetShow(win_struct *win, gboolean show)
{
	GtkWidget *w = (win != NULL) ? win->statusbar : NULL;
	if(w == NULL)
	    return;

	win->freeze_count++;

	if(win->statusbar_map_state && !show)
	{
	    gtk_widget_hide(w);
	    win->statusbar_map_state = FALSE;
	}
	else if(!win->statusbar_map_state&& show)
	{
	    gtk_widget_show(w);
	    win->statusbar_map_state = TRUE;
	}

	WinUpdate(win);

	win->freeze_count--;
}


/*
 *	Sets the Win's status bar progress.
 */
void WinStatusProgress(
	win_struct *win, gfloat val, gboolean allow_gtk_iteration
)
{
	GtkProgress *progress;
	GtkWidget *w = (win != NULL) ? win->statusbar_progress : NULL;
	if(w == NULL)
	    return;

	progress = GTK_PROGRESS(w);

	/* Do activity? */
	if(val < 0.0f)
	{
	    GtkAdjustment *adj = progress->adjustment;

	    /* Get new value based on unknown position */
	    val = gtk_progress_get_value(progress) + 1.0f;
	    if(val > adj->upper)
		val = adj->lower;

	    /* Update progress bar for `activity mode' (unknown limit)
	     * and set new value            
	     */
	    gtk_progress_set_activity_mode(progress, TRUE);
	    gtk_progress_set_show_text(progress, FALSE);
	    gtk_progress_set_value(progress, val);

	    /* Reset last progress position to -1.0, indicating that
	     * there is no known last position since we are doing
	     * activity mode
	     */
	    win->statusbar_progress_last = -1.0f;
	}
	else
	{
	    /* Clip value to 0.0, 1.0 */
	    if(val > 1.0f)
		val = 1.0f;

	    /* Reset last progress position if it is greater than the
	     * given value, implying the progress has warped back to
	     * the beginning
	     */
	    if(win->statusbar_progress_last > val)
		win->statusbar_progress_last = -1.0f;

	    /* Has value not sufficiently changed? (less than 0.01
	     * change in value)
	     */
	    if((win->statusbar_progress_last > 0.0f) &&
	       ((val - win->statusbar_progress_last) < 0.01f)
	    )
		return;

	    /* Update progress */ 
	    gtk_progress_set_activity_mode(progress, FALSE);
	    gtk_progress_set_format_string(
		progress, "%p%%"
	    );
	    /* Display text only if value is positive */
	    gtk_progress_set_show_text(
		progress, (val > 0.0f) ? TRUE : FALSE 
	    );
	    gtk_progress_bar_update(GTK_PROGRESS_BAR(w), val); 

	    /* Record last position */
	    win->statusbar_progress_last = val;
	}

	/* Flush events? */
	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}

/*
 *	Sets the Win's status bar message.
 */
void WinStatusMessage(
	win_struct *win, const gchar *mesg, gboolean allow_gtk_iteration
)
{
	GtkWidget *w = (win != NULL) ? win->statusbar_label : NULL;
	if(w == NULL)
	    return;

	gtk_label_set_text(
	    GTK_LABEL(w), (mesg != NULL) ? mesg : ""
	);  

	while((gtk_events_pending() > 0) && allow_gtk_iteration)
	    gtk_main_iteration();
}

/*
 *	Checks if the Win is mapped.
 */
gboolean WinIsMapped(win_struct *win)
{
	GtkWidget *w = (win != NULL) ? win->toplevel : NULL;
	return((w != NULL) ? GTK_WIDGET_MAPPED(w) : FALSE);
}

/*
 *	Maps the Win.
 */
void WinMap(win_struct *win)
{
	GtkWidget *w = (win != NULL) ? win->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_show_raise(w);
}

/*
 *	Unmaps the Win.
 */
void WinUnmap(win_struct *win)
{
	GtkWidget *w = (win != NULL) ? win->toplevel : NULL;
	if(w == NULL)
	    return;

	gtk_widget_hide(w);
}

/*
 *	Deletes the Win.
 */
void WinDelete(win_struct *win)
{
	if(win == NULL)
	    return;

	/* Interrupt all child process and delete stdout and stderr
	 * files
	 */
	WinScanProcessStop(win);
	WinUpdateProcessStop(win);

	win->freeze_count++;

	/* Right-click menus */
	GTK_WIDGET_DESTROY(win->scan_list_menu);
	win->scan_list_menu = NULL;
	GTK_WIDGET_DESTROY(win->results_list_menu);
	win->results_list_menu = NULL;

	/* Menu Bar */
	GTK_WIDGET_DESTROY(win->menubar);
	win->menubar = NULL;
	GTK_WIDGET_DESTROY(win->menubar_handle);
	win->menubar_handle = NULL;

	/* Tool Bar */
	ToolBarDelete(win->toolbar);
	win->toolbar = NULL;
	GTK_WIDGET_DESTROY(win->toolbar_handle);
	win->toolbar_handle = NULL;

	/* Lists */
	GTK_WIDGET_DESTROY(win->scan_clist);
	win->scan_clist = NULL;
	GTK_WIDGET_DESTROY(win->add_btn);
	win->add_btn = NULL;
	GTK_WIDGET_DESTROY(win->edit_btn);
	win->edit_btn = NULL;
	GTK_WIDGET_DESTROY(win->remove_btn);
	win->remove_btn = NULL;
	GTK_WIDGET_DESTROY(win->location_entry);
	win->location_entry = NULL;
	GTK_WIDGET_DESTROY(win->recursive_check);
	win->recursive_check = NULL;
	GTK_WIDGET_DESTROY(win->executables_only_check);
	win->executables_only_check = NULL;
	GTK_WIDGET_DESTROY(win->results_location_label);
	win->results_location_label = NULL;
	GTK_WIDGET_DESTROY(win->results_clist);
	win->results_clist = NULL;
	GTK_WIDGET_DESTROY(win->db_find_entry);
	win->db_find_entry = NULL;
	GTK_WIDGET_DESTROY(win->db_ctree);
	win->db_ctree = NULL;
	GTK_WIDGET_DESTROY(win->db_stats_label);
	win->db_stats_label = NULL;
	GTK_WIDGET_DESTROY(win->db_refresh_btn);
	win->db_refresh_btn = NULL;
	GTK_WIDGET_DESTROY(win->db_location_btn);
	win->db_location_btn = NULL;
	GTK_WIDGET_DESTROY(win->db_update_net_btn);
	win->db_update_net_btn = NULL;
	GTK_WIDGET_DESTROY(win->anim_icon);
	win->anim_icon = NULL;
	GTK_WIDGET_DESTROY(win->start_btn);
	win->start_btn = NULL;
	GTK_WIDGET_DESTROY(win->stop_btn);
	win->stop_btn = NULL;
	GTK_WIDGET_DESTROY(win->close_btn);
	win->close_btn = NULL;
	GTK_WIDGET_DESTROY(win->notebook);
	win->notebook = NULL;
	GTK_WIDGET_DESTROY(win->lists_vbox);
	win->lists_vbox = NULL;

	/* Status Bar */
	GTK_WIDGET_DESTROY(win->statusbar);
	win->statusbar = NULL;

	/* Main VBox */
	GTK_WIDGET_DESTROY(win->main_vbox);
	win->main_vbox = NULL;

	/* Toplevel */
	GTK_WIDGET_DESTROY(win->toplevel);
	win->toplevel = NULL;

	/* Accel group */
	GTK_ACCEL_GROUP_UNREF(win->accelgrp);
	win->accelgrp = NULL;

	/* OPIDs list */
	WinOPIDListDelete(win->opid);
	win->opid = NULL;

	win->freeze_count--;

	g_free(win);
}
