#include <sys/stat.h>
#include <glib.h>

#include "../include/fio.h"
#include "../include/disk.h"

#include "obj.h"
#include "objfio.h"
#include "config.h"


GList *ObjListOpen(
	const gchar *filename,
	gint (*progress_cb)(gpointer, gulong, gulong),
	gpointer client_data
);
gint ObjListSave(
	const gchar *filename,
	GList *obj_list,
	gint (*progress_cb)(gpointer, gulong, gulong),
	gpointer client_data  
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Opens the objects from the specified file.
 */
GList *ObjListOpen(
	const gchar *filename,
	gint (*progress_cb)(gpointer, gulong, gulong),
	gpointer client_data  
)
{
	gboolean aborted = FALSE;
	gchar *parm;
	FILE *fp;
	gulong file_size;
	struct stat stat_buf;
	GList *obj_list = NULL;
	obj_struct *obj = NULL;

	if(STRISEMPTY(filename))
	    return(obj_list);

	/* Open file for reading */
	fp = FOpen(filename, "rb");
	if(fp == NULL)
	    return(obj_list);

	/* Get file statistics */
	if(fstat(fileno(fp), &stat_buf))
	    file_size = 0l;
	else
	    file_size = (gulong)stat_buf.st_size;

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(
		client_data, 0l, file_size
	    ))
		aborted = TRUE;
	}

	/* Begin reading file */
	parm = NULL;
	while(!aborted)
	{
	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    client_data, (gulong)ftell(fp), file_size
		))
		{
		    aborted = TRUE;
		    break;
		}
	    }

	    /* Read next parameter */
	    parm = FSeekNextParm(
		fp, parm,
		CFG_COMMENT_CHAR,
		CFG_DELIMINATOR_CHAR
	    );
	    if(parm == NULL)
		break;

	    /* Begin handling by parameter */

	    /* BeginObject */
	    if(!g_strcasecmp(parm, "BeginObject"))
	    {
		gchar *v = FGetString(fp);
		obj = ObjNew();
		if(obj != NULL)
		{
		    obj_list = g_list_append(obj_list, obj);

		    g_free(obj->name);
		    obj->name = STRDUP(v);
		}
		g_free(v);
	    }
	    /* Path */
	    else if(!g_strcasecmp(parm, "Path"))
	    {
		gchar *v = FGetString(fp);
		if(obj != NULL)
		{
		    g_free(obj->path);
		    obj->path = STRDUP(v);
		}
		g_free(v);
	    }
	    /* Recursive */
	    else if(!g_strcasecmp(parm, "Recursive"))
	    {
		gint v[1];
		FGetValuesI(fp, v, 1);
		if(obj != NULL)
		    obj->recursive = (v[0]) ? TRUE : FALSE;
	    }
	    /* ExecutablesOnly */
	    else if(!g_strcasecmp(parm, "ExecutablesOnly"))
	    {
		gint v[1];
		FGetValuesI(fp, v, 1);
		if(obj != NULL)
		    obj->executables_only = (v[0]) ? TRUE : FALSE;
	    }
	    /* LastRunned */
	    else if(!g_strcasecmp(parm, "LastRunned"))
	    {
		glong v[1];
		FGetValuesL(fp, v, 1);
		if(obj != NULL)
		    obj->last_runned = (gulong)v[0];
	    }
	    /* EndObject */
	    else if(!g_strcasecmp(parm, "EndObject"))
	    {
		FSeekNextLine(fp);

		/* Reset contexts */
		obj = NULL;
	    }
	    /* All else unsupported parameter or wrong context */
	    else
	    {
		FSeekNextLine(fp);
	    }
	}

	/* Delete parameter */
	g_free(parm);

	/* Close file */
	FClose(fp);

	/* Report final progress */
	if((progress_cb != NULL) && !aborted)
	    progress_cb(client_data, file_size, file_size);

	return(obj_list);
}

/*
 *	Saves the object list to the specified file.
 */
gint ObjListSave(
	const gchar *filename,
	GList *obj_list,
	gint (*progress_cb)(gpointer, gulong, gulong),
	gpointer client_data 
)
{
	gboolean aborted = FALSE;
	gint i, nobjs;
	gchar *parent_path;
	FILE *fp;
	GList *glist;
	obj_struct *obj;

	if(STRISEMPTY(filename))
	    return(-1);

	nobjs = g_list_length(obj_list);

	/* Get parent directory and create it as needed */
	parent_path = STRDUP(GetParentDir(filename));
	if(parent_path != NULL)
	{
	    rmkdir(parent_path, S_IRUSR | S_IWUSR | S_IXUSR);
	    g_free(parent_path);
	    parent_path = NULL;
	}

	/* Open file for writing */
	fp = FOpen(filename, "wb");
	if(fp == NULL)
	    return(-1);

	/* Report initial progress */
	if(progress_cb != NULL)
	{
	    if(progress_cb(client_data, 0l, (gulong)nobjs))
		aborted = TRUE;
	}

	/* Begin writing file */
	for(glist = obj_list, i = 0;
	    (glist != NULL) && !aborted;
	    glist = g_list_next(glist), i++
	)
	{
	    obj = OBJ(glist->data);
	    if(obj == NULL)
		continue;

	    /* Report progress */
	    if(progress_cb != NULL)
	    {
		if(progress_cb(
		    client_data, (gulong)i, (gulong)nobjs
		))
		{
		    aborted = TRUE;
		    break;
		}
	    }


	    /* BeginObject */
	    fprintf(
		fp,
		"BeginObject = %s\n",
		obj->name
	    );
	    /* Path */
	    if(!STRISEMPTY(obj->path))
		fprintf(
		    fp,
		    "\tPath = %s\n",
		    obj->path
		);
	    /* Recursive */
	    fprintf(
		fp,
		"\tRecursive = %i\n",
		obj->recursive
	    );
	    /* ExecutablesOnly */
	    fprintf(
		fp, 
		"\tExecutablesOnly = %i\n",
		obj->executables_only
	    );
	    /* LastRunned */
	    fprintf(
		fp,
		"\tLastRunned = %ld\n",
		obj->last_runned
	    );
	    /* EndObject */
	    fprintf(
		fp,
		"EndObject\n"
	    );
	}

	/* Close file */
	FClose(fp);

	/* Report final progress */   
	if((progress_cb != NULL) && !aborted)
	    progress_cb(client_data, (gulong)nobjs, (gulong)nobjs);

	return(aborted ? -4 : 0);
}
