#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "obj.h"


obj_struct *ObjNew(void);
obj_struct *ObjCopy(const obj_struct *obj);
void ObjDelete(obj_struct *obj);

GList *ObjDDEBufferParse(const guint8 *buf, gint buf_len);
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj  
);
GList *ObjDDEBufferParsePath(const guint8 *buf, gint buf_len);
guint8 *ObjDDEBufferAppendPath(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new Object.
 */
obj_struct *ObjNew(void)
{
	return(
	    OBJ(g_malloc0(sizeof(obj_struct)))
	);

}

/*
 *	Coppies the Object.
 */
obj_struct *ObjCopy(const obj_struct *obj)
{
	const obj_struct *src = obj;
	obj_struct *tar = ObjNew();
	if(tar == NULL)
	    return(NULL);

	tar->name = STRDUP(src->name);
	tar->path = STRDUP(src->path);
	tar->recursive = src->recursive;
	tar->executables_only = src->executables_only;
	tar->last_runned = src->last_runned;

	return(tar);
}

/*
 *	Deletes the Object.
 */
void ObjDelete(obj_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);
	g_free(obj);
}


/*
 *	Parses the DDE buffer into a list of Objects.
 */
GList *ObjDDEBufferParse(const guint8 *buf, gint buf_len)
{
	gint segment_cnt = 0, len;
	gchar *segment;
	const gchar *s;
	const guint8 *buf_end = buf + buf_len;
	GList *glist = NULL;
	obj_struct *obj = NULL;

	if(buf == NULL)
	    return(glist);

	/* Iterate through buffer */
	while(buf < buf_end)
	{
	    /* Calculate length of this segment (excluding deliminator) */
	    for(s = (const gchar *)buf;
		s < (const gchar *)buf_end;
		s++
	    )
	    {
		if(*s == '\0')
		    break;
	    }
	    len = s - (const gchar *)buf;

	    /* Copy this segment */
	    segment = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(segment == NULL)
		break;
	    memcpy(segment, (const gchar *)buf, len * sizeof(gchar));
	    segment[len] = '\0';

	    /* Seek buf to next segment */
	    buf += len + 1;


	    /* Parse this segment by the segment counter */
	    /* Path */
	    if(segment_cnt == 0)
	    {
		/* Create a new Object and append it to the list */
		obj = ObjNew();
		if(obj != NULL)
		{
		    glist = g_list_append(glist, obj);

		    /* Seek s past "://" deliminator to the start of
		     * the path
		     */
		    for(s = segment; *s != '\0'; s++)
		    {
		        if(*s == ':')
		        {
			    s++;
			    if(*s == '/')
				s++;
			    if(*s == '/')
				s++;

			    while((*s != '\0') && (*s != '/'))
			        s++;

			    break;
		        }
		        else if(*s == '/')
		        {
			    break;
		        }
		    }
		    obj->path = STRDUP(s);
		}
	    }
	    /* Name */
	    else if(segment_cnt == 1)
	    {
		if(obj != NULL)
		{
		    obj->name = STRDUP(segment);
		}
	    }
	    /* Recursive */
	    else if(segment_cnt == 2)
	    {
		if(obj != NULL)
		{
		    obj->recursive = ATOI(segment) ? TRUE : FALSE;
		}
	    }
	    /* Executables Only */
	    else if(segment_cnt == 3)
	    {
		if(obj != NULL)
		{     
		    obj->executables_only = ATOI(segment) ? TRUE : FALSE;
		}
	    }
	    /* Last Runned */
	    else if(segment_cnt == 4)
	    {
		if(obj != NULL)
		{     
		    obj->last_runned = (gulong)ATOL(segment);
		}
	    }

	    segment_cnt++;
	    if(segment_cnt > 4)
		segment_cnt = 0;

	    g_free(segment);
	}

	/* Number of segments for last object not complete? */
	if((segment_cnt != 0) && (obj != NULL))
	{
	    /* Try to complete object values */
	    if(obj->name == NULL)
	    {
		s = obj->path;
		s = (s != NULL) ? strrchr(s, '/') : NULL;
		if(s != NULL)
		    s++;
		else
		    s = obj->path;
		g_free(obj->name);
		obj->name = STRDUP(s);
	    }

	}

	return(glist);
}

/*
 *	Appends the Object to the DDE buffer.
 */
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj
)
{
	gchar *s;

	if((obj == NULL) || (buf_len == NULL))
	    return(buf);

#define APPEND_BUF(_s_)		{		\
 const gint	i = *buf_len,			\
		len = STRLEN(_s_);		\
						\
 *buf_len = i + len + 1;			\
 buf = (guint8 *)g_realloc(			\
  buf, (*buf_len) * sizeof(guint8)		\
 );						\
 if(buf != NULL) {				\
  memcpy(buf + i, (_s_), len * sizeof(guint8));	\
  buf[i + len] = '\0';				\
 } else {					\
  *buf_len = 0;					\
 }						\
}

	s = g_strdup_printf("file://%s", obj->path);
	APPEND_BUF(s);
	g_free(s);

	APPEND_BUF(obj->name);

	s = g_strdup_printf("%i", obj->recursive);
	APPEND_BUF(s);
	g_free(s);

	s = g_strdup_printf("%i", obj->executables_only);
	APPEND_BUF(s);
	g_free(s);

	s = g_strdup_printf("%ld", obj->last_runned);
	APPEND_BUF(s);
	g_free(s);

	return(buf);
#undef APPEND_BUF
}

/*
 *	Same as ObjDDEBufferParse() except that only the path is
 *	parsed from the buffer.
 */
GList *ObjDDEBufferParsePath(const guint8 *buf, gint buf_len)
{
	gint len;
	gchar *url;
	const gchar *s;
	const guint8 *buf_end = buf + buf_len;
	GList *glist = NULL;
	obj_struct *obj;

	if(buf == NULL)
	    return(glist);

	/* Iterate through buffer */
	while(buf < buf_end)
	{                   
	    /* Create a new Object and append it to the list */
	    obj = ObjNew();
	    if(obj == NULL)
		break;

	    glist = g_list_append(glist, obj);

	    /* Parse URL */
	    s = (const gchar *)buf;
	    while(s < (const gchar *)buf_end)
	    {
		if(*s == '\0')
		    break;
		s++;
	    }
	    len = s - (const gchar *)buf;
		 
	    url = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(url != NULL)
	    {
		const gchar *s2, *s3;

		memcpy(url, buf, len * sizeof(gchar));
		url[len] = '\0';

		/* Seek s2 past "://" deliminator to the start of the
		 * path
		 */
		for(s2 = url; *s2 != '\0'; s2++)
		{
		    if(*s2 == ':')  
		    {
			if(*s2 == ':')
			    s2++;
			if(*s2 == '/')
			    s2++;
			if(*s2 == '/')
			    s2++;

			while((*s2 != '\0') && (*s2 != '/'))
			    s2++;

			break;
		    }
		    else if(*s2 == '/')
		    {
			break;
		    }
		}
		 
		/* Get/copy path from URL */
		obj->path = STRDUP(s2);

		/* Get/copy name from path */
		s2 = obj->path;
		if(s2 != NULL)
		{
		    for(s3 = s2; *s3 != '\0'; s3++);

		    while(s3 > s2)
		    {
			if(*s3 == '/')
			{
			    s3++;
			    break;
			}
			s3--;
		    }

		    obj->name = STRDUP(s3);
		}
		 
		g_free(url);
	    }
	     
	    buf += len;

	    buf++;                      /* Skip deliminator */
	}
	 
	return(glist);
}

/*
 *	Same as ObjDDEBufferAppend() except that only the path is
 *	appended to the buffer.
 */
guint8 *ObjDDEBufferAppendPath(
	guint8 *buf, gint *buf_len,
	const obj_struct *obj
)
{
	gint i, len;
	gchar *url;

	if((obj == NULL) || (buf_len == NULL))
	    return(buf); 

	/* Get current length and starting position */
	len = i = *buf_len;

	/* Format URL */
	url = g_strdup_printf("file://%s", obj->path);

	/* Calculate new buffer length needed */
	len += STRLEN(url);
	len++;                          /* Null terminating byte / deliminator */

	/* Increase buffer allocation */
	buf = (guint8 *)g_realloc(
	    buf,
	    len * sizeof(guint8)
	);
	if(buf == NULL)
	{
	    *buf_len = 0;
	    return(buf);
	}               

	/* Copy URL to the buffer */
	if(url != NULL)
	{
	    const gint url_len = STRLEN(url);
	    memcpy(buf + i, url, url_len);
	    i += url_len;

	    g_free(url);
	}

	/* Null terminating byte */
	buf[i] = '\0';

	*buf_len = len;
	return(buf);
}

