# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Definition of ASTER_RUN class.
"""


import sys
import os
import re
import time
import getpass
import traceback
import pprint
from ConfigParser import SafeConfigParser, NoOptionError

from asrun.i18n         import _
from asrun.mystring     import print3
from asrun.option       import AsRunParser, OptionGroup, SUPPRESS_HELP
from asrun.timer        import ASTER_TIMER
from asrun.utils        import get_absolute_dirname, remove_subdirs
from asrun.mystring     import convert
from asrun.rcfile       import read_rcfile
from asrun.installation import aster_root, confdir, localedir


class RunAsterError(Exception):
   """
   Base exception for ASTER_RUN
   """

class ASTER_RUN:
   # Main class  (__doc__ is used as help string for the parser)
   u"""%(program)s action [options] [arguments]

  Functions :
"""
   def __init__(self):
      """
      Initializations
      """
      self.ExitOnFatalError = True
      self.PrintExitCode    = True
      self.__initialized    = False
      self.__dbg_count      = 0
      self._val_pid         = 0
      # ----- verification de la plate-forme
      if os.name != 'posix':
         print3(_(u"""This program runs only under Linux/Unix system'
... but its license allows you to port it under another system and propose your
modifications to the author !
Porting should be limited to 'os.system' calls and other
functions of the 'os' module which are not platform independent.
Contact : http://www.code-aster.org"""))
         self.Sortie(4)

      # ----- verification de la version de Python
      if sys.hexversion < 0x020400F0:
         print3(_(u"This script requires Python 2.4 or higher, sorry !"))
         self.Sortie(4)

      # ----- package informations
      from asrun.__pkginfo__ import version, copyright
      self.program     = 'as_run'
      self.__version__ = version
      
      # ----- options, liste des options du parser à stocker dans options
      self.options = { 'verbose' : False, 'silent' : False }
      # always use _add_option to extend options_list
      self.options_list = ['debug', 'verbose', 'silent', 'force', 'num_job', 'debug_stderr']
      
      # ----- informations about actions : module, method
      # always use set_action to define a new action
      self.actions_info = {}
      # store current action
      self.current_action = ''

      # ----- user preferences and defaults
      self.user_vars = {
         'editor' :
            {  'val'  : 'nedit',
               'help' : _(u'editor command') },
         'devel_server_user' :
            {  'val'  : '',
               'help' : _(u'login on the development server') + os.linesep + \
               _(u'# (name/ip address is usually set in /etc/codeaster/asrun)') },
         'remote_copy_protocol' :
            {  'val'  : 'RCP',
               'help' : _(u'remote protocol used to copy files and directories') },
         'remote_shell_protocol' :
            {  'val'  : 'RSH',
               'help' : _(u'remote protocol used for shell commands') },
      }
      self._add_option(*self.user_vars.keys())

      # ----- formats
      self.fmt = {
         'usage'     : u'   - %s :'+os.linesep+'      %s'+os.linesep,
         'verb1'     : u' %-15s = %s',
         'msg+cod'   : os.linesep+u'%-18s %s'+os.linesep,
         'msg_info'  : u'<INFO>%s %s',
         'silent'    : u'      %s %s',
         'title'     : os.linesep+u'-'*80+os.linesep+u'%s %s'+os.linesep,
         'diag'      : os.linesep+u'-'*60+os.linesep+\
                       u'--- DIAGNOSTIC JOB : %s'+\
                       os.linesep+'-'*60+os.linesep,
         'exit'      : os.linesep+u'EXIT_CODE='+'%d'
      }

      # ----- diagnostic
      self.diag = ''

      # ----- list of files/directories to delete on exit
      self.to_delete = []

      # ----- lists of text to print on exit depending on diagnostic
      # this allows to keep important messages when a lot of output is generated
      # and to print them once again at end
      self.print_on_exit = {}

      # ----- do not print timer info on exit by default
      self.timer = None
      self.print_timer = False

      # ----- parser
      p = AsRunParser(run=self,
         usage=self.__doc__ % vars(self),
         version=self.program + ' ' + self.__version__ + os.linesep + copyright)
      self.parser = p
      p.add_option('-v', '--verbose',
         action='store_true', dest='verbose', default=self.options['verbose'],
         help=_(u'increase verbosity'))
      p.add_option('--silent',
         action='store_true', dest='silent', default=self.options['silent'],
         help=_(u'run as silent as possible'))
      p.add_option('-g', '--debug',
         action='store_true', dest='debug', default=False,
         help=_(u'print debugging information'))
      p.add_option('--nodebug_stderr',
         action='store_false', dest='debug_stderr', default=True,
         help=_(u'disable printing of debugging information to stderr'))
      p.add_option('-f', '--force',
         action='store_true', dest='force', default=False,
         help=_(u'force operations which can be cached (download, ' \
                'compilation...)'))
      p.add_option('--num_job',
         action='store', dest='num_job', default=str(os.getpid()),
         help=SUPPRESS_HELP)
      p.add_option('--display',
         action='store', dest='display', default=None,
         help=_(u'value of DISPLAY variable (NOTE : some functions read it from a file)'))
      p.add_option('--rcdir',
         action='store', dest='rcdir', default=None, metavar='SUFFIX',
         help=_(u"""use resources directory $HOME/.astkrc_'SUFFIX'"""))

      # ----- add user's preferences as options
      for k, dV in self.user_vars.items():
         p.add_option('--'+k,
            action='store', dest=k, default=None,
            help=dV['help'].replace(os.linesep, ' ').replace('#', ''))

      # ----- user resource directory
      suff = ''
      mat = [s for s in sys.argv if s.find('--rcdir=') >= 0]
      if len(mat) > 0:
         suff = mat[0].replace('--rcdir=', '')
      if suff == '':
         self.rcdir = '.astkrc'
      else:
         self.rcdir = '.astkrc_%s' % suff

      # ----- read master configuration file and user preferences
      self.user_rc = os.path.join(os.getenv('HOME'), self.rcdir, 'config')
      self.config = {}
      self._read_rc(os.path.join(confdir, 'asrun'), self.config, mcsimp=['noeud',])
      self._read_rc(os.path.join(confdir, 'aster'), self.config, optionnal=True, mcsimp=['vers',])
      self._read_rc(os.path.join(confdir, 'agla'),  self.config, optionnal=True)
      self._init_rc()

      # ----- prepare tmp_user and cache_dir
      # no folder in cache_dir
      tmp_user = os.path.join(self.config['rep_tmp'],
            'astk_'+getpass.getuser())
      self.config['tmp_user'] = tmp_user
      self.config['cache_dir'] = os.path.join(tmp_user, 'cache')
      # safer as os.path.exists for multiple executions
      try:
         os.makedirs(tmp_user)
      except OSError:
         pass
      if not os.access(tmp_user, os.W_OK):
         print3(_(u'no write access to %s') % tmp_user)
         self.Sortie(4)

      # safer as os.path.exists for multiple executions
      try:
         os.mkdir(self.config['cache_dir'])
      except OSError:
         pass

      # ----- check for 'flasheur'
      flash = os.path.join(os.getenv('HOME'), self.config['flash_srv'])
      self.config['flasheur'] = flash
      # safer as os.path.exists for multiple executions
      try:
         os.makedirs(flash)
      except OSError:
         pass
      if not os.access(flash, os.W_OK):
         print3(_(u'no write access to %s') % flash)
         self.Sortie(4)

      # ----- init timer
      self.timer = ASTER_TIMER()
      self.LoadExtensions()
      self.CheckExtensions()


   def ParseArgs(self):
      """
      Call the arguments parser, returns options and arguments.
      """
      opts, args = self.parser.parse_args()
      # ----- all should have been initialized !
      self.__initialized = True
      # exceptions
      os.environ['ASTER_ROOT'] = aster_root
      
      # ----- stocke les options
      for option in self.options_list:
         self.options[option] = getattr(opts, option)

      # ----- local/remote mode : nolocal allows to force remote mode
      if self['nolocal']:
         self.options['local'] = False

      if opts.display != None:
         os.environ['DISPLAY'] = opts.display
         self.DBG(u'set DISPLAY to', opts.display)
      
      # ----- migration
      if self.options['version_dev'] is not None:
         from warnings import warn
         warn('--version_dev=... is deprecated. Use --vers=... instead.', DeprecationWarning, stacklevel=2)
         if self.options['aster_vers'] != self.options['version_dev']:
            warn('--vers=%s will be overwritten by --version_dev into --vers=%s' \
                 % (self.options['aster_vers'], self.options['version_dev']), DeprecationWarning, stacklevel=2)
         self.options['aster_vers'] = self.options['version_dev']

      return opts, args


   def _init_rc(self):
      """
      Read user preferences or create the file the first time
      """
      ficrc = self.user_rc
      # ----- check for ressources directory
      # safer as os.path.exists for multiple executions
      try:
           os.makedirs(os.path.dirname(ficrc))
      except OSError:
         pass
      
      if not os.path.isfile(ficrc):
         # ----- create ressources file
         content = []
         content.append(_(u"# as_run : user preferences file"))
         content.append("")
         content.append(_(u"# You can override here all values of $ASTER_ROOT/etc/codeaster/asrun"))
         content.append("")
         for var in self.user_vars.keys():
            content.append("# %s" % self.user_vars[var]['help'])
            content.append("%s : %s" % (var, self.user_vars[var]['val']))
            content.append("")
         txt = os.linesep.join(content)
         open(ficrc, 'w').write(convert(txt))
      # ----- read ressource file
      self._read_rc(ficrc, self.config)


   def _read_rc(self, ficrc, destdict, optionnal=False, mcsimp=None):
      """
      Read a ressource file and store variables to 'destdict'.
      """
      if os.path.isfile(ficrc):
         read_rcfile(ficrc, destdict, mcsimp=mcsimp)
      elif not optionnal:
         print3(_(u'file not found : %s') % ficrc)
         self.Sortie(4)


   def _add_action(self, action, info):
      """
      Set informations about an action
      Prefer use this method over extend manually the dictionnary.
      """
      if not type(action) in (str, unicode):
         self.Mess(_(u'invalid type (expect %s not %s)') % \
               ('string', type(action)), '<F>_PROGRAM_ERROR')
      if not type(info) is dict:
         self.Mess(_(u'invalid type (expect %s not %s)') % \
               ('dict', type(info)), '<F>_PROGRAM_ERROR')
      if action in self.actions_info.keys():
         self.Mess(_(u'action already defined : %s') % action,
               '<F>_PROGRAM_ERROR')
      else:
         self.actions_info[action] = info


   def _add_option(self, *args):
      """
      Add one or more option which will be available through getitem.
      Prefer use this method over extend manually the list.
      """
      if not type(args) in (list, tuple):
         self.Mess(_(u'invalid type (expect %s not %s)') % \
               ('string', type(args)), '<F>_PROGRAM_ERROR')
      for opt in args:
         if opt in self.options_list:
            self.Mess(_(u'option already defined : %s') % opt,
               '<F>_PROGRAM_ERROR')
         else:
            self.options_list.append(opt)


   def LoadExtensions(self):
      """
      Initialisations des extensions.
      """
      import asrun.maintenance
      import asrun.execute
      import asrun.services
      import asrun.job
      import asrun.rex
      import asrun.bddetudes
      asrun.maintenance.SetParser(self)
      asrun.execute.SetParser(self)
      asrun.services.SetParser(self)
      asrun.job.SetParser(self)
      asrun.rex.SetParser(self)
      asrun.bddetudes.SetParser(self)


   def CheckExtensions(self):
      """
      Initialisations des éventuelles extensions.
      """
      user_extensions    = os.path.join(os.getenv('HOME'), self.rcdir, 'as_run.extensions')
      cwd_extensions     = 'as_run.extensions'
      config = SafeConfigParser()
      l_read = config.read([user_extensions, cwd_extensions])

      l_ext = config.sections()
      for extension in l_ext:
         try:
            filename = os.path.splitext(config.get(extension, 'module'))[0]
            if self['verbose']:
               print3(_(u'Loading extension %s from %s...') % (extension, filename))
            module = __import__(filename, globals(), locals(), ['SetParser'])
            init_function = getattr(module, 'SetParser')
            init_function(self)
            if self['verbose']:
               print3(_(u'Extension %s loaded') % extension)
         except (ImportError, NoOptionError), msg:
            print3(_(u'Extension %s not loaded (reason : %s)') % (extension, msg))


   def SetActions(self, actions_descr, **dico):
      """
      Add new actions to the parser configuration
         actions_descr  : description of each action :
            key   = action name,
            value = {syntax, help, method}
      Optionnal arguments :
         actions_order  : order to display actions in help (default to '' :
            arbitrary order of dico.keys method)
         group_options  : add a option group to list options (default to True)
         group_title    : title of the option group (default to '')
         actions_group_title : list actions names in the title between
            parenthesis (default to True)
         options_descr  : description of each option (default to {}) :
            key   = option name,
            value = {args, kwargs} passed to add_option parser method
         stored_options : list of options which should be available using
            getitem (default to all "dest" from options_descr)
      """
      # ----- default values
      # actions_order
      if dico.has_key('actions_order'):
         lacts = dico['actions_order']
      else:
         lacts = actions_descr.keys()
         lacts.sort()
      # title
      if dico.has_key('group_title'):
         title = dico['group_title']+' ('
      else:
         title = ''
      # actions_group_title
      if not dico.has_key('actions_group_title'):
         dico['actions_group_title'] = True
      #  group_options
      if not dico.has_key('group_options'):
         dico['group_options'] = True
      # options_descr
      if not dico.has_key('options_descr'):
         dico['options_descr'] = {}
      # stored_options
      if not dico.has_key('stored_options'):
         dico['stored_options'] = [d.values()[1]['dest'] \
               for d in dico['options_descr'].values()]
      # ----- check types
      # dict
      for obj in (actions_descr, dico['options_descr']):
         if not type(obj) is dict:
            self.Mess(_(u'invalid type (expect %s not %s)') % \
                  ('dict', type(obj)), '<F>_PROGRAM_ERROR')
      # list
      for obj in (lacts, dico['stored_options']):
         if not type(obj) is list:
            self.Mess(_(u'invalid type (expect %s not %s)') % \
                  ('list', type(obj)), '<F>_PROGRAM_ERROR')
      prem = True
      for act in lacts:
         descr = actions_descr[act]
         # ----- actions_info
         self._add_action(act, { 'method' : descr['method'] })
         # ----- parser
         if descr['help'] != SUPPRESS_HELP:
            new_usage = self.parser.get_usage()+ \
               (self.fmt['usage'] % \
                  (descr['help'].capitalize(), u'as_run --'+act+' '+descr['syntax']))
            self.parser.set_usage(new_usage)
         self.parser.add_option('--'+act,
               action='store_const_once', dest='action', const=act,
               help=SUPPRESS_HELP)
         if not prem:
            title = title+', '
         prem = False
         title = title+('%r' % act)
      title = title+') '
      if not dico['actions_group_title']:
         title = dico['group_title']
      # ----- specific options
      if dico['group_options']:
         group = OptionGroup(self.parser, title=title)
         self.parser.add_option_group(group)
      else:
         group = self.parser
      for opt, descr in dico['options_descr'].items():
         group.add_option(*descr['args'], **descr['kwargs'])
      # ----- add options to the options list to store
      self._add_option(*dico['stored_options'])


   def PrintConfig(self):
      """
      Imprime la liste et la valeur des paramètres
      """
      self.parser.print_version()
      print3(os.linesep+_(u'Parameters (config attribute)')+' :')
      for key, val in self.config.items():
         print3(self.fmt['verb1'] % (key, val))
      print3(os.linesep+_(u'Options (options attribute)')+' :')
      for key, val in self.options.items():
         print3(self.fmt['verb1'] % (key, val))


   def ToDelete(self, path):
      """
      Add 'path' to to_delete list.
      """
      if not path in self.to_delete:
         self.to_delete.append(path)


   def DoNotDelete(self, path):
      """
      Remove 'path' from to_delete list.
      """
      if path in self.to_delete:
         self.to_delete.remove(path)


   def _clear_tmp(self):
      """
      Empty cache directory if necessary (remove files for which last access
      is older than 'deltat' seconds), and delete all directories
      from 'self.to_delete'.
      """
      # to avoid to delete current directory (it may freeze some systems...)
      os.chdir(os.environ.get('HOME', '/'))
      ct = time.time()
      deltat = 24*3600
      # ----- config is not set during __init__
      if self.__initialized:
         if self['verbose']:
            print3(_(u'Clear cache directory')+' '+self['cache_dir']+'...')
         # ----- avoid deleting such folders : /, /usr or /home !!!
         if re.search('^/.+/.*', self['cache_dir']):
            for root, dirs, files in os.walk(self['cache_dir'], topdown=False):
               for name in files:
                  if ct-os.stat(os.path.join(root, name)).st_atime > deltat:
                     os.remove(os.path.join(root, name))
                  for name in dirs:
                     try:
                        os.rmdir(os.path.join(root, name))
                     except OSError:
                        pass
         # ----- delete each directory in list_dirs
         if self['verbose']:
            print3(_(u'Clear temporary files/directories'))
         # if a Delete system function has been added
         if hasattr(self, 'Delete'):
            to_del = [d for d in remove_subdirs(self.to_delete) if d.startswith('/')]
            for d in to_del:
               getattr(self, 'Delete')(d)


   def __getitem__(self, key):
      """
      Méthode pour accéder facilement aux paramètres de configuration ou à la
      valeur d'une option.
      Les options surchargent les paramètres de configuration.
      """
      if hasattr(self, '__'+key+'__'):
         return getattr(self, '__'+key+'__')
      elif self.options.get(key) != None:
         return self.options[key]
      elif self.config.get(key) != None:
         return self.config[key]
      elif self.__initialized:
         self.Mess(_(u"""'%s' does not exist in config or options.
 Perhaps it must be defined in %s.""") % (key, self.user_rc), \
               '<F>_PROGRAM_ERROR')


   def get(self, key, default=None):
      """
      Semblable à getitem avec la possibilité de retourner une valeur par défaut
      si la valeur n'est pas trouvée.
      """
      if hasattr(self, '__'+key+'__'):
         return getattr(self, '__'+key+'__')
      elif self.options.get(key) != None:
         return self.options[key]
      elif self.config.get(key) != None:
         return self.config[key]
      else:
         return default


   def get_pid(self, num=None):
      """
      Return a jobid based on self['num_job'].
      Counter is incremented at each call.
      """
      if num is None:
         self._val_pid += 1
         num = self._val_pid
      assert type(num) in (int, long), _(u'get_pid: integer argument required!')
      return '%04d-%s' % (num, self['num_job'])


   def PostConf(self):
      """
      Ajuste la configuration dynamiquement (car dépendance au contexte et
      pas seulement au fichier 'config').
      Paramètres déclenchants :
         serv_as_node, only_serv_as_node
      """
      if not self.__initialized or not hasattr(self, 'GetHostName'):
         print3(_(u'Warning : ASTER_RUN object not ready for dynamic adjustement !'))
         return
      
      # add localhost as a compute node
      if self.config.get('serv_as_node', False):
         l_node = self.config.get('noeud', '').split()
         localhost = self.GetHostName()
         localhost_short = localhost.split('.')[0]
         if not localhost in l_node and not localhost_short in l_node:
            l_node.insert(0, localhost_short)
         if self.config.get('only_serv_as_node', False):
            l_node = [localhost_short,]
         self.config['noeud'] = ' '.join(l_node)

      # trace installation parameter
      self.DBG(u'installation dirs (aster_root, confdir, localedir, LANG)',
               aster_root, confdir, localedir, os.environ.get('LANG', 'undefined'))


   def GetGrav(self, cod):
      """
      Return the severity behind 'cod' as a number to allow comparison
      """
      dgrav = { '?' : -9, '_' : -9, 'OK' : 0, 'A' : 1,
                'NOOK' : 2, 'NO_TEST_RESU' : -1,
                'S' : 4,
                'E' : 6, 'NO_RESU_FILE' : 6,
                'F' : 10 }
      g = dgrav['?']
      mat = re.search('<(.)>', cod)
      if cod in dgrav.keys():
         g = dgrav[cod]
      elif re.search('NOOK', cod):
         g = dgrav['NOOK']
      elif mat != None:
         try:
            g = dgrav[mat.group(1)]
         except KeyError:
            pass
      return g


   def Mess(self, msg, cod='', store=False):
      """
      Print a message sur stdout,
      'cod' is an error code (format "<.>_...")
         <E> : continue,
         <F> : stop,
         <A> : alarm, continue,
         '' or INFO : for an info,
         SILENT : like an info without <INFO> string,
         TITLE : add a separator.
      If cod='<F>' (without a description), if exists last <E> error is
      transformed into <F>, else <F>_ABNORMAL_ABORT.
      If store is True, 'msg' is stored in print_on_exit dictionnary.
      """
      # ----- gravite
      g0 = self.GetGrav(self.diag)
      g1 = self.GetGrav(cod)
      coderr = cod
      if cod == '<F>':
         if g0 < self.GetGrav('<E>'):
            coderr =  '<F>_ABNORMAL_ABORT'
         else:
            coderr = self.diag.replace('<E>', '<F>')
      if g1 >= self.GetGrav('<A>'):
         self.DBG(u'Warning or error raised :', '%s %s' % (cod, msg),
                  print_traceback=True)

      # ----- message
      if cod == '' or cod == 'INFO':
         fmt = self.fmt['msg_info']
         coderr = ''
      elif cod == 'SILENT':
         fmt = self.fmt['silent']
         coderr = ''
      elif cod == 'TITLE':
         fmt = self.fmt['title']
         coderr = ''
      else:
         fmt = self.fmt['msg+cod']
         # unknown flag
         if g1 == -9:
            coderr = '<I> '+coderr
      print3(fmt % (coderr, msg))
      sys.stdout.flush()
      # ----- store in print_on_exit
      if store:
         k = '?'
         msg2 = msg
         mat = re.search('<(.)>', cod)
         if mat != None and mat.group(1) in ('A', 'S', 'E', 'F'):
            k = mat.group(1)
            msg2 = self.fmt['msg+cod'] % (coderr, msg)
         elif cod in ('OK', 'NOOK'):
            k = cod
         if not self.print_on_exit.has_key(k):
            self.print_on_exit[k] = [msg2]
         else:
            self.print_on_exit[k].append(msg2)

      # ----- diagnostic le plus défavorable
      if g1 > g0:
         self.diag = coderr
         if g1 == self.GetGrav('<F>'):
            self.Sortie(4)


   def Sortie(self, exit_code):
      """
      Exit by printing diagnostic if defined or exit code if not null
      """
      # print important messages
      titles = {
         '?'      : _(u'Important messages previously printed :')+os.linesep,
         'OK'     : _(u'Successful messages previously printed:')+os.linesep,
         'NOOK'   : _(u'NOOK messages previously printed:')+os.linesep,
         'A'      : _(u'<A> Alarms previously raised :')+os.linesep,
         'S'      : _(u'<S> errors previously raised :')+os.linesep,
         'E'      : _(u'<E> errors previously raised :')+os.linesep,
         'F'      : _(u'<F> errors previously raised :')+os.linesep,
      }
      for k, tit in titles.items():
         lmsg = self.print_on_exit.get(k, [])
         if len(lmsg)>0:
            print3(self.fmt['title'] % (tit, os.linesep.join(lmsg)))
      
      # stop all timers
      if self.print_timer and hasattr(self, 'timer'):
         print3(os.linesep, self.timer)
      if self.print_timer: # related to non silent actions
         print3(self.program + ' ' + self.__version__)

      if self.diag:
         # change <E> to <F>
         self.diag = self.diag.replace('<E>', '<F>')
         print3(self.fmt['diag'] % self.diag)
      if self.PrintExitCode or exit_code != 0:
         print3(self.fmt['exit'] % exit_code)
      self._clear_tmp()
      # helps to locate error in PROGRAM_ERROR case or in verbose mode
      if self.diag == '<F>_PROGRAM_ERROR' \
            or not self.ExitOnFatalError \
            or (exit_code != 0 and (self['debug'] or self['verbose'])):
         raise RunAsterError, exit_code
      else:
         sys.exit(exit_code)


   def CheckOK(self, tole='<S>'):
      """
      Exit one error more severe than 'tole' occurred.
      """
      if self.GetGrav(self.diag) >= self.GetGrav(tole):
         self.Sortie(4)


   def ReinitDiag(self, diag='?'):
      """
      Reinitialize diagnostic (for example if a global diagnostic is stored)
      """
      self.diag = diag


   def _printDBG(self, *args, **kargs):
      """
      Print debug information to stderr.
      """
      print_traceback = kargs.get('print_traceback', True)
      file            = kargs.get('file', sys.stderr)
      stack_id        = -3 - kargs.get('stack_id', 0)      # -1 : _printDBG, -2 : DBG ou ASSERT
      all             = kargs.get('all', False)
      try:
         self.__dbg_count += 1
         form = """
>>> %(num)09d  %(orig)s
%(content)s
%(traceback)s"""
         formTB = """
Traceback:
%s
"""
         stack = traceback.format_stack(limit=10)
         try:
            ls = []
            for a in args:
               if type(a) in (str, unicode):
                  ls.append(a)
               else:
                  ls.append(pprint.pformat(a))
            content = os.linesep.join(ls)
            if not all and len(content) > 800:
               content = content[:800] + os.linesep + '[...]'
         except None:
            content = str(args)
         dinfo = {
            'num'       : self.__dbg_count,
            'orig'      : stack[stack_id],
            'content'   : content,
            'traceback' : '',
         }
         try:
            mat = re.search('File [\'\"]*(.*?)[\'\"]*, *line ([0-9]+), *in (.*)', dinfo['orig'])
            dinfo['orig'] = '[%s@%s:%s]' % (mat.group(3), mat.group(1), mat.group(2))
         except None:
            pass
         if print_traceback:
            dinfo['traceback'] = formTB % (''.join(stack))
         print3(form % dinfo, file=file)
         file.flush()
      except None:
         pass


   def DBG(self, *args, **kargs):
      """
      Print debug information to stderr.
      """
      print_traceback = kargs.get('print_traceback', False)
      file            = kargs.get('file', sys.stderr)
      stack_id        = kargs.get('stack_id', 0)
      all             = kargs.get('all', False)
      if ((not file.isatty()) or self['debug'] or self['verbose']) \
            and (self['debug_stderr'] or file != sys.stderr):
         self._printDBG(print_traceback=print_traceback, file=file, stack_id=stack_id, all=all, *args)

   def ASSERT(self, condition):
      """
      Print errors to stdout and stderr.
      """
      if not condition:
         for f in (sys.stdout, sys.stderr):
            self._printDBG(u'Assertion failed', print_traceback=True, file=f, stack_id=0)


def AsRunFactory(*args, **kargs):
   """
   Initialization of an ASTER_RUN object
   """
   #XXX pre 1.8 backward compatibility : astk_serv_root argument removed
   if len(args) != 0:
      from warnings import warn
      warn('AsRunFactory : astk_serv_root argument is deprecated', DeprecationWarning, stacklevel=2)

   from asrun.system import ASTER_SYSTEM
      
   run = ASTER_RUN()
   run.options['debug']   = False
   run.options['debug_stderr'] = True
   run.options['verbose'] = False
   run.options['force']   = False
   run.options['num_job'] = str(os.getpid())
   for opt, value in kargs.items():
      if run.options.has_key(opt):
         run.options[opt] = value
      elif run.config.has_key(opt):
         run.config[opt] = value
   run.system = ASTER_SYSTEM(run, **kargs)
   run.print_timer = False

   return run


