# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Tools for developpers :
   - search messages
"""


import sys
import os
import re
import pprint
import cPickle
from glob import glob

from asrun.installation import aster_root
from asrun.i18n         import _
from asrun.mystring     import print3
from asrun.config       import ASTER_CONFIG
from asrun.build        import ASTER_BUILD, unigest2dict
from asrun.progress     import Progress


class MessageError(Exception):
   pass

class MESSAGE_MANAGER(object):
   """
   Classe pour récupérer des informations sur le catalogue de messages.
   """
   def __init__(self, repref, fort=('bibfor', 'bibf90'), pyt='bibpyt',
                      cache_dict=None, verbose=True, force=False,
                      ignore_comments=True, debug=False,
                      surch_fort=[], surch_pyt=[], unig=None):
      """
      Initialisations
      """
      if type(fort) not in (list, tuple):
         fort = [fort,]
      if type(surch_fort) not in (list, tuple):
         surch_fort = [surch_fort,]
      if type(surch_pyt) not in (list, tuple):
         surch_pyt = [surch_pyt,]
      self.repref          = repref
      self.fort            = [os.path.join(repref, rep) for rep in fort]
      self.pyt             = os.path.join(repref, pyt)
      self.surch_fort      = [os.path.abspath(d) for d in surch_fort]
      self.surch_pyt       = [os.path.abspath(d) for d in surch_pyt]
      self.verbose         = verbose
      self.debug           = debug
      self.force           = force
      self.message_dir     = 'Messages'
      self.cache_dict      = cache_dict or os.path.join('/tmp', 'messages_dicts.pick')
      self.ignore_comments = ignore_comments
      self.wrk_fort        = os.path.join(os.getcwd(), 'F')
      self.wrk_pyt         = os.path.join(os.getcwd(), 'PY')
      
      self.unig            = unig
      # init ASTER_CONFIG, ASTER_BUILD objects
      fconf = os.path.join(self.repref, 'config.txt')
      self.conf_obj  = ASTER_CONFIG(fconf, run=None)
      self.build_obj = ASTER_BUILD(run=None, conf=self.conf_obj)
      if unig:
         self.unig = unigest2dict(unig, self.conf_obj)
      
      if self.verbose:
         print3('Repertoires :\n   REF=%s\n   FORTRAN=%s\n   PYTHON=%s' % (self.repref, self.fort, self.pyt))

      self.l_cata            = self._get_list_cata()
      self.l_src, self.l_pyt = self._get_list_src()
#      self._build_regexp()
      self._build_dict()
      self.print_stats()


   def _filename_cata(self, cata):
      """
      Retourne le nom du fichier associé à un catalogue."""
      name = os.path.join(self.pyt, self.message_dir, cata + '.py')
      return name


   def _id2filename(self, msgid):
      """
      Retourne modelisa5 à partir de MODELISA5_46"""
      mat = re.search('([A-Z]+[0-9]*)_([0-9]+)', msgid)
      if mat == None:
         raise MessageError, 'invalid message id : %s' % msgid
      cata, num = mat.groups()
      return self._filename_cata(cata.lower()), int(num)


   def _filename2id(self, fcata, num):
      """
      Retourne MODELISA5_46 à partir de modelisa5"""
      return '%s_%d' % (os.path.splitext(os.path.basename(fcata))[0].upper(), num)


   def makedirs(self):
      """Crée les répertoires pour les fichiers modifiés
      """
      for d in (self.wrk_fort, self.wrk_pyt):
         if not os.path.isdir(d):
            os.makedirs(d)


   def _get_list_cata(self):
      """
      Retourne la liste des catalogues de messages.
      """
      re_mess = re.compile('#@.*Messages')
      s_cata = set(glob(os.path.join(self.pyt, self.message_dir, '*.py')))
      s_suppr = set()
      l_surch = []
      # ajouter la surcharge et retirer le catalogue d'origine
      for dsrc in self.surch_pyt:
         for f in glob(os.path.join(dsrc, '*.py')) + glob(os.path.join(dsrc, '*', '*.py')):
            txt = open(f, 'r').read()
            if re_mess.search(txt):
               l_surch.append(os.path.abspath(f))
               fsup = os.path.join(self.pyt, self.message_dir, os.path.basename(f))
               if os.path.exists(fsup):
                  s_suppr.add(fsup)
      s_cata.update(l_surch)
      if len(l_surch) > 0:
         print3('%5d catalogues en surcharge : ' % len(l_surch))
         print3(os.linesep.join(l_surch))

      s_cata = set([os.path.abspath(f) for f in s_cata \
            if not os.path.basename(f) in ['__init__.py', 'context_info.py', 'fermetur.py']])
      l_suppr = []
      if self.unig and len(self.unig['py']) > 0:
         l_suppr = [os.path.abspath(os.path.join(self.repref, f)) \
                                 for f in self.unig['py'] if f.find(self.message_dir) > -1]
         s_suppr.update(l_suppr)
      l_suppr = list(s_suppr)
      l_suppr.sort()
      if len(l_suppr) > 0:
         print3('%5d catalogue(s) supprime(s) :' % len(l_suppr))
         if self.verbose: print3(os.linesep.join(l_suppr))
      l_cata = list(s_cata.difference(s_suppr))
      l_cata.sort()
      return l_cata


   def check_cata(self):
      """
      Vérifie les catalogues.
      """
      print3('%5d catalogues dans %s + %s' \
            % (len(self.l_cata), self.pyt, self.surch_pyt))
      all_msg = []
      for f in self.l_cata:
         try:
            cata = CATA(f)
            all_msg.extend([self._filename2id(f, i) for i in cata])
            if self.verbose:
               print3(cata)
         except MessageError, msg:
            print3(msg)
            self.l_cata.remove(f)
      all_msg = set(all_msg)
      
      # messages jamais appelés
      used = set(self.d_msg_call.keys())
      unused =  list(all_msg.difference(used))
      unused.sort()
      
      union = used.union(all_msg)
      not_found = list(used.difference(all_msg))
      not_found.sort()
      print3('%6d messages dans les catalogues' % len(all_msg))
      print3('dont %6d messages appeles presents dans les catalogues' % (len(used) - len(not_found)))
      print3('  et %6d messages inutilises' % len(unused))
      print3('   + %6d messages appeles absents des catalogues' % len(not_found))
      #print '%6d messages total (union pour verif)' % len(union)

      return unused, not_found


   def _build_regexp(self):
      """
      Construit la liste des expressions régulières pour la recherche des messages.
      """
      l_regexp = []
      for cata in self.l_cata:
         cata = os.path.splitext(os.path.basename(cata))[0]
         l_regexp.append('%s_[0-9]+' % cata.upper())
      self.regexp = re.compile('|'.join(l_regexp))


   def _get_list_src(self):
      """
      Retourne la liste des routines fortran et python.
      """
      l_f   = self._get_list_fort()
      l_pyt = self._get_list_python()
      if self.verbose:
         print3('%5d routines fortran dans %s + %s' % (len(l_f), self.fort, self.surch_fort))
         print3('%5d modules python   dans %s + %s' % (len(l_pyt), self.pyt, self.surch_pyt))
      return l_f, l_pyt


   def _get_list_fort(self):
      """
      Retourne la liste des fichiers sources fortran/fortran90."""
      s_f = set()
      for dsrc in self.fort:
         s_f.update(glob(os.path.join(dsrc, '*.f')))
         s_f.update(glob(os.path.join(dsrc, '*', '*.f')))
         s_f.update(glob(os.path.join(dsrc, '*.F')))
         s_f.update(glob(os.path.join(dsrc, '*', '*.F')))
      d_f = {}
      for f in s_f:
         assert d_f.get(os.path.basename(f)) is None, 'ERROR : %s  (old : %s)' % (f, d_f[os.path.basename(f)])
         d_f[os.path.basename(f)] = f
      # surcharge
      s_surch = set()
      for dsrc in self.surch_fort:
         s_surch.update(glob(os.path.join(dsrc, '*.f')))
         s_surch.update(glob(os.path.join(dsrc, '*', '*.f')))
         s_surch.update(glob(os.path.join(dsrc, '*.F')))
         s_surch.update(glob(os.path.join(dsrc, '*', '*.F')))
      if len(s_surch) > 0:
         l_surch = list(s_surch)
         l_surch.sort()
         print3('%5d sources en surcharge : ' % len(l_surch))
         print3(os.linesep.join(l_surch))
      # retirer des sources originaux ceux de la surcharge...
      s_suppr = set()
      for f in s_surch:
         fexist = d_f.get(os.path.basename(f))
         if fexist:
            s_suppr.add(fexist)
            if self.verbose: print3('suppression :', fexist)
      # ... et ceux de l'unigest
      if self.unig:
         iunig = 0
         for f in self.unig['f'] + self.unig['f90']:
            iunig += 1
            s_suppr.add(d_f.get(os.path.basename(f), ''))
            if self.verbose: print3('suppression :', f)
         if iunig > 0:
            print3('%5d source(s) supprime(s).' % iunig)
      
      s_f.difference_update(s_suppr)
      s_f.update(s_surch)
      l_f = [os.path.abspath(f) for f in s_f]
      l_f.sort()
      return l_f


   def _get_list_python(self):
      """
      Retourne la liste des fichiers python
      """
      s_pyt = set()
      s_pyt.update(glob(os.path.join(self.pyt, '*.py')))
      s_pyt.update(glob(os.path.join(self.pyt, '*', '*.py')))
      d_py = {}
      for f in s_pyt:
         key = self.build_obj.GetCModif('py', f)
         assert d_py.get(key) is None, 'ERROR : %s  (old : %s)' % (key, d_py[key])
         d_py[key] = f
      # surcharge
      s_surch = set()
      for dsrc in self.surch_pyt:
         s_surch.update(glob(os.path.join(dsrc, '*.py')))
         s_surch.update(glob(os.path.join(dsrc, '*', '*.py')))
      if len(s_surch) > 0:
         l_surch = list(s_surch)
         l_surch.sort()
         print3('%5d module(s) python en surcharge : ' % len(l_surch))
         print3(os.linesep.join(l_surch))
      # retirer des sources originaux ceux de la surcharge...
      s_suppr = set()
      for f in s_surch:
         key = self.build_obj.GetCModif('py', f)
         fexist = d_py.get(key)
         if fexist:
            s_suppr.add(fexist)
            if self.verbose: print3('suppression :', fexist)
      # ... et ceux de l'unigest
      if self.unig:
         iunig = 0
         for f in self.unig['py']:
            iunig += 1
            fabs = os.path.abspath(os.path.join(self.repref, f))
            key = self.build_obj.GetCModif('py', fabs)
            s_suppr.add(d_py.get(key, ''))
            if self.verbose: print3('suppression :', fabs)
         if iunig > 0:
            print3('%5d module(s) python supprime(s).' % iunig)

      s_pyt.difference_update(s_suppr)
      s_pyt.update(s_surch)
      l_pyt = [os.path.abspath(f) for f in s_pyt]
      l_pyt.sort()
      return l_pyt


   def search_message(self, fich):
      """
      Retourne les messages utilisés dans 'fich'.
      """
      try:
         txt = open(fich, 'r').read()
      except IOError, msg:
         print3(_(u'Error with file %s : %s') % (fich, msg))
         return []
      ext = os.path.splitext(fich)[-1]
      txt = clean_source(txt, ext, ignore_comments=self.ignore_comments, wrap=True)
      
      if os.path.splitext(fich)[-1] != '.py':
         expr  = 'CALL\s+(U2ME|UTEXC).*?,[ \'\"]+(.*?)[\'\" ]+[,\)]+'
      else:
         expr = '(UTMESS|GetText)\s*\(.*?,[ \'\"]+(.*?)[\'\" ]+[,\)]+'
      all_msg = re.findall(expr, txt)
      l_msg = []
      for found in all_msg:
         sub, msg = found
         if msg.startswith('FERMETUR_'):
            pass
         elif re.search('^[A-Z0-9]+_[0-9]+$', msg) is None:
            print3('Invalid', msg, fich)
         else:
            l_msg.append(msg)
#      l_msg = self.regexp.findall(txt)
      
      # verif
      l_res = []
      for msg in l_msg:
         spl = msg.split('_')
         assert len(spl) == 2, 'ERROR invalid : %s' % msg
         msg = '%s_%d' % (spl[0], int(spl[1]))
         l_res.append(msg)
      
      return l_res


   def _build_dict(self):
      """
      Construit les dictionnaires :
         fichier source : liste des messsages appelés
         message : liste des fichiers appelant ce message
      """
      # est-ce dans le cache ?
      if not self.force and os.path.isfile(self.cache_dict) \
                        and os.stat(self.cache_dict).st_size > 0:
         if self.verbose:
            print3('Load dicts from cache (%s)...' % self.cache_dict)
         pick = open(self.cache_dict, 'r')
         self.d_msg_used = cPickle.load(pick)
         self.d_msg_call = cPickle.load(pick)
         pick.close()

      else:   
         self.d_msg_used = {}
         self.d_msg_call = {}
         l_all = self.l_src + self.l_pyt
         if self.verbose:
            p = Progress(maxi=len(l_all), format='%5.1f %%',
                        msg='Analyse des sources... ')
         for i, f in enumerate(l_all):
            if self.verbose:
               p.Update(i)
#             key = f.replace(self.repref, '')
            key = re.sub('^%s/*', '', f)
            lm = self.search_message(f)
            if len(lm) > 0:
               self.d_msg_used[key] = lm
               for msg in lm:
                  self.d_msg_call[msg] = self.d_msg_call.get(msg, []) + [key,]
         
         if self.verbose:
            p.End()
            #pprint.pprint(self.d_msg_used)
         
         pick = open(self.cache_dict, 'w')
         cPickle.dump(self.d_msg_used, pick)
         cPickle.dump(self.d_msg_call, pick)
         pick.close()


   def print_stats(self):
      """
      Affiche les stats sur les données lues/construites
      """
      if self.verbose:
         print3('%6d messages appelés dans les sources' % len(self.d_msg_call.keys()))
         print3('%6d fichiers sources appellent le catalogue de messages' % len(self.d_msg_used.keys()))


   def move(self, oldid, dest, reuse_hole=True):
      """
      Déplace le message 'oldid' dans le catalogue 'dest'.
      """
      if self.verbose:
         print3('--- moving "%s" into "%s"' % (oldid, dest))
      # catalogue objects
      old_f, old_num = self._id2filename(oldid)
      new_f = self._filename_cata(dest.lower())
      
      # have these catalogues been already changed ?
      fmod = os.path.join(self.wrk_pyt, os.path.basename(old_f))
      if os.path.isfile(fmod):
         print3('from %s' % fmod)
         old_f = fmod
      fmod = os.path.join(self.wrk_pyt, os.path.basename(new_f))
      if os.path.isfile(fmod):
         print3('from %s' % fmod)
         new_f = fmod
         
      old_cata = CATA(old_f)
      new_cata = CATA(new_f)
      if self.verbose:
         print3(old_cata)
         print3(new_cata)
      new_num = new_cata.get_new_id(reuse_hole)
      if new_num < 0:
         raise MessageError,  'no message id available in %s' % new_f
      newid = self._filename2id(new_f, new_num)
      
      # check message existence
      if old_cata[old_num] == None:
         raise MessageError, 'message %s inexistant !' % oldid
      
      new_cata[new_num] = old_cata[old_num]
      del old_cata[old_num]

      # write modified catalogues
      self.makedirs()
      fout = os.path.join(self.wrk_pyt, os.path.basename(old_f))
      content = old_cata.content()
      open(fout, 'w').write(content)
      fout = os.path.join(self.wrk_pyt, os.path.basename(new_f))
      content = new_cata.content()
      open(fout, 'w').write(content)
      print3('Nouveau fichier : %s' % fout)
      
      # modify source using 'oldid' message
      l_src = self.d_msg_call.get(oldid, [])
      for f in l_src:
         ext = os.path.splitext(f)[-1]
         rdest = self.wrk_fort
         if ext == '.py':
            rdest = self.wrk_pyt
         # already changed ?
         fmod = os.path.join(rdest, os.path.basename(f))
         if os.path.isfile(fmod):
            print3('from %s' % fmod)
            f = fmod
         txt = open(os.path.join(self.repref, f), 'r').read()
         new = re.sub('%s([\'\"]+)' % oldid, newid + r'\1', txt)
         fout = os.path.join(rdest, os.path.basename(f))
         open(fout, 'w').write(new)
         print3('Nouveau fichier : %s' % fout)


   def who_use(self, msg):
      """
      Qui utilise le message 'msg' ?
      """
      return tuple(self.d_msg_call.get(msg.upper(), []))


   def get_key(self, sub):
      """
      Retourne la clé dans d_msg_used correspondant à 'sub'.
      Seule la première est retournée si jamais il y en avait plusieurs.
      """
      l_allsub = self.d_msg_used.keys()
      l_sub = [f for f in l_allsub if f.split(os.sep)[-1] == sub]
      if len(l_sub) > 1:
         print3('Plusieurs routines correspondent : %s' % ', '.join(l_sub))
         print3('On utilise la première valeur.')
      if len(l_sub) == 0:
         result = None
      else:
         result = l_sub[0]
      return result


   def which_msg(self, sub):
      """
      Quels sont les messages utilisés par la routine 'sub' ?
      """
      key = self.get_key(sub)
      return tuple(self.d_msg_used.get(key, []))


template_cata_header = """# -*- coding: utf-8 -*-
#            CONFIGURATION MANAGEMENT OF EDF VERSION
# ======================================================================
# COPYRIGHT (C) 1991 - 2007  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY  
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY  
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR     
# (AT YOUR OPTION) ANY LATER VERSION.                                                  
#                                                                       
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT   
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF            
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU      
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.                              
#                                                                       
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE     
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,         
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.        
# ======================================================================

def _(x) : return x

cata_msg = {"""

template_cata_footer = """
}
"""

template_cata_msg = '''
%(msgid)s : _(u"""%(text)s"""),'''


class CATA:
   """
   Classe représentant un catalogue de messages.
   Méthodes attendues :
      - nombre de messages,
      - indice du dernier messages,
      - indice libre,
      - ajouter/supprimer un message,
      - ...
   """
   def __init__(self, fcata):
      """
      Initialisation
      """
      self.fcata = fcata
      self.cata_msg = {}
      try:
         d = {}
         execfile(fcata, d)
         self.cata_msg = d['cata_msg']
      except Exception, msg:
         print3('-'*80+'\n', msg, '\n'+'-'*80)
         raise MessageError, 'ERROR with : %s' % self.fcata


   def get_nb_msg(self):
      """
      Nombre de messages.
      """
      return len(self.cata_msg)


   def get_last_id(self):
      """
      Indice du dernier message.
      """
      return max(self.cata_msg.keys() or [0,])


   def get_new_id(self, reuse_hole=True):
      """
      Indice libre. Si 'end', on prend systématiquement à la fin,
      sinon on cherche aussi dans les trous.
      """
      if not reuse_hole:
         start = self.get_last_id()
      else:
         start = 1
      all = set(range(start, 100))
      free = all.difference(self.cata_msg.keys())
      if len(free) == 0:
         new = -1
      else:
         new = min(free)
      return new


   def __getitem__(self, key):
      """
      Retourne le contenu du message ou None.
      """
      return self.cata_msg.get(key, None)


   def __setitem__(self, key, msg):
      """
      Ajoute le message 'msg' à l'indice 'key'.
      """
      if self[key] != None:
         raise MessageError, 'message %s already exists !' % key
      self.cata_msg[key] = msg


   def __delitem__(self, key):
      """
      Supprime le message d'indice 'key'.
      """
      if self[key] == None:
         raise MessageError, 'message %s does not exist !' % key
      del self.cata_msg[key]


   def __repr__(self):
      """
      Résumé du catalogue.
      """
      return '%3d messages (last=%3d, next=%3d) dans %s' % \
            (self.get_nb_msg(), self.get_last_id(), self.get_new_id(), self.fcata)


   def __iter__(self):
      """
      Itération sur les id des messages.
      """
      return iter(self.cata_msg)


   def get_cmodif(self):
      """
      Retourne la carte MODIF.
      """
      cmodif = None
      fobj = open(self.fcata, 'r')
      for line in fobj:
         if re.search('#@ +MODIF|#@ +AJOUT', line):
            cmodif = line.replace(os.linesep, '')
            break
      fobj.close()
      if cmodif == None:
         raise MessageError, 'invalid header "#@ MODIF/AJOUT..." in %s' % self.fcata
      return cmodif


   def content(self):
      """
      Contenu du catalogue "reconstruit".
      """
      txt = [self.get_cmodif(),]
      txt.append(template_cata_header)
      l_ind = self.cata_msg.keys()
      l_ind.sort()
      for msgid in l_ind:
         txt.append(template_cata_msg % {'msgid' : msgid, 'text' : self.cata_msg[msgid]})
      txt.append(template_cata_footer)
      return os.linesep.join(txt)


def clean_source(content, ext, ignore_comments, wrap):
   """
   Nettoie un fichier source.
   """
   if ignore_comments:
      assert ext in ('.f', '.F', '.py'), 'unknown type : %s' % str(ext)
      if ext in ('.f', '.F'):
         reg_ign = re.compile('(^[A-Z!#].*$)', re.MULTILINE)
      elif ext == '.py':
         reg_ign = re.compile('(#.*$)', re.MULTILINE)
      content = reg_ign.sub('', content).expandtabs()
      if wrap and ext in ('.f', '.F'):
         content = ''.join([lin[6:] for lin in content.splitlines()])
   return content


def GetMessageInfo(run, *args):
   """
   Return info about Code_Aster messages.
   """
   if not run.get('aster_vers'):
      run.parser.error(_(u"You must define 'default_vers' in 'aster' configuration file or use '--vers' option."))
   REPREF = os.path.join(aster_root, run['aster_vers'])
   if not run.get('config'):
      fconf = os.path.join(REPREF, 'config.txt')
   else:
      fconf = os.path.abspath(run['config'])
   conf = ASTER_CONFIG(fconf, run)

   if not run['local']:
      run.Mess(_(u'This operation only works on local source files. "local" mode forced'), '<A>_ALARM')

   bibfor = [conf['SRCFOR'][0],]
   if conf['SRCF90'][0] != '':
      bibfor.append(conf['SRCF90'][0])
   else:
      run.Mess(_(u'Fortran 90 source files are not used on this machine (see %s)') \
            % conf.filename, '<A>_ALARM')
   bibpyt = conf['SRCPY'][0]

   args = list(args)
   named_actions = ('check', 'move',)
   action = None
   
   if len(args) > 0 and args[0] in named_actions:
      action = args.pop(0)
   elif len(args) == 0:
      run.Mess(_(u'You must choose an operation from %s or give a subroutine name or a message ID') \
            % repr(named_actions), '<F>_INVALID_ARGUMENT')
   
   # surcharge
   surch_fort = run.get('surch_fort', [])
   if surch_fort:
      surch_fort = surch_fort.split(',')
   surch_pyt = run.get('surch_pyt', [])
   if surch_pyt:
      surch_pyt = surch_pyt.split(',')
   
   msgman = MESSAGE_MANAGER(repref=REPREF, fort=bibfor, pyt=bibpyt, 
                            cache_dict=os.path.join(run['cache_dir'], 'messages_dicts.pick'),
                            force=run['force'], verbose=run['verbose'],
                            debug=run['debug'],
                            surch_fort=surch_fort,
                            surch_pyt =surch_pyt,
                            unig      =run.get('unigest', None),
                            )
   
   if action == 'check':
      unused, not_found = msgman.check_cata()
      if not run['silent']:
         print3('Messages inutilises :')
         print3(' '.join(unused))
         print3('Messages inexistants :')
         print3(' '.join(not_found))
      if len(unused) + len(not_found) > 0:
         run.Sortie(4)

   elif action == 'move':
      if len(args) != 2:
         run.parser.error(
               _(u"'--%s %s' requires 2 arguments") % (run.current_action, action))
      msgman.move(*args)
   
   else:
      print3()
      fmt = '%12s : %s'
      for obj in args:
         l_msg = list(set(msgman.which_msg(obj)))
         if len(l_msg) > 0:
            l_msg.sort()
            print3(fmt % (obj, l_msg))
         l_sub = list(set(msgman.who_use(obj)))
         if len(l_sub) > 0:
            l_sub.sort()
            print3(fmt % (obj, l_sub))


if __name__ == '__main__':

   # Pour une utilisation hors de as_run :
   #from asrun.devtools import *
   
   REPREF = '/opt/aster/NEW10'
   
   msgman = MESSAGE_MANAGER(repref=REPREF)
   l = msgman._get_list_python()
#   print msgman.search_message('/tmp/amumph.F')



