/*
 * Chan_Misdn -- Channel Driver for Asterisk
 *
 * Interface to Asterisk
 *
 * Copyright (C) 2004, Christian Richter
 *
 * Christian Richter <crich@beronet.com>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */


#include "chan_misdn_config.h"

#include <stdio.h>
#include <pthread.h>
#include <string.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <errno.h>
#include <unistd.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/file.h>


#include <asterisk/channel.h>
#include <asterisk/config.h>
#include <asterisk/logger.h>
#include <asterisk/module.h>
#include <asterisk/pbx.h>
#include <asterisk/options.h>
#include <asterisk/io.h>
#include <asterisk/frame.h>
#include <asterisk/translate.h>
#include <asterisk/cli.h>
#include <asterisk/musiconhold.h>
#include <asterisk/dsp.h>
#include <asterisk/translate.h>
#include <asterisk/config.h>
#include <asterisk/file.h>
#include <asterisk/callerid.h>
#include <asterisk/indications.h>
#include <asterisk/app.h>
#include <asterisk/features.h>
#include "chan_misdn.h"
#include "isdn_lib.h"
#include "crypt.h"

#ifdef ASTERISK_STABLE
#include <asterisk/channel_pvt.h>
#include <asterisk/utils.h>
#else
#include <asterisk/strings.h>
#endif

/* #define MISDN_DEBUG 1 */

#ifdef ASTERISK_STABLE
#define MISDN_ASTERISK_TECH_PVT(ast) ast->pvt->pvt
#define MISDN_ASTERISK_PVT(ast) ast->pvt
#else
#define MISDN_ASTERISK_TECH_PVT(ast) ast->tech_pvt
#define MISDN_ASTERISK_PVT(ast) 1
#endif

static  char *desc = "Channel driver for mISDN Support (Bri/Pri)";
static  char *type = "mISDN";
#ifdef ASTERISK_STABLE
static  char *tdesc = "This driver enables the asterisk to use hardware which is supported by the new  isdn4linux System mISDN";
#endif

/* FIXME do we still need this? */
int tracing = 0 ;
/* FIXME please!! */
int default_clearl3;


static int usecnt=0;

char **misdn_key_vector=NULL;
int misdn_key_vector_size=0;


#if 0
static int misdn_dynamic_crypting=0;
#endif

/* Only alaw and mulaw is allowed for now */
static int prefformat =  AST_FORMAT_ALAW ; /*  AST_FORMAT_SLINEAR ;  AST_FORMAT_ULAW | */


static ast_mutex_t usecnt_lock; 

#if 0
extern int misdn_crypt_debug;
#endif

/* we set this after loading the config since it is read at least a billion times */
int misdn_debug=0;

struct chan_list dummy_cl;

struct chan_list *cl_te=NULL;
pthread_mutex_t cl_te_lock;


struct misdn_lib *mgr_te;
enum te_event_response_e
cb_te_events(struct misdn_lib *mgr, enum te_event_e event, struct misdn_bchannel *bc, void *user_data);

void send_cause2ast(struct ast_channel *ast, struct misdn_bchannel*bc);

void cl_queue_chan(struct chan_list **list, struct chan_list *chan);
void cl_dequeue_chan(struct chan_list **list, struct chan_list *chan);
struct chan_list *cl_find_chan_by_bc_te(struct chan_list *list, struct misdn_bchannel *bc);
void * audio_thread( void * data);
void chan_misdn_log(char *tmpl, ...);
void chan_misdn_trace_call(struct ast_channel *chan, int debug, char *tmpl, ...);


int check_ret (struct misdn_lib *mgr,struct misdn_bchannel * bc, enum te_event_e event, int ret);
static int start_bc_tones(struct chan_list *cl);
static int stop_bc_tones(struct chan_list *cl);
static void release_chan(struct misdn_bchannel *bc);

static int misdn_set_opt_exec(struct ast_channel *chan, void *data);


/*************** Helpers *****************/


static struct chan_list * get_chan_by_ast(struct ast_channel *ast)
{
	struct chan_list *tmp;
  
	for (tmp=cl_te; tmp; tmp = tmp->next) {
		if ( tmp->ast == ast ) return tmp;
	}
  
	return NULL;
}


static struct chan_list * get_chan_by_ast_name(char *name)
{
	struct chan_list *tmp;
  
	for (tmp=cl_te; tmp; tmp = tmp->next) {
		if ( tmp->ast  && strcmp(tmp->ast->name,name) == 0) return tmp;
	}
  
	return NULL;
}

struct misdn_bchannel *misdn_get_bc_te_by_ast(struct ast_channel *ast)
{
	struct chan_list *chan = get_chan_by_ast(ast);
	if (chan)
		return chan->bc_te;
	return NULL;
}

static char* tone2str(struct misdn_bchannel *bc)
{
	static struct {
		char name[16];
		enum tone_e tone;
	} *tone, tone_buf[] = {
		{"NOTONE",TONE_NONE},
		{"DIAL",TONE_DIAL},
		{"BUSY",TONE_BUSY},
		{"ALERT",TONE_ALERTING},
		{"",TONE_NONE}
	};
  
  
	for (tone=&tone_buf[0]; tone->name[0]; tone++) {
		if (tone->tone == bc->tone) return tone->name;
	}
	return NULL;
}


static void print_bearer(struct misdn_bchannel *bc) 
{
	switch (bc->capability) {
	case INFO_CAPABILITY_SPEECH:
		if (misdn_debug>1) chan_misdn_log(" --> Bearer: Speech\n");
		break;
	case INFO_CAPABILITY_AUDIO_3_1K:
		if (misdn_debug>1) chan_misdn_log(" --> Bearer: Audio 3.1k\n");
		break;
	case INFO_CAPABILITY_DIGITAL_UNRESTRICTED:
		if (misdn_debug>1) chan_misdn_log(" --> Bearer: Unres Digital\n");
		break;
	case INFO_CAPABILITY_DIGITAL_RESTRICTED:
		if (misdn_debug>1) chan_misdn_log(" --> Bearer: Res Digital\n");
		break;
	default: 
		if (misdn_debug>1) chan_misdn_log(" --> Unknown Bearer\n");
		break;
	}
	switch(bc->law) {
	case INFO_CODEC_ALAW:
		if (misdn_debug>1) chan_misdn_log(" --> Codec: Alaw\n");
		break;
	case INFO_CODEC_ULAW:
		if (misdn_debug>1) chan_misdn_log(" --> Codec: Ulaw\n");
		break;
	}
}
/*************** Helpers END *************/





void send_digit_to_chan(struct chan_list *cl, char digit )
{
	static const char* dtmf_tones[] = {
		"!941+1336/100,!0/100",	/* 0 */
		"!697+1209/100,!0/100",	/* 1 */
		"!697+1336/100,!0/100",	/* 2 */
		"!697+1477/100,!0/100",	/* 3 */
		"!770+1209/100,!0/100",	/* 4 */
		"!770+1336/100,!0/100",	/* 5 */
		"!770+1477/100,!0/100",	/* 6 */
		"!852+1209/100,!0/100",	/* 7 */
		"!852+1336/100,!0/100",	/* 8 */
		"!852+1477/100,!0/100",	/* 9 */
		"!697+1633/100,!0/100",	/* A */
		"!770+1633/100,!0/100",	/* B */
		"!852+1633/100,!0/100",	/* C */
		"!941+1633/100,!0/100",	/* D */
		"!941+1209/100,!0/100",	/* * */
		"!941+1477/100,!0/100" };	/* # */
	struct ast_channel *chan=cl->ast; 
  
	if (digit >= '0' && digit <='9')
		ast_playtones_start(chan,0,dtmf_tones[digit-'0'], 0);
	else if (digit >= 'A' && digit <= 'D')
		ast_playtones_start(chan,0,dtmf_tones[digit-'A'+10], 0);
	else if (digit == '*')
		ast_playtones_start(chan,0,dtmf_tones[14], 0);
	else if (digit == '#')
		ast_playtones_start(chan,0,dtmf_tones[15], 0);
	else {
		/* not handled */
		ast_log(LOG_DEBUG, "Unable to handle DTMF tone '%c' for '%s'\n", digit, chan->name);
    
    
	}
}
/*** CLI HANDLING ***/
static int misdn_set_debug(int fd, int argc, char *argv[])
{
	if (argc != 4 )return RESULT_SHOWUSAGE; 
  
	misdn_debug = atoi(argv[3]);
	ast_cli(fd,"changing debug level to %d\n",misdn_debug);

	return 0;
}


static int misdn_set_crypt_debug(int fd, int argc, char *argv[])
{
	if (argc != 5 )return RESULT_SHOWUSAGE; 

#if 0  
	misdn_crypt_debug = atoi(argv[4]);
	ast_cli(fd,"changing debug level to %d\n",misdn_crypt_debug);
#endif
	
	return 0;
}

static int misdn_flush_stack (int fd, int argc, char *argv[])
{
	struct misdn_lib *mgr = mgr_te;
	int port, i;
	struct misdn_stack *stack;
  
	if (argc != 4)
		return RESULT_SHOWUSAGE;
  
	port = atoi(argv[3]);

	for (stack=mgr->stack_list;
	     stack;
	     stack=stack->next ) {
		if (stack->port == port) {
			for (i=0; i< stack->b_num; i++) {
				struct misdn_bchannel *mybc=&stack->bc_te[i];
				mybc->in_use=0;
			}
		}
	}
  
  
	return 0;
}

static int misdn_restart_port (int fd, int argc, char *argv[])
{
	int port;
  
	if (argc != 4)
		return RESULT_SHOWUSAGE;
  
	port = atoi(argv[3]);

	misdn_lib_port_restart(mgr_te, port);
  
	return 0;
}

static int misdn_port_up (int fd, int argc, char *argv[])
{
	int port;
	
	if (argc != 4)
		return RESULT_SHOWUSAGE;
	
	port = atoi(argv[3]);
	
	misdn_lib_get_port_up(mgr_te, port);
  
	return 0;
}


static int misdn_show_config (int fd, int argc, char *argv[])
{
	char buffer[BUFFERSIZE];
	enum misdn_cfg_elements elem;
	int linebreak;

	int onlyport = -1;
	if (argc >= 4) {
		onlyport = atoi(argv[3]);
	}
	
	if (argc == 3 || onlyport == 0) {
		ast_cli(fd,"Misdn General-Config: \n"); 
		ast_cli(fd," -> VERSION: " CHAN_MISDN_VERSION "\n");
		
		for (elem = MISDN_GEN_FIRST + 1, linebreak = 1; elem < MISDN_GEN_LAST; elem++, linebreak++) {
			misdn_cfg_get_config_string( 0, elem, buffer, BUFFERSIZE);
			ast_cli(fd, "%-30s%s", buffer, !(linebreak % 2) ? "\n" : "");
		}
	}

	if (onlyport < 0) {
		int port = misdn_cfg_get_next_port(0);
		for (; port > 0; port = misdn_cfg_get_next_port(port)) {
			ast_cli(fd, "\n[PORT %d]\n", port);
			for (elem = MISDN_CFG_FIRST + 1, linebreak = 1; elem < MISDN_CFG_LAST; elem++, linebreak++) {
				misdn_cfg_get_config_string( port, elem, buffer, BUFFERSIZE);
				ast_cli(fd, "%-30s%s", buffer, !(linebreak % 2) ? "\n" : "");
			}	
			ast_cli(fd, "\n");
		}
	}
	
	if (onlyport > 0) {
		if (misdn_cfg_is_port_valid(onlyport)) {
			ast_cli(fd, "[PORT %d]\n", onlyport);
			for (elem = MISDN_CFG_FIRST + 1, linebreak = 1; elem < MISDN_CFG_LAST; elem++, linebreak++) {
				misdn_cfg_get_config_string( onlyport, elem, buffer, BUFFERSIZE);
				ast_cli(fd, "%-30s%s", buffer, !(linebreak % 2) ? "\n" : "");
			}	
			ast_cli(fd, "\n");
		} else {
			ast_cli(fd, "Port %d is not active!\n", onlyport);
		}
	}
	return 0;
}



struct state_struct {
	enum misdn_chan_state state;
	char txt[255] ;
} ;

struct state_struct state_array[] = {
	{MISDN_NOTHING,"NOTHING"}, /* at beginning */
	{MISDN_WAITING4DIGS,"WAITING4DIGS"}, /*  when waiting for infos */
	{MISDN_EXTCANTMATCH,"EXTCANTMATCH"}, /*  when asterisk couldnt match our ext */
	{MISDN_DIALING,"DIALING"}, /*  when pbx_start */
	{MISDN_PROGRESS,"PROGRESS"}, /*  when pbx_start */
	{MISDN_CALLING,"CALLING"}, /*  when misdn_call is called */
	{MISDN_ALERTING,"ALERTING"}, /*  when Alerting */
	{MISDN_BUSY,"BUSY"}, /*  when BUSY */
	{MISDN_CONNECTED,"CONNECTED"}, /*  when connected */
	{MISDN_BRIDGED,"BRIDGED"}, /*  when bridged */
	{MISDN_CLEANING,"CLEANING"}, /* when hangup from * but we were connected before */
	{MISDN_HUNGUP_FROM_MISDN,"HUNGUP_FROM_MISDN"}, /* when DISCONNECT/RELEASE/REL_COMP  cam from misdn */
	{MISDN_HOLDED,"HOLDED"}, /* when DISCONNECT/RELEASE/REL_COMP  cam from misdn */
	{MISDN_HOLD_DISCONNECT,"HOLD_DISCONNECT"}, /* when DISCONNECT/RELEASE/REL_COMP  cam from misdn */
	{MISDN_HUNGUP_FROM_AST,"HUNGUP_FROM_AST"} /* when DISCONNECT/RELEASE/REL_COMP came out of */
	/* misdn_hangup */
};




char *misdn_get_ch_state(struct chan_list *p) 
{
	int i;
	if( !p) return NULL;
  
	for (i=0; i< sizeof(state_array)/sizeof(struct state_struct); i++) {
		if ( state_array[i].state == p->state) return state_array[i].txt; 
	}
  
	return NULL;
}

static int misdn_reload (int fd, int argc, char *argv[])
{
	ast_cli(fd, "Reloading mISDN Config\n");
	chan_misdn_log("Dynamic Crypting Activation is not support during reload at the moment\n");
	
	misdn_cfg_reload();

	/* FIXME */
	misdn_cfg_get( 0, MISDN_GEN_CLEAR_L3, &default_clearl3, sizeof(int));
	{
		char tempbuf[BUFFERSIZE];
		misdn_cfg_get( 0, MISDN_GEN_TRACEFILE, tempbuf, BUFFERSIZE);
		if (strlen(tempbuf))
			tracing = 1;
	}

	misdn_cfg_get( 0, MISDN_GEN_DEBUG, &misdn_debug, sizeof(int));
	
	return 0;
}

static void print_bc_info (int fd, struct chan_list* help, struct misdn_bchannel* bc)
{
	struct ast_channel *ast=help->ast;
	ast_cli(fd,
		"* Pid:%d Prt:%d Ch:%d Mode:%s Org:%s dad:%s oad:%s ctx:%s state:%s\n",
		bc->pid, bc->stack->port, bc->channel,
		bc->stack->mode==NT_MODE?"NT":"TE",
		help->orginator == ORG_AST?"*":"I",
		ast?ast->exten:NULL,
		ast?AST_CID_P(ast):NULL,
		ast?ast->context:NULL,
		misdn_get_ch_state(help)
		);
	if (misdn_debug > 0)
		ast_cli(fd,
			"  --> astname: %s\n"
			"  --> ch_l3id: %x\n"
			"  --> bc_addr: %x\n"
			"  --> bc_l3id: %x\n"
			"  --> display: %s\n"
#if 0
			"  --> curprx: %d\n"
			"  --> curptx: %d\n"
			"  --> crypt: %d\n"
#endif
			"  --> activated: %d\n"
			"  --> hdlc: %d\n"
			"  --> echo_cancel: %d\n"
			"  --> echo_tics: %d\n"
			"  --> notone : rx %d tx:%d\n"
			"  --> bc_hold: %d holded_bc :%d\n",
			help->ast->name,
			help->l3id,
			bc->addr,
			bc?bc->l3_id:-1,
			bc->display,

#if 0
			bc->curprx,
			bc->curptx,
			bc->crypt,
#endif
			bc->active,
			bc->digital,
			bc->ec_enable,
			bc->ec_tics, 
			help->norxtone,help->notxtone,
			bc->holded, help->holded_bc?1:0
			);
  
}


static int misdn_show_cls (int fd, int argc, char *argv[])
{
	struct chan_list *help=cl_te;
  
	ast_cli(fd,"Chan List: %p\n",cl_te); 
  
	for (;help; help=help->next) {
		struct misdn_bchannel *bc=help->bc_te;   
		struct ast_channel *ast=help->ast;
		if (misdn_debug > 2) ast_cli(fd, "Bc:%p Ast:%p\n", bc, ast);
		if (bc) {
			print_bc_info(fd, help, bc);
		} else if ( (bc=help->holded_bc) ) {
			chan_misdn_log("ITS A HOLDED BC:\n");
			print_bc_info(fd, help,  bc);
		} else {
			ast_cli(fd,"* Channel in unknown STATE !!! Exten:%s, Callerid:%s\n", ast->exten, AST_CID_P(ast));
		}
	}
  
  
	return 0;
}



static int misdn_show_cl (int fd, int argc, char *argv[])
{
	struct chan_list *help=cl_te;

	if (argc != 4)
		return RESULT_SHOWUSAGE;
  
	for (;help; help=help->next) {
		struct misdn_bchannel *bc=help->bc_te;   
		struct ast_channel *ast=help->ast;
    
		if (bc && ast) {
			if (!strcasecmp(ast->name,argv[3])) {
				print_bc_info(fd, help, bc);
				break; 
			}
		} 
	}
  
  
	return 0;
}

pthread_mutex_t lock;
int MAXTICS=8;

static int misdn_set_tics (int fd, int argc, char *argv[])
{
	if (argc != 4)
		return RESULT_SHOWUSAGE;
  
	MAXTICS=atoi(argv[3]);
  
	return 0;
}


static int misdn_show_fullstacks (int fd, int argc, char *argv[])
{
	struct misdn_lib *mgr = mgr_te;
  
	struct misdn_stack *stack;
	ast_cli(fd, "BEGIN STACK_LIST:\n");
	for (stack=mgr->stack_list;
	     stack;
	     stack=stack->next ) {
		int i;
		ast_cli(fd, "* Stack Addr: Uid %x Port %d Type %s Prot. %s Link %s\n",stack->upper_id, stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN");
		for (i=0; i< stack->b_num; i++) {
			struct misdn_bchannel *mybc=&stack->bc_te[i];
			ast_cli(fd," --> bchan: addr %x channel %d pid %d cr %x tone %s inuse %d\n", mybc->addr,mybc->channel, mybc?mybc->pid:-1,mybc?mybc->l3_id:-1 , tone2str(mybc), mybc->in_use);
		}
	}

	return 0;
}

static int misdn_show_stacks (int fd, int argc, char *argv[])
{
	struct misdn_lib *mgr = mgr_te;
	struct misdn_stack *stack;
    
	ast_cli(fd, "BEGIN STACK_LIST:\n");
	for (stack=mgr->stack_list;
	     stack;
	     stack=stack->next ) {
		ast_cli(fd, "* Stack Addr: Port %d Type %s Prot. %s L2Link %s L1Link:%s\n", stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN", stack->l1link?"UP":"DOWN");
	}

	return 0;

}

static int misdn_show_port (int fd, int argc, char *argv[])
{
	struct misdn_lib *mgr = mgr_te;
	struct misdn_stack *stack;
	int i;
	int port;
  
	if (argc != 4)
		return RESULT_SHOWUSAGE;
  
	port = atoi(argv[3]);
  
	ast_cli(fd, "BEGIN STACK_LIST:\n");
	for (stack=mgr->stack_list;
	     stack;
	     stack=stack->next ) {
		if (stack->port == port) {
			ast_cli(fd, "* Stack Addr: %x Port %d Type %s Prot. %s L2Link %s L1Link:%s\n",stack->upper_id, stack->upper_id & IF_CONTRMASK, stack->mode==NT_MODE?"NT":"TE", stack->ptp?"PTP":"PMP", stack->l2link?"UP":"DOWN", stack->l1link?"UP":"DOWN");
			for (i=0; i <stack->b_num; i++) {
				ast_cli(fd,"Idx: %d stack->chan: %d Chan %d InUse:%d\n",i,stack->channels[i], i+1, stack->bc_te[i].in_use);
			}
			/* misdn_lib_get_port_info(mgr, port); */
		}
	}

	return 0;
}

static int misdn_send_digit (int fd, int argc, char *argv[])
{
	char *channame; 
	char *msg; 
  
	if (argc != 5)
		return RESULT_SHOWUSAGE;
  
	channame = argv[3];
	msg = argv[4];

	ast_cli(fd, "Sending %s to %s\n",msg, channame);
  
	{
		struct chan_list *tmp=get_chan_by_ast_name(channame);
    
		if (!tmp) {
			ast_cli(fd, "Sending %s to %s failed Channel does not exist\n",msg, channame);
			return 0; 
		} else {
#if 1
			int i;
			int msglen = strlen(msg);
			for (i=0; i<msglen; i++) {
				ast_cli(fd, "Sending: %c\n",msg[i]);
				send_digit_to_chan(tmp, msg[i]);
				/* res = ast_safe_sleep(tmp->ast, 250); */
				usleep(250000);
				/* res = ast_waitfor(tmp->ast,100); */
			}
#else
			int res;
			res = ast_dtmf_stream(tmp->ast,NULL,msg,250);
#endif
		}
	}
  
	return 0; 
}

static int misdn_toggle_echocancel (int fd, int argc, char *argv[])
{
	char *channame; 

	if (argc != 4)
		return RESULT_SHOWUSAGE;
	
	channame = argv[3];
  
	ast_cli(fd, "Toggling EchoCancel on %s\n", channame);
  
	{
		struct chan_list *tmp=get_chan_by_ast_name(channame);
    
		if (!tmp) {
			ast_cli(fd, "Toggling EchoCancel %s failed Channel does not exist\n", channame);
			return 0; 
		} else {
			tmp->bc_te->ec_enable=tmp->bc_te->ec_enable?0:1;

			if (tmp->bc_te->ec_enable) {
				manager_ec_enable(tmp->bc_te);
			} else {
				manager_ec_disable(tmp->bc_te);
			}
		}
	}
  
	return 0; 
}



static int misdn_send_display (int fd, int argc, char *argv[])
{
	char *channame; 
	char *msg; 
  
	if (argc != 5)
		return RESULT_SHOWUSAGE;
  
	channame = argv[3];
	msg = argv[4];

	ast_cli(fd, "Sending %s to %s\n",msg, channame);
	{
		struct chan_list *tmp;
		tmp=get_chan_by_ast_name(channame);
    
		if (tmp && tmp->bc_te) {
			strcpy(tmp->bc_te->display, msg);
			misdn_lib_send_event(mgr_te, tmp->bc_te, EVENT_INFORMATION);
		} else {
			ast_cli(fd,"No such channel %s\n",channame);
			return RESULT_FAILURE;
		}
	}

	return RESULT_SUCCESS ;
}




static char *complete_ch_helper(char *line, char *word, int pos, int state, int rpos)
{
	struct ast_channel *c;
	int which=0;
	char *ret;
	if (pos != rpos)
		return NULL;
	c = ast_channel_walk_locked(NULL);
	while(c) {
		if (!strncasecmp(word, c->name, strlen(word))) {
			if (++which > state)
				break;
		}
		ast_mutex_unlock(&c->lock);
		c = ast_channel_walk_locked(c);
	}
	if (c) {
		ret = strdup(c->name);
		ast_mutex_unlock(&c->lock);
	} else
		ret = NULL;
	return ret;
}

static char *complete_ch(char *line, char *word, int pos, int state)
{
	return complete_ch_helper(line, word, pos, state, 3);
}

static struct ast_cli_entry cli_send_digit =
{ {"misdn","send","digit", NULL},
  misdn_send_digit,
  "Sends DTMF Digit to mISDN Channel", /*  Short description */
  "Usage: misdn send digit <channel> \"<msg>\" \n"
  "       Send <digit> to <channel> as DTMF Tone\n"
  "       when channel is a mISDN channel\n",
  complete_ch
};


static struct ast_cli_entry cli_toggle_echocancel =
{ {"misdn","toggle","echocancel", NULL},
  misdn_toggle_echocancel,
  "Toggles EchoCancel on mISDN Channel", /*  Short description */
  "Usage: misdn toggle echocancel <channel>\n", 
  complete_ch
};



static struct ast_cli_entry cli_send_display =
{ {"misdn","send","display", NULL},
  misdn_send_display,
  "Sends Text to mISDN Channel", /*  Short description */
  "Usage: misdn send display <channel> \"<msg>\" \n"
  "       Send <msg> to <channel> as Display Message\n"
  "       when channel is a mISDN channel\n",
  complete_ch
};


static struct ast_cli_entry cli_show_config =
{ {"misdn","show","config", NULL},
  misdn_show_config,
  "Shows internal mISDN config, read from cfg-file", /*  Short description */
  "Usage: misdn show config [port number | 0]\n"
};
 

static struct ast_cli_entry cli_reload =
{ {"misdn","reload", NULL},
  misdn_reload,
  "Reloads internal mISDN config, read from cfg-file", /*  Short description */
  "Usage: misdn reload\n"
};

static struct ast_cli_entry cli_set_tics =
{ {"misdn","set","tics", NULL},
  misdn_set_tics,
  "", /*  Short description */
  "\n"
};


static struct ast_cli_entry cli_show_cls =
{ {"misdn","show","channels", NULL},
  misdn_show_cls,
  "Shows internal mISDN chan_list", /*  Short description */
  "Usage: misdn show channels\n"
};

static struct ast_cli_entry cli_show_cl =
{ {"misdn","show","channel", NULL},
  misdn_show_cl,
  "Shows internal mISDN chan_list", /*  Short description */
  "Usage: misdn show channels\n",
  complete_ch
};

static struct ast_cli_entry cli_show_fullstacks =
{ {"misdn","show","fullstacks", NULL},
  misdn_show_fullstacks,
  "Shows internal mISDN stack_list with bchannels", /*  Short description */
  "Usage: misdn show fullstacks\n"
};

static struct ast_cli_entry cli_flush_stack =
{ {"misdn","flush","stack", NULL},
  misdn_flush_stack,
  "Flushes the in_use flag", /*  Short description */
  "Usage: misdn flush stack\n"
};

static struct ast_cli_entry cli_restart_port =
{ {"misdn","restart","port", NULL},
  misdn_restart_port,
  "Restarts the given port", /*  Short description */
  "Usage: misdn restart port\n"
};


static struct ast_cli_entry cli_port_up =
{ {"misdn","port","up", NULL},
  misdn_port_up,
  "Tries to establish L1 on the given port", /*  Short description */
  "Usage: misdn port up <port>\n"
};


static struct ast_cli_entry cli_show_stacks =
{ {"misdn","show","stacks", NULL},
  misdn_show_stacks,
  "Shows internal mISDN stack_list", /*  Short description */
  "Usage: misdn show stacks\n"
};

static struct ast_cli_entry cli_show_port =
{ {"misdn","show","port", NULL},
  misdn_show_port,
  "Shows detailed information for given port", /*  Short description */
  "Usage: misdn show port <port>\n"
};



static struct ast_cli_entry cli_set_debug =
{ {"misdn","set","debug", NULL},
  misdn_set_debug,
  "Sets Debuglevel of chan_misdn, at the moment, level={1,2}", /*  Short description */
  "Usage: misdn set debug <level>\n"
};

static struct ast_cli_entry cli_set_crypt_debug =
{ {"misdn","set","crypt","debug", NULL},
  misdn_set_crypt_debug,
  "Sets CryptDebuglevel of chan_misdn, at the moment, level={1,2}", /*  Short description */
  "Usage: misdn set crypt debug <level>\n"
};
/*** CLI END ***/


/*****************************/
/*** AST Indications Start ***/
/*****************************/

static int misdn_call(struct ast_channel *ast, char *dest, int timeout)
{
	int port=1;
	char *prt; 
	char *ext;
	char dest_cp[128]="";
	char group[128]="";
	char *tokb;
	int r;
	struct misdn_bchannel *newbc=NULL;
	char *opts;
  
	if (!ast) {
		chan_misdn_log( "! IND : CALL Without Asterisk Channel !?!?!\n");
		if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
		ast->hangupcause=41;
		ast_setstate(ast, AST_STATE_DOWN);
		return -1;
	}
  
	if (!dest) {
		chan_misdn_log( "! IND : CALL dad:%s tech:%s WITHOUT PORT, check extension.conf\n",ast->exten,ast->name, ast->context);
		if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
		ast->hangupcause=41;
		ast_setstate(ast, AST_STATE_DOWN);
		return -1;
	}
	
	strcpy(dest_cp, dest);
  
	if (misdn_debug>0) chan_misdn_log( "* CALL: %s\n",dest_cp);
    
	prt=strtok_r(dest_cp,"/", &tokb);
  
	if (prt) {
		if (prt[0]=='g' && prt[1]==':' ) {
			/* We make a group call lets checkout which ports are in my group */
			/*  */
			prt +=2;
			strcpy(group,prt);
			if (misdn_debug >1) chan_misdn_log( " --> Group Call group: %s\n",group);
		} else {
			port=atoi(prt);
		}
		ext=strtok_r(NULL,"/", &tokb);
		if (!ext) {
			chan_misdn_log( " --> ! IND : CALL dad:%s tech:%s WITH WRONG ARGS, check extension.conf\n",ast->exten,ast->name, ast->context);
			if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
			ast_setstate(ast, AST_STATE_DOWN);
			ast->hangupcause=41;
			return -1;
		} else {
			opts=strtok_r(NULL,"/", &tokb);
		}
	} else {
		chan_misdn_log( " --> ! IND : CALL dad:%s tech:%s WITHOUT PORT/Group, check extension.conf\n",ast->exten,ast->name, ast->context);
		if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
		ast_setstate(ast, AST_STATE_DOWN);
		ast->hangupcause=41;
		return -1;
	}

  
	if ( !ast_strlen_zero(group) ) {
		for ( port=misdn_cfg_get_next_port(0); port > 0;
		      port=misdn_cfg_get_next_port(port) ) {
			
			char cfg_group[BUFFERSIZE];
			misdn_cfg_get( port, MISDN_CFG_NAME, cfg_group, BUFFERSIZE);
			if (!strcasecmp(cfg_group, group)) {
				
#ifdef MISDNUSER_JOLLY
				int l1;
				misdn_cfg_get( 0, MISDN_GEN_L1_INFO_OK, &l1, sizeof(l1));
				chan_misdn_log("def_l1:%d, portup:%d\n",
					       l1,
					       misdn_lib_port_up(mgr_te,port));
				
				if ( l1 && misdn_lib_port_up(mgr_te, port) )
#endif
				{
					
					newbc=misdn_lib_get_free_bc(mgr_te, port );
					if (newbc) {
						break;
					}
				}
				
			}
		}
		if (!newbc) {
			if (misdn_debug > 0) chan_misdn_log(" --> ! No free channel chan %p even after Group Call\n",ast);
			if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
			ast->hangupcause=34;      
			ast_setstate(ast, AST_STATE_DOWN);
			return -1;
			
		}
	} else {
		newbc=misdn_lib_get_free_bc(mgr_te, port );
		if (!newbc) {
			if (misdn_debug > 0) chan_misdn_log(" --> ! No free channel chan %p\n",ast);
			if (misdn_debug > 0) chan_misdn_log(" --> SEND: State Down\n");
			ast->hangupcause=34;
			ast_setstate(ast, AST_STATE_DOWN);
			return -1;
      
		}
	}
  
  
	if (misdn_debug>0) chan_misdn_log( " --> * dad:%s tech:%s ctx:%s\n",ext,ast->name, ast->context);
	
	if ((ast->_state != AST_STATE_DOWN) && (ast->_state != AST_STATE_RESERVED)) {
		ast_log(LOG_WARNING, " --> ! misdn_call called on %s, neither down nor reserved\n", ast->name);
		ast->hangupcause=41;
		return -1;
	}


	{
		char context[BUFFERSIZE];
		
		misdn_cfg_get( port, MISDN_CFG_CONTEXT, context, sizeof(ast->context));
		strcpy(ast->context,context);
		if (misdn_debug>1) chan_misdn_log(" --> * Setting Context to %s\n",context);
		misdn_cfg_get( port, MISDN_CFG_LANGUAGE, ast->language, BUFFERSIZE);
		
		misdn_cfg_get( port, MISDN_CFG_TXGAIN, &newbc->txgain, sizeof(int));
		misdn_cfg_get( port, MISDN_CFG_RXGAIN, &newbc->rxgain, sizeof(int));
		
		misdn_cfg_get( port, MISDN_CFG_TE_CHOOSE_CHANNEL, &(newbc->stack->te_choose_channel), sizeof(int));


		{
			char callerid[BUFFERSIZE];
			misdn_cfg_get( port, MISDN_CFG_CALLERID, callerid, BUFFERSIZE);
			if ( ! ast_strlen_zero(callerid) ) {
				if (misdn_debug>0) chan_misdn_log(" --> * Setting Cid to %s\n", callerid);
				strcpy(newbc->oad,callerid);
				
				misdn_cfg_get( port, MISDN_CFG_DIALPLAN, &newbc->dnumplan, sizeof(int));
				switch (newbc->dnumplan) {
				case NUMPLAN_INTERNATIONAL:
				case NUMPLAN_NATIONAL:
				case NUMPLAN_SUBSCRIBER:
				case NUMPLAN_UNKNOWN:
					/* Maybe we should cutt off the prefix if present ? */
					break;
				default:
					chan_misdn_log(" --> !!!! Wrong dialplan setting, please see the misdn.conf sample file\n ");
					break;
				}
				
			}
		}
		
		
#ifndef ASTERISK_STABLE
		{
			char buf[256];
			ast_group_t pg,cg;

			misdn_cfg_get( port, MISDN_CFG_PICKUPGROUP, &pg, sizeof(pg));
			misdn_cfg_get( port, MISDN_CFG_CALLGROUP, &cg, sizeof(cg));
			
			if (misdn_debug>1) chan_misdn_log(" --> * CallGrp:%s PickupGrp:%s\n",ast_print_group(buf,sizeof(buf),cg),ast_print_group(buf,sizeof(buf),pg));
			ast->pickupgroup=pg;
			ast->callgroup=cg;
		}
#endif

		/* Will be overridden by asterisk in head! */
		{
			int pres;
			
			misdn_cfg_get( port, MISDN_CFG_PRES, &pres, sizeof(int));
			newbc->pres=pres?0:1;
			
		}
		
#ifndef ASTERISK_STABLE
		int def_callingpres;
		misdn_cfg_get( 0, MISDN_CFG_USE_CALLINGPRES, &def_callingpres, sizeof(int));
		if ( def_callingpres) {
			switch (ast->cid.cid_pres){
			case AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED:
				newbc->pres=1;
				break;
				
			case AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN:
				newbc->pres=0;
				break;
			default:
				newbc->pres=0;
			}
		}
#endif

		
		{
			int ec, ectr;
			
			misdn_cfg_get( port, MISDN_CFG_ECHOCANCEL, &ec, sizeof(int));
			
			misdn_cfg_get( port, MISDN_CFG_ECHOTRAINING, &ectr, sizeof(int));
			if (ec == 1 ) {
				newbc->ec_enable=1;
			} else if ( ec > 1 ) {
				newbc->ec_enable=1;
				newbc->ec_deftaps=ec;
			}

			if ( !ectr ) {
				newbc->ec_training=0;
			}
		}
		
	} 
	
	
	
	
	if (misdn_debug > 2) chan_misdn_log(" --> * adding2newbc ext %s\n",ext);
	if (ext) strcpy(newbc->dad,ext);
	if (misdn_debug > 2) chan_misdn_log(" --> * adding2newbc callerid %s\n",AST_CID_P(ast));
	if (ast_strlen_zero(newbc->oad) && AST_CID_P(ast) ) {
#ifdef ASTERISK_STABLE
		char *cid_num , *cid_name;
		ast_callerid_parse(ast->callerid,&cid_name,&cid_num);		if (cid_num) strcpy(newbc->oad,cid_num);
#else
		if (AST_CID_P(ast)) strcpy(newbc->oad,AST_CID_P(ast));
#endif
	}
	
	{
		struct chan_list *ch=MISDN_ASTERISK_TECH_PVT(ast);
		if (!ch) { ast_verbose("No chan_list in misdn_call"); return -1;}
		ch->bc_te = newbc;
		ch->orginator=ORG_AST;
		ch->ast = ast;
		
		MISDN_ASTERISK_TECH_PVT(ast) = ch ;
		
		
      
#ifdef ASTERISK_STABLE
		if (ast_test_flag(ast, AST_FLAG_DIGITAL) || newbc->digital) {
			newbc->capability=INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
			newbc->digital=1; 
			if (misdn_debug>1) chan_misdn_log(" --> * Call with flag Digital\n");
		}
#else
		newbc->capability=ast->transfercapability;
		pbx_builtin_setvar_helper(ast,"TRANSFERCAPABILITY",ast_transfercapability2str(newbc->capability));
		if ( ast->transfercapability == INFO_CAPABILITY_DIGITAL_UNRESTRICTED) {
			newbc->digital=1;
			if (misdn_debug>1) chan_misdn_log(" --> * Call with flag Digital\n");
		}
      
#endif

		/* Finally The Options Override Everything */
		if (opts) misdn_set_opt_exec(ast,opts);

		switch (newbc->pres) {
		case 0:
			if (misdn_debug > 0) chan_misdn_log(" --> Number Screened\n");
			break;
		case 1:
			if (misdn_debug > 0) chan_misdn_log(" --> Number Not Screened\n");
			break;
		default:
			if (misdn_debug > 0) chan_misdn_log(" --> Other Screened\n");
		}
		
		cl_queue_chan(&cl_te, ch) ;
		ch->state=MISDN_CALLING;

		chan_misdn_trace_call(ast,1,"*->I: EVENT_CALL\n" );
		
		r=misdn_lib_send_event(mgr_te, newbc, EVENT_SETUP );
		
		/** we should have l3id after sending setup **/
		ch->l3id=newbc->l3_id;
		
	}
	
	/* newbc->channel=1; */
	/* chan_misdn_log("Sending Setup Port:%d (stack->port:%d)\n",port,newbc->stack->port); */
    
    
	/* manager_bchannel_setup(newbc); */
    
	if (check_ret(mgr_te, newbc, EVENT_SETUP, r) ) {
		chan_misdn_log(" --> * Theres no Channel at the moment .. !\n");
		if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Down pid:%d\n",newbc?newbc->pid:-1);
		ast->hangupcause=34;
		ast_setstate(ast, AST_STATE_DOWN);
		return -1;
	}
	
	if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Dialing pid:%d\n",newbc?newbc->pid:1);

	ast_setstate(ast, AST_STATE_DIALING);
	/* ast_setstate(ast, AST_STATE_RING); */
	ast->hangupcause=16;
	return 0; 
 
}


int misdn_answer(struct ast_channel *ast)
{
	struct chan_list *p;

	
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

	chan_misdn_trace_call(ast,1,"*->I: EVENT_ANSWER\n");
	/* printf("Got Answer\n"); */
	if (misdn_debug>0) chan_misdn_log("* ANSWER:\n");
	
	if (!p) {
		chan_misdn_log(" --> Channel not connected ??\n");
		ast_queue_hangup(ast);
	}

	if (!p->bc_te) {
		if (misdn_debug>0) chan_misdn_log(" --> Got Answer, but theres no bc obj ??\n");

		ast_queue_hangup(ast);

	}

	{
		char *tmp_key = pbx_builtin_getvar_helper(p->ast, "CRYPT_KEY");
    
		if (tmp_key ) {
			if (misdn_debug>0) chan_misdn_log(" --> Connection will be BF crypted\n");
			strcpy(p->bc_te->crypt_key,tmp_key);
		} else {
			if (misdn_debug>2) chan_misdn_log(" --> Connection is without BF encryption\n");
		}
    
	}
	
	p->state = MISDN_CONNECTED;
	misdn_lib_send_event(mgr_te, p->bc_te, EVENT_CONNECT);
	start_bc_tones(p);
	
  
	return 0;
}

int misdn_digit(struct ast_channel *ast, char digit )
{
	struct chan_list *p;


	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;


	if (misdn_debug>0) chan_misdn_log("* IND : Digit %c\n",digit);

	if (!p->bc_te) {
		chan_misdn_log(" --> !! Got Digit Event withut having bchannel Object\n");
		return -1;
	}
  

#if 0
	if (p->bc_te->stack->mode==TE_MODE) 
		misdn_crypt_event(p, CRYPT_EVENT_AST_DIGIT, (void*) &digit);
#endif
  
	switch (p->state ) {
	case MISDN_CALLING:
	{
		struct misdn_bchannel *bc=p->bc_te;
		char buf[8];
		buf[0]=digit;
		buf[1]=0;
      
		strcat(bc->infos_pending,buf);
	}
	break;
	case MISDN_CALLING_ACKNOWLEDGE:
	{
		struct misdn_bchannel *bc=p->bc_te;
		bc->info_dad[0]=digit;
		bc->info_dad[1]=0;
      
		strcat(bc->dad,bc->info_dad);
		strcpy(p->ast->exten, bc->dad);
      
		misdn_lib_send_event(mgr_te, bc, EVENT_INFORMATION);
	}
	break;
    
	default:
		if ( p->bc_te->send_dtmf ) {
			send_digit_to_chan(p,digit);
			/* char digbuf[3]={digit,'\n',0}; */
			/* ast_dtmf_stream(p->ast, NULL, digbuf, 250) ; */
		}
    
		/* chan_misdn_log(""); */
		break;
	}
	return 0;
}


int misdn_fixup(struct ast_channel *oldast, struct ast_channel *ast)
{
	struct chan_list *p;

  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

	if (misdn_debug>0) chan_misdn_log("* IND: Got Fixup State:%s Holded:%d L3id:%x\n", misdn_get_ch_state(p), p->holded, p->l3id);
  
	/* p_old->state=MISDN_HOLD_DISCONNECT;  */
  
	p->ast = ast ;
	p->state=MISDN_CONNECTED;
  
	return 0;
}

int misdn_soption(struct ast_channel *ast, int opt, void *data, int len )
{
	struct chan_list *p;

  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
	if (misdn_debug>0) chan_misdn_log("* IND : Got option %d\n",opt);
	return 0;
}

int misdn_qoption(struct ast_channel *ast, int opt, void *data, int *len )
{
	struct chan_list *p;

  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
	if (misdn_debug>0) chan_misdn_log("* IND : Got Query option %d\n",opt);
	return 0;
}

int misdn_transfer (struct ast_channel *ast, char *dest)
{
	struct chan_list *p;

  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
	if (misdn_debug>0) chan_misdn_log("* IND : Got Transfer %s\n",dest);
	return 0;
}



int misdn_indication(struct ast_channel *ast, int cond)
{
	struct chan_list *p;

  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) {
		chan_misdn_log("Returnded -1 in misdn_indication\n");
		return -1;
	}
	p = MISDN_ASTERISK_TECH_PVT(ast) ;
  
	if (misdn_debug>0) chan_misdn_log("* IND : Indication from %s\n",ast->exten);
  
	switch (cond) {
	case AST_CONTROL_BUSY:
		if (misdn_debug>0) chan_misdn_log("* IND :\tbusy\n");
    
		/* manager_send_tone(); */
    
		if (p) p->state=MISDN_BUSY;
    
		if (p && p->bc_te) {
			/* misdn_lib_send_event(mgr_te, p->bc_te, EVENT_ALERTING); */
			if (misdn_debug>0) chan_misdn_log(" --> * SEND: State Busy pid:%d\n",p->bc_te?p->bc_te->pid:-1);
			ast_setstate(ast,AST_STATE_BUSY);

			p->bc_te->out_cause=17;
			if (p->state != MISDN_CONNECTED) {
				misdn_lib_send_event(mgr_te, p->bc_te, EVENT_DISCONNECT);
				manager_send_tone(mgr_te, p->bc_te, TONE_BUSY);
			} else {
				chan_misdn_log(" --> !! Got Busy in Connected State !?! port:%d ast:%s\n",
					       p->bc_te->stack->port, ast->name);
			}
      
		}
		break;
	case AST_CONTROL_RING:
		if (p && p->bc_te) {
			if (misdn_debug>0)    chan_misdn_log(" --> * IND :\tring pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
	case AST_CONTROL_RINGING:

		if (p && p->state == MISDN_ALERTING) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\tringing pid:%d but I ws Ringing before, so ignoreing it\n",p->bc_te?p->bc_te->pid:-1);
			break;
		}
		if (p) p->state=MISDN_ALERTING;
    
		if (p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\tringing pid:%d\n",p->bc_te?p->bc_te->pid:-1);
      
			misdn_lib_send_event(mgr_te, p->bc_te, EVENT_ALERTING);
			
			manager_send_tone(mgr_te, p->bc_te, TONE_ALERTING);
			if (misdn_debug > 0 ) chan_misdn_log(" --> * SEND: State Ring pid:%d\n",p->bc_te?p->bc_te->pid:-1);
			ast_setstate(ast,AST_STATE_RINGING);
		}
		/* ast_playtones_start(ast, 0, "1111", 0); */
		/* tone_zone_play_tone(ast, ZT_TONE_RINGTONE); */
		break;
	case AST_CONTROL_ANSWER:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\tanswer pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
	case AST_CONTROL_TAKEOFFHOOK:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> *\ttakeoffhook pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
	case AST_CONTROL_OFFHOOK:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> *\toffhook pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break; 
	case AST_CONTROL_FLASH:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> *\tflash pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
	case AST_CONTROL_PROGRESS:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\tprogress pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		if (p->bc_te && p->bc_te->stack->mode == NT_MODE) {
			/* misdn_lib_send_event(mgr_te, p->bc_te, EVENT_PROCEEDING); */
			/* start_bc_tones(p->bc_te); */
			/* ast_setstate(ast,AST_STATE_RINGING); */
		}
    
		break;
	case AST_CONTROL_CONGESTION:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\tcongestion pid:%d\n",p->bc_te?p->bc_te->pid:-1);

			p->bc_te->out_cause=42;
			if (p->state != MISDN_CONNECTED) {
				misdn_lib_send_event(mgr_te, p->bc_te, EVENT_RELEASE_COMPLETE);
				/* manager_send_tone(mgr_te, p->bc_te, TONE_BUSY); */
			} else {
				chan_misdn_log(" --> !! Got Congestion in Connected State !?! port:%d ast:%s\n",
					       p->bc_te->stack->port, ast->name);
			}
      
		}
		break;
	case -1 :
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\t-1! pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		} else {
			if (misdn_debug>0) chan_misdn_log(" --> * IND :\t-1!\n");
		}
		/* misdn_answer(ast); */
		break;
#ifndef ASTERISK_STABLE
	case AST_CONTROL_HOLD:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> *\tHOLD pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
	case AST_CONTROL_UNHOLD:
		if (p && p->bc_te) {
			if (misdn_debug>0) chan_misdn_log(" --> *\tUNHOLD pid:%d\n",p->bc_te?p->bc_te->pid:-1);
		}
		break;
#endif
	default:
		if (p && p->bc_te) {
			chan_misdn_log(" --> * Unknown Indication:%d pid:%d\n",cond,p->bc_te?p->bc_te->pid:-1);
		} else {
			chan_misdn_log(" --> * Unknown Indication :%d\n",cond);
		}
	}
  
	return 0;
}

/*static void misdn_destroy(struct chan_list *ch)
  {
  }*/


int misdn_hangup(struct ast_channel *ast)
{
	struct chan_list *p;
	struct misdn_bchannel *bc_te=NULL;
  
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;
	

	chan_misdn_trace_call(ast,1,"*->I: EVENT_HANGUP cause=%d\n",ast->hangupcause);
	
	ast_log(LOG_DEBUG, "misdn_hangup(%s)\n", ast->name);
	
	if (ast->_state == AST_STATE_RESERVED) {
		/* between request and call */
		MISDN_ASTERISK_TECH_PVT(ast)=NULL;
		return 0;
	}
  
	if (!p) {
		if (misdn_debug>2) chan_misdn_log("misdn_hangup called, without chan_list obj.\n");
		return 0 ;
	}
  
	bc_te=p->bc_te;
  
	if (bc_te) {
		char *varcause=NULL;
		bc_te->cause=ast->hangupcause?ast->hangupcause:16;
    
		if ( (varcause=pbx_builtin_getvar_helper(ast, "HANGUPCAUSE")) ||
		     (varcause=pbx_builtin_getvar_helper(ast, "PRI_CAUSE"))) {
			int tmpcause=atoi(varcause);
			bc_te->out_cause=tmpcause?tmpcause:16;
		}
    
		if (misdn_debug > 0) chan_misdn_log("* IND : HANGUP\tpid:%d ctx:%s dad:%s oad:%s State:%s\n",p->bc_te?p->bc_te->pid:-1, ast->context, ast->exten, AST_CID_P(ast), misdn_get_ch_state(p));
		if (misdn_debug > 1) chan_misdn_log(" --> l3id:%x\n",p->l3id);
		if (misdn_debug > 0) chan_misdn_log(" --> cause:%d\n",bc_te->cause);
		if (misdn_debug > 0) chan_misdn_log(" --> out_cause:%d\n",bc_te->out_cause);
    
		/* manager_send_tone(mgr_te, bc_te, TONE_BUSY); */
		switch (p->state) {
		case MISDN_CALLING:
			p->state=MISDN_CLEANING;
			misdn_lib_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
			break;
		case MISDN_HOLDED:
		case MISDN_DIALING:
			manager_send_tone(mgr_te, bc_te, TONE_BUSY);
			p->state=MISDN_CLEANING;
			
			misdn_lib_send_event(mgr_te, bc_te, EVENT_RELEASE_COMPLETE);
      
			break;
      
		case MISDN_ALERTING:
			if (misdn_debug > 1) chan_misdn_log(" --> * State Alerting\n");

			if (p->orginator != ORG_AST) 
				manager_send_tone(mgr_te, bc_te, TONE_BUSY);
      
			p->state=MISDN_CLEANING;
			misdn_lib_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
			break;
		case MISDN_CONNECTED:
			/*  Alerting or Disconect */
			if (misdn_debug > 1) chan_misdn_log(" --> * State Connected\n");
			
			manager_send_tone(mgr_te, bc_te, TONE_BUSY);
			misdn_lib_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
      
			p->state=MISDN_CLEANING; /* MISDN_HUNGUP_FROM_AST; */
			break;

		case MISDN_CLEANING:
			break;
      
		case MISDN_HOLD_DISCONNECT:
			/* need to send release here */
			if (misdn_debug > 1) chan_misdn_log(" --> state HOLD_DISC\n");
			stop_bc_tones(p);
      
			if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc_te->cause);
			if (misdn_debug > 0) chan_misdn_log(" --> out_cause %d\n",bc_te->out_cause);
			misdn_lib_send_event(mgr_te,bc_te,EVENT_RELEASE);
			break;
		default:
			/*  Alerting or Disconect */
      
			stop_bc_tones(p);

			if (bc_te->stack->mode == NT_MODE)
				misdn_lib_send_event(mgr_te, bc_te, EVENT_RELEASE);
			else
				misdn_lib_send_event(mgr_te, bc_te, EVENT_DISCONNECT);
			p->state=MISDN_CLEANING; /* MISDN_HUNGUP_FROM_AST; */
		}
    
	}
	
	MISDN_ASTERISK_TECH_PVT(ast)=NULL;
	p->ast=NULL;
  
	return 0;
}

struct ast_frame  *misdn_read(struct ast_channel *ast)
{
	struct chan_list *tmp;
	
	char blah[255];
	int len =0 ;
	
	if (!ast) return NULL;
	tmp = MISDN_ASTERISK_TECH_PVT(ast);
	if (!tmp) return NULL;
	
	read(tmp->pipe[0],blah,sizeof(blah));
	
	len = ibuf_usedcount(tmp->bc_te->astbuf);
	ibuf_memcpy_r(tmp->ast_rd_buf, tmp->bc_te->astbuf,len);
	
	//manager_flip_buf_bits(tmp->ast_rd_buf, len);
	
	tmp->frame.frametype  = AST_FRAME_VOICE;
	tmp->frame.subclass = AST_FORMAT_ALAW;
	tmp->frame.datalen = len;
	tmp->frame.samples = len ;
	tmp->frame.mallocd =0 ;
	tmp->frame.offset= 0 ;
	tmp->frame.src = NULL;
	tmp->frame.data = tmp->ast_rd_buf ;

	chan_misdn_trace_call(tmp->ast,3,"*->I: EVENT_READ len=%d\n",len);
	
	return &tmp->frame;
}

int misdn_write(struct ast_channel *ast, struct ast_frame *frame)
{
	struct chan_list *p;
	int i  = 0;

	
	/* ast_verbose("Txing frm: %d name:%s\n",frame->samples,ast->name); */
	
	
	if (!ast || ! MISDN_ASTERISK_PVT(ast)) return -1;
	p = MISDN_ASTERISK_TECH_PVT(ast) ;

  
	if (p->bc_te && p->bc_te->tone != TONE_NONE)
		manager_send_tone(mgr_te,p->bc_te,TONE_NONE);


	if (p->holded ) {
		if (misdn_debug>4)chan_misdn_log("misdn_write: Returning because holded\n");
		return 0;
	}
  
	if (p->notxtone) {
		if (misdn_debug>4)chan_misdn_log("misdn_write: Returning because notxone\n");
		return 0;
	}
	
	if ( !(frame->subclass & prefformat)) {
		chan_misdn_log("Got Unsupported Frame with Format:%d\n", frame->subclass);
	}
  
#if MISDN_DEBUG
	{
		int i, max=5>frame->samples?frame->samples:5;
    
		printf("write2mISDN %p %d bytes: ", p, frame->samples);
    
		for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame->data)[i]);
		printf ("\n");
	}
#endif

	
	if (p->bc_te) {
		/*if ( !p->bc_te->digital ) {
		  manager_flip_buf_bits(frame->data, frame->samples);
		  }*/
		
		/* ast_verbose("Txing frm: %d port %d\n",frame->samples, p->bc_te->stack->port); */
		
		chan_misdn_trace_call(ast,3,"*->I: EVENT_WRITE len=%d\n",frame->samples);
		
		i= manager_tx2misdn_frm(p->bc_te, frame->data, frame->samples);
	} else {
		chan_misdn_log ("NO bc_te !!\n");
	}

	return 0;
}



int misdn_bridge (struct ast_channel *c0, struct ast_channel *c1, int flags,
		  struct ast_frame **fo, struct ast_channel **rc)
{
	struct chan_list *ch1,*ch2;
	struct ast_channel *carr[2], *who;
	int to=-1;
	struct ast_frame *f;
	/*   return 1; */
  
	ch1=get_chan_by_ast(c0);
	ch2=get_chan_by_ast(c1);

	/* if (!default_misdn_bridging || ch1->bc_te->digital ) return 1; */
	
	carr[0]=c0;
	carr[1]=c1;
  
  
	if (ch1 && ch2 ) ;
	else
		return -1;
  

	int bridging;
	misdn_cfg_get( 0, MISDN_GEN_BRIDGING, &bridging, sizeof(int));
	if (bridging) {
		int ecwb;
		misdn_cfg_get( ch1->bc_te->stack->port, MISDN_CFG_ECHOCANCELWHENBRIDGED, &ecwb, sizeof(int));
		if ( !ecwb ) {
			chan_misdn_log("Disabling Echo Cancellor when Bridged\n");
			ch1->bc_te->ec_enable=0;
			manager_ec_disable(ch1->bc_te);
		}
		misdn_cfg_get( ch2->bc_te->stack->port, MISDN_CFG_ECHOCANCELWHENBRIDGED, &ecwb, sizeof(int));
		if ( !ecwb ) {
			chan_misdn_log("Disabling Echo Cancellor when Bridged\n");
			ch2->bc_te->ec_enable=0;
			manager_ec_disable(ch2->bc_te);
		}
		
		/* trying to make a mISDN_dsp conference */
		/* ph_control(p_m_b_addr, (p_m_conf)?CMX_CONF_JOIN:CMX_CONF_SPLIT, p_m_conf); */
		chan_misdn_log("I SEND: Making conference with Number:%d\n", (ch1->bc_te->pid<<1) +1);
		manager_ph_control(ch1->bc_te, CMX_RECEIVE_OFF, 0);
		manager_ph_control(ch2->bc_te, CMX_RECEIVE_OFF, 0);
    

		manager_ph_control(ch1->bc_te, CMX_CONF_JOIN, (ch1->bc_te->pid<<1) +1);
		manager_ph_control(ch2->bc_te, CMX_CONF_JOIN, (ch1->bc_te->pid<<1) +1);
	}
  
	if (misdn_debug > 0) chan_misdn_log("* Makeing Native Bridge between %s and %s\n", ch1->bc_te->oad, ch2->bc_te->oad);
  
	while(1) {
		to=-1;
		who = ast_waitfor_n(carr, 2, &to);
		f = ast_read(who);
    
		if (!f || f->frametype == AST_FRAME_CONTROL) {
			/* got hangup .. */
			*fo=f;
			*rc=who;
      
			break;
		}
    
    
		if (who == c0) {
			ast_write(c1,f);
		}
		else {
			ast_write(c0,f);
		}
    
	}
  
	if (bridging) {
		manager_ph_control(ch1->bc_te, CMX_RECEIVE_ON, 0);
		manager_ph_control(ch2->bc_te, CMX_RECEIVE_ON, 0);
    
		chan_misdn_log("I SEND: Splitting conference with Number:%d\n", (ch1->bc_te->pid<<1) +1);
		manager_ph_control(ch1->bc_te, CMX_CONF_SPLIT, (ch1->bc_te->pid<<1) +1);
		manager_ph_control(ch2->bc_te, CMX_CONF_SPLIT, (ch1->bc_te->pid<<1) +1);
	}
  
	return 0;
}

/** AST INDICATIONS END **/

static int start_bc_tones(struct chan_list* cl)
{
	manager_bchannel_activate(cl->bc_te);
	manager_send_tone(mgr_te, cl->bc_te ,TONE_NONE);
	cl->notxtone=0;
	cl->norxtone=0;
	return 0;
}

static int stop_bc_tones(struct chan_list *cl)
{
	if (cl->bc_te) {
		/* manager_send_tone(mgr_te, bc, TONE_NONE); */
		manager_bchannel_deactivate(cl->bc_te);
	}
	cl->notxtone=1;
	cl->norxtone=1;
  
	return 0;
}

static int chan_avail(char *port)
{
	int prt=-1;
	char group[128]="";
  
	if (port) {
		if (port[0]=='g' && port[1]==':' ) {
			/* We make a group call lets checkout which ports are in my group */
			/*  */
			port +=2;
			strcpy(group,port);
			if (misdn_debug >0) chan_misdn_log( "Checking Availbl. Chan in Group: %s\n",group);
		} else {
			prt=atoi(port);
		}


    
		if ( !ast_strlen_zero(group) ) {
			int prt=0;
			
			for (prt=misdn_cfg_get_next_port(0);
			     prt >0;
			     prt=misdn_cfg_get_next_port(prt)) {
				char cfg_group[BUFFERSIZE];
				misdn_cfg_get( prt, MISDN_CFG_NAME, cfg_group, BUFFERSIZE);
				
				if (!strcasecmp(cfg_group, group)) {
					int c=0;
					if ((c=misdn_lib_free_bc_avail(mgr_te,prt )))
						return c;
				}
			}
			
		} else {
			int c=0;
			if ((c=misdn_lib_free_bc_avail(mgr_te, prt ))) return c;
		}
    
	}
  
	return 0;
}


struct chan_list *init_chan_list()
{
	struct chan_list *cl=malloc(sizeof(struct chan_list));
	
	if (!cl) {
		chan_misdn_log("misdn_request: malloc failed!");
		return NULL;
	}
	
	memset(cl,0,sizeof(struct chan_list));

	return cl;
	
}

static struct ast_channel *misdn_request(const char *type, int format, void *data, int *cause)
{
	/* struct chan_list *p; */
	struct ast_channel *tmp = NULL;
	char buf[128];
	char buf2[128], *oad=NULL, *port;
	char *tokb;
	int c=0;
	struct chan_list *cl=init_chan_list(cl);
	
	sprintf(buf,"%s/%s",type,(char*)data);
	strcpy(buf2,data);
	port=strtok_r(buf2,"/", &tokb);
	if (port)
		oad=strtok_r(NULL,"/", &tokb);
	
	if (!(c=chan_avail(port))) {
		chan_misdn_log("Sorry on group/port %s is no channel free\n", port);
		return NULL;
	}
	
	tmp = misdn_new( cl, AST_STATE_RESERVED, buf, "","",oad, format,c);
	
	return tmp;
}


#ifndef ASTERISK_STABLE
struct ast_channel_tech misdn_tech = {
	.type="mISDN",
	.description="Channel driver for mISDN Support (Bri/Pri)",
	.capabilities= AST_FORMAT_ALAW ,
	.requester=misdn_request,
	.send_digit=misdn_digit,
	.call=misdn_call,
	.bridge=misdn_bridge, 
	.hangup=misdn_hangup,
	.answer=misdn_answer,
	.read=misdn_read,
	.write=misdn_write,
	.indicate=misdn_indication,
	.fixup=misdn_fixup,
	.properties=0
	/* .transfer=misdn_transfer */
};

struct ast_channel_tech misdn_tech_wo_bridge = {
	.type="mISDN",
	.description="Channel driver for mISDN Support (Bri/Pri)",
	.capabilities=AST_FORMAT_ALAW ,
	.requester=misdn_request,
	.send_digit=misdn_digit,
	.call=misdn_call,
	.hangup=misdn_hangup,
	.answer=misdn_answer,
	.read=misdn_read,
	.write=misdn_write,
	.indicate=misdn_indication,
	.fixup=misdn_fixup,
	.properties=0
	/* .transfer=misdn_transfer */
};
#endif


struct ast_channel *misdn_new(struct chan_list *chlist, int state, char * name, char * context, char *exten, char *callerid, int format, int c)
{
	struct ast_channel *tmp;
  
	tmp = ast_channel_alloc(1);
  
	if (tmp) {
		if (misdn_debug>1)chan_misdn_log(" --> * NEW CHANNEL dad:%s oad:%s ctx:%s\n",exten,callerid, context);
    
		snprintf(tmp->name, sizeof(tmp->name), "%s-%d", name,c);
		tmp->type = type;
    
		tmp->nativeformats = prefformat;
#ifdef ASTERISK_STABLE
		tmp->pvt->pvt = chlist;
		tmp->pvt->rawreadformat = format;
		tmp->pvt->rawwriteformat = format;
		
		tmp->pvt->call = misdn_call;
		tmp->pvt->hangup = misdn_hangup;
		tmp->pvt->read = misdn_read;
		tmp->pvt->write = misdn_write;
		tmp->pvt->answer = misdn_answer;
		tmp->pvt->indicate = misdn_indication; 

		int bridging;
		misdn_cfg_get( 0, MISDN_GEN_BRIDGING, &bridging, sizeof(int));
		if (bridging)
			tmp->pvt->bridge = misdn_bridge;
    
		tmp->pvt->fixup = misdn_fixup;
		tmp->pvt->setoption = misdn_soption;
		tmp->pvt->queryoption = misdn_qoption;
		tmp->pvt->transfer = misdn_transfer;
		tmp->pvt->send_digit = misdn_digit; 
#else
		tmp->readformat = format;
		tmp->writeformat = format;
    
		tmp->tech_pvt = chlist;

		int bridging;
		misdn_cfg_get( 0, MISDN_GEN_BRIDGING, &bridging, sizeof(int));
		if (bridging)
			tmp->tech = &misdn_tech;
		else
			tmp->tech = &misdn_tech_wo_bridge;
    
#endif
    
		tmp->writeformat = format;
		tmp->readformat = format;
		tmp->priority=1;
    
    
		strncpy(tmp->context, context, sizeof(tmp->context)-1);
		strncpy(tmp->exten, exten,  sizeof(tmp->exten) - 1);

		if (callerid) {
#ifndef ASTERISK_STABLE
			char *cid_name, *cid_num;
      
			ast_callerid_parse(callerid, &cid_name, &cid_num);
			if (cid_name)
				tmp->cid.cid_name=strdup(cid_name);
			if (cid_num)
				tmp->cid.cid_num=strdup(cid_num);
#else
			AST_CID_P(tmp)=strdup(callerid);
#endif
		}

		{
			if (pipe(chlist->pipe)<0)
				perror("Pipe failed\n");
			
			tmp->fds[0]=chlist->pipe[0];
			
		}
		misdn_cfg_get( 0, MISDN_CFG_LANGUAGE, tmp->language, sizeof(tmp->language));
		/* i->owner = tmp; */

		if (chlist->bc_te) {

			if (misdn_cfg_is_port_valid(chlist->bc_te->stack->port) ) {
#ifndef ASTERISK_STABLE
				{
					char buf[256];
					ast_group_t pg,cg;
					
					misdn_cfg_get( chlist->bc_te->stack->port, MISDN_CFG_PICKUPGROUP, &pg, sizeof(pg));
					misdn_cfg_get( chlist->bc_te->stack->port, MISDN_CFG_CALLGROUP, &cg, sizeof(cg));
					
					if (misdn_debug>1) chan_misdn_log(" --> * CallGrp:%s PickupGrp:%s\n",ast_print_group(buf,sizeof(buf),cg),ast_print_group(buf,sizeof(buf),pg));
					tmp->pickupgroup=pg;
					tmp->callgroup=cg;
				}
#endif
				misdn_cfg_get( chlist->bc_te->stack->port, MISDN_CFG_TXGAIN, &chlist->bc_te->txgain, sizeof(int));
				misdn_cfg_get( chlist->bc_te->stack->port, MISDN_CFG_RXGAIN, &chlist->bc_te->rxgain, sizeof(int));
				chan_misdn_log(" --> rxgain:%d txgain:%d\n",chlist->bc_te->rxgain,chlist->bc_te->txgain);
				
				
			} else {
				chan_misdn_log(" --> !! Config Not found in misdn_new port:%d\n",chlist->bc_te->stack->port);
			}

		} else {
			if (misdn_debug>2) chan_misdn_log(" --> Not Setting Pickupgroup, we have no bc yet\n");
		}
    
		/*ast_mutex_lock(&usecnt_lock);
		  usecnt++;
		  ast_mutex_unlock(&usecnt_lock);
		  ast_update_use_count(); */

		ast_setstate(tmp, state);
		if (state == AST_STATE_RING)
			tmp->rings = 1;
		else
			tmp->rings = 0;
    
    
	} else
		ast_log(LOG_WARNING, "Unable to allocate channel structure\n");

  
  
	return tmp;
}



int misdn_tx2ast_frm(struct chan_list * tmp, char * buf,  int len )
{
	struct ast_frame frame;

	/* If in hold state we drop frame .. */
	if (tmp->holded || tmp->state == MISDN_CLEANING ) return 0; 

	if (tmp->norxtone) {
		if (misdn_debug>2)
			chan_misdn_log("misdn_tx2ast_frm: Returning because norxtone\n");
		return 0;
	}
  
	frame.frametype  = AST_FRAME_VOICE;
	frame.subclass = AST_FORMAT_ALAW;
	frame.datalen = len;
	frame.samples = len ;
	frame.mallocd =0 ;
	frame.offset= 0 ;
	frame.src = NULL;
	frame.data = buf ;

	if (tmp->faxdetect) {
		struct ast_frame *f,*f2;
		if (tmp->trans)
			f2=ast_translate(tmp->trans, &frame,0);
		else {
			chan_misdn_log("No T-Path found\n");
			return 0;
		}

		/* chan_misdn_log("Processing Frame"); */
		f = ast_dsp_process(tmp->ast, tmp->dsp, f2);
		if (f && (f->frametype == AST_FRAME_DTMF)) {
			ast_log(LOG_DEBUG, "Detected inband DTMF digit: %c", f->subclass);
			if (f->subclass == 'f') {
				/* Fax tone -- Handle and return NULL */
				struct ast_channel *ast = tmp->ast;
				if (!tmp->faxhandled) {
					tmp->faxhandled++;
					if (strcmp(ast->exten, "fax")) {
						if (ast_exists_extension(ast, ast_strlen_zero(ast->macrocontext)? ast->context : ast->macrocontext, "fax", 1, AST_CID_P(ast))) {
							if (option_verbose > 2)
								ast_verbose(VERBOSE_PREFIX_3 "Redirecting %s to fax extension\n", ast->name);
							/* Save the DID/DNIS when we transfer the fax call to a "fax" extension */
							pbx_builtin_setvar_helper(ast,"FAXEXTEN",ast->exten);
							if (ast_async_goto(ast, ast->context, "fax", 1))
								ast_log(LOG_WARNING, "Failed to async goto '%s' into fax of '%s'\n", ast->name, ast->context);
						} else
							ast_log(LOG_NOTICE, "Fax detected, but no fax extension ctx:%s exten:%s\n",ast->context, ast->exten);
					} else
						ast_log(LOG_DEBUG, "Already in a fax extension, not redirecting\n");
				} else
					ast_log(LOG_DEBUG, "Fax already handled\n");
				frame.frametype = AST_FRAME_NULL;
				frame.subclass = 0;
				f = &frame;
			}
		}
	}
  
	if (tmp && tmp->ast && MISDN_ASTERISK_PVT (tmp->ast) && MISDN_ASTERISK_TECH_PVT(tmp->ast) ) {
#if MISDN_DEBUG
		int i, max=5>len?len:5;
    
		printf("write2* %p %d bytes: ",tmp, len);
    
		for (i=0; i<  max ; i++) printf("%2.2x ",((char*) frame.data)[i]);
		printf ("\n");
#endif
		/* if (tmp->ast->_state == AST_STATE_UP) */
		/* ast_mutex_lock(&tmp->ast->lock); */
		if (misdn_debug>8) chan_misdn_log("Queueing %d bytes 2 Asterisk\n",len);

		/*if (tmp->bc_te->nojitter)
			manager_flip_buf_bits(frame.data, frame.samples);
		*/
		
		ast_queue_frame(tmp->ast,&frame);
		/* ast_mutex_unlock(&tmp->ast->lock); */
    
	}  else {
		chan_misdn_log ("No ast || ast->pvt || ch\n");
	}
  
	return 0;
}

/** Channel Queue ***/

struct chan_list *cl_find_chan_by_l3id(struct chan_list *list, unsigned long l3id)
{
	struct chan_list *help=list;
	for (;help; help=help->next) {
		/* if (help->bc_te == bc && !help->holded) return help; */
		if (help->l3id == l3id) return help;
	}
  
	if (misdn_debug>3) chan_misdn_log("$$$ find_chan: No channel found with l3id:%x\n",l3id);
  
	return NULL;
}


struct chan_list *cl_find_chan_by_bc_te(struct chan_list *list, struct misdn_bchannel *bc)
{
	struct chan_list *help=list;
	for (;help; help=help->next) {
		/* if (help->bc_te == bc && !help->holded) return help; */
		if (help->bc_te == bc) return help;
	}
  
	if (misdn_debug>3) chan_misdn_log("$$$ find_chan: No channel found for oad:%s dad:%s\n",bc->oad,bc->dad);
  
	return NULL;
}

struct chan_list *cl_find_chan_by_bc_te_holded(struct chan_list *list, struct misdn_bchannel *bc)
{
	struct chan_list *help=list;
	for (;help; help=help->next)
		if (help->holded_bc == bc && help->state == MISDN_HOLDED) return help;
  
	if (misdn_debug>3) chan_misdn_log("$$$ find_chan_holded: No channel found for oad:%s dad:%s\n",
					  bc->oad,bc->dad);

	return NULL;
}

void cl_queue_chan(struct chan_list **list, struct chan_list *chan)
{
	if (misdn_debug>3) chan_misdn_log("* Queuing chan %p\n",chan);
  
	pthread_mutex_lock(&cl_te_lock);
	if (!*list) {
		/* if (misdn_debug>3) chan_misdn_log("* List is empty so new chan is Listroot\n"); */
		*list = chan;
	} else {
		struct chan_list *help=*list;
		for (;help->next; help=help->next); 
		help->next=chan;
		/* if (misdn_debug> 3) chan_misdn_log("* just adding Chan to the End of List\n"); */
	}
	chan->next=NULL;
	pthread_mutex_unlock(&cl_te_lock);
}

void cl_dequeue_chan(struct chan_list **list, struct chan_list *chan) 
{
	/* if (misdn_debug>3)chan_misdn_log("* Dequeuing chan %p from List %p\n",chan, list); */
  
	if (chan->dsp) 
		ast_dsp_free(chan->dsp);
	if (chan->trans)
		ast_translator_free_path(chan->trans);
  
  
	pthread_mutex_lock(&cl_te_lock);
	if (!*list) {
		/* if (misdn_debug >2) chan_misdn_log("List is empty ?\n"); */
		pthread_mutex_unlock(&cl_te_lock);
		return;
	}
  
	if (*list == chan) {
		/* if (misdn_debug>2) chan_misdn_log("* Its the first Chan\n"); */
		*list=(*list)->next;
		pthread_mutex_unlock(&cl_te_lock);
		return ;
	}
  
	{
		struct chan_list *help=*list;
		for (;help->next; help=help->next) {
			if (help->next == chan) {
				/* if (misdn_debug>2)chan_misdn_log("* Its not the first Chan\n"); */
				help->next=help->next->next;
				pthread_mutex_unlock(&cl_te_lock);
				return;
			}
		}
	}
	/* chan_misdn_log("Nothin dequed from chan list\n"); */
	pthread_mutex_unlock(&cl_te_lock);
}

/** Channel Queue End **/

pthread_mutex_t release_lock;

/** Isdn asks us to release channel, pendant to misdn_hangup **/
static void release_chan(struct misdn_bchannel *bc) {
	ast_mutex_lock(&release_lock);
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		if (!ch) ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);
		
		if (ch) {
			if (ch->ast)
				chan_misdn_trace_call(ch->ast,1,"I->*: EVENT_RELEASE\n");
			
			close(ch->pipe[0]);
			close(ch->pipe[1]);
			
			if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
				if (misdn_debug>0) chan_misdn_log("* RELEASING CHANNEL pid:%d ctx:%s dad:%s oad:%s state: %s\n",bc?bc->pid:-1, ch->ast->context, ch->ast->exten,AST_CID_P(ch->ast),misdn_get_ch_state(ch));
				/* ast_setstate(ch->ast, AST_STATE_BUSY); */
				if (misdn_debug>2) chan_misdn_log(" --> * State Down\n");
				/* copy cause */
				send_cause2ast(ch->ast,bc);
				
				MISDN_ASTERISK_TECH_PVT(ch->ast)=NULL;
				
      
				if (ch->ast->_state != AST_STATE_RESERVED) {
					if (misdn_debug> 2) chan_misdn_log(" --> Setting AST State to down\n");
					ast_setstate(ch->ast, AST_STATE_DOWN);
				}
      
				switch(ch->state) {
				case MISDN_EXTCANTMATCH:
				case MISDN_WAITING4DIGS:
				{
					if (misdn_debug > 2 ) chan_misdn_log(" --> * State Wait4dig | ExtCantMatch\n");
					/* ast_queue_hangup(ch->ast); */
					ast_hangup(ch->ast);
	  
				}
				break;
	
				case MISDN_CALLING:
				case MISDN_CALLING_ACKNOWLEDGE:
				case MISDN_DIALING:
				case MISDN_PROGRESS:
					if (misdn_debug > 1 ) chan_misdn_log("* --> In State Calling|Dialing\n");
					if (misdn_debug > 1 ) chan_misdn_log("* --> Queue Hangup\n");
	

					ast_queue_control(ch->ast, AST_CONTROL_HANGUP);
					break;
				case MISDN_CLEANING:
					/*  this state comes out of ast so we mustnt call a ast function ! */
					if (misdn_debug > 1 ) chan_misdn_log("* --> In StateCleaning\n");
					/* ast_hangup(ch->ast); */
	
					break;
				case MISDN_HOLD_DISCONNECT:
					if (misdn_debug > 1 ) chan_misdn_log("* --> In HOLD_DISC\n");
					break;
				default:
					if (misdn_debug > 1 ) chan_misdn_log("* --> In State Default\n");
					if (misdn_debug > 1 ) chan_misdn_log("* --> Queue Hangup\n");
	
	
					if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
						ast_queue_hangup(ch->ast);
					} else {
						chan_misdn_log ("!! Not really queued!\n");
					}
				}
			}
			cl_dequeue_chan(&cl_te, ch);
		
			free(ch);
		} else {
			/*  chan is already cleaned, so exiting  */
		}
	}
	ast_mutex_unlock(&release_lock);
	
}
/*** release end **/

void misdn_transfer_bc(struct chan_list *tmp_ch, struct chan_list *holded_chan)
{
	tmp_ch->state=MISDN_HOLD_DISCONNECT;
  
	ast_moh_stop(AST_BRIDGED_P(holded_chan->ast));

	holded_chan->state=MISDN_CONNECTED;
	holded_chan->holded=0;
	misdn_lib_transfer(mgr_te, holded_chan->bc_te?holded_chan->bc_te:holded_chan->holded_bc);
  
	ast_channel_masquerade(holded_chan->ast, AST_BRIDGED_P(tmp_ch->ast));
}


/** State Machine for sending ISDN Messages to mISDN **/

int check_ret_te(struct misdn_lib *mgr,struct misdn_bchannel * bc, enum te_event_e event, int ret)
{
	switch (event) {
	case EVENT_SETUP:
		switch(ret) {
		case -ENOCHAN:
			return 1;
		default:
			break;
		}
		break;
	default:
		break;
	}
	return 0;
}

int check_ret (struct misdn_lib *mgr,struct misdn_bchannel * bc, enum te_event_e event, int ret)
{
	if (bc->stack->mode == TE_MODE) return check_ret_te(mgr, bc, event, ret);
  
	switch (event) {
	case EVENT_SETUP_ACKNOWLEDGE:

		switch(ret) {
		case -ENOCHAN:
			misdn_lib_send_event(mgr, bc, EVENT_RELEASE_COMPLETE);
			return 1;
			break;
		}
		break;
	case EVENT_SETUP:
		switch(ret) {
		case -ENOCHAN:
			return 1;
			break;
		}
		break;
	default:
		break;
	}
 
	return 0;
}

/** Send mISDN State Machine End **/

void do_immediate_setup(struct misdn_bchannel *bc,struct chan_list *ch , struct ast_channel *ast)
{
	char predial[256]="";
	char *p = predial;
  
	struct ast_frame fr;
  
	strncpy(predial, ast->exten, sizeof(predial) -1 );
  
	ch->state=MISDN_DIALING;

	/* manager_bchannel_setup(bc); */
  
	if (bc->stack->mode == NT_MODE) {
		int ret; 
		ret = misdn_lib_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
	} else {
		int ret;
		/* ret = misdn_lib_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE ); */
		if (bc->stack->ptp) {
			ret = misdn_lib_send_event(mgr_te, bc, EVENT_SETUP_ACKNOWLEDGE );
		} else {
			ret = misdn_lib_send_event(mgr_te, bc, EVENT_PROCEEDING );
		}
	}

	manager_send_tone(mgr_te, bc,TONE_DIAL);  
  
	if (misdn_debug >0)
		chan_misdn_log("* Starting Ast ctx:%s dad:%s oad:%s with 's' extension\n", ast->context, ast->exten, AST_CID_P(ast));
  
	strcpy(ast->exten,"s");
  
	if (ast_pbx_start(ast)<0) {
		ast=NULL;
		manager_send_tone(mgr_te,bc,TONE_BUSY);
		if (bc->stack->mode == NT_MODE)
			misdn_lib_send_event(mgr_te, bc, EVENT_RELEASE_COMPLETE );
		else
			misdn_lib_send_event(mgr_te, bc, EVENT_DISCONNECT );
	}
  
  
	while (!ast_strlen_zero(p) ) {
		fr.frametype = AST_FRAME_DTMF;
		fr.subclass = *p ;
		fr.src=NULL;
		fr.data = NULL ;
		fr.datalen = 0;
		fr.samples = 0 ;
		fr.mallocd =0 ;
		fr.offset= 0 ;

		if (ch->ast && MISDN_ASTERISK_PVT(ch->ast) && MISDN_ASTERISK_TECH_PVT(ch->ast)) {
			ast_queue_frame(ch->ast, &fr);
		}
		p++;
	}
}



void send_cause2ast(struct ast_channel *ast, struct misdn_bchannel*bc) {
	
	ast->hangupcause=bc->cause;
	
	switch ( bc->cause) {
		
	case 1: /** Congestion Cases **/
	case 2:
	case 3:
 	case 4:
 	case 22:
 	case 27:
		if (misdn_debug > 0) chan_misdn_log(" --> * SEND: Queue Congestion pid:%d\n", bc?bc->pid:-1);
		
		ast_queue_control(ast, AST_CONTROL_CONGESTION);
		break;
		
	case 21:
	case 17: /* user busy */
		if (misdn_debug > 0) chan_misdn_log(" --> * SEND: Queue Busy pid:%d\n", bc?bc->pid:-1);
		
		ast_queue_control(ast, AST_CONTROL_BUSY);
		
		break;
	}
}

/************************************************************/
/*  Receive Events from isdn_lib  here                     */
/************************************************************/
enum te_event_response_e
cb_te_events(struct misdn_lib *mgr, enum te_event_e event, struct misdn_bchannel *bc, void *user_data)
{
  
	if (event != EVENT_BCHAN_DATA) { /*  Debug Only Non-Bchan */
		if (misdn_debug >0) chan_misdn_log("I IND :%s\tpid:%d\tmode:%s\taddr:%x\tport:%d\n",manager_isdn_get_info( event),bc?bc->pid:-1, bc->stack->mode==NT_MODE?"NT":"TE", bc->addr,bc->stack->port);
    
		if (misdn_debug >0) chan_misdn_log(" --> l3id:%x cause:%d dad:%s oad:%s channel:%d port:%d\n",bc->l3_id, bc->cause, bc->dad,bc->oad,bc->channel, bc->stack->port);
	}
  
	switch (event) {
	case EVENT_NEW_L3ID:
#if 1
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		if (ch) {
			ch->l3id=bc->l3_id;
		} else {
			chan_misdn_log("Chan not existing at the moment\n");
		}
	}
#endif 
	break;
    
	case EVENT_DTMF_TONE:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		/*  sending INFOS as DTMF-Frames :) */
		struct ast_frame fr;
		memset(&fr, 0 , sizeof(fr));
		fr.frametype = AST_FRAME_DTMF;
		fr.subclass = bc->dtmf ;
		fr.src=NULL;
		fr.data = NULL ;
		fr.datalen = 0;
		fr.samples = 0 ;
		fr.mallocd =0 ;
		fr.offset= 0 ;

		if (!ch) {
			chan_misdn_log("!! Got DTMF Tone without channel on port:%d\n",bc->stack->port);
			return -1;
		}

		if (ch->ast && MISDN_ASTERISK_PVT(ch->ast) && MISDN_ASTERISK_TECH_PVT(ch->ast)) { 
			if ( misdn_debug > 1) chan_misdn_log(" --> * SEND: DTMF :%c\n",bc->dtmf);
	
			ast_queue_frame(ch->ast, &fr);
	
		}

#if 0
		if (misdn_dynamic_crypting)
			misdn_crypt_event(ch, CRYPT_EVENT_MISDN_DIGIT, (void*) & bc->dtmf);
#endif
	}
    
	break;
	case EVENT_STATUS:
		if (misdn_debug>0) chan_misdn_log(" --> cause:%d\n",bc->cause);
		break;
    
	case EVENT_INFORMATION:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);

		if (ch) {
			int stop_tone;
			misdn_cfg_get( 0, MISDN_GEN_STOP_TONE, &stop_tone, sizeof(int));
			if ( stop_tone && bc->tone != TONE_NONE) {
				/* if (misdn_debug > 2) chan_misdn_log(" --> Stoping Tones after first digit\n"); */
				manager_send_tone(mgr,bc,TONE_NONE);
			}
		}
		
		if (misdn_debug >1) chan_misdn_log(" --> keypad: %s info_dad: %s\n",bc->keypad,bc->info_dad);
		if (ch->state == MISDN_WAITING4DIGS ) {
			/*  Ok, incomplete Setup, waiting till extension exists */
	
			strcat(bc->dad,bc->info_dad);
			if (!ch->ast) { chan_misdn_log("!! ERROR: Infos without ast-obj??\n");return RESPONSE_ERR;}
			strcpy(ch->ast->exten, bc->dad);


			/* chan_misdn_log("Can Match Extension: dad:%s oad:%s\n",bc->dad,bc->oad); */
			if (misdn_debug >0) chan_misdn_log("Can Match Extension: dad:%s oad:%s\n",bc->dad,bc->oad);


			char bc_context[BUFFERSIZE];
			misdn_cfg_get( bc->stack->port, MISDN_CFG_CONTEXT, bc_context, BUFFERSIZE);
			if(!ast_canmatch_extension(ch->ast, bc_context, bc->dad, 1, bc->oad)) {
				chan_misdn_log("Extension can never match, so disconnecting\n");
				manager_send_tone(mgr,bc,TONE_BUSY);
				ch->state=MISDN_EXTCANTMATCH;
				bc->out_cause=1;
				if (bc->stack->mode == NT_MODE)
					misdn_lib_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
				else
					misdn_lib_send_event(mgr, bc, EVENT_DISCONNECT );
				break;
			}
			if (ast_exists_extension(ch->ast, bc_context, bc->dad, 1, bc->oad)) {
				ch->state=MISDN_DIALING;
	  
				/* manager_send_tone(mgr, bc,TONE_NONE); */
				/* stop_bc_tones(ch); */
				manager_send_tone(mgr,bc,TONE_NONE);
				if (misdn_debug >0) chan_misdn_log(" --> * Starting Ast ctx:%s dad:%s oad:%s\n", ch->ast->context, ch->ast->exten, AST_CID_P(ch->ast));
				if (ast_pbx_start(ch->ast)<0) {
					chan_misdn_log("ast_pbx_start returned <0 in INFO\n");
					manager_send_tone(mgr,bc,TONE_BUSY);
					if (bc->stack->mode == NT_MODE)
						misdn_lib_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
					else
						misdn_lib_send_event(mgr, bc, EVENT_DISCONNECT );
				}
			}
	
		} else {
			/*  sending INFOS as DTMF-Frames :) */
			struct ast_frame fr;
			fr.frametype = AST_FRAME_DTMF;
			fr.subclass = bc->info_dad[0] ;
			fr.src=NULL;
			fr.data = NULL ;
			fr.datalen = 0;
			fr.samples = 0 ;
			fr.mallocd =0 ;
			fr.offset= 0 ;

			if (ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
	  
				int digits;
				misdn_cfg_get( 0, MISDN_GEN_APPEND_DIGITS2EXTEN, &digits, sizeof(int));
				if (ch->state != MISDN_CONNECTED && digits) {
					/* if (misdn_debug>1) chan_misdn_log(" --> Appending Digits %s\n",bc->info_dad); */
					strcat(bc->dad,bc->info_dad);
					strcpy(ch->ast->exten, bc->dad);
					ast_cdr_update(ch->ast);
				} else {
					/* if (misdn_debug>1) chan_misdn_log(" --> Not Appending Digits %s\n",bc->info_dad); */
				}
	  
				ast_queue_frame(ch->ast, &fr);

			}
			else
				chan_misdn_log("Ast Channel doesn't exist anymore @ ast_queue_frame??\n");
		}
	}
	break;
	case EVENT_SETUP:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		
		if (ch && ch->state != MISDN_NOTHING && ch->state != MISDN_HOLDED) {
			if (misdn_debug>0) chan_misdn_log(" --> Ignoring Call we have already one\n");
			return RESPONSE_IGNORE_SETUP_WITHOUT_CLOSE; /*  Ignore MSNs which are not in our List */
		}
	}
	
	int msn_valid = misdn_cfg_is_msn_valid(bc->stack->port, bc->dad);
	if (bc->stack->mode == TE_MODE && ! msn_valid) {
		if (misdn_debug>0) chan_misdn_log(" --> Ignoring Call, its not in our MSN List\n");
		return RESPONSE_IGNORE_SETUP; /*  Ignore MSNs which are not in our List */
	}
    
	if (misdn_debug>0) chan_misdn_log(" --> Keypad: %s\n",bc->keypad);
	
	print_bearer(bc);
    
	{
		struct chan_list *ch=init_chan_list();
		struct ast_channel *chan;
		char name[128];
		if (!ch) { chan_misdn_log("cb_te_events: malloc for chan_list"); return 0;}
		
		ch->bc_te = bc;
		ch->l3id=bc->l3_id;
		ch->orginator = ORG_MISDN;
		
		{
			char prefix[BUFFERSIZE]="";
			char tmp[255];
			/* if ( !strlen(bc->dad) && strlen(bc->keypad)) strcpy(bc->dad,bc->keypad); */
			switch( bc->onumplan ) {
			case NUMPLAN_INTERNATIONAL:
				misdn_cfg_get( bc->stack->port, MISDN_CFG_INTERNATPREFIX, prefix, BUFFERSIZE);
				if (misdn_debug>0) chan_misdn_log(" --> orig Numplan: Internat. prefix %s\n", prefix);
				break;
	  
			case NUMPLAN_NATIONAL:
				misdn_cfg_get( bc->stack->port, MISDN_CFG_NATPREFIX, prefix, BUFFERSIZE);
				if (misdn_debug>0) chan_misdn_log(" --> orig Numplan: Nat. prefix %s\n", prefix);
				break;
	  
	  
			case NUMPLAN_SUBSCRIBER:
				/*  dunno what to do here ? */
				if (misdn_debug>0) chan_misdn_log(" --> orig Numplan: Subscriber\n");
				break;

			case NUMPLAN_UNKNOWN:
				if (misdn_debug>0) chan_misdn_log(" --> orig Numplan: Unknown\n");
				break;
			default:
				if (misdn_debug>0) chan_misdn_log(" --> orig Numplan: No Numplan\n");
				break;
			}
	
			strcpy(tmp,prefix);
			strcat(tmp,bc->oad);
			strcpy(bc->oad,tmp);
	
			if (!ast_strlen_zero(bc->oad))
				sprintf(name,"mISDN/%d/%s",bc->stack->port,bc->oad);
			else
				sprintf(name,"mISDN/%d",bc->stack->port);


			if (!ast_strlen_zero(bc->dad))
				strcpy(bc->orig_dad,bc->dad);
			
			if ( ast_strlen_zero(bc->dad) && !ast_strlen_zero(bc->keypad)) strcpy(bc->dad,bc->keypad);
			prefix[0] = 0;
			
			switch( bc->dnumplan ) {
			case NUMPLAN_INTERNATIONAL:
				misdn_cfg_get( bc->stack->port, MISDN_CFG_INTERNATPREFIX, prefix, BUFFERSIZE);
				if (misdn_debug>0) chan_misdn_log(" --> dest Numplan: Internat. prefix %s\n", prefix);
				break;
				
			case NUMPLAN_NATIONAL:
				misdn_cfg_get( bc->stack->port, MISDN_CFG_NATPREFIX, prefix, BUFFERSIZE);
				if (misdn_debug>0) chan_misdn_log(" --> dest Numplan: Nat. prefix %s\n", prefix);
				break;
				
				
			case NUMPLAN_SUBSCRIBER:
				/* dunno what to do here ? */
				if (misdn_debug>0) chan_misdn_log(" --> dest Numplan: Subscriber\n");
				break;
				
			case NUMPLAN_UNKNOWN:
				if (misdn_debug>0) chan_misdn_log(" --> dest Numplan: Unknown\n");
				break;
			default:
				if (misdn_debug>0) chan_misdn_log(" --> dest Numplan: No Numplan\n");
				break;
			}
			
			strcpy(tmp,prefix);
			strcat(tmp,bc->dad);
			strcpy(bc->dad,tmp);
		
			char bc_context[BUFFERSIZE];
			misdn_cfg_get( bc->stack->port, MISDN_CFG_CONTEXT, bc_context, BUFFERSIZE);
			chan=misdn_new(ch, AST_STATE_RING,name ,bc_context, bc->dad, bc->oad, AST_FORMAT_ALAW,  bc->channel>0?bc->channel:misdn_lib_free_bc_avail(mgr_te, bc->stack->port));
			
		}

		ch->ast = chan;

		chan_misdn_trace_call(chan,1,"I->*: EVENT_SETUP\n");
		
#ifndef ASTERISK_STABLE
		if ( bc->pres ) {
			if (misdn_debug > 0) chan_misdn_log(" --> Number Not Screened\n");
			chan->cid.cid_pres=AST_PRES_ALLOWED_USER_NUMBER_NOT_SCREENED;
		}  else {
			if (misdn_debug > 0) chan_misdn_log(" --> Number Screened\n");
			chan->cid.cid_pres=AST_PRES_ALLOWED_USER_NUMBER_PASSED_SCREEN;
		}
      
		pbx_builtin_setvar_helper(chan, "TRANSFERCAPABILITY", ast_transfercapability2str(bc->capability));
		chan->transfercapability=bc->capability;
      
		switch (bc->capability) {
		case INFO_CAPABILITY_DIGITAL_UNRESTRICTED:
			if (misdn_debug >0 ) chan_misdn_log(" --> Setting Digital\n");
			pbx_builtin_setvar_helper(chan,"CALLTYPE","DIGITAL");
			break;
		default:
			pbx_builtin_setvar_helper(chan,"CALLTYPE","SPEECH");
		}
#else
		if (bc->digital) {
			ast_set_flag(chan,AST_FLAG_DIGITAL);
			if (misdn_debug >0 ) chan_misdn_log(" --> Setting Digital\n");
			pbx_builtin_setvar_helper(chan,"CALLTYPE","DIGITAL");
		} else {
			pbx_builtin_setvar_helper(chan,"CALLTYPE","SPEECH");
		}
#endif

      
		/** queue new chan **/
		cl_queue_chan(&cl_te, ch) ;
      


#ifndef ASTERISK_STABLE
		/* Check for Pickup Request first */
		if (!strcmp(chan->exten, ast_pickup_ext())) {
			int ret;/** Sending SETUP_ACK**/
			ret = misdn_lib_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
	
			if (ast_pickup_call(chan)) {
				if (misdn_debug>0)chan_misdn_log("Nothing to pick up\n");
	  
				ast_hangup(chan);
			} else {
	  
				if (misdn_debug>0)chan_misdn_log("I picked Up, senbing ACK to phone port:%d\n",bc->stack->port);
				/* ret = misdn_lib_send_event(mgr, bc, EVENT_ALERTING); */
				ch->state = MISDN_CALLING_ACKNOWLEDGE;
	  
				ch->ast=NULL;
	  
				ast_setstate(chan, AST_STATE_DOWN);
				ast_hangup(chan); 
	  
				break;
			}
		}
#endif
		/*
		  added support for s extension hope it will help those poor cretains
		  which haven't overlap dial.
		*/
		{
			
			misdn_cfg_get( bc->stack->port, MISDN_CFG_LANGUAGE, chan->language, sizeof(chan->language));
			int ai;
			misdn_cfg_get( bc->stack->port, MISDN_CFG_ALWAYS_IMMEDIATE, &ai, sizeof(ai));
			if ( ai ) {
				if (misdn_debug>0)chan_misdn_log(" --> Doing always_immediate Setup \n");
				do_immediate_setup(bc, ch , chan);
				break;
			}

			int immediate;
			misdn_cfg_get( bc->stack->port, MISDN_CFG_IMMEDIATE, &immediate, sizeof(int));
			
			if (ast_strlen_zero(bc->orig_dad) && immediate ) {
				if (misdn_debug>0)chan_misdn_log(" --> Doing immediate Setup\n");
				do_immediate_setup(bc, ch , chan);
				break;
			}
			
		}

		/* chan_misdn_log("Can Match Extension: Ast->exten:%s Ast->context:%s Context:%s dad:%s oad:%s\n", ch->ast->exten, ch->ast->context, get_context(bc), bc->dad,bc->oad); */

		/** Now after we've finished configuring our channel object
		    we'll jump into the dialplan **/
		
		char bc_context[BUFFERSIZE];
		misdn_cfg_get( bc->stack->port, MISDN_CFG_CONTEXT, bc_context, BUFFERSIZE);
		if(!ast_canmatch_extension(ch->ast, bc_context, bc->dad, 1, bc->oad)) {
			chan_misdn_log("Extension can never match, so disconnecting\n");
			manager_send_tone(mgr,bc,TONE_BUSY);
			ch->state=MISDN_EXTCANTMATCH;
			bc->out_cause=1;
			if (bc->stack->mode == NT_MODE)
				misdn_lib_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
			else
				misdn_lib_send_event(mgr, bc, EVENT_DISCONNECT );
			break;
		}
      
		if (ast_exists_extension(ch->ast, bc_context, bc->dad, 1, bc->oad)) {
			ch->state=MISDN_DIALING;
	
			if (bc->stack->mode == NT_MODE) {
				int ret; 
				ret = misdn_lib_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
			} else {
				int ret;
				ret= misdn_lib_send_event(mgr, bc, EVENT_PROCEEDING );
			}
	
			if (misdn_debug >0) chan_misdn_log(" --> * Starting Ast ctx:%s dad:%s oad:%s\n", chan->context, chan->exten, AST_CID_P(chan));
			if (ast_pbx_start(chan)<0) {
				chan_misdn_log("ast_pbx_start returned <0 in SETUP\n");
				chan=NULL;
				manager_send_tone(mgr,bc,TONE_BUSY);
				if (bc->stack->mode == NT_MODE)
					misdn_lib_send_event(mgr, bc, EVENT_RELEASE_COMPLETE );
				else
					misdn_lib_send_event(mgr, bc, EVENT_DISCONNECT );
			}
		} else {
			int ret= misdn_lib_send_event(mgr, bc, EVENT_SETUP_ACKNOWLEDGE );
			if ( check_ret (mgr, bc, EVENT_SETUP_ACKNOWLEDGE, ret)) {
				misdn_lib_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
			}
			/*  send tone to phone :) */
	
			int stop_tone;
			misdn_cfg_get( 0, MISDN_GEN_STOP_TONE, &stop_tone, sizeof(int));
			if ( (!ast_strlen_zero(bc->dad)) && stop_tone ) 
				manager_send_tone(mgr, bc,TONE_NONE);
			else
				manager_send_tone(mgr, bc,TONE_DIAL);
	
			ch->state=MISDN_WAITING4DIGS;
		}
      
	}
	break;
	case EVENT_SETUP_ACKNOWLEDGE:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		if (bc->stack->mode == TE_MODE ) {
	
			if (ch) {
				/* should i do something here? */
			}
		}
      
		ch->state = MISDN_CALLING_ACKNOWLEDGE;
		if (!ast_strlen_zero(bc->infos_pending)) {
			/* TX Pending Infos */
			strcat(bc->dad,bc->infos_pending);
			strcpy(ch->ast->exten, bc->dad);
			strcpy(bc->info_dad, bc->infos_pending);
			strcpy(bc->infos_pending,"");

			misdn_lib_send_event(mgr_te, bc, EVENT_INFORMATION);
		}
	}
	break;
	case EVENT_PROCEEDING:
		;
		int bconn;
		misdn_cfg_get( bc->stack->port, MISDN_CFG_EARLY_BCONNECT, &bconn, sizeof(int));
		if ( !bc->digital && bconn) {
			struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
			if (ch && ch->ast) {
				start_bc_tones(ch);  /*  Only */
				/* if (misdn_debug >0) chan_misdn_log(" --> * SEND: Queue Progress pid:%d\n", bc?bc->pid:-1); */
	
			}
		}
      
		break;
	case EVENT_PROGRESS:
		if (bc->stack->mode == TE_MODE ) {
			struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
			if (ch && ch->ast) {
				start_bc_tones(ch);  /*  Only */
				if (misdn_debug >0) chan_misdn_log(" --> * SEND: Queue Progress pid:%d\n", bc?bc->pid:-1);

				ast_queue_control(ch->ast, AST_CONTROL_PROGRESS);

				ch->state=MISDN_PROGRESS;
			}
		}
		break;
    
	case EVENT_RELEASE:
		if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc->cause);
    
		{
			struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
			if (ch ){
	
				switch ( bc->cause) {
					
				case -1:
					/*
					  OK, it really sucks, this is a RELEASE from NT-Stack So we take
					  it and return easylie, It seems that we've send a DISCONNECT
					  before, so we should RELEASE_COMPLETE after that Disconnect
					  (looks like ALERTING State at misdn_hangup !!
					*/
					return RESPONSE_OK;
					break;
				}
				
				
				bc->out_cause=16;
	
				stop_bc_tones(ch);
				release_chan(bc);
			}
		}
		break;
	case EVENT_RELEASE_COMPLETE:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		
		if (ch ){
			if (misdn_debug > 0) chan_misdn_log("--> cause %d\n",bc->cause);
			stop_bc_tones(ch);
			release_chan(bc);
		} else {
			chan_misdn_log("Didn't RELEAS_COMPLETED cause there was no chan_list ??\n");
		}
      
	}
	break;
	case EVENT_ALERTING:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
      
		if (ch) ch->state = MISDN_ALERTING;
      
		if (ch && ch->ast) {
			chan_misdn_trace_call(ch->ast,1,"I->*: EVENT_ALERTING\n");
			if (misdn_debug>0) chan_misdn_log(" --> * SEND: Queue Frame Ringing pid:%d\n", bc?bc->pid:-1);
			
			ast_queue_control(ch->ast, AST_CONTROL_RINGING);
			ast_setstate(ch->ast, AST_STATE_RINGING);

			int bconn;
			misdn_cfg_get( bc->stack->port, MISDN_CFG_EARLY_BCONNECT, &bconn, sizeof(int));
			if ( !bc->digital && bconn) {
				start_bc_tones(ch);  /*  Only */
			}
		}
		
		
	}
	break;
	case EVENT_CONNECT:
		misdn_lib_send_event(mgr,bc,EVENT_CONNECT_ACKNOWLEDGE);
	case EVENT_CONNECT_ACKNOWLEDGE:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		bc->state=STATE_CONNECTED;
		
		ch->l3id=bc->l3_id;
		
		start_bc_tones(ch);
		if (ch && ch->ast) {
			chan_misdn_trace_call(ch->ast,1,"I->*: EVENT_CONNECT\n");
			
			
			if (misdn_debug > 0) chan_misdn_log(" --> * SEND: Queue Answer pid:%d\n", bc?bc->pid:-1);
			
			ch->state = MISDN_CONNECTED;
			ast_queue_control(ch->ast, AST_CONTROL_ANSWER);
			
		}
	}
	break;
	case EVENT_DISCONNECT:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		struct chan_list *holded_ch = cl_find_chan_by_bc_te_holded(cl_te, bc);
		
		if (ch && (misdn_debug > 1)) {
			chan_misdn_log (" --> ch->state %s, bc->holded %d\n",misdn_get_ch_state(ch),bc->holded );
		}
		if (holded_ch && ch->state == MISDN_CONNECTED) {
			if (misdn_debug > 0) chan_misdn_log("Found holded_bc so transfering\n");
			misdn_transfer_bc(ch, holded_ch) ;
			bc->l3_id = ch->l3id;
			misdn_lib_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
			break;
		}
		/* lets have a try */
		if ( bc->stack->mode == NT_MODE) {
			manager_send_tone(mgr_te,bc,TONE_BUSY);
		}else {
			if (ch) stop_bc_tones(ch);
		}
		
		if (misdn_debug > 0) chan_misdn_log(" --> cause %d\n",bc->cause);

		if (ch && ch->ast && MISDN_ASTERISK_PVT(ch->ast)) {
			send_cause2ast(ch->ast,bc);
		}
		
		bc->out_cause=16;
		misdn_lib_send_event(mgr,bc,EVENT_RELEASE);
	}
	break;
	case EVENT_BCHAN_DATA:
	{
		struct chan_list *tmp=cl_find_chan_by_bc_te(cl_te, bc);
		if (!tmp) break; 

		if (tmp->ast) 
			chan_misdn_trace_call(tmp->ast,3,"I->*: EVENT_B_DATA len=%d\n",bc->bframe_len);
		
		if ( tmp->bc_te->digital || bc->nojitter) {
			/* chan_misdn_log("Flipping Buf\n"); */
			misdn_tx2ast_frm(tmp, bc->bframe, bc->bframe_len );
		} else {
			int len=bc->bframe_len;
			
			if (bc->bframe_len > ibuf_freecount(bc->astbuf)) {
				ast_log(LOG_DEBUG, "sbuf overflow!\n");
				len=ibuf_freecount(bc->astbuf);
			}
			
			ibuf_memcpy_w(bc->astbuf, bc->bframe, len);
			
			{
				char blah[1]="\0";
				
				
				{
					struct timeval tv;
					gettimeofday(&tv,NULL);
					/* chan_misdn_log("Time [misdn_write]: %d.%d delta:%d len:%d\n",tv.tv_sec,tv.tv_usec, tv.tv_usec-bc->time_usec, bc->bframe_len); */
					
					if (tv.tv_usec % 10000 > 0 ) {
						write(tmp->pipe[1], blah,sizeof(blah));
						bc->time_usec=tv.tv_usec;
					}
				}
				

			}
		}
		
	}
	break;
	case EVENT_TIMEOUT:
		break; /* Ignore now .. */
		{
			struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
			if (ch)
				switch (ch->state) {
				case MISDN_CALLING:
					chan_misdn_log("GOT TIMOUT AT CALING pid:%d\n", bc?bc->pid:-1);
					break;
				case MISDN_DIALING:
				case MISDN_PROGRESS:
					break;
				default:
					misdn_lib_send_event(mgr,bc,EVENT_RELEASE_COMPLETE);
				}
		}
		break;
	case EVENT_CLEANUP:
	{
		struct chan_list *ch=cl_find_chan_by_bc_te(cl_te, bc);
		if (ch) stop_bc_tones(ch);
		release_chan(bc);
	}
	break;
    
	/***************************/
	/** Suplementary Services **/
	/***************************/
	case EVENT_RETRIEVE:
	{
		struct chan_list *ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);

		if (ch ) {

			/** really ? **/
			ch->l3id=bc->l3_id;
	
			ch->state = MISDN_CONNECTED;
			ch->holded = 0;
	
			if (AST_BRIDGED_P(ch->ast)) {
				ast_moh_stop(AST_BRIDGED_P(ch->ast));
	
			} else { /* Other ASt End not there anymore ? */
				chan_misdn_log("Other ASt End not there anymore ?\n");
			}

			ch->bc_te = ch->holded_bc ;
			ch->holded_bc = NULL;

			if ( misdn_lib_send_event(mgr, bc, EVENT_RETRIEVE_ACKNOWLEDGE) < 0)
				misdn_lib_send_event(mgr, bc, EVENT_RETRIEVE_REJECT);

			start_bc_tones(ch);
		
		} else {
			chan_misdn_log("Got Retrieve without holded Channel ? .. Rejecting\n");
			misdn_lib_send_event(mgr, bc, EVENT_RETRIEVE_REJECT);
		}
	}
	break;
    
	case EVENT_HOLD:
	{
		struct chan_list *ch=cl_find_chan_by_l3id(cl_te, bc->l3_id);
		if (ch && ch->ast) {
			int hold_allowed;
			misdn_cfg_get( bc->stack->port, MISDN_CFG_HOLD_ALLOWED, &hold_allowed, sizeof(int));
			
			if (!hold_allowed) {
				chan_misdn_log("Hold not allowed on port:%d\n", bc->stack->port);
				misdn_lib_send_event(mgr, bc, EVENT_HOLD_REJECT);
				break;
			}
			
			if (AST_BRIDGED_P(ch->ast)){
				ast_moh_start(AST_BRIDGED_P(ch->ast), NULL);
				misdn_lib_send_event(mgr, bc, EVENT_HOLD_ACKNOWLEDGE);
				
				ch->state = MISDN_HOLDED;
				ch->holded = 1;
				ch->holded_bc = bc ;
				ch->bc_te=NULL;
			}
			else {
				misdn_lib_send_event(mgr, bc, EVENT_HOLD_REJECT);
				chan_misdn_log("We aren't bridged to anybody\n");
			}
		} else {
			misdn_lib_send_event(mgr, bc, EVENT_HOLD_REJECT);
			chan_misdn_log("No ast !\n");
		}
	} 
	break;
	default:
		chan_misdn_log("Got Unknown Event\n");
		break;
	}
  
	return RESPONSE_OK;
}

/** TE STUFF END **/

/******************************************
 *
 *   Asterisk Channel Endpoint END
 *
 *
 *******************************************/


int load_module()
{
	char ports[256]="";
	
	int max=misdn_lib_maxports_get();

	if (max<=0) {
		ast_log(LOG_ERROR, "Unable to initialize mISDN\n");
		return -1;
	}


	misdn_cfg_init(max);

	/* FIXME */
	misdn_cfg_get( 0, MISDN_GEN_CLEAR_L3, &default_clearl3, sizeof(int));
	{
		char tempbuf[BUFFERSIZE];
		misdn_cfg_get( 0, MISDN_GEN_TRACEFILE, tempbuf, BUFFERSIZE);
		if (strlen(tempbuf))
			tracing = 1;
	}

	misdn_cfg_get( 0, MISDN_GEN_DEBUG, &misdn_debug, sizeof(int));


#ifndef ASTERISK_STABLE 
	{
		if (ast_channel_register(&misdn_tech)) {
			ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
			unload_module();
			return -1;
		}
	}
#else
	if (ast_channel_register(type, tdesc, prefformat, misdn_request)) {
		ast_log(LOG_ERROR, "Unable to register channel class %s\n", type);
		unload_module();
		return -1;
	}
#endif
  
	ast_cli_register(&cli_send_display);
	ast_cli_register(&cli_send_digit);
	ast_cli_register(&cli_toggle_echocancel);
	ast_cli_register(&cli_set_tics);

	ast_cli_register(&cli_show_cls);
	ast_cli_register(&cli_show_cl);
	ast_cli_register(&cli_show_config);
	ast_cli_register(&cli_show_port);
	ast_cli_register(&cli_show_stacks);
	ast_cli_register(&cli_show_fullstacks);
	ast_cli_register(&cli_flush_stack);
	ast_cli_register(&cli_restart_port);
	ast_cli_register(&cli_port_up);
	ast_cli_register(&cli_set_debug);
	ast_cli_register(&cli_set_crypt_debug);
	ast_cli_register(&cli_reload);

  
	ast_register_application("misdn_set_opt", misdn_set_opt_exec, "misdn_set_flags",
				 "misdn_set_opt(:<opt><optarg>:<opt><optarg>..):\n"
				 "Sets mISDN opts. and optargs\n"
				 "\n"
		);
  
	pthread_mutex_init(&cl_te_lock, NULL);
	pthread_mutex_init(&release_lock, NULL);
	
  
	msg_init();
	debug_init(0 , NULL, NULL, NULL);
	/* debug_init( 0xffffffff & ~DBGM_MSG, NULL, NULL, NULL); */
  
	misdn_cfg_get_ports_string(ports);
	chan_misdn_log("Got : %s from get_ports\n",ports);
	{
		mgr_te=misdn_lib_init(ports, cb_te_events, NULL);
		if (!mgr_te) chan_misdn_log("No te ports initialized\n");
	}

#if 0
	if (misdn_dynamic_crypting)
		crypt_init();
#endif
	return 0;
}



int unload_module()
{
	/* First, take us out of the channel loop */
	chan_misdn_log("-- Unregistering mISDN Channel Driver --\n");

	ast_cli_unregister(&cli_send_display);
	ast_cli_unregister(&cli_send_digit);
	ast_cli_unregister(&cli_toggle_echocancel);
	ast_cli_unregister(&cli_set_tics);
  
	ast_cli_unregister(&cli_show_cls);
	ast_cli_unregister(&cli_show_cl);
	ast_cli_unregister(&cli_show_config);
	ast_cli_unregister(&cli_show_port);
	ast_cli_unregister(&cli_show_stacks);
	ast_cli_unregister(&cli_show_fullstacks);
	ast_cli_unregister(&cli_flush_stack);
	ast_cli_unregister(&cli_restart_port);
	ast_cli_unregister(&cli_set_debug);
	ast_cli_unregister(&cli_set_crypt_debug);
	ast_cli_unregister(&cli_reload);
	ast_cli_unregister(&cli_port_up);
	/* ast_unregister_application("misdn_crypt"); */
	ast_unregister_application("misdn_set_opt");
  
#ifndef ASTERISK_STABLE
	ast_channel_unregister(&misdn_tech);
#else
	ast_channel_unregister(type);
#endif
  
	misdn_cfg_destroy();
	misdn_lib_destroy(mgr_te);
  
	return 0;
}

int usecount()
{
	int res;
	ast_mutex_lock(&usecnt_lock);
	res = usecnt;
	ast_mutex_unlock(&usecnt_lock);
	return res;
}

char *description()
{
	return desc;
}

char *key()
{
	return ASTERISK_GPL_KEY;
}

void chan_misdn_log(char *tmpl, ...)
{
	va_list ap;
	char buf[1024]; /* maxbuflog */
  
	va_start(ap, tmpl);
  
	vsprintf( buf, tmpl, ap );
  
	va_end(ap);

	ast_console_puts(buf);
	
	if (tracing) {
		time_t tm = time(NULL);
		char *tmp=ctime(&tm),*p;
		char file[BUFFERSIZE];
		misdn_cfg_get( 0, MISDN_GEN_TRACEFILE, file, BUFFERSIZE);
		FILE *fp= fopen(file, "a+");

		p=strchr(tmp,'\n');
		if (p) *p=':';
    
		if (!fp) {
			ast_console_puts("Error opening Tracefile: ");
			ast_console_puts(strerror(errno));
			ast_console_puts("\n");
			return ;
		}
		
		fputs(tmp,fp);
		fputs(" ", fp);
		fputs(buf, fp);

		fclose(fp);
	}
}


void chan_misdn_trace_call(struct ast_channel *chan, int debug, char *tmpl, ...)
{
	va_list ap;
	char buf[1024]; /* maxbuflog */
	char name[1024];

	int trace;
	misdn_cfg_get( 0, MISDN_GEN_TRACE_CALLS, &trace, sizeof(int));
	if (!trace) return ;
	
	if (misdn_debug < debug) return ; 
	
	char tracedir[BUFFERSIZE];
	misdn_cfg_get( 0, MISDN_GEN_TRACE_DIR, tracedir, BUFFERSIZE);
#ifdef ASTERISK_STABLE
	sprintf(name,"%s/%s.%s",tracedir,  chan->uniqueid, chan->callerid );
#else
	sprintf(name,"%s/%s.%s",tracedir, chan->uniqueid, chan->cid.cid_num );
#endif
	
	va_start(ap, tmpl);
	
	vsprintf( buf, tmpl, ap );
	
	va_end(ap);
	
	time_t tm = time(NULL);
	char *tmp=ctime(&tm),*p;
	FILE *fp= fopen(name, "a");
	int fd;
	
	if (!fp) {
		ast_console_puts("Error opening Tracefile");
		ast_console_puts(strerror(errno));
		ast_console_puts("\n");
		return ;
	}
	
	fd=fileno(fp) ;
	
	flock(fd, LOCK_EX);
	
	p=strchr(tmp,'\n');
	if (p) *p=':';
	
	
	
	fputs(tmp,fp);
	fputs(" ", fp);
	fputs(buf, fp);

	flock(fd, LOCK_UN);
	
	fclose(fp);
	
}


/*** SOME APPS ;)***/

static int misdn_set_opt_exec(struct ast_channel *chan, void *data)
{
	struct chan_list *ch = MISDN_ASTERISK_TECH_PVT(chan);
	char *tok,*tokb;
	int  keyidx=0;
	int rxgain=0;
	int txgain=0;
	
	if (!data || ast_strlen_zero((char *)data)) {
		ast_log(LOG_WARNING, "misdn_volume_change Requires arguments\n");
		return -1;
	}

	for (tok=strtok_r((char*)data, ":",&tokb);
	     tok;
	     tok=strtok_r(NULL,":",&tokb) ) {
    
		switch(tok[0]) {

		case 'd' :
			strncpy(ch->bc_te->display,++tok,84);
			if (misdn_debug>0) chan_misdn_log("SETOPT: Display:%s\n",ch->bc_te->display);
			break;
			
		case 'n':
			if (misdn_debug>0) chan_misdn_log("SETOPT: No DSP\n");
			ch->bc_te->nodsp=1;
			break;

		case 'j':
			if (misdn_debug>0) chan_misdn_log("SETOPT: No jitter\n");
			ch->bc_te->nojitter=1;
			break;
      
		case 'v':
			tok++;

			switch ( tok[0] ) {
			case 'r' :
				rxgain=atoi(++tok);
				if (rxgain<-8) rxgain=-8;
				if (rxgain>8) rxgain=8;
				ch->bc_te->rxgain=rxgain;
				if (misdn_debug>0) chan_misdn_log("SETOPT: Volume:%d\n",rxgain);
				break;
			case 't':
				txgain=atoi(++tok);
				if (txgain<-8) txgain=-8;
				if (txgain>8) txgain=8;
				ch->bc_te->txgain=txgain;
				if (misdn_debug>0) chan_misdn_log("SETOPT: Volume:%d\n",txgain);
				break;
			}
			break;
      
		case 'c':
			keyidx=atoi(++tok);
      
			if (keyidx > misdn_key_vector_size  || keyidx < 0 ) {
				ast_log(LOG_WARNING, "You entered the keyidx: %d but we have only %d keys\n",keyidx, misdn_key_vector_size );
				continue; 
			}
      
			strcpy(ch->bc_te->crypt_key, misdn_key_vector[keyidx]);
			chan_misdn_log("SETOPT: crypt with key:%s\n",misdn_key_vector[keyidx]);
			break;

		case 'e':
			if (misdn_debug >0) chan_misdn_log("SETOPT: EchoCancel\n");
			ch->bc_te->ec_enable=1;
			ch->bc_te->orig=ch->orginator;
			tok++;
			if (tok) {
				ch->bc_te->ec_deftaps=atoi(tok);
			}
			break;
      
		case 'h':
			if (misdn_debug >0) chan_misdn_log("SETOPT: Digital\n");
			if (strlen(tok) > 1 && tok[1]=='1') {
				if (misdn_debug >0) chan_misdn_log("SETOPT: Digital TRANS_DIGITAL\n");
				ch->bc_te->digital=1;
				ch->bc_te->digital_trans=1;
				ch->bc_te->capability=INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
			} else {
				ch->bc_te->digital=1;
				ch->bc_te->digital_trans=0;
				ch->bc_te->capability=INFO_CAPABILITY_DIGITAL_UNRESTRICTED;
			}
			break;
            
		case 's':
			if (misdn_debug>0) chan_misdn_log("SETOPT: Send DTMF\n");
			ch->bc_te->send_dtmf=1;
			break;

		case 'f':
			if (misdn_debug>0) chan_misdn_log("SETOPT: Faxdetect\n");
			ch->faxdetect=1;
			break;

		case 'p':
			if (misdn_debug>0) chan_misdn_log("SETOPT: callerpres: %s\n",&tok[1]);
			/* CRICH: callingpres!!! */
			if (strstr(tok,"allowed") ) {
				ch->bc_te->pres=0;
			} else if (strstr(tok,"not_screened")) {
				ch->bc_te->pres=1;
			}
			
			
			break;
      
      
		default:
			break;
		}
	}
  
	if (ch->faxdetect) {

		if (!ch->dsp) ch->dsp = ast_dsp_new();
		if (ch->dsp) ast_dsp_set_features(ch->dsp, DSP_FEATURE_DTMF_DETECT| DSP_FEATURE_FAX_DETECT);
		if (!ch->trans) ch->trans=ast_translator_build_path(AST_FORMAT_SLINEAR, AST_FORMAT_ALAW);
	}
  
	return 0;
}
