/*
 * Chan_Misdn -- Channel Driver for Asterisk
 *
 * Interface to mISDN
 *
 * Copyright (C) 2004, Christian Richter
 *
 * Originally written by Andreas Eversberg,
 * Programmer of the famous Jolly Pbx which inspired
 * me a lot.
 *
 * Christian Richter <crich@beronet.com>
 *
 * This program is free software, distributed under the terms of
 * the GNU General Public License
 */


#ifndef mISDN_HEAD_SIZE
#define mISDN_HEAD_SIZE (sizeof(mISDN_head_t))
#endif

#ifndef IFRAME_HEAD_SIZE
#define IFRAME_HEAD_SIZE 16
#endif


#define IEPRINTF if(DEBG) printf

#ifndef mISDN_HEAD_SIZE
#define mISDN_HEAD_SIZE (sizeof(mISDN_head_t))
#endif

#ifndef IFRAME_HEAD_SIZE
#define IFRAME_HEAD_SIZE 16
#endif



/* support stuff */
static void strnncpy(unsigned char *dest, unsigned char *src, int len, int dst_len)
{
  if (len > dst_len-1)
		len = dst_len-1;
	strncpy((char *)dest, (char *)src, len);
	dest[len] = '\0';
}


/* IE_COMPLETE */
void enc_ie_complete(unsigned char **ntmode, msg_t *msg, int complete, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);

	if (complete<0 || complete>1)
	{
		IEPRINTF("%s: ERROR: complete(%d) is out of range.\n", __FUNCTION__, complete);
		return;
	}

	if (complete)
		IEPRINTF("    complete=%d\n", complete);

	if (complete)
	{
		p = msg_put(msg, 1);
		if (nt)
		{
			*ntmode = p;
		} else
			qi->sending_complete = p - (unsigned char *)qi - sizeof(Q931_info_t);
		p[0] = IE_COMPLETE;
	}
}

void dec_ie_complete(unsigned char *p, Q931_info_t *qi, int *complete, int nt, bchannel_te_t *bc)
{
	*complete = 0;
	if (!nt)
	{
		if (qi->sending_complete)
			*complete = 1;
	} else
	if (p)
		*complete = 1;

	if (*complete)
		IEPRINTF("    complete=%d\n", *complete);
}


/* IE_BEARER */
void enc_ie_bearer(unsigned char **ntmode, msg_t *msg, int coding, int capability, int mode, int rate, int multi, int user, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (coding<0 || coding>3)
	{
		IEPRINTF("%s: ERROR: coding(%d) is out of range.\n", __FUNCTION__, coding);
		return;
	}
	if (capability<0 || capability>31)
	{
		IEPRINTF("%s: ERROR: capability(%d) is out of range.\n", __FUNCTION__, capability);
		return;
	}
	if (mode<0 || mode>3)
	{
		IEPRINTF("%s: ERROR: mode(%d) is out of range.\n", __FUNCTION__, mode);
		return;
	}
	if (rate<0 || rate>31)
	{
		IEPRINTF("%s: ERROR: rate(%d) is out of range.\n", __FUNCTION__, rate);
		return;
	}
	if (multi>127)
	{
		IEPRINTF("%s: ERROR: multi(%d) is out of range.\n", __FUNCTION__, multi);
		return;
	}
	if (user>31)
	{
		IEPRINTF("%s: ERROR: user L1(%d) is out of range.\n", __FUNCTION__, rate);
		return;
	}
	if (rate!=24 && multi>=0)
	{
		IEPRINTF("%s: WARNING: multi(%d) is only possible if rate(%d) would be 24.\n", __FUNCTION__, multi, rate);
		multi = -1;
	}

	IEPRINTF("    coding=%d capability=%d mode=%d rate=%d multi=%d user=%d\n", coding, capability, mode, rate, multi, user);

	l = 2 + (multi>=0) + (user>=0);
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->bearer_capability = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_BEARER;
	p[1] = l;
	p[2] = 0x80 + (coding<<5) + capability;
	p[3] = 0x80 + (mode<<5) + rate;
	if (multi >= 0)
		p[4] = 0x80 + multi;
	if (user >= 0)
		p[4+(multi>=0)] = 0xa0 + user;
}

void dec_ie_bearer(unsigned char *p, Q931_info_t *qi, int *coding, int *capability, int *mode, int *rate, int *multi, int *user, int nt, bchannel_te_t *bc)
{
	*coding = -1;
	*capability = -1;
	*mode = -1;
	*rate = -1;
	*multi = -1;
	*user = -1;

	if (!nt)
	{
		p = NULL;
		if (qi->bearer_capability)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->bearer_capability + 1;
	}
	if (!p)
		return;
	if (p[0] < 3)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*coding = (p[1]&0x60) >> 5;
	*capability = p[1] & 0x1f;
	*mode = (p[2]&0x60) >> 5;
	*rate = p[2] & 0x1f;

	if (p[0]>=3 && *rate==0x18)
	{
		*multi = p[3] & 0x7f;
		if (p[0]>=4)
			*user = p[4] & 0x1f;
	} else
	{
		if (p[0]>=3)
			*user = p[3] & 0x1f;
	}

	IEPRINTF("    coding=%d capability=%d mode=%d rate=%d multi=%d user=%d\n", *coding, *capability, *mode, *rate, *multi, *user);
}


/* IE_CALL_ID */
void enc_ie_call_id(unsigned char **ntmode, msg_t *msg, unsigned char *callid, int callid_len, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	char debug[25];
	int i;

	if (!callid || callid_len<=0)
	{
		return;
	}
	if (callid_len>8)
	{
		IEPRINTF("%s: ERROR: callid_len(%d) is out of range.\n", __FUNCTION__, callid_len);
		return;
	}

	i = 0;
	while(i < callid_len)
	{
	  //UPRINT(debug+(i*3), " %02x", callid[i]);
		i++;
	}
		
	IEPRINTF("    callid%s\n", debug);

	l = callid_len;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->call_id = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CALL_ID;
	p[1] = l;
	memcpy(p+2, callid, callid_len);
}

void dec_ie_call_id(unsigned char *p, Q931_info_t *qi, unsigned char *callid, int *callid_len, int nt, bchannel_te_t *bc)
{
	char debug[25];
	int i;

	*callid_len = 0;

	if (!nt)
	{
		p = NULL;
		if (qi->call_id)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->call_id + 1;
	}
	if (!p)
		return;
	if (p[0] > 8)
	{
		IEPRINTF("%s: ERROR: message too long (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*callid_len = p[0];
	memcpy(callid, p, *callid_len);

	i = 0;
	while(i < *callid_len)
	{
	  //UPRINT(debug+(i*3), " %02x", callid[i]);
		i++;
	}
		
	IEPRINTF("    callid%s\n", debug);
}


/* IE_CALLED_PN */
void enc_ie_called_pn(unsigned char **ntmode, msg_t *msg, int type, int plan, unsigned char *number, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (type<0 || type>7)
	{
		IEPRINTF("%s: ERROR: type(%d) is out of range.\n", __FUNCTION__, type);
		return;
	}
	if (plan<0 || plan>15)
	{
		IEPRINTF("%s: ERROR: plan(%d) is out of range.\n", __FUNCTION__, plan);
		return;
	}
	if (!number[0])
	{
		IEPRINTF("%s: ERROR: number is not given.\n", __FUNCTION__);
		return;
	}

	IEPRINTF("    type=%d plan=%d number='%s'\n", type, plan, number);

	l = 1+strlen((char *)number);
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->called_nr = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CALLED_PN;
	p[1] = l;
	p[2] = 0x80 + (type<<4) + plan;
	strncpy((char *)p+3, (char *)number, strlen((char *)number));
}

void dec_ie_called_pn(unsigned char *p, Q931_info_t *qi, int *type, int *plan, unsigned char *number, int number_len, int nt, bchannel_te_t *bc)
{
	*type = -1;
	*plan = -1;
	*number = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->called_nr)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->called_nr + 1;
	}
	if (!p)
		return;
	if (p[0] < 2)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*type = (p[1]&0x70) >> 4;
	*plan = p[1] & 0xf;
	strnncpy(number, p+2, p[0]-1, number_len);

	IEPRINTF("    type=%d plan=%d number='%s'\n", *type, *plan, number);
}


/* IE_CALLING_PN */
void enc_ie_calling_pn(unsigned char **ntmode, msg_t *msg, int type, int plan, int present, int screen, unsigned char *number, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (type<0 || type>7)
	{
		IEPRINTF("%s: ERROR: type(%d) is out of range.\n", __FUNCTION__, type);
		return;
	}
	if (plan<0 || plan>15)
	{
		IEPRINTF("%s: ERROR: plan(%d) is out of range.\n", __FUNCTION__, plan);
		return;
	}
	if (present>3)
	{
		IEPRINTF("%s: ERROR: present(%d) is out of range.\n", __FUNCTION__, present);
		return;
	}
	if (present >= 0) if (screen<0 || screen>3)
	{
		IEPRINTF("%s: ERROR: screen(%d) is out of range.\n", __FUNCTION__, screen);
		return;
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d number='%s'\n", type, plan, present, screen, number);

	l = 1;
	if (number) if (number[0])
		l += strlen((char *)number);
	if (present >= 0)
		l += 1;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->calling_nr = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CALLING_PN;
	p[1] = l;
	if (present >= 0)
	{
		p[2] = 0x00 + (type<<4) + plan;
		p[3] = 0x80 + (present<<5) + screen;
		if (number) if (number[0])
			strncpy((char *)p+4, (char *)number, strlen((char *)number));
	} else
	{
		p[2] = 0x80 + (type<<4) + plan;
		if (number) if (number[0])
			strncpy((char *)p+3, (char *)number, strlen((char *)number));
	}
}

void dec_ie_calling_pn(unsigned char *p, Q931_info_t *qi, int *type, int *plan, int *present, int *screen, unsigned char *number, int number_len, int nt, bchannel_te_t *bc)
{
	*type = -1;
	*plan = -1;
	*present = -1;
	*screen = -1;
	*number = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->calling_nr)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->calling_nr + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*type = (p[1]&0x70) >> 4;
	*plan = p[1] & 0xf;
	if (!(p[1] & 0x80))
	{
		if (p[0] < 2)
		{
			IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
			return;
		}
		*present = (p[2]&0x60) >> 5;
		*screen = p[2] & 0x3;
		strnncpy(number, p+3, p[0]-2, number_len);
	} else
	{
		strnncpy(number, p+2, p[0]-1, number_len);
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d number='%s'\n", *type, *plan, *present, *screen, number);
}


/* IE_CONNECTED_PN */
void enc_ie_connected_pn(unsigned char **ntmode, msg_t *msg, int type, int plan, int present, int screen, unsigned char *number, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (type<0 || type>7)
	{
		IEPRINTF("%s: ERROR: type(%d) is out of range.\n", __FUNCTION__, type);
		return;
	}
	if (plan<0 || plan>15)
	{
		IEPRINTF("%s: ERROR: plan(%d) is out of range.\n", __FUNCTION__, plan);
		return;
	}
	if (present>3)
	{
		IEPRINTF("%s: ERROR: present(%d) is out of range.\n", __FUNCTION__, present);
		return;
	}
	if (present >= 0) if (screen<0 || screen>3)
	{
		IEPRINTF("%s: ERROR: screen(%d) is out of range.\n", __FUNCTION__, screen);
		return;
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d number='%s'\n", type, plan, present, screen, number);

	l = 1;
	if (number) if (number[0])
		l += strlen((char *)number);
	if (present >= 0)
		l += 1;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->connected_nr = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CONNECT_PN;
	p[1] = l;
	if (present >= 0)
	{
		p[2] = 0x00 + (type<<4) + plan;
		p[3] = 0x80 + (present<<5) + screen;
		if (number) if (number[0])
			strncpy((char *)p+4, (char *)number, strlen((char *)number));
	} else
	{
		p[2] = 0x80 + (type<<4) + plan;
		if (number) if (number[0])
			strncpy((char *)p+3, (char *)number, strlen((char *)number));
	}
}

void dec_ie_connected_pn(unsigned char *p, Q931_info_t *qi, int *type, int *plan, int *present, int *screen, unsigned char *number, int number_len, int nt, bchannel_te_t *bc)
{
	*type = -1;
	*plan = -1;
	*present = -1;
	*screen = -1;
	*number = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->connected_nr)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->connected_nr + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*type = (p[1]&0x70) >> 4;
	*plan = p[1] & 0xf;
	if (!(p[1] & 0x80))
	{
		if (p[0] < 2)
		{
			IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
			return;
		}
		*present = (p[2]&0x60) >> 5;
		*screen = p[2] & 0x3;
		strnncpy(number, p+3, p[0]-2, number_len);
	} else
	{
		strnncpy(number, p+2, p[0]-1, number_len);
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d number='%s'\n", *type, *plan, *present, *screen, number);
}


/* IE_CAUSE */
void enc_ie_cause(unsigned char **ntmode, msg_t *msg, int location, int cause, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (location<0 || location>7)
	{
		IEPRINTF("%s: ERROR: location(%d) is out of range.\n", __FUNCTION__, location);
		return;
	}
	if (cause<0 || cause>127)
	{
		IEPRINTF("%s: ERROR: cause(%d) is out of range.\n", __FUNCTION__, cause);
		return;
	}

	IEPRINTF("    location=%d cause=%d\n", location, cause);

	l = 2;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->cause = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CAUSE;
	p[1] = l;
	p[2] = 0x80 + location;
	p[3] = 0x80 + cause;
}
void enc_ie_cause_standalone(unsigned char **ntmode, msg_t *msg, int location, int cause, int nt, bchannel_te_t *bc)
{
	unsigned char *p = msg_put(msg, 4);
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	if (ntmode)
		*ntmode = p+1;
	else
		qi->cause = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_CAUSE;
	p[1] = 2;
	p[2] = 0x80 + location;
	p[3] = 0x80 + cause;
}


void dec_ie_cause(unsigned char *p, Q931_info_t *qi, int *location, int *cause, int nt, bchannel_te_t *bc)
{
	*location = -1;
	*cause = -1;

	if (!nt)
	{
		p = NULL;
		if (qi->cause)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->cause + 1;
	}
	if (!p)
		return;
	if (p[0] < 2)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*location = p[1] & 0x0f;
	*cause = p[2] & 0x7f;

	IEPRINTF("    location=%d cause=%d\n", *location, *cause);
}


/* IE_CHANNEL_ID */
void enc_ie_channel_id(unsigned char **ntmode, msg_t *msg, int exclusive, int channel, int nt, bchannel_te_t *bc)
{
  unsigned char *p;
  Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
  int l;
  int pri = bc->stack->pri;
  
	if (exclusive<0 || exclusive>1)
	{
		IEPRINTF("%s: ERROR: exclusive(%d) is out of range.\n", __FUNCTION__, exclusive);
		return;
	}
	if (channel<0 || channel>0xff)
	  {
		IEPRINTF("%s: ERROR: channel(%d) is out of range.\n", __FUNCTION__, channel);
		return;
	}

	IEPRINTF("    exclusive=%d channel=%d\n", exclusive, channel);
	
	if (pri) {
	  
	  if (channel == 0) return ; //ANY_CHANNEL

	  l=3;
	  p = msg_put(msg, l+2);
	  if (nt)
	    *ntmode = p+1;
	  else
	    qi->channel_id = p - (unsigned char *)qi - sizeof(Q931_info_t);
	  p[0] = IE_CHANNEL_ID;
	  p[1] = l;
	  p[2] = 0x80 + 0x20 + (exclusive<<3);
	  p[3] = 0x80 + 3; /* CCITT, Number, B-type */
	  p[4] = 0xff + channel;
	  
	  
	} else {
	  
	  l = 1;
	  p = msg_put(msg, l+2);
	  if (nt)
	    *ntmode = p+1;
	  else
	    qi->channel_id = p - (unsigned char *)qi - sizeof(Q931_info_t);
	  p[0] = IE_CHANNEL_ID;
	  p[1] = l;
	  p[2] = 0x80 + (exclusive<<3) + channel;
	}
}

void dec_ie_channel_id(unsigned char *p, Q931_info_t *qi, int *exclusive, int *channel, int nt, bchannel_te_t *bc)
{
  int pri = bc->stack->pri;
  
	*exclusive = -1;
	*channel = -1;

	if (!nt)
	{
		p = NULL;
		if (qi->channel_id)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->channel_id + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	if (p[1] & 0x40)
	{
		IEPRINTF("%s: ERROR: using channels of other interfaces is not supported.\n", __FUNCTION__);
		return;
	}
	
	if (p[1] & 0x04)
	{
		IEPRINTF("%s: ERROR: using d-channel is not supported.\n", __FUNCTION__);
		return;
	}
	
	*exclusive = (p[1]&0x08) >> 3;
	
	if (pri) { //PRI - TE
	  
	  
	  if (p[0] < 3) return ; //IE too short
	  
	  if ( !(p[1] & 0x20))
	    {
	      IEPRINTF("%s: ERROR: primary rate interface is not supported.\n", __FUNCTION__);
	      return;
	    }

	  if (p[2] &0x10)  return ; //channel map not supported
	  
	  *channel = p[3] & 0x7f; //0x03;
	  
	} else {
	  if ( p[1] & 0x20)
	    {
	      IEPRINTF("%s: ERROR: primary rate interface is not supported.\n", __FUNCTION__);
	      return;
	    }
	  
	  *channel = p[1] & 0x03;
	}
	
	
	IEPRINTF("    exclusive=%d channel=%d\n", *exclusive, *channel);
}


/* IE_DATE */
void enc_ie_date(unsigned char **ntmode, msg_t *msg, time_t ti, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	struct tm *tm;

	tm = localtime(&ti);
	if (!tm)
	{
		IEPRINTF("%s: ERROR: gettimeofday() returned NULL.\n", __FUNCTION__);
		return;
	}

	IEPRINTF("    year=%d month=%d day=%d hour=%d minute=%d\n", tm->tm_year%100, tm->tm_mon+1, tm->tm_mday, tm->tm_hour, tm->tm_min);

	l = 5;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->date = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_DATE;
	p[1] = l;
	p[2] = tm->tm_year % 100;
	p[3] = tm->tm_mon + 1;
	p[4] = tm->tm_mday;
	p[5] = tm->tm_hour;
	p[6] = tm->tm_min;
}


/* IE_DISPLAY */
void enc_ie_display(unsigned char **ntmode, msg_t *msg, unsigned char *display, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (!display[0])
	{
		IEPRINTF("%s: ERROR: display text not given.\n", __FUNCTION__);
		return;
	}

	if (strlen((char *)display) > 80)
	{
		IEPRINTF("%s: WARNING: display text too long (max 80 chars), cutting.\n", __FUNCTION__);
		display[80] = '\0';
	}

	IEPRINTF("    display='%s'\n", display);

	l = strlen((char *)display);
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->display = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_DISPLAY;
	p[1] = l;
	strncpy((char *)p+2, (char *)display, strlen((char *)display));
}

void dec_ie_display(unsigned char *p, Q931_info_t *qi, unsigned char *display, int display_len, int nt, bchannel_te_t *bc)
{
	*display = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->display)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->display + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	strnncpy(display, p+1, p[0], display_len);

	IEPRINTF("    display='%s'\n", display);
}


/* IE_KEYPAD */
void enc_ie_keypad(unsigned char **ntmode, msg_t *msg, unsigned char *keypad, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (!keypad[0])
	{
		IEPRINTF("%s: ERROR: keypad info not given.\n", __FUNCTION__);
		return;
	}

	IEPRINTF("    keypad='%s'\n", keypad);

	l = strlen((char *)keypad);
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->keypad = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_KEYPAD;
	p[1] = l;
	strncpy((char *)p+2, (char *)keypad, strlen((char *)keypad));
}

void dec_ie_keypad(unsigned char *p, Q931_info_t *qi, unsigned char *keypad, int keypad_len, int nt, bchannel_te_t *bc)
{
	*keypad = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->keypad)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->keypad + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	strnncpy(keypad, p+1, p[0], keypad_len);

	IEPRINTF("    display='%s'\n", keypad);
}


/* IE_NOTIFY */
void enc_ie_notify(unsigned char **ntmode, msg_t *msg, int notify, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (notify<0 || notify>0x7f)
	{
		IEPRINTF("%s: ERROR: notify(%d) is out of range.\n", __FUNCTION__, notify);
		return;
	}

	IEPRINTF("    notify=%d\n", notify);

	l = 1;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->notify = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_NOTIFY;
	p[1] = l;
	p[2] = 0x80 + notify;
}

void dec_ie_notify(unsigned char *p, Q931_info_t *qi, int *notify, int nt, bchannel_te_t *bc)
{
	*notify = -1;

	if (!nt)
	{
		p = NULL;
		if (qi->notify)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->notify + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*notify = p[1] & 0x7f;

	IEPRINTF("    notify=%d\n", *notify);
}


/* IE_PROGRESS */
void enc_ie_progress(unsigned char **ntmode, msg_t *msg, int coding, int location, int progress, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (coding<0 || coding>0x03)
	{
		IEPRINTF("%s: ERROR: coding(%d) is out of range.\n", __FUNCTION__, coding);
		return;
	}
	if (location<0 || location>0x0f)
	{
		IEPRINTF("%s: ERROR: location(%d) is out of range.\n", __FUNCTION__, location);
		return;
	}
	if (progress<0 || progress>0x7f)
	{
		IEPRINTF("%s: ERROR: progress(%d) is out of range.\n", __FUNCTION__, progress);
		return;
	}

	IEPRINTF("    coding=%d location=%d progress=%d\n", coding, location, progress);

	l = 2;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->progress = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_PROGRESS;
	p[1] = l;
	p[2] = 0x80 + (coding<<5) + location;
	p[3] = 0x80 + progress;
}

void dec_ie_progress(unsigned char *p, Q931_info_t *qi, int *coding, int *location, int *progress, int nt, bchannel_te_t *bc)
{
	*coding = -1;
	*location = -1;
	*progress = -1;

	if (!nt)
	{
		p = NULL;
		if (qi->progress)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->progress + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*coding = (p[1]&0x60) >> 5;
	*location = p[1] & 0x0f;
	*progress = p[2] & 0x7f;

	IEPRINTF("    coding=%d location=%d progress=%d\n", *coding, *location, *progress);
}


/* IE_REDIR_NR (redirecting = during MT_SETUP) */
void enc_ie_redir_nr(unsigned char **ntmode, msg_t *msg, int type, int plan, int present, int screen, int reason, unsigned char *number, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (type<0 || type>7)
	{
		IEPRINTF("%s: ERROR: type(%d) is out of range.\n", __FUNCTION__, type);
		return;
	}
	if (plan<0 || plan>15)
	{
		IEPRINTF("%s: ERROR: plan(%d) is out of range.\n", __FUNCTION__, plan);
		return;
	}
	if (present > 3)
	{
		IEPRINTF("%s: ERROR: present(%d) is out of range.\n", __FUNCTION__, present);
		return;
	}
	if (present >= 0) if (screen<0 || screen>3)
	{
		IEPRINTF("%s: ERROR: screen(%d) is out of range.\n", __FUNCTION__, screen);
		return;
	}
	if (reason > 0x0f)
	{
		IEPRINTF("%s: ERROR: reason(%d) is out of range.\n", __FUNCTION__, reason);
		return;
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d readon=%d number='%s'\n", type, plan, present, screen, reason, number);

	l = 1;
	if (number)
		l += strlen((char *)number);
	if (present >= 0)
	{
		l += 1;
		if (reason >= 0)
			l += 1;
	}
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->redirect_nr = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_REDIR_NR;
	p[1] = l;
	if (present >= 0)
	{
		if (reason >= 0)
		{
			p[2] = 0x00 + (type<<4) + plan;
			p[3] = 0x00 + (present<<5) + screen;
			p[4] = 0x80 + reason;
			if (number)
				strncpy((char *)p+5, (char *)number, strlen((char *)number));
		} else
		{
			p[2] = 0x00 + (type<<4) + plan;
			p[3] = 0x80 + (present<<5) + screen;
			if (number)
				strncpy((char *)p+4, (char *)number, strlen((char *)number));
		}
	} else
	{
		p[2] = 0x80 + (type<<4) + plan;
		if (number) if (number[0])
			strncpy((char *)p+3, (char *)number, strlen((char *)number));
	}
}

void dec_ie_redir_nr(unsigned char *p, Q931_info_t *qi, int *type, int *plan, int *present, int *screen, int *reason, unsigned char *number, int number_len, int nt, bchannel_te_t *bc)
{
	*type = -1;
	*plan = -1;
	*present = -1;
	*screen = -1;
	*reason = -1;
	*number = '\0';

	if (!nt)
	{
		p = NULL;
		if (qi->redirect_nr)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->redirect_nr + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*type = (p[1]&0x70) >> 4;
	*plan = p[1] & 0xf;
	if (!(p[1] & 0x80))
	{
		*present = (p[2]&0x60) >> 5;
		*screen = p[2] & 0x3;
		if (!(p[2] & 0x80))
		{
			*reason = p[3] & 0x0f;
			strnncpy(number, p+4, p[0]-3, number_len);
		} else
		{
			strnncpy(number, p+3, p[0]-2, number_len);
		}
	} else
	{
		strnncpy(number, p+2, p[0]-1, number_len);
	}

	IEPRINTF("    type=%d plan=%d present=%d screen=%d reason=%d number='%s'\n", *type, *plan, *present, *screen, *reason, number);
}


/* IE_REDIR_DN (redirection = during MT_NOTIFY) */
void enc_ie_redir_dn(unsigned char **ntmode, msg_t *msg, int type, int plan, int present, unsigned char *number, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
//	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	if (type<0 || type>7)
	{
		IEPRINTF("%s: ERROR: type(%d) is out of range.\n", __FUNCTION__, type);
		return;
	}
	if (plan<0 || plan>15)
	{
		IEPRINTF("%s: ERROR: plan(%d) is out of range.\n", __FUNCTION__, plan);
		return;
	}
	if (present > 3)
	{
		IEPRINTF("%s: ERROR: present(%d) is out of range.\n", __FUNCTION__, present);
		return;
	}

	IEPRINTF("    type=%d plan=%d present=%d number='%s'\n", type, plan, present, number);

	l = 1;
	if (number)
		l += strlen((char *)number);
	if (present >= 0)
		l += 1;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
//#warning REINSERT redir_dn, when included in te-mode
		/*qi->redir_dn = p - (unsigned char *)qi - sizeof(Q931_info_t)*/;
	p[0] = IE_REDIR_DN;
	p[1] = l;
	if (present >= 0)
	{
		p[2] = 0x00 + (type<<4) + plan;
		p[3] = 0x80 + (present<<5);
		if (number)
			strncpy((char *)p+4, (char *)number, strlen((char *)number));
	} else
	{
		p[2] = 0x80 + (type<<4) + plan;
		if (number)
			strncpy((char *)p+3, (char *)number, strlen((char *)number));
	}
}

void dec_ie_redir_dn(unsigned char *p, Q931_info_t *qi, int *type, int *plan, int *present, unsigned char *number, int number_len, int nt, bchannel_te_t *bc)
{
	*type = -1;
	*plan = -1;
	*present = -1;
	*number = '\0';

	if (!nt)
	{
		p = NULL;
//#warning REINSERT redir_dn, when included in te-mode
//		if (qi->redir_dn)
//			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->redir_dn + 1;
	}
	if (!p)
		return;
	if (p[0] < 1)
	{
		IEPRINTF("%s: ERROR: message too short (%d).\n", __FUNCTION__, p[0]);
		return;
	}

	*type = (p[1]&0x70) >> 4;
	*plan = p[1] & 0xf;
	if (!(p[1] & 0x80))
	{
		*present = (p[2]&0x60) >> 5;
		strnncpy(number, p+3, p[0]-2, number_len);
	} else
	{
		strnncpy(number, p+2, p[0]-1, number_len);
	}

	IEPRINTF("    type=%d plan=%d present=%d number='%s'\n", *type, *plan, *present, number);
}


/* IE_FACILITY */
void enc_ie_facility(unsigned char **ntmode, msg_t *msg, unsigned char *facility, int facility_len, int nt, bchannel_te_t *bc)
{
	unsigned char *p;
	Q931_info_t *qi = (Q931_info_t *)(msg->data + IFRAME_HEAD_SIZE);
	int l;

	char debug[768];
	int i;

	if (!facility || facility_len<=0)
	{
		return;
	}

	i = 0;
	while(i < facility_len)
	{
	  //UPRINT(debug+(i*3), " %02x", facility[i]);
		i++;
	}
		
	IEPRINTF("    facility%s\n", debug);

	l = facility_len;
	p = msg_put(msg, l+2);
	if (nt)
		*ntmode = p+1;
	else
		qi->facility = p - (unsigned char *)qi - sizeof(Q931_info_t);
	p[0] = IE_FACILITY;
	p[1] = l;
	memcpy(p+2, facility, facility_len);
}

void dec_ie_facility(unsigned char *p, Q931_info_t *qi, unsigned char *facility, int *facility_len, int nt, bchannel_te_t *bc)
{
	char debug[768];
	int i;

	*facility_len = 0;

	if (!nt)
	{
		p = NULL;
		if (qi->facility)
			p = (unsigned char *)qi + sizeof(Q931_info_t) + qi->facility + 1;
	}
	if (!p)
		return;

	*facility_len = p[0];
	memcpy(facility, p, *facility_len);

	i = 0;
	while(i < *facility_len)
	{
	  //UPRINT(debug+(i*3), " %02x", facility[i]);
		i++;
	}
	debug[i*3] = '\0';
		
	IEPRINTF("    facility%s\n", debug);

}


