// This file is part of PUMA.
// Copyright (C) 1999-2003  The PUMA developer team.
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#ifndef __Builder_h__
#define __Builder_h__

#include "Puma/ErrorCollector.h"
#include "Puma/ErrorSink.h"
#include "Puma/PtrStack.h"
#include "Puma/CTree.h"

namespace Puma {


class Token;

class Builder : public PtrStack<CTree> {
protected:
  ErrorCollector ec;
  static unsigned long int token_counter;
  
public:
  // Container class to shift nodes from one level to another
  class Container : public CT_List {
  public:
    static const char *NodeId ();
    const char *NodeName () const { return NodeId (); }
  };

protected:
  Builder ()/* : token_counter (0)*/ {}

public:
  // release a syntax tree
  static void destroy (CTree *);

  // print out collected error messages
  void errors (ErrorSink &);

  // callback functions used by the parser
  ErrorSink &err () const;
  void save_state ();
  void forget_state ();
  void restore_state ();

  // Helper functions
  CTree *token (Token *);
  CTree *error ();

  int nodes () const;
  CTree *get_node (int = 0) const;
  
  void setTokenCounter (unsigned long);
  unsigned long getTokenCounter () const;

protected:
  void Delete ();

  // put all child nodes into a container, 
  // the next level can handle the contents
  CTree *container () const;

  // put all children into a list node
  CTree *list (CT_List *) const;
  CTree *copy_list (CT_List *, Container *) const;
};

inline int Builder::nodes () const
 { return (int)Length (); }
 
inline CTree *Builder::get_node (int i) const
 { return (CTree*)Get (i); }

inline CTree *Builder::container () const
 { return list (new Container); }

inline ErrorSink &Builder::err () const { return (ErrorSink&)ec; }
inline void Builder::errors (ErrorSink &e) { ec.Shift (e); }

inline void Builder::save_state ()    { New (); }
inline void Builder::forget_state ()  { Reject (); }
inline void Builder::restore_state () { Destroy (); }

inline void Builder::setTokenCounter (unsigned long num) 
 { token_counter = num; }
inline unsigned long Builder::getTokenCounter () const 
 { return token_counter; }

} // namespace Puma

#endif /* __Builder_h__ */
