// This file is part of the AspectC++ compiler 'ac++'.
// Copyright (C) 1999-2003  The 'ac++' developers (see aspectc.org)
//                                                                
// This program is free software;  you can redistribute it and/or 
// modify it under the terms of the GNU General Public License as 
// published by the Free Software Foundation; either version 2 of 
// the License, or (at your option) any later version.            
//                                                                
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of 
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the  
// GNU General Public License for more details.                   
//                                                                
// You should have received a copy of the GNU General Public      
// License along with this program; if not, write to the Free     
// Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, 
// MA  02111-1307  USA                                            

#include "JoinPointPlan.h"
#include "ACUnit.h"
#include "IntroductionUnit.h"
#include "AspectInfo.h"
#include "WeaverBase.h"
#include "LineDirectiveMgr.h"

#include "Puma/CProtection.h"
#include "Puma/ErrorSink.h"
#include "Puma/ManipCommander.h"
#include "Puma/CFunctionInfo.h"
#include "Puma/CArgumentInfo.h"
#include "Puma/ACSliceInfo.h"
#include "Puma/CClassDatabase.h"

// issue an advice in the plan for this code join point
void JPP_Code::issue (int index) {
  AdviceInfo *ai = _unsorted[index];
  const Condition &cond = *_conds[index];
  switch (ai->type()) {
  case JPL_AdviceCode::ADVICE_BEFORE:
    _jpa.do_before (ai, cond);
    break;
  case JPL_AdviceCode::ADVICE_AROUND:
    _jpa.do_around (ai, cond);
    break;
  case JPL_AdviceCode::ADVICE_AFTER:
    _jpa.do_after (ai, cond);
    break;
  }
}

// consider an advice in the plan for this code join point
bool JPP_Code::consider (AdviceInfo *ai, const Condition &cond) {
  _unsorted.append (ai);
  _conds.append (&cond);
  return true;
}

bool JPP_Code::check(ErrorSink &err) {
  if (!plan ()) {
    err << sev_error << "order directive not resolveable";
    return false;
  }
  for (int i = 0; i < items (); i++) {
    ACAspectInfo* ai = &item (i);
    for (int j = 0; j < _unsorted.length(); j++) {
      AdviceInfo* a = _unsorted[j];
      if (a && (TI_Aspect::of (a->aspect())->acnode() == ai)) {
        issue(j);
        _unsorted[j] = 0;
      }
    }
  }
  for (int j = 0; j < _unsorted.length(); j++)
    if (_unsorted[j] != 0) {
      issue(j);
    }
  return true;
}

// issue an introduction in the plan for this class join point
void JPP_Class::issue (JPL_Introduction *ii) {
  switch (ii->introduced ()->slice_type ()) {
  case JPL_ClassSlice::CS_OLD_BASE:
    _base.append (ii);
    break;
  case JPL_ClassSlice::CS_OLD_OTHER:
    _other.append (ii);
    break;
  case JPL_ClassSlice::CS_NORMAL: { // slices might contribute new members and base classes
    CObjectInfo *obj = TI_ClassSlice::of (*ii->introduced ())->assoc_obj ();
    assert (obj && obj->Tree ());
    assert (obj->Tree ()->NodeName () == CT_ClassSliceDecl::NodeId ());
    CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
    if (csd->base_clause ())
      _base.append (ii);
    if (csd->members () && csd->members ()->Entries ())
      _other.append (ii);
    }
    break;
  case JPL_ClassSlice::CS_ERROR:
    break;
  }
}

// consider an introduction in the plan for this class join point
bool JPP_Class::consider (JPL_Introduction *ii) {
  switch (ii->introduced ()->slice_type ()) {
  case JPL_ClassSlice::CS_OLD_BASE:
  case JPL_ClassSlice::CS_OLD_OTHER:
  case JPL_ClassSlice::CS_NORMAL:
    _unsorted.append (ii);
    return true;
  default:
    return false;
  }
}

bool JPP_Class::check(ErrorSink &err) {
  if (!plan ()) {
    err << sev_error << "order directive not resolveable";
    return false;
  }
  for (int i = 0; i < items (); i++) {
    ACAspectInfo* ai = &item (i);
    for (int j = 0; j < _unsorted.length(); j++) {
      if (_unsorted[j] && (((TI_Aspect*)_unsorted[j]->parent()->transform_info ())->acnode() == ai)) {
        issue(_unsorted[j]);
        _unsorted[j] = 0;
      }
    }
  }
  for (int j = 0; j < _unsorted.length(); j++)
    if (_unsorted[j] != 0)
      issue(_unsorted[j]);

  return true;
}


void JPP_Class::gen_intros (list<Unit*> &units, ErrorStream &err,
  CStructure *target, LineDirectiveMgr &lmgr, bool non_inline) const {

  // ... some unit for formatting
  ACUnit ws (err);
  ws << " " << endu;
    
  // handle all intros
  for (int i = 0; i < otherIntros (); i++) {
    JPL_Introduction *ii = otherIntro (i);
    
    // create the new unit
    IntroductionUnit *unit = 
      new IntroductionUnit (err, (Unit*)target->Tree ()->token ()->belonging_to ());
    unit->intro (ii);

    // at least on token is needed in the unit for 'move' (see below)
    unit->append (*((Token*)ws.first ())->duplicate ());

    // TODO: member intros from slices are ignored here!
    if (ii->introduced ()->slice_type () == JPL_ClassSlice::CS_NORMAL) {
      CObjectInfo *obj = TI_ClassSlice::of (*ii->introduced ())->assoc_obj ();
      assert (obj && obj->Tree ());
      assert (obj->Tree ()->NodeName () == CT_ClassSliceDecl::NodeId ());
      ACSliceInfo *acsi = obj->ClassDB ()->SliceInfo (obj)->definition ();
      if (non_inline)
        gen_intro_non_inline (unit, ii, acsi, err, lmgr, target);
      else
        gen_intro_inline (unit, ii, acsi, err, lmgr, target);
    }
    else if ((ii->introduced ()->prot () == CProtection::PROT_NONE && non_inline) ||
              (ii->introduced ()->prot () != CProtection::PROT_NONE && !non_inline)) {
      if (!non_inline) {
        // add privat/public/protected
        ACUnit prot (err);
        prot << WeaverBase::protection_string (ii->introduced ()->prot ()) << "  " << endu;
        unit->move ((Token*)unit->last (), prot);
      }
      
      // obtain the pattern for this intro
      const Unit &pattern = ii->introduced ()->pattern ();
  
      CT_Intro *intro = (CT_Intro*)TI_Introduction::of (*ii)->tree ()->Decl ();
      Token *curr = (Token*)ii->introduced ()->pattern ().first ();
      ACUnit dir (err);
      lmgr.directive (dir, (Unit*)&pattern, curr);
      dir << endu;
      if (!dir.empty ())
        unit->move ((Token*)unit->last (), dir);
      Location loc = ((Token*)pattern.first ())->location ();
      instantiate_intro (err, unit, intro, pattern, target, loc);
    }

    // if there was no introduction return 0
    if (unit->first () == unit->last ())
      delete unit;
    else
      units.push_back (unit);
  }
}


void JPP_Class::gen_intro_inline (Unit *unit, JPL_Introduction *ii,
  ACSliceInfo *acsi, ErrorStream &err, LineDirectiveMgr &lmgr,
  CStructure *target) const {
  
  CObjectInfo *obj = TI_ClassSlice::of (*ii->introduced ())->assoc_obj ();
  CT_ClassSliceDecl *csd = (CT_ClassSliceDecl*)obj->Tree ();
  
  // add "private:\n"
  ACUnit slice_start (err);
  if (ii->introduced ()->prot () == CProtection::PROT_PRIVATE)
    slice_start << "  private:" << endl;
  else
    slice_start << "  public:" << endl;
  // add "  typedef <target-name> <slice-name>;\n"
  if (!obj->isAnonymous ()) {
    slice_start << "  typedef " << target->Name () << " "
                << obj->Name () << ";" << endl;
  }
  slice_start << endu;
  unit->move ((Token*)unit->last (), slice_start);
  // generate the intro itself from the stored pattern
  const Unit &pattern = *acsi->pattern ();
  Token *curr = (Token*)pattern.first ();
  Location loc = curr->location ();
  ACUnit dir (err);
  lmgr.directive (dir, (Unit*)&pattern, curr);
  dir << endu;
  if (!dir.empty ())
    unit->move ((Token*)unit->last (), dir);
  for (int m = 0; m < csd->members ()->Entries (); m++) {
    Unit tmp_pattern;
    tmp_pattern.name (pattern.name ());
    CT_Intro *intro = (CT_Intro*)csd->members ()->Entry (m);
    for (int i = 0; i < intro->Entries (); i++) {
      tmp_pattern.append (*curr->duplicate ());
      curr = (Token*)pattern.next (curr);
    }
    instantiate_intro (err, unit, intro, tmp_pattern, target, loc);
  }
}

void JPP_Class::gen_intro_non_inline (Unit *unit, JPL_Introduction *ii,
  ACSliceInfo *acsi, ErrorStream &err, LineDirectiveMgr &lmgr,
  CStructure *target) const {
  
  // generate the intros from the stored non-inline member patterns
  for (int m = 0; m < acsi->members (); m++) {
    CT_Intro *intro = acsi->member (m);
    const Unit &pattern = *acsi->member_pattern (m);
    Token *curr = (Token*)pattern.first ();
    Location loc = curr->location ();
    ACUnit dir (err);
    lmgr.directive (dir, (Unit*)&pattern, curr);
    dir << endu;
    if (!dir.empty ())
      unit->move ((Token*)unit->last (), dir);
    instantiate_intro (err, unit, intro, pattern, target, loc);
  }
}


void JPP_Class::gen_base_intro (IntroductionUnit &intro_unit,
  JPL_Introduction *ii, bool first) const {

  // generate the code for this entry
  intro_unit << (first ? ": " : ", ");

  if (ii->introduced ()->slice_type () == JPL_ClassSlice::CS_NORMAL) {
    // slice-based base class intro
    CObjectInfo *obj = TI_ClassSlice::of (*ii->introduced ())->assoc_obj ();
    assert (obj && obj->Tree ());
    assert (obj->Tree ()->NodeName () == CT_ClassSliceDecl::NodeId ());
    ACSliceInfo *acsi = obj->ClassDB ()->SliceInfo (obj)->definition ();
    const Unit &base = *acsi->base_pattern ();
    for (Token *curr = (Token*)base.first (); curr;
      curr = (Token*)base.next (curr)) {
      intro_unit << *curr << " ";
    }
  }
  else {
    // old-style base class intro with 'baseclass(virtual Bla)'
    // get all the needed information
    CObjectInfo *obj = TI_ClassSlice::of (*ii->introduced ())->assoc_obj ();
    CProtection::Type prot = obj->Protection ();
    bool virt = obj->isVirtual ();
    CTypeInfo *type = obj->FunctionInfo ()->Argument (0u)->TypeInfo ();
      
    if (virt) intro_unit << "virtual ";
    switch (prot) {
      case CProtection::PROT_PRIVATE:   intro_unit << "private "; break;
      case CProtection::PROT_PROTECTED: intro_unit << "protected "; break;
      case CProtection::PROT_PUBLIC:    intro_unit << "public "; break;
      default:
        assert (false);
    }
    intro_unit << *type;
  }
  intro_unit << endu;
}


void JPP_Class::gen_base_intros (list<Unit*> &units, ErrorStream &err,
  CClassInfo *target, LineDirectiveMgr &lmgr) const {

  for (int i = 0; i < _base.length (); i++) {
    // get the current introduction
    JPL_Introduction *ii = _base.lookup (i);

    // create the new unit
    IntroductionUnit *unit = 
      new IntroductionUnit (err, (Unit*)target->Tree ()->token ()->belonging_to ());
    unit->intro (ii);

    // generate the code for this base class introduction
    gen_base_intro (*unit, ii, (i == 0));

    // store the result for the caller
    units.push_back (unit);
  }
}

void JPP_Class::instantiate_intro (ErrorStream &err, Unit *unit,
  CT_Intro *intro, const Unit &pattern, CStructure *target,
  Location &loc) const {
    
  // create a unit with the target class name
  ACUnit target_name (err);
  target_name << target->Name () << endu;
  ACUnit target_qual_name (err);
  target_qual_name << target->QualName () << endu;

  // ... some units for formatting
  ACUnit nl (err);
  nl << endl << endu;
  ACUnit ws (err);
  ws << " " << endu;
  
  int i = 0, e = 0;
  // create a formatted unit and replace the advice name with the target name
  for (Token *curr = (Token*)pattern.first (); curr;
       curr = (Token*)pattern.next (curr), e++) {
    Token *tok = curr;
    if (curr->location ().line () != loc.line ()) {
      for (int l = loc.line (); l < curr->location ().line (); l++)
        unit->append (*((Token*)nl.first ())->duplicate ());
      loc = curr->location ();
    }
    else
      unit->append (*((Token*)ws.first ())->duplicate ());
    if (i < intro->NameIndices () && e == intro->NameIndex (i)) {
      // target class name is used instead of intro token
      if (intro->NameQual (i))
        *unit += target_qual_name;
      else
        *unit += target_name;
      while (e < intro->NameToIndex (i)) {
        e++;
        curr = (Token*)pattern.next (curr);
      }
      i++;
    }
    else
      unit->append (*tok->duplicate ());
  }
}
