/*
    Copyright (C) 2002-4 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: mtc_slave.cc,v 1.24 2004/12/06 12:28:02 pauld Exp $
*/

#include <errno.h>
#include <poll.h>
#include <sys/types.h>
#include <unistd.h>
#include <pbd/error.h>
#include <pbd/failed_constructor.h>
#include <pbd/pthread_utils.h>

#include <midi++/port.h>
#include <ardour/slave.h>
#include <ardour/session.h>
#include <ardour/audioengine.h>
#include <ardour/cycles.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;
using namespace MIDI;

MTC_Slave::MTC_Slave (Session& s, MIDI::Port& p) 
	: session (s),
	  port (p)
{
	reset ();

	port.input()->mtc_time.connect (slot (*this, &MTC_Slave::update_mtc_time));
	port.input()->mtc_qtr.connect (slot (*this, &MTC_Slave::update_mtc_qtr));
	port.input()->mtc_status.connect (slot (*this, &MTC_Slave::update_mtc_status));
}

MTC_Slave::~MTC_Slave()
{
}

void
MTC_Slave::update_mtc_qtr (Parser& p)
{
	cycles_t cnow = get_cycles ();
	jack_nframes_t now = session.engine().frame_time();
	jack_nframes_t qtr;
	static cycles_t last_qtr = 0;

	qtr = (session.frames_per_smpte_frame() / 4);
	mtc_frame += qtr;
	last_qtr = cnow;

	current.guard1++;
	current.position = mtc_frame;
	current.timestamp = now;
	current.guard2++;

	last_inbound_frame = now;
}

void
MTC_Slave::update_mtc_time (const byte *msg, bool was_full)
{
	jack_nframes_t now = session.engine().frame_time();
	jack_nframes_t frame_rate = session.frame_rate();

	mtc_frame = (msg[3] * 60 * 60 * frame_rate) + 
		(msg[2] * 60 * frame_rate) +
		(msg[1] * frame_rate) +
		(msg[0] * session.frames_per_smpte_frame());

	if (was_full) {

	 	current.guard1++;
		current.position = mtc_frame;
		current.timestamp = 0;
		current.guard2++;
	
		session.request_locate (mtc_frame, false);
		update_mtc_status (MIDI::Parser::MTC_Stopped);

		reset ();

	} else {

		if (first_mtc_frame == 0) {
			first_mtc_frame = mtc_frame;
			first_mtc_time = now;
		} 

		current.guard1++;
		current.position = mtc_frame;
		current.timestamp = now;
		current.guard2++;
	}

	last_inbound_frame = now;
}

void
MTC_Slave::handle_locate (const MIDI::byte* mmc_tc)
{
	MIDI::byte mtc[4];
	
	mtc[3] = mmc_tc[0] & 0xf; /* hrs only */
	mtc[2] = mmc_tc[1];
	mtc[1] = mmc_tc[2];
	mtc[0] = mmc_tc[3];

	update_mtc_time (mtc, true);
}

void
MTC_Slave::update_mtc_status (MIDI::Parser::MTC_Status status)
{

#ifdef DEBUG_MTC
	cerr << "received new MTC status\n";
#endif

	switch (status) {
	case MTC_Stopped:
#ifdef DEBUG_MTC
		cerr << "MTC stopped ...\n";
#endif 
		mtc_speed = 0.0f;
		mtc_frame = 0;

		current.guard1++;
		current.position = mtc_frame;
		current.timestamp = 0;
		current.guard2++;

		// session.request_transport_speed (0);
		break;

	case MTC_Forward:
#ifdef DEBUG_MTC
		cerr << "MTC starts forward ...\n";
#endif
		mtc_speed = 0.0f;
		mtc_frame = 0;

		current.guard1++;
		current.position = mtc_frame;
		current.timestamp = 0;
		current.guard2++;

		break;

	case MTC_Backward:
#ifdef DEBUG_MTC
		cerr << "MTC starts backward ...\n";
#endif
		mtc_speed = 0.0f;
		mtc_frame = 0;

		current.guard1++;
		current.position = mtc_frame;
		current.timestamp = 0;
		current.guard2++;

		break;
	}
}

void
MTC_Slave::read_current (SafeTime *st) const
{
	int tries = 0;
	do {
		if (tries == 10) {
			error << _("MTC Slave: atomic read of current time failed, sleeping!") << endmsg;
			usleep (20);
			tries = 0;
		}

		*st = current;
		tries++;

	} while (st->guard1 != st->guard2);
}

bool
MTC_Slave::locked () const
{
	return port.input()->mtc_locked();
}

bool
MTC_Slave::ok() const
{
	return true;
}

bool 
MTC_Slave::speed_and_position (float& speed, jack_nframes_t& pos)
{
	jack_nframes_t now = session.engine().frame_time();
	SafeTime last;
	jack_nframes_t frame_rate;
	jack_nframes_t elapsed;
	float speed_now;

	read_current (&last);

	if (first_mtc_time == 0) {
		// cerr << "MTC not started\n";
		speed = 0;
		pos = last.position;
		return true;
	}
	
	/* no timecode for 1/4 second ? conclude that its stopped */

	if (last_inbound_frame && now > last_inbound_frame && now - last_inbound_frame > session.frame_rate() / 4) {

		cerr << "no MTC received for more than 0.25 seconds"
		     << endl;

		mtc_speed = 0;
		pos = last.position;
		update_mtc_status (MIDI::Parser::MTC_Stopped);
		return false;
	}

	frame_rate = session.frame_rate();

	speed_now = (float) ((last.position - first_mtc_frame) / (double) (now - first_mtc_time));

	accumulator[accumulator_index++] = speed_now;

	if (accumulator_index >= accumulator_size) {
		have_first_accumulated_speed = true;
		accumulator_index = 0;
	}

	if (have_first_accumulated_speed) {
		float total = 0;

		for (int32_t i = 0; i < accumulator_size; ++i) {
			total += accumulator[i];
		}

		mtc_speed = total / accumulator_size;

	} else {

		mtc_speed = speed_now;

	}

	// cerr << "smoothed speed = " << mtc_speed << endl;

	if (mtc_speed == 0.0f) {

		elapsed = 0;

	} else {
	
		/* scale elapsed time by the current MTC speed */
		
		if (last.timestamp && (now > last.timestamp)) {
			elapsed = (jack_nframes_t) floor (mtc_speed * (now - last.timestamp));
		} else {
			elapsed = 0; /* XXX is this right? */
		}
	}

	/* now add the most recent timecode value plus the estimated elapsed interval */

	pos =  elapsed + last.position;

	// cerr << "pos = " << pos << " using " << elapsed << " since TC @ " << mtc_speed << endl;
	
	speed = mtc_speed;
	return true;
}

jack_nframes_t
MTC_Slave::resolution() const
{
	return session.frames_per_smpte_frame();
}

void
MTC_Slave::reset ()
{
	/* XXX massive thread safety issue here. MTC could
	   be being updated as well call this. but this
	   supposed to be a realtime-safe call.
	*/
	
	port.input()->reset_mtc_state ();
	
	last_inbound_frame = 0;
	current.guard1++;
	current.position = 0;
	current.timestamp = 0;
	current.guard2++;
	first_mtc_frame = 0;
	first_mtc_time = 0;

	accumulator_index = 0;
	have_first_accumulated_speed = false;
	mtc_speed = 0;
}
