/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: insert.h,v 1.36 2004/12/11 23:57:44 pauld Exp $
*/

#ifndef __ardour_insert_h__
#define __ardour_insert_h__

#include <vector>
#include <string>
#include <exception>

#include <sigc++/signal_system.h>
#include <ardour/ardour.h>
#include <ardour/redirect.h>
#include <ardour/plugin_state.h>

class XMLNode;

namespace MIDI {
	class Port;
}

namespace ARDOUR {

class Session;
class Plugin;
class Route;

class Insert : public Redirect
{
  public:
	Insert(Session& s, Placement p);
	Insert(Session& s, string name, Placement p);

	Insert(Session& s, Placement p, int imin, int imax, int omin, int omax);
	
	virtual ~Insert() { }

	virtual void run (vector<Sample *>& bufs, uint32_t nbufs, jack_nframes_t nframes, jack_nframes_t offset) = 0;
	virtual void activate () {}
	virtual void deactivate () {}

	virtual int  require_io (int32_t, int32_t, bool) = 0;
};

class PortInsert : public Insert 
{
  public:
	PortInsert (Session&, Placement);
	PortInsert (Session&, const XMLNode&);
	PortInsert (const PortInsert&);
	~PortInsert ();

	XMLNode& get_state(void);
	int set_state(const XMLNode&);

	void init ();
	void run (vector<Sample *>& bufs, uint32_t nbufs, jack_nframes_t nframes, jack_nframes_t offset);

	jack_nframes_t latency();

	int require_io (int32_t, int32_t, bool);
};

struct PluginInsertState : public RedirectState
{
    PluginInsertState (std::string why) 
	    : RedirectState (why) {}
    ~PluginInsertState() {}

    PluginState plugin_state;
};

class PluginInsert : public Insert
{
  public:
	PluginInsert (Session&, Plugin&, Placement);
	PluginInsert (Session&, const XMLNode&);
	PluginInsert (const PluginInsert&);
	~PluginInsert ();

	static const string port_automation_node_name;
	
	XMLNode& get_state(void);
	int set_state(const XMLNode&);

	StateManager::State* state_factory (std::string why) const;
	Change restore_state (StateManager::State&);

	void run (vector<Sample *>& bufs, uint32_t nbufs, jack_nframes_t nframes, jack_nframes_t offset);
	void activate ();
	void deactivate ();

	void set_block_size (jack_nframes_t nframes);

	uint32_t output_streams() const;
	uint32_t input_streams() const;

	int require_io (int32_t, int32_t, bool hard_input_limit);

	void reset_midi_control (MIDI::Port*, bool);
	void send_all_midi_feedback ();

	void set_parameter (uint32_t port, float val);

	AutoState get_port_automation_state (uint32_t port);
	void set_port_automation_state (uint32_t port, AutoState);
	
	float default_parameter_value (uint32_t which);

	Plugin& plugin(uint32_t num=0) const {
		if (num < _plugins.size()) { 
			return *_plugins[num];
		} else {
			return *_plugins[0]; // we always have one
		}
	}

	int set_count (uint32_t num);
	uint32_t get_count () { return _plugins.size(); }

	string describe_parameter (uint32_t);

	jack_nframes_t latency();

	void transport_stopped (jack_nframes_t now);
	void automation_snapshot (jack_nframes_t now);

  protected:
	void store_state (PluginInsertState&) const;

  private:

	void parameter_changed (uint32_t, float);
	
	vector<Plugin*> _plugins;
	void automation_run (vector<Sample *>& bufs, uint32_t nbufs, jack_nframes_t nframes, jack_nframes_t offset);
	void connect_and_run (vector<Sample *>& bufs, uint32_t nbufs, jack_nframes_t nframes, jack_nframes_t offset, bool with_auto, jack_nframes_t now = 0);

	void init ();
	void set_automatable ();
	void auto_state_changed (uint32_t which);
	void automation_list_creation_callback (uint32_t, AutomationList&);

	Plugin* plugin_factory (Plugin&);
};

}; /* namespace ARDOUR */

#endif /* __ardour_insert_h__ */
