/*
    Copyright (C) 1999-2002 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: session_time.cc,v 1.8 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <unistd.h>

#include <ardour/timestamps.h>

#include <pbd/error.h>

#include <ardour/ardour.h>
#include <ardour/configuration.h>
#include <ardour/audioengine.h>
#include <ardour/session.h>
#include <ardour/tempo.h>

#include "i18n.h"

using namespace ARDOUR;
using namespace SigC;

/* BBT TIME*/

void
Session::bbt_time (jack_nframes_t when, BBT_Time& bbt)
{
	_tempo_map->bbt_time (when, bbt);
}

/* SMPTE TIME */


void
Session::set_smpte_offset (jack_nframes_t off)
{
	_smpte_offset = off;
	last_smpte_valid = false;
	 SMPTEOffsetChanged (); /* EMIT SIGNAL */
}

void
Session::smpte_time (jack_nframes_t when, SMPTE_Time& smpte)
{
	if (last_smpte_valid && when == last_smpte_when) {
		smpte = last_smpte;
		return;
	}

	jack_nframes_t offset_sample;
	jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
                                                                                                                   
        if (when < _smpte_offset) {
                offset_sample = (_smpte_offset - when);
                smpte.negative = true;
        } else {
                offset_sample =  when - _smpte_offset;
                smpte.negative = false;
        }

	left = offset_sample;
        hrs = (int) floor (left / (frame_rate() * 60.0f * 60.0f));
        left -= (jack_nframes_t) floor (hrs * frame_rate() * 60.0f * 60.0f);
        mins = (int) floor (left / (frame_rate() * 60.0f));
        left -= (jack_nframes_t) floor (mins * frame_rate() * 60.0f);
        secs = (int) floor (left / frame_rate() * 1.0f);
        left -= (jack_nframes_t) floor (secs * frame_rate() * 1.0f);
        frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
                                                                                                                   
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;

	last_smpte_when = when;
	last_smpte = smpte;
	last_smpte_valid = true;
}

void
Session::smpte_time_subframes (jack_nframes_t when, SMPTE_Time& smpte)
{
	if (last_smpte_valid && when == last_smpte_when) {
                smpte = last_smpte;
                return;
        }
                                                                                                                   
        jack_nframes_t offset_sample;
        jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
	long subframes;
                                                                                                                   
        if (when < _smpte_offset) {
                offset_sample = (_smpte_offset - when);
                smpte.negative = true;
        } else {
                offset_sample =  when - _smpte_offset;
                smpte.negative = false;
        }
                                                                                                                   
        left = offset_sample;
        hrs = (int) floor (left / (frame_rate() * 60.0f * 60.0f));
        left -= (jack_nframes_t) floor (hrs * frame_rate() * 60.0f * 60.0f);
        mins = (int) floor (left / (frame_rate() * 60.0f));
        left -= (jack_nframes_t) floor (mins * frame_rate() * 60.0f);
        secs = (int) floor (left / frame_rate() * 1.0f);
        left -= (jack_nframes_t) floor (secs * frame_rate() * 1.0f);
        frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
	left -= (jack_nframes_t) floor (frames * frame_rate() / smpte_frames_per_second);
	/* eighty bits in a SMPTE frame */
	subframes = (int) floor (left / (frame_rate() / (smpte_frames_per_second * 80)));
	
        smpte.subframes = subframes;
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;
                                                                                                                   
        last_smpte_when = when;
        last_smpte = smpte;
        last_smpte_valid = true;

}

void
Session::smpte_duration (jack_nframes_t when, SMPTE_Time& smpte) const
{
        jack_nframes_t left;
        long hrs;
        long mins;
        long secs;
        long frames;
        long subframes;

	left = when;
        hrs = (int) floor (left / (frame_rate() * 60.0f * 60.0f));
        left -= (jack_nframes_t) floor (hrs * frame_rate() * 60.0f * 60.0f);
        mins = (int) floor (left / (frame_rate() * 60.0f));
        left -= (jack_nframes_t) floor (mins * frame_rate() * 60.0f);
        secs = (int) floor (left / frame_rate() * 1.0f);
        left -= (jack_nframes_t) floor (secs * frame_rate() * 1.0f);
        frames = (int) floor (left / (frame_rate() / smpte_frames_per_second));
        left -= (jack_nframes_t) floor (frames * frame_rate() / smpte_frames_per_second);
        /* eighty bits in a SMPTE frame */
        subframes = (int) floor (left / (frame_rate() / (smpte_frames_per_second * 80)));
                                                                                                                   
        smpte.subframes = subframes;
        smpte.frames = frames;
        smpte.seconds = secs;
        smpte.minutes = mins;
        smpte.hours = hrs;

/*  i've left the original here.. i can't really test the new one.
it was only accurate to frames. maybe it was intentional ..nm

	double sample;
	double secs;
	double computed_secs;
	double approx_secs;
	long hrs;
	long mins;
	
	sample = when;

	approx_secs = (sample / frame_rate());
	secs = approx_secs;
	hrs =  (long) floor (secs / 3600.0);
	secs -= (hrs * 3600.0);
	mins = (long) floor (secs / 60.0);
	secs -= (mins * 60.0);

	computed_secs = (hrs * 3600.0) + (mins * 60.0) + floor (secs);

	smpte.frames = (long) ((approx_secs - computed_secs) * smpte_frames_per_second);
	smpte.seconds = (long) secs;
	smpte.minutes = mins;
	smpte.hours = hrs;

*/
}

void
Session::smpte_duration_string (char* buf, jack_nframes_t when) const
{
	SMPTE_Time smpte;

	smpte_duration (when, smpte);
	sprintf (buf, "%02ld:%02ld:%02ld:%02ld", smpte.hours, smpte.minutes, smpte.seconds, smpte.frames);
}

void
Session::smpte_time (SMPTE_Time &t)

{
	smpte_time (_transport_frame, t);
}

int
Session::jack_sync_callback (jack_transport_state_t state,
			     jack_position_t* pos)
{
	switch (state) {
	case JackTransportStopped:
		if (_transport_frame != pos->frame) {
			request_locate (pos->frame, false);
			return FALSE;
		} else {
			return TRUE;
		}
		
	case JackTransportStarting:
		return post_transport_work == 0;

	case JackTransportRolling:
		warning << _("JACK didn't wait for us!") << endmsg;
		break;

	default:
		error << compose (_("Unknown JACK transport state %1 in sync callback"), state)
		      << endmsg;
	} 

	return TRUE;
}

void
Session::jack_timebase_callback (jack_transport_state_t state,
				 jack_nframes_t nframes,
				 jack_position_t* pos,
				 int new_position)
{
	BBT_Time bbt;

	/* frame info */

	pos->frame = _transport_frame;
	pos->valid = JackPositionTimecode;

	/* BBT info */
	
	if (_tempo_map) {

		TempoMap::Metric metric (_tempo_map->metric_at (_transport_frame));
		_tempo_map->bbt_time_with_metric (_transport_frame, bbt, metric);
		
		pos->bar = bbt.bars;
		pos->beat = bbt.beats;
		pos->tick = bbt.ticks;

		// XXX still need to set bar_start_tick

		pos->beats_per_bar = metric.meter().beats_per_bar();
		pos->beat_type = metric.meter().note_divisor();
		pos->ticks_per_beat = Meter::ticks_per_beat;
		pos->beats_per_minute = metric.tempo().beats_per_minute();

		pos->valid = jack_position_bits_t (pos->valid | JackPositionBBT);
	}

#if 0
	/* SMPTE info */

	t.smpte_offset = _smpte_offset;
	t.smpte_frame_rate = smpte_frames_per_second;

	if (_transport_speed) {

		if (auto_loop) {

			Location* location = _locations.auto_loop_location();

			if (location) {

				t.transport_state = JackTransportLooping;
				t.loop_start = location->start();
				t.loop_end = location->end();
				t.valid = jack_transport_bits_t (t.valid | JackTransportLoop);

			} else {

				t.loop_start = 0;
				t.loop_end = 0;
				t.transport_state = JackTransportRolling;

			}

		} else {

			t.loop_start = 0;
			t.loop_end = 0;
			t.transport_state = JackTransportRolling;

		}

	} 

#endif		
}

jack_nframes_t
Session::convert_to_frames_at (jack_nframes_t position, AnyTime& any)
{
	double secs;
	
	switch (any.type) {
	case AnyTime::BBT:
		return _tempo_map->frame_time (position, any.bbt);
		break;

	case AnyTime::SMPTE:
		/* XXX need to handle negative values */
		secs = any.smpte.hours * 60 * 60;
		secs += any.smpte.minutes * 60;
		secs += any.smpte.seconds;
		secs += any.smpte.frames / smpte_frames_per_second;
		return _smpte_offset + (jack_nframes_t) floor (secs / frame_rate());
		break;

	case AnyTime::Seconds:
		return (jack_nframes_t) floor (any.seconds / frame_rate());
		break;

	case AnyTime::Frames:
		return any.frames;
		break;
	}

	return any.frames;
}
