/*
    Copyright (C) 2000-2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: region.cc,v 1.62 2004/02/29 23:33:56 pauld Exp $
*/

#include <cmath>
#include <climits>

#include <sigc++/bind.h>
#include <sigc++/class_slot.h>

#include <pbd/lockmonitor.h>
#include <pbd/xml++.h>

#include <ardour/region.h>
#include <ardour/playlist.h>
#include <ardour/session.h>

#include "i18n.h"

using namespace ARDOUR;

Change Region::FadeChanged = ARDOUR::new_change ();
Change Region::SyncOffsetChanged = ARDOUR::new_change ();
Change Region::MuteChanged = ARDOUR::new_change ();
Change Region::OpacityChanged = ARDOUR::new_change ();
Change Region::LockChanged = ARDOUR::new_change ();
Change Region::LayerChanged = ARDOUR::new_change ();

SigC::Signal1<void,Region *> Region::RegionCreated;

Region::Region (jack_nframes_t start, jack_nframes_t length, const string& name, layer_t layer, Region::Flag flags)
{
	/* basic Region constructor */

	_id = ARDOUR::new_id();
	_flags = flags;
	_sync_offset = 0;
	_playlist = 0;
	_read_data_count = 0;
	_frozen = 0;
	pending_changed = Change (0);

	_name = name;
	_start = start; 
	_length = length; 
	_position = 0; 
	_layer = layer;
	_current_state_id = 0;
	_read_data_count = 0;
	_first_edit = true;

	// cerr << "region from basic constructor " << _name << " [" << this << ']' << endl;
}

Region::Region (const Region& other, jack_nframes_t offset, jack_nframes_t length, const string& name, layer_t layer, Flag flags)
{
	/* create a new Region, that is part of an existing one */

	_id = ARDOUR::new_id();
	_frozen = 0;
	pending_changed = Change (0);
	_sync_offset = 0;
	_playlist = 0;
	_read_data_count = 0;

	_start = other._start + offset; 
	_length = length; 
	_name = name;
	_position = 0; 
	_layer = layer; 
	_flags = flags;
	_current_state_id = 0;
	_first_edit = true;

	// cerr << "region from partial copy constructor " << _name << " [" << this << ',' << &other << ']' << endl;

}

Region::Region (const Region &other, const string& name)
{
	_id = ARDOUR::new_id();
	_frozen = 0;
	pending_changed = Change (0);
	_sync_offset = 0;
	_playlist = 0;
	_read_data_count = 0;

	_name = name;

	_start = other._start; 
	_length = other._length; 
	_position = other._position; 
	_layer = other._layer; 
	_flags = other._flags;
	_current_state_id = 0;
	_first_edit = true;

	// cerr << "region from renamed copy constructor " << _name << " (was " << &other << ")" << " [" << this << ']' << endl;
}

Region::Region (const Region &other)
{
	/* Pure copy constructor */

	_id = other.id();
	_frozen = 0;
	pending_changed = Change (0);
	_sync_offset = 0;
	_playlist = 0;
	_read_data_count = 0;
	_first_edit = true;

	other._first_edit = true;

	if (other._extra_xml) {
		_extra_xml = new XMLNode (*other._extra_xml);
	} else {
		_extra_xml = 0;
	}

	_start = other._start;
	_length = other._length; 
	_name = other._name;
	_position = other._position; 
	_layer = other._layer; 

	_flags = other._flags;
	_current_state_id = 0;

	// cerr << "region from copy constructor " << _name << " [" << this << ',' << &other << ']' << endl;

	/* NOTE: no RegionCreated signal emitted here. This is the copy constructor */
}

Region::Region (const XMLNode& node)
{
	_id = 0;
	_frozen = 0;
	pending_changed = Change (0);
	_sync_offset = 0;
	_playlist = 0;
	_read_data_count = 0;
	_start = 0; 
	_length = 0;
	_name = "error: XML did not reset this";
	_position = 0; 
	_layer = 0;
	_flags = Flag (0);
	_current_state_id = 0;

	if (set_state (node)) {
		throw failed_constructor();
	}

	// cerr << "Region from XML: " << _name << " [" << this << ']' << endl;
}

Region::~Region ()
{
}

void
Region::set_playlist (Playlist* pl)
{
	_playlist = pl;
}

void
Region::store_state (RegionState& state) const
{
	state._start = _start;
	state._length = _length;
	state._position = _position;
	state._flags = _flags;
	state._sync_offset = _sync_offset;
	state._layer = _layer;
	state._name = _name;
}	

Change
Region::restore_and_return_flags (RegionState& state)
{
	Change what_changed = Change (0);

	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		
		if (_start != state._start) {
			what_changed = Change (what_changed|StartChanged);	
			_start = state._start;
		}
		if (_length != state._length) {
			what_changed = Change (what_changed|LengthChanged);
			_length = state._length;
		}
		if (_position != state._position) {
			what_changed = Change (what_changed|PositionChanged);
			_position = state._position;
		} 
		if (_sync_offset != state._sync_offset) {
			_sync_offset = state._sync_offset;
			what_changed = Change (what_changed|SyncOffsetChanged);
		}
		if (_layer != state._layer) {
			what_changed = Change (what_changed|LayerChanged);
			_layer = state._layer;
		}
		if (_name != state._name) {
			what_changed = Change (what_changed|NameChanged);
		}

		unsigned int old_flags = _flags;
		_flags = Flag (state._flags);
		
		if ((old_flags ^ state._flags) & Muted) {
			what_changed = Change (what_changed|MuteChanged);
		}
		if ((old_flags ^ state._flags) & Opaque) {
			what_changed = Change (what_changed|OpacityChanged);
		}
		if ((old_flags ^ state._flags) & Locked) {
			what_changed = Change (what_changed|LockChanged);
		}

		_first_edit = state._first_edit;
	}

	return what_changed;
}

void
Region::set_name (string str)

{
	if (_name != str) {
		_name = str; 
		send_change (NameChanged);
	}
}

void
Region::set_length (jack_nframes_t len, void *src)
{
	if (_flags & Locked) {
		return;
	}

	if (_length != len && len != 0) {

		if (!verify_length (len)) {
			return;
		}
		
		_length = len;

		rename_after_first_edit ();
		recompute_at_end ();
		maybe_uncopy ();

		if (!_frozen) {
			char buf[64];
			sprintf (buf, "length set to %u", len);
			save_state (buf);
		}

		send_change (LengthChanged);
	}
}

void
Region::maybe_uncopy ()
{
}

void
Region::rename_after_first_edit ()
{
	if (_first_edit && _playlist) {
		_name = _playlist->session().new_region_name (_name);
		send_change (NameChanged);
		_first_edit = false;
	}
}
	
void
Region::set_position (jack_nframes_t pos, void *src)
{
	if (_flags & Locked) {
		return;
	}

	if (_position != pos) {
		_position = pos;

		if (!_frozen) {
			char buf[64];
			sprintf (buf, "position set to %u", pos);
			save_state (buf);
		}
	}

	/* do this even if the position is the same. this helps out
	   a GUI that has moved its representation already.
	*/

	send_change (PositionChanged);
}

void
Region::set_position_on_top (jack_nframes_t pos, void *src)
{
	if (_flags & Locked) {
		return;
	}

	if (_position != pos) {
		_position = pos;

		if (!_frozen) {
			char buf[64];
			sprintf (buf, "position set to %u", pos);
			save_state (buf);
		}
	}

	_playlist->raise_region_to_top (*this);

	/* do this even if the position is the same. this helps out
	   a GUI that has moved its representation already.
	*/
	
	send_change (PositionChanged);
}

void
Region::nudge_position (long n, void *src)
{
	if (_flags & Locked) {
		return;
	}

	if (n == 0) {
		return;
	}
	
	if (n > 0) {
		if (_position > max_frames - n) {
			_position = max_frames;
		} else {
			_position += n;
		}
	} else {
		if (_position < (jack_nframes_t) -n) {
			_position = 0;
		} else {
			_position += n;
		}
	}

	if (!_frozen) {
		char buf[64];
		sprintf (buf, "position set to %u", _position);
		save_state (buf);
	}

	send_change (PositionChanged);
}

void
Region::set_start (jack_nframes_t pos, void *src)
{
	if (_flags & Locked) {
		return;
	}
	/* This just sets the start, nothing else. It effectively shifts
	   the contents of the Region within the overall extent of the Source,
	   without changing the Region's position or length
	*/

	if (_start != pos) {

		if (!verify_start (pos)) {
			return;
		}

		_start = pos;

		rename_after_first_edit ();

		if (!_frozen) {
			char buf[64];
			sprintf (buf, "start set to %u", pos);
			save_state (buf);
		}

		send_change (StartChanged);
	}
}

void
Region::trim_start (jack_nframes_t new_position, void *src)
{
	if (_flags & Locked) {
		return;
	}
	jack_nframes_t new_start;
	long start_shift;
	
	if (new_position > _position) {
		start_shift = new_position - _position;
	} else {
		start_shift = -(_position - new_position);
	}

	if (start_shift > 0) {

		if (_start > max_frames - start_shift) {
			new_start = max_frames;
		} else {
			new_start = _start + start_shift;
		}

		if (!verify_start (new_start)) {
			return;
		}

	} else if (start_shift < 0) {

		if (_start < (jack_nframes_t) -start_shift) {
			new_start = 0;
		} else {
			new_start = _start + start_shift;
		}
	} else {
		return;
	}

	if (new_start == _start) {
		return;
	}
	
	_start = new_start;
	rename_after_first_edit ();

	if (!_frozen) {
		char buf[64];
		sprintf (buf, "slipped start to %u", _start);
		save_state (buf);
	}

	send_change (StartChanged);
}

void
Region::trim_front (jack_nframes_t new_position, void *src)
{
	if (_flags & Locked) {
		return;
	}

	jack_nframes_t end = _position + _length - 1;

	if ((new_position < end) && ((new_position + _start) >= _position)) {
		jack_nframes_t newlen;

		if (new_position > _position) {
			newlen = _length - (new_position - _position);
		} else {
			newlen = _length + (_position - new_position);
		}

		trim_to_internal (new_position, newlen, src);
		recompute_at_start ();
	}
}

void
Region::trim_end (jack_nframes_t new_position, void *src)
{
	if (_flags & Locked) {
		return;
	}
	if (new_position > _position) {
		trim_to_internal (_position, new_position - _position, this);
		recompute_at_end ();
	}
}

void
Region::trim_to (jack_nframes_t position, jack_nframes_t length, void *src)
{
	if (_flags & Locked) {
		return;
	}
	trim_to_internal (position, length, src);
	recompute_at_start ();
	recompute_at_end ();
}

void
Region::trim_to_internal (jack_nframes_t position, jack_nframes_t length, void *src)

{
	long start_shift;
	jack_nframes_t new_start;

	if (_flags & Locked) {
		return;
	}

	if (position > _position) {
		start_shift = position - _position;
	} else {
		start_shift = -(_position - position);
	}

	if (start_shift > 0) {

		if (_start > max_frames - start_shift) {
			new_start = max_frames;
		} else {
			new_start = _start + start_shift;
		}


	} else if (start_shift < 0) {

		if (_start < (jack_nframes_t) -start_shift) {
			new_start = 0;
		} else {
			new_start = _start + start_shift;
		}
	} else {
		new_start = _start;
	}

	if (!verify_start_and_length (new_start, length)) {
		return;
	}

	Change what_changed = Change (0);

	if (_start != new_start) {
		_start = new_start;
		what_changed = Change (what_changed|StartChanged);
	}
	if (_length != length) {
		_length = length;
		what_changed = Change (what_changed|LengthChanged);
	}
	if (_position != position) {
		_position = position;
		what_changed = Change (what_changed|PositionChanged);
	}
	
	if (what_changed & (StartChanged|LengthChanged)) {
		rename_after_first_edit ();
	} 

	if (what_changed) {
		
		if (!_frozen) {
			char buf[64];
			sprintf (buf, "trimmed to %u-%u", _position, _position+_length-1);
			save_state (buf);
		}

		send_change (what_changed);
	}
}	

void
Region::set_muted (bool yn)
{
	if (muted() != yn) {
		if (!_frozen) {
			char buf[64];
			if (yn) {
				sprintf (buf, "muted");
				_flags = Flag (_flags|Muted);
			} else {
				sprintf (buf, "unmuted");
				_flags = Flag (_flags & ~Muted);
			}
			save_state (buf);
		}

		send_change (MuteChanged);
	}
}

void
Region::set_opaque (bool yn)
{
	if (opaque() != yn) {
		if (!_frozen) {
			char buf[64];
			if (yn) {
				sprintf (buf, "opaque");
				_flags = Flag (_flags|Opaque);
			} else {
				sprintf (buf, "translucent");
				_flags = Flag (_flags & ~Opaque);
			}
			save_state (buf);
		}
		send_change (OpacityChanged);
	}
}

void
Region::set_locked (bool yn)
{
	if (locked() != yn) {
		if (!_frozen) {
			char buf[64];
			if (yn) {
				sprintf (buf, "locked");
				_flags = Flag (_flags|Locked);
			} else {
				sprintf (buf, "unlocked");
				_flags = Flag (_flags & ~Locked);
			}
			save_state (buf);
		}
		send_change (LockChanged);
	}
}

void
Region::set_sync_offset (jack_nframes_t n)
{
	if (_sync_offset != n && n < _length) {
		_sync_offset = n;
		
		if (!_frozen) {
			char buf[64];
			maybe_uncopy ();
			sprintf (buf, "sync offset set to %u", _sync_offset);
			save_state (buf);
		}

		send_change (SyncOffsetChanged);
	}
}

void
Region::raise ()
{
	if (_playlist == 0) {
		return;
	}

	_playlist->raise_region (*this);
}

void
Region::lower ()
{
	if (_playlist == 0) {
		return;
	}

	_playlist->lower_region (*this);
}

void
Region::raise_to_top ()
{
	if (_playlist == 0) {
		return;
	}

	_playlist->raise_region_to_top (*this);
}

void
Region::lower_to_bottom ()
{
	if (_playlist == 0) {
		return;
	}

	_playlist->lower_region_to_bottom (*this);
}

void
Region::set_layer (layer_t l)
{
	if (_layer != l) {
		_layer = l;

		if (!_frozen) {
			char buf[64];
			sprintf (buf, "layer set to %lu", _layer);
			save_state (buf);
		}

		send_change (LayerChanged);
	}
}

XMLNode&
Region::_get_state (bool isshort)
{
	XMLNode *node = new XMLNode ("Region");
	char buf[64];
	
	sprintf (buf, "%Lu", _id);
	node->add_property ("id", buf);
	node->add_property ("name", _name);
	sprintf (buf, "%u", _start);
	node->add_property ("start", buf);
	sprintf (buf, "%u", _length);
	node->add_property ("length", buf);
	sprintf (buf, "%u", _position);
	node->add_property ("position", buf);

	/* note: flags are stored by derived classes */

	sprintf (buf, "%d", (int) _layer);
	node->add_property ("layer", buf);
	sprintf (buf, "%u", _sync_offset);
	node->add_property ("sync-offset", buf);

	return *node;
}

XMLNode&
Region::get_short_state ()
{
	return _get_state (true);
}

XMLNode&
Region::get_state ()
{
	return _get_state (false);
}

int
Region::set_state (const XMLNode& node)
{
	const XMLNodeList& nlist = node.children();
	const XMLProperty *prop;

	if (_extra_xml) {
		delete _extra_xml;
		_extra_xml = 0;
	}

	if ((prop = node.property ("id")) == 0) {
		error << _("Session: XMLNode describing a Region is incomplete (no id)") << endmsg;
		return -1;
	}

	sscanf (prop->value().c_str(), "%Lu", &_id);

	if ((prop = node.property ("name")) == 0) {
		error << _("Session: XMLNode describing a Region is incomplete (no name)") << endmsg;
		return -1;
	}

	_name = prop->value();

	if ((prop = node.property ("start")) != 0) {
		_start = (jack_nframes_t) atoi (prop->value().c_str());
	}

	if ((prop = node.property ("length")) != 0) {
		_length = (jack_nframes_t) atoi (prop->value().c_str());
	}

	if ((prop = node.property ("position")) != 0) {
		_position = (jack_nframes_t) atoi (prop->value().c_str());
	}

	if ((prop = node.property ("layer")) != 0) {
		_layer = (layer_t) atoi (prop->value().c_str());
	}

	/* note: derived classes set flags */

	if ((prop = node.property ("sync-offset")) != 0) {
		_sync_offset = (jack_nframes_t) atoi (prop->value().c_str());
	}
	
	for (XMLNodeConstIterator niter = nlist.begin(); niter != nlist.end(); ++niter) {
		
		XMLNode *child;
		
		child = (*niter);
		
		if (child->name () == "extra") {
			_extra_xml = new XMLNode (*child);
			break;
		}
	}

	_first_edit = true;

	return 0;
}

void
Region::freeze ()
{
	_frozen++;
}

void
Region::thaw (const string& why)
{
	Change what_changed = Change (0);

	{
		LockMonitor lm (lock, __LINE__, __FILE__);

		if (_frozen && --_frozen > 0) {
			return;
		}

		if (pending_changed) {
			what_changed = pending_changed;
			pending_changed = Change (0);
		}
	}

	if (what_changed == Change (0)) {
		return;
	}
		
	save_state (why);
	StateChanged (what_changed);
}

void
Region::send_change (Change what_changed)
{
	{
		LockMonitor lm (lock, __LINE__, __FILE__);
		if (_frozen) {
			pending_changed = Change (pending_changed|what_changed);
			return;
		} 
	}

	StateManager::send_state_changed (what_changed);
}

