/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: insert.h,v 1.21 2004/02/22 23:04:01 taybin Exp $
*/

#ifndef __ardour_insert_h__
#define __ardour_insert_h__

#include <vector>
#include <string>

#include <sigc++/signal_system.h>
#include <ardour/ardour.h>
#include <ardour/redirect.h>
#include <ardour/plugin_state.h>

#include <glib.h>

class XMLNode;

namespace LADSPA {
	class Plugin;
}

namespace MIDI {
	class Port;
}

namespace ARDOUR {

class Session;

class Insert : public Redirect
{
  public:
	Insert(Session& s, Redirect::Placement p);
	Insert(Session& s, string name, Redirect::Placement p);
	Insert(Session& s, Redirect::Placement p, int imin, int imax, int omin, int omax);
	
	virtual ~Insert() { }

	virtual void run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset) = 0;
	virtual void activate () {}
	virtual void deactivate () {}
};

class PortInsert : public Insert 
{
  public:
	PortInsert (Session&, Redirect::Placement);
	PortInsert (Session&, const XMLNode&);
	PortInsert (const PortInsert&);
	~PortInsert ();

	XMLNode& get_state(void);
	int set_state(const XMLNode&);

	void init ();
	void run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset);

	jack_nframes_t latency();
};

struct PluginInsertState : public RedirectState
{
    PluginInsertState (std::string why) 
	    : RedirectState (why) {}
    ~PluginInsertState() {}

    PluginState plugin_state;
};

class PluginInsert : public Insert
{
  public:
	PluginInsert (Session&, LADSPA::Plugin&, Redirect::Placement, unsigned int count=1);
	PluginInsert (Session&, const XMLNode&);
	PluginInsert (const PluginInsert&);
	~PluginInsert ();

	static const string port_automation_node_name;
	
	XMLNode& get_state(void);
	int set_state(const XMLNode&);

	StateManager::State* state_factory (std::string why) const;
	Change restore_state (StateManager::State&);

	void run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset);
	void activate ();
	void deactivate ();

	void reset_midi_control (MIDI::Port*, bool);

	void set_control_port (unsigned long port, float val);

	void set_port_automation_playback (unsigned long port, bool yn);
	bool get_port_automation_playback (unsigned long port) const;
	void set_port_automation_record (unsigned long port, AutomationList::Mode);
	AutomationList::Mode get_port_automation_record (unsigned long port) const;
	
	float default_parameter_value (unsigned long which);

	LADSPA::Plugin& plugin(unsigned int num=0) const
		{
			if (num < _plugins.size()) return *_plugins[num];
			else return *_plugins[0]; // we always have one
		}

	void set_count (unsigned int num);
	unsigned int get_count () { return _plugins.size(); }
	
	string describe_parameter (unsigned long);

	jack_nframes_t latency();

	SigC::Signal0<void> AutomationPlaybackChanged;
	SigC::Signal0<void> AutomationRecordChanged;

  protected:
	void store_state (PluginInsertState&) const;

  private:

	void control_port_changed (unsigned long, float);
	
	vector<LADSPA::Plugin*> _plugins;
	void automation_run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset);
	void connect_and_run (vector<Sample *>& bufs, jack_nframes_t nframes, jack_nframes_t offset, bool with_auto, jack_nframes_t now = 0);

	bool* _port_automation_playback;
	AutomationList::Mode* _port_automation_record;

	void init ();
	void set_automatable ();
};

}; /* namespace ARDOUR */

#endif /* __ardour_insert_h__ */
