/*
    Copyright (C) 2000 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: crossfade.h,v 1.6 2004/02/20 03:14:33 pauld Exp $
*/

#ifndef __ardour_overlap_h__
#define __ardour_overlap_h__

#include <vector>
#include <sigc++/signal_system.h>

#include <pbd/undo.h>

#include <ardour/ardour.h>
#include <ardour/curve.h>
#include <ardour/audioregion.h>
#include <ardour/state_manager.h>

namespace ARDOUR {

class AudioRegion;
class Playlist;

struct CrossfadeState : public StateManager::State {
    CrossfadeState (std::string reason) : StateManager::State (reason) {}

    UndoAction fade_in_memento;
    UndoAction fade_out_memento;
    bool active;
};

class Crossfade : public Stateful, public StateManager
{
  public:
	ARDOUR::AudioRegion& right() const { return *_right; }
	ARDOUR::AudioRegion& left() const { return *_left; }
	
	Crossfade (ARDOUR::AudioRegion& r, ARDOUR::AudioRegion& l, bool active = true);
	Crossfade (const ARDOUR::Playlist&, XMLNode&);
	virtual ~Crossfade();

	bool operator== (const ARDOUR::Crossfade&);

	XMLNode& get_state (void);
	int set_state (const XMLNode&);

	jack_nframes_t read_at (Sample *buf, Sample *mixdown_buffer, 
				float *gain_buffer, jack_nframes_t position, jack_nframes_t cnt, 
				unsigned int chan_n,
				jack_nframes_t read_frames = 0,
				jack_nframes_t skip_frames = 0);
	
	void refresh ();
	void update();

	bool involves (ARDOUR::AudioRegion& region) const {
		return _right == &region || _left == &region;
	}

	bool involves (ARDOUR::AudioRegion& a, ARDOUR::AudioRegion& b) const {
		return (_right == &a && _left == &b) || (_right == &b && _left == &a);
	}

	jack_nframes_t length() const;

	SigC::Signal0<void>            Changed;
	SigC::Signal1<void,Crossfade*> Invalidated;
	SigC::Signal0<void>            GoingAway;

	OverlapType coverage (jack_nframes_t start, jack_nframes_t end) const;

	UndoAction get_memento() const;	

	static void set_buffer_size (jack_nframes_t);

	bool active () const { return _active; }
	void set_active (bool yn);

	Curve& fade_in() { return _fade_in; } 
	Curve& fade_out() { return _fade_out; }

	SigC::Signal0<void> ActiveChanged;

  private:
	friend struct CrossfadeComparePtr;

	ARDOUR::AudioRegion* _right;
	ARDOUR::AudioRegion* _left;
	bool _active;
	bool _in_update;
	OverlapType overlap_type;

	StateManager::State* state_factory (std::string why) const;
	Change restore_state (StateManager::State&);

	Curve _fade_in;
	Curve _fade_out;
	float *fade_out_vector;
	float *fade_in_vector;

	jack_nframes_t last_length;

	static Sample* crossfade_buffer_left;
	static Sample* crossfade_buffer_right;

	void member_changed (ARDOUR::Change);
	void fade_in_changed (ARDOUR::Change);
	void fade_out_changed (ARDOUR::Change);
	void fill ();
};


} // namespace ARDOUR

#endif /* __ardour_overlap_h__ */	
