/*
    Copyright (C) 2001 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: editor_cd.cc,v 1.6 2004/02/22 23:03:59 taybin Exp $
*/

#include <algorithm>
#include <cmath>
#include <fstream>

#include "editor.h"
#include "ardour_ui.h"

#include "i18n.h"

using namespace ARDOUR;

struct LocationSortByStart {
    bool operator() (Location *a, Location *b) {
	    return a->start() < b->start();
    }
};

void
Editor::export_cd_markers_callback (Locations::LocationList& locations, const string& path)
{
	ofstream out (path.c_str());
	unsigned int trk;

	if (!out) {
		error << compose(_("Editor: cannot open \"%1\" as export file for CD track markers"), path) << endmsg;
		return;
	}

	Locations::LocationList::iterator i;
	Locations::LocationList temp;

	for (i = locations.begin(); i != locations.end(); ++i) {
		if ((*i)->is_cd_marker()) {
			temp.push_back (*i);
		}
	}

	if (temp.size()) {
		LocationSortByStart cmp;
		temp.sort (cmp);
		
		for (trk = 1, i = temp.begin(); i != temp.end(); ++i, ++trk) {
			double cd_block = session->frame_rate() / 75.0;
			double blocks;

			blocks = floor ((*i)->start() / cd_block);
			out << trk << ' ' << blocks << endl;
		}
	}
	
}

void
Editor::do_export_cd_markers (const string& path)
{
	if (session == 0) {
		return;
	}

	session->locations()->apply (*this, &Editor::export_cd_markers_callback, path);
}

void
Editor::finish_export_cd_markers (bool status)
{
	export_cd_marker_status = status;
	Gtk::Main::quit ();
}

void
Editor::start_export_cd_markers ()
{
	Gtk::FileSelection fileselector;

	fileselector.show_all ();

	export_cd_marker_status = false;

	fileselector.get_ok_button()->clicked.connect (bind (slot (*this, &Editor::finish_export_cd_markers), true));
	fileselector.get_cancel_button()->clicked.connect (Gtk::Main::quit.slot());

	ARDOUR_UI::instance()->allow_focus (true);
	Gtk::Main::run ();
	ARDOUR_UI::instance()->allow_focus (false);

	if (export_cd_marker_status) {
		do_export_cd_markers (fileselector.get_filename());
	}
}

