/*
    Copyright (C) 2003 Paul Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: crossfade_view.cc,v 1.9 2004/02/29 23:33:54 pauld Exp $
*/

#include <ardour/region.h>

#include "canvas-simplerect.h"
#include "canvas-curve.h"
#include "crossfade_view.h"
#include "rgb_macros.h"
#include "audio_time_axis.h"
#include "public_editor.h"
#include "regionview.h"
#include "utils.h"
#include "doi.h"

using namespace SigC;
using namespace ARDOUR;

SigC::Signal1<void,CrossfadeView*> CrossfadeView::GoingAway;

CrossfadeView::CrossfadeView (GtkCanvasGroup *parent, 
			      AudioTimeAxisView &tv, 
			      Crossfade& xf, 
			      double spu,
			      GdkColor& basic_color,
			      AudioRegionView& lview,
			      AudioRegionView& rview)
			      

	: TimeAxisViewItem ("xf.name()", parent, tv, spu, basic_color, xf.right().position(), 
			    xf.length(), TimeAxisViewItem::Visibility (TimeAxisViewItem::ShowFrame)),
	  crossfade (xf),
	  left_view (lview),
	  right_view (rview)
	
{
	_valid = true;

	fade_in = gtk_canvas_item_new (GTK_CANVAS_GROUP(group),
				       gtk_canvas_line_get_type(),
				       "fill_color_rgba", RGBA_TO_UINT(0,239,139,255),
					"width_pixels", (guint) 1,
				       NULL);

	fade_out = gtk_canvas_item_new (GTK_CANVAS_GROUP(group),
					gtk_canvas_line_get_type(),
					"fill_color_rgba", RGBA_TO_UINT(244,0,139,255),
					"width_pixels", (guint) 1,
					NULL);
	
	// // cerr << "xfade in @ " << fade_in << " out @ " << fade_out << endl;

	set_height (get_time_axis_view().height);

	/* no frame around the xfade */

	gtk_canvas_item_set (frame, "outline_what", 0, NULL);

	gtk_object_set_data (GTK_OBJECT(group), "crossfadeview", this);
	gtk_signal_connect (GTK_OBJECT(group), "event",
			    (GtkSignalFunc) PublicEditor::canvas_crossfade_view_event,
			    this);

	crossfade_changed ();
	active_changed ();

	crossfade.Changed.connect (slot (*this, &CrossfadeView::crossfade_changed));
	crossfade.ActiveChanged.connect (slot (*this, &CrossfadeView::active_changed));
}

CrossfadeView::~CrossfadeView ()
{
	 GoingAway (this) ; /* EMIT_SIGNAL */
}

std::string
CrossfadeView::get_item_name ()
{
	return "xfade";
//	return crossfade.name();
}

void
CrossfadeView::reset_width_dependent_items (double pixel_width)
{
	TimeAxisViewItem::reset_width_dependent_items (pixel_width);

	if (pixel_width < 5) {
		gtk_canvas_item_hide (fade_in);
		gtk_canvas_item_hide (fade_out);
	} else {

		if (crossfade.active()) {
			redraw ();
		}

		/* call this to make we get the visibility of curves
		   correct.
		*/

		active_changed ();
	}
}

void
CrossfadeView::set_height (double height)
{
	TimeAxisViewItem::set_height (height);
	crossfade_changed ();
}

void
CrossfadeView::crossfade_changed ()
{
	redraw ();

	set_position (crossfade.right().first_frame(), this);
	set_duration (crossfade.length(), this);
}

void
CrossfadeView::redraw ()
{
	GtkCanvasPoints* points; 
	int npoints;
	float* vec;

	npoints = get_time_axis_view().editor.frame_to_pixel (crossfade.length());
	double h = get_time_axis_view().height - 4.0;
	
	if (npoints <= 1) {
		gtk_canvas_item_hide (group);
		return;
	} else {
		gtk_canvas_item_show (group);
	}

	if (h < 0) {
		/* no space allocated yet */
		return;
	}
 
	points = gtk_canvas_points_new (npoints);
	vec = new float[npoints];

	crossfade.fade_in().get_vector (0, crossfade.length(), vec, npoints);
	for (int i = 0, pci = 0; i < npoints; ++i) {
		points->coords[pci++] = i;
		points->coords[pci++] = 2.0 + h - (h * vec[i]);
	}
	gtk_canvas_item_set (fade_in, "points", points, NULL);

	crossfade.fade_out().get_vector (0, crossfade.length(), vec, npoints);
	for (int i = 0, pci = 0; i < npoints; ++i) {
		points->coords[pci++] = i;
		points->coords[pci++] = 2.0 + h - (h * vec[i]);
	}
	gtk_canvas_item_set (fade_out, "points", points, NULL);

	delete [] vec;

	gtk_canvas_points_unref (points);

	active_changed();

	gtk_canvas_item_raise_to_top (group);
}

void
CrossfadeView::active_changed ()
{
	if (crossfade.active()) {
		gtk_canvas_item_set (frame, "fill_color_rgba", RGBA_TO_UINT (132,124,249,164), NULL);
		gtk_canvas_item_show (fade_in);
		gtk_canvas_item_show (fade_out);
	} else {
		gtk_canvas_item_set (frame, "fill_color_rgba", RGBA_TO_UINT (132,124,249,54), NULL);
		gtk_canvas_item_hide (fade_in);
		gtk_canvas_item_hide (fade_out);
	}
}

void
CrossfadeView::set_valid (bool yn)
{
	_valid = yn;
}

AudioRegionView&
CrossfadeView::upper_regionview () const
{
	if (left_view.region.layer() > right_view.region.layer()) {
		return left_view;
	} else {
		return right_view;
	}
}
