# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package ArchZoom::FileHighlight;

use Arch::Util qw(run_cmd load_file);
use ArchZoom::Util;

sub new ($;$) {
	my $class = shift;
	my $filters = shift || [];

	my $self = {
		filters => $filters,
	};
	return bless $self, $class;
}

sub process_file ($$;$) {
	my $self = shift;
	my $file_name = shift;
	my $content_ref = shift;
	load_file($file_name, \$content_ref) unless defined $content_ref;

	return undef if -B $file_name;

	foreach (@{$self->{filters}}) {
		# make sure we actually copy $_ and not work in-place
		my $filter = $_;
		my %args = ();
		if ($filter =~ /(.*)\((.*)\)/) {
			$filter = $1;
			my $args = $2;
			%args = map { /^(.+?)=(.*)$/? ($1 => $2): ($_ => 1) }
				split(/[^:\w=]+/, $args);
		}
		my $method = "_process_$filter";
		unless ($self->can($method)) {
			warn qq(ArchZoom::FileHighlight: unknown filter "$filter"\n);
			next;
		}
		my $html_ref = $self->$method($file_name, $content_ref, %args);
		return $html_ref if $html_ref;
	}
	$self->_process_none($file_name, $content_ref);
}

sub _process_enscript ($$$%) {
	my $self = shift;
	my $file_name = shift;
	my $content_ref = shift;
	my %args = @_;

	my @enscript_args = qw(enscript --output - --quiet --pretty-print);
	push @enscript_args, "--color" unless $args{"mono"};
	push @enscript_args, "--language", "html", $file_name;
	my $html = run_cmd(@enscript_args);
	return undef unless $html;

	$html =~ s!^.*<PRE>\n?!!s; $html =~ s!</PRE>.*$!!s;
	for (1 .. 3) {
		my $dot = $_ == 3? ".": "[^<]";
		$html =~ s!<B><FONT COLOR="#A020F0">($dot*?)</FONT></B>!<span class=syntax_keyword>$1</span>!sg;
		$html =~ s!<B><FONT COLOR="#DA70D6">($dot*?)</FONT></B>!<span class=syntax_builtin>$1</span>!sg;
		$html =~ s!<I><FONT COLOR="#B22222">($dot*?)</FONT></I>!<span class=syntax_comment>$1</span>!sg;
		$html =~ s!<B><FONT COLOR="#5F9EA0">($dot*?)</FONT></B>!<span class=syntax_special>$1</span>!sg;
		$html =~ s!<B><FONT COLOR="#0000FF">($dot*?)</FONT></B>!<span class=syntax_funcdef>$1</span>!sg;
		$html =~ s!<B><FONT COLOR="#228B22">($dot*?)</FONT></B>!<span class=syntax_vartype>$1</span>!sg;
		$html =~ s!<B><FONT COLOR="#BC8F8F">($dot*?)</FONT></B>!<span class=syntax_string>$1</span>!sg;
		$html =~ s!<FONT COLOR="#228B22"><B>($dot*?)</FONT></B>!<span class=syntax_vartype>$1</span>!sg;
		$html =~ s!<FONT COLOR="#BC8F8F"><B>($dot*?)</FONT></B>!<span class=syntax_string>$1</span>!sg;
		$html =~ s!<FONT COLOR="#B8860B">($dot*?)</FONT>!<span class=syntax_constant>$1</span>!sg;
	}
	$html =~ s!<B>(.*?)</B>!<span class=syntax_keyword>$1</span>!sg;
	$html =~ s!<I>(.*?)</I>!<span class=syntax_comment>$1</span>!sg;
	$html =~ s!</FONT></B>!!sg;  # enscript bug with perl highlightling
	return \$html;
}

sub _match_file_extension ($$) {
	my $file_name = shift;
	my $args = shift;

	while (my ($ext, $value) = each %$args) {
		return 1 if $value && $file_name =~ /\.$ext(\.in)?$/i;
	}
	return 0;
}

sub _process_internal ($$$%) {
	my $self = shift;
	my $file_name = shift;
	my $content_ref = shift;
	my %args = @_;

	if (%args) {
		if (exists $args{':xml'}) {
			my $value = delete $args{':xml'};
			$args{$_} = $value foreach qw(html htm sgml xml rss glade);
		}
		return undef unless _match_file_extension($file_name, \%args);
	}

	print STDERR "internal highlighting for $file_name\n" if $ENV{DEBUG};
	my $html = htmlize($$content_ref);
	if ($file_name =~ /\.(ac|am|conf|m4|pl|pm|po|py|rb|sh)(\.in)?$/i || $html =~ /^#!/) {
		$html =~ s!^([ \t]*)(#.*)!$1<span class="syntax_comment">$2</span>!mg;
	}
	if ($file_name =~ /\.(c|cc|cpp|cxx|c\+\+|h|hpp|idl|php|xpm)$/i) {
		$html =~ s!(^|[^\\])(//.*)!$1<span class="syntax_comment">$2<\/span>!g;
		$html =~ s!(^|[^\\])(/\*.*?\*/)!$1<span class="syntax_comment">$2<\/span>!sg;
	}
	if ($file_name =~ /\.m4(\.in)?$/i) {
		$html =~ s!(\bdnl\b.*)!<span class="syntax_comment">$1<\/span>!g;
	}
	if ($file_name =~ /\.(html|htm|sgml|xml|rss|glade)(\.in)?$/i) {
		$html =~ s!(&lt;\!--.*?--&gt;)!<span class="syntax_comment">$1<\/span>!sg;
		$html =~ s!(&lt;/?\w+.*?&gt;)!<span class="syntax_keyword">$1<\/span>!sg;
		while ($html =~ s!(>(?:&lt;[\w-]+)?\s+)([\w-]+)(=)("[^"]*"|'[^']'|[^\s]*)!$1<span class="syntax_special">$2<\/span>$3<span class="syntax_string">$4<\/span>!sg) {}
	}
	return \$html;
}

sub _process_none ($$$%) {
	my $self = shift;
	my $file_name = shift;
	my $content_ref = shift;
	my %args = @_;

	if (%args) {
		return undef unless _match_file_extension($file_name, \%args);
	}

	my $html = htmlize($$content_ref);
	return \$html;
}

1;
