/*
 *  Aquamarine the KDE window decorator
 *
 *  Copyright (c) 2006 Dennis Kasprzyk <onestone@beryl-project.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include <qdir.h>

#include <kconfig.h>
#include <ksimpleconfig.h>
#include <kdebug.h>
#include <kglobal.h>
#include <kstandarddirs.h>
#include <kinstance.h>
#include <kshortcut.h>
#include <kipc.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <stdlib.h>

extern "C" {
#include <beryl-settings-backend.h>
}

static KInstance *instance = NULL;

const static char *edgeName[] = {
    "Left",
    "Right",
    "Top",
    "Bottom",
    "TopLeft",
    "TopRight",
    "BottomLeft",
    "BottomRight"
};

const static char *fspLevel[] = {
    "None",
    "Low",
    "Normal",
    "High",
    "Extreme"
};

typedef struct _ConfigFiles {
	KSimpleConfig	*beryl;
	KConfig			*kwin;
	KConfig			*global;
	Bool			modified;
} ConfigFiles;

typedef enum {
	OptionInt,
	OptionBool,
	OptionKey,
	OptionSpecial
} SpecialOptionType;

struct _SpecialOption {
	QString berylName;
	QString berylPlugin;
	QString kdeName;
	bool global;
	SpecialOptionType type;
} const specialOptions[] = {
	{"close_window", NULL, "Window Close", true, OptionKey},
	{"run", NULL, "Run Command", true, OptionKey},
	{"main_menu", NULL, "Popup Launch Menu", true, OptionKey},
	{"lower_window", NULL, "Window Lower", true, OptionKey},
	{"toggle_window_maximized", NULL, "Window Maximize", true, OptionKey},
	{"minimize_window", NULL, "Window Minimize", true, OptionKey},
	{"toggle_window_maximized_horizontally", NULL, "Window Maximize Horizontal", true, OptionKey},
	{"toggle_window_maximized_vertically", NULL, "Window Maximize Vertical", true, OptionKey},
	{"show_desktop", NULL, "Toggle Showing Desktop", true, OptionKey},
	{"run_command_screenshot", NULL, "Desktop Screenshot", true, OptionKey},
	{"run_command_window_screenshot", NULL, "Window Screenshot", true, OptionKey},
	{"window_menu", NULL, "Window Operations Menu", true, OptionKey},
	{"toggle_window_shaded", NULL, "Window Shade", true, OptionKey},
	{"raise_window", NULL, "Window Raise", true, OptionKey},
	{"toggle_window_fullscreen", NULL, "Window Fullscreen", true, OptionKey},
	{"run_command11", NULL, "Kill Window", true, OptionKey},
	{"initiate", "move", "Window Move", true, OptionKey},
	{"initiate", "resize", "Window Resize", true, OptionKey},
	{"rotate_right", "rotate", "Switch to Next Desktop", true, OptionKey},
	{"rotate_left", "rotate", "Switch to Previous Desktop", true, OptionKey},
	{"rotate_to_1", "rotate", "Switch to Desktop 1", true, OptionKey},
	{"rotate_to_2", "rotate", "Switch to Desktop 2", true, OptionKey},
	{"rotate_to_3", "rotate", "Switch to Desktop 3", true, OptionKey},
	{"rotate_to_4", "rotate", "Switch to Desktop 4", true, OptionKey},
	{"rotate_to_5", "rotate", "Switch to Desktop 5", true, OptionKey},
	{"rotate_to_6", "rotate", "Switch to Desktop 6", true, OptionKey},
	{"rotate_to_7", "rotate", "Switch to Desktop 7", true, OptionKey},
	{"rotate_to_8", "rotate", "Switch to Desktop 8", true, OptionKey},
	{"rotate_to_9", "rotate", "Switch to Desktop 9", true, OptionKey},
	{"rotate_to_10", "rotate", "Switch to Desktop 10", true, OptionKey},
	{"rotate_to_11", "rotate", "Switch to Desktop 11", true, OptionKey},
	{"rotate_to_12", "rotate", "Switch to Desktop 12", true, OptionKey},

	{"rotate_right_window", "rotate", "Window to Next Desktop", true, OptionKey},
	{"rotate_left_window", "rotate", "Window to Previous Desktop", true, OptionKey},
	{"rotate_to_1_window", "rotate", "Window to Desktop 1", true, OptionKey},
	{"rotate_to_2_window", "rotate", "Window to Desktop 2", true, OptionKey},
	{"rotate_to_3_window", "rotate", "Window to Desktop 3", true, OptionKey},
	{"rotate_to_4_window", "rotate", "Window to Desktop 4", true, OptionKey},
	{"rotate_to_5_window", "rotate", "Window to Desktop 5", true, OptionKey},
	{"rotate_to_6_window", "rotate", "Window to Desktop 6", true, OptionKey},
	{"rotate_to_7_window", "rotate", "Window to Desktop 7", true, OptionKey},
	{"rotate_to_8_window", "rotate", "Window to Desktop 8", true, OptionKey},
	{"rotate_to_9_window", "rotate", "Window to Desktop 9", true, OptionKey},
	{"rotate_to_10_window", "rotate", "Window to Desktop 10", true, OptionKey},
	{"rotate_to_11_window", "rotate", "Window to Desktop 11", true, OptionKey},
	{"rotate_to_12_window", "rotate", "Window to Desktop 12", true, OptionKey},

	{"plane_up", "plane", "Switch One Desktop Up", true, OptionKey},
	{"plane_down", "plane", "Switch One Desktop Down", true, OptionKey},
	{"plane_left", "plane", "Switch One Desktop to the Left", true, OptionKey},
	{"plane_right", "plane", "Switch One Desktop to the Right", true, OptionKey},

	{"plane_to_1", "plane", "Switch to Desktop 1", true, OptionKey},
	{"plane_to_2", "plane", "Switch to Desktop 2", true, OptionKey},
	{"plane_to_3", "plane", "Switch to Desktop 3", true, OptionKey},
	{"plane_to_4", "plane", "Switch to Desktop 4", true, OptionKey},
	{"plane_to_5", "plane", "Switch to Desktop 5", true, OptionKey},
	{"plane_to_6", "plane", "Switch to Desktop 6", true, OptionKey},
	{"plane_to_7", "plane", "Switch to Desktop 7", true, OptionKey},
	{"plane_to_8", "plane", "Switch to Desktop 8", true, OptionKey},
	{"plane_to_9", "plane", "Switch to Desktop 9", true, OptionKey},
	{"plane_to_10", "plane", "Switch to Desktop 10", true, OptionKey},
	{"plane_to_11", "plane", "Switch to Desktop 11", true, OptionKey},
	{"plane_to_12", "plane", "Switch to Desktop 12", true, OptionKey},

	{"next", "switcher", "Walk Through Windows", true, OptionKey},
	{"prev", "switcher", "Walk Through Windows (Reverse)", true, OptionKey},

	{"autoraise", NULL, "AutoRaise", false, OptionBool},
	{"raise_on_click", NULL, "ClickRaise", false, OptionBool},
	{"snapoff_maximized", "move", "MoveResizeMaximizedWindows", false, OptionBool},

	{"autoraise_delay", NULL, "AutoRaiseInterval", false, OptionInt},

	{"command_screenshot", NULL, NULL, true, OptionSpecial},
	{"command_window_screenshot", NULL, NULL, true, OptionSpecial},
	{"unmaximize_window", NULL, NULL, true, OptionSpecial},
	{"maximize_window", NULL, NULL, true, OptionSpecial},
	{"maximize_window_horizontally", NULL, NULL, true, OptionSpecial},
	{"maximize_window_vertically", NULL, NULL, true, OptionSpecial},
	{"command11", NULL, NULL, true, OptionSpecial},
	{"click_to_focus", NULL, NULL, false, OptionSpecial},
	{"fsp_level", NULL, NULL, false, OptionSpecial},
	{"number_of_desktops", NULL, "Number", false, OptionSpecial},
	{"resize_mode", "resize", NULL, true, OptionSpecial},
	{"map_on_shutdown", NULL, NULL, true, OptionSpecial},
};

#define N_SOPTIONS (sizeof (specialOptions) / sizeof (struct _SpecialOption))

static bool isIntegratedOption(BerylSetting * setting)
{

	for (unsigned int i = 0; i < N_SOPTIONS; i++)
	{
		if (setting->name == specialOptions[i].berylName &&
			QString(setting->parent->name) == specialOptions[i].berylPlugin)
			return true;
	}
	return false;
}

static void KdeIntToBeryl (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	int val = cfg->readNumEntry(specialOptions[num].kdeName);

	beryl_setting_value_set_int(&setting->value,&val);
}

static void KdeBoolToBeryl (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	Bool val = (cfg->readBoolEntry(specialOptions[num].kdeName))?TRUE:FALSE;

	beryl_setting_value_set_bool(&setting->value,&val);
}

static void KdeKeyToBeryl (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	KKey key(cfg->readEntry(specialOptions[num].kdeName));

	gboolean kde_enabled = !key.isNull();

	int kde_keysym = key.sym();
	int kde_keymod = 0;

	if (key.modFlags() & KKey::SHIFT)
		kde_keymod |= ShiftMask;
	if (key.modFlags() & KKey::CTRL)
		kde_keymod |= ControlMask;
	if (key.modFlags() & KKey::ALT)
		kde_keymod |= CompAltMask;
	if (key.modFlags() & KKey::WIN)
		kde_keymod |= CompSuperMask;


	if (kde_enabled)
	{
		beryl_setting_value_set_keysym(&setting->value,&kde_keysym);
		beryl_setting_value_set_keymods(&setting->value,&kde_keymod);
		beryl_setting_value_set_key_enabled(&setting->value,&kde_enabled);
	}
	else
	{
		char * sval = NULL;
		if (cFiles->beryl->hasKey(specialOptions[num].berylName + "_key (Integrated)"))
			sval = strdup(cFiles->beryl->readEntry(specialOptions[num].berylName + "_key (Integrated)").ascii());

		int keysym = 0;
		int keymods = 0;
		gchar * spos = NULL;
		if (sval) keymods = beryl_settings_get_mods_and_endptr(sval,&spos);
		if (spos && *spos)
		{
			keysym=XStringToKeysym(spos);
		}
		beryl_setting_value_set_keysym(&setting->value,&keysym);
		beryl_setting_value_set_keymods(&setting->value,&keymods);
		beryl_setting_value_set_key_enabled(&setting->value,&kde_enabled);
		delete sval;
	}
}

static void readIntegratedOption(ConfigFiles *cFiles, BerylSetting * setting)
{
	int option = 0;
	for (unsigned int i = 0; i < N_SOPTIONS; i++)
	{
		if (setting->name == specialOptions[i].berylName &&
			QString(setting->parent->name) == specialOptions[i].berylPlugin)
		{
			option = i;
			break;
		}
	}
	setting->is_default=FALSE;
	switch (specialOptions[option].type)
	{
		case OptionInt:
			KdeIntToBeryl (cFiles, setting, option);
			break;
		case OptionBool:
			KdeBoolToBeryl (cFiles, setting, option);
			break;
		case OptionKey:
			KdeKeyToBeryl (cFiles, setting, option);
			break;
		case OptionSpecial:
			if (specialOptions[option].berylName == "command_screenshot")
			{
				setting->value.value.as_string =
					g_strdup("ksnapshot");
			}
			else if (specialOptions[option].berylName == "command_window_screenshot")
			{
				setting->value.value.as_string =
					g_strdup("ksnapshot -c");
			}
			else if (specialOptions[option].berylName == "command11")
			{
				setting->value.value.as_string =
					g_strdup("xkill");
			}
			else if (specialOptions[option].berylName == "map_on_shutdown")
			{
				int val = TRUE;
				beryl_setting_value_set_bool(&setting->value,&val);
			}
			else if (specialOptions[option].berylName == "unmaximize_window"
			   || specialOptions[option].berylName == "maximize_window"
			   || specialOptions[option].berylName == "maximize_window_horizontally"
			   || specialOptions[option].berylName == "maximize_window_vertically")
			{
				int val = 0;
				beryl_setting_value_set_keysym(&setting->value,&val);
				beryl_setting_value_set_keymods(&setting->value,&val);
				beryl_setting_value_set_key_enabled(&setting->value,&val);
			}
			else if (specialOptions[option].berylName == "click_to_focus")
			{
				Bool val = (cFiles->kwin->readEntry("FocusPolicy") == "ClickToFocus")?TRUE:FALSE;
				beryl_setting_value_set_bool(&setting->value,&val);
			}
			else if (specialOptions[option].berylName == "fsp_level")
			{
				int level = cFiles->kwin->readNumEntry("FocusStealingPreventionLevel");
				level = MAX(0,MIN(4,level));
				setting->value.value.as_string = g_strdup(fspLevel[level]);
			}
			else if (specialOptions[option].berylName == "number_of_desktops")
			{
				cFiles->kwin->setGroup("Desktops");
				KdeIntToBeryl (cFiles, setting, option);
				cFiles->kwin->setGroup("Windows");
			}
			else if (specialOptions[option].berylName == "resize_mode")
			{
				QString mode = cFiles->kwin->readEntry("ResizeMode");
				QString imode;
				QString result = "Normal";
				if (cFiles->beryl->hasKey(specialOptions[option].berylName + " (Integrated)"))
					imode = cFiles->beryl->readEntry(specialOptions[option].berylName + " (Integrated)");

				if (mode == "Opaque")
				{
					result = "Normal";
					if (imode == "Stretch")
						result = "Stretch";
				}
				else if (mode == "Transparent")
				{
					result = "Outline";
					if (imode == "Filled Outline")
						result = "Filled Outline";
				}
				setting->value.value.as_string = g_strdup(result.ascii());
			}
			break;
		default:
			break;
	}
}

gboolean get_setting_is_integrated(BerylSetting * setting)
{
	return beryl_settings_context_get_de_integration_enabled(setting->parent->context)
			&& isIntegratedOption(setting);
}


gboolean get_setting_is_read_only(BerylSetting * setting)
{
	if (!beryl_settings_context_get_de_integration_enabled(setting->parent->context)
			|| !isIntegratedOption(setting))
		return FALSE;

	int option = 0;
	for (unsigned int i = 0; i < N_SOPTIONS; i++)
	{
		if (setting->name == specialOptions[i].berylName &&
			QString(setting->parent->name) == specialOptions[i].berylPlugin)
		{
			option = i;
			break;
		}
	}
	switch (specialOptions[option].type)
	{
		case OptionSpecial:
			if (specialOptions[option].berylName == "command_screenshot")
			{
				return TRUE;
			}
			else if (specialOptions[option].berylName == "command_window_screenshot")
			{
				return TRUE;			}
			else if (specialOptions[option].berylName == "command11")
			{
				return TRUE;
			}
			else if (specialOptions[option].berylName == "map_on_shutdown")
			{
				return TRUE;
			}
			else if (specialOptions[option].berylName == "unmaximize_window"
			   || specialOptions[option].berylName == "maximize_window"
			   || specialOptions[option].berylName == "maximize_window_horizontally"
			   || specialOptions[option].berylName == "maximize_window_vertically")
			{
				return TRUE;
			}
			break;
		default:
			break;
	}
	return FALSE;
}


GSList * get_existing_profiles(void)
{
	if (!instance)
		instance = new KInstance("beryl-kconfig");
	QDir dir(KGlobal::dirs()->saveLocation("config", QString::null, false),"berylrc.*");
	QStringList files = dir.entryList();

	GSList * ret = NULL;

	QStringList::iterator it;
	for (it = files.begin(); it != files.end(); it++)
	{
		QString str = (*it);
		if (str.length() > 8)
		{
			QString profile = str.right(str.length() - 8);
			if (!profile.isEmpty())
			{
				gchar * p = g_strdup(profile.ascii());
				ret=g_slist_append(ret,p);
			}
		}
	}
	return ret;
}

void read_setting(BerylSettingsContext * c, BerylSetting * setting)
{
	ConfigFiles *cFiles = (ConfigFiles *)c->backend_private_ptr;
	KSimpleConfig *cfg = cFiles->beryl;

	QString key(setting->name);
	QString group(setting->parent->name);
	if (setting->is_screen)
		group += "_screen";
	cfg->setGroup(group);

	if (beryl_settings_context_get_de_integration_enabled(c) && isIntegratedOption(setting))
	{
		readIntegratedOption(cFiles, setting);
		return;
	}


	if (setting->type != BERYL_SETTING_TYPE_BINDING && !cfg->hasKey(key))
		return;

	setting->is_default=FALSE;
	switch (setting->type)
	{
		case BERYL_SETTING_TYPE_STRING:
			setting->value.value.as_string =
					g_strdup(cfg->readEntry(key,setting->value.value.as_string).ascii());
			break;
		case BERYL_SETTING_TYPE_FLOAT:
			{
				double val = cfg->readDoubleNumEntry(key);
				// stupid rounding issue
				val = MAX(val,setting->info.for_float.min);
				val = MIN(val,setting->info.for_float.max);
				beryl_setting_value_set_float(&setting->value,&val);
			}
			break;
		case BERYL_SETTING_TYPE_INT:
			{
				int val = cfg->readNumEntry(key);
				beryl_setting_value_set_int(&setting->value,&val);
			}
			break;
		case BERYL_SETTING_TYPE_BOOL:
			{

				Bool val = (cfg->readBoolEntry(key))?TRUE:FALSE;
				beryl_setting_value_set_bool(&setting->value,&val);
			}
			break;
		case BERYL_SETTING_TYPE_COLOR:
			{
				QValueList<int> list = cfg->readIntListEntry(key);
				BerylSettingColorValue color;
				color.color.red = list[0];
				color.color.green = list[1];
				color.color.blue = list[2];
				color.color.alpha = list[3];
				beryl_setting_value_set_color(&setting->value,&color);
			}
			break;
		case BERYL_SETTING_TYPE_LIST:
			{
				switch(setting->info.for_list.list_of_type)
				{
					case BERYL_SETTING_TYPE_BOOL:
						{
							QValueList<int> list = cfg->readIntListEntry(key);

							beryl_setting_list_clear(setting);

							QValueList<int>::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
								Bool val = ((*it))?TRUE:FALSE;
                            	beryl_setting_value_set_bool(value,&val);
							}
						}
						break;
					case BERYL_SETTING_TYPE_INT:
						{
							QValueList<int> list = cfg->readIntListEntry(key);

							beryl_setting_list_clear(setting);

							QValueList<int>::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
                            	beryl_setting_value_set_int(value,&(*it));
							}
						}
						break;
					case BERYL_SETTING_TYPE_STRING:
						{
							QStringList list = cfg->readListEntry(key,';');

							beryl_setting_list_clear(setting);

							QStringList::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
                            	value->value.as_string = g_strdup((*it).ascii());
								value->parent->is_default=FALSE;
							}
						}
						break;
					case BERYL_SETTING_TYPE_FLOAT:
						{
							QStringList list = cfg->readListEntry(key,';');

							beryl_setting_list_clear(setting);

							QStringList::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
								double val = (*it).toDouble();
								beryl_setting_value_set_float(value,&val);
							}
						}
						break;
					case BERYL_SETTING_TYPE_COLOR:
						{
							QStringList list = cfg->readListEntry(key,';');

							beryl_setting_list_clear(setting);

							QStringList::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
								QStringList colorstr = QStringList::split(",",(*it));

								BerylSettingColorValue color;
								color.color.red = colorstr[0].toInt();
								color.color.green = colorstr[1].toInt();
								color.color.blue = colorstr[2].toInt();
								color.color.alpha = colorstr[3].toInt();
								beryl_setting_value_set_color(value,&color);
							}
						}
						break;
					case BERYL_SETTING_TYPE_BINDING:
						{
							QStringList list = cfg->readListEntry(key,';');

							beryl_setting_list_clear(setting);

							QStringList::iterator it;
							for (it = list.begin(); it != list.end(); it++)
							{
								BerylSettingValue * value = beryl_setting_list_append(setting);
								QStringList binding = QStringList::split(",",(*it));

								int val;
								Bool bval;

								val = binding[0].toInt();
								beryl_setting_value_set_buttonmods(value,&val);
								val = binding[1].toInt();
								beryl_setting_value_set_button(value,&val);
								val = binding[2].toInt();
								beryl_setting_value_set_keymods(value,&val);
								val = binding[3].toInt();
								beryl_setting_value_set_keysym(value,&val);
								val = binding[4].toInt();
								beryl_setting_value_set_edgemask(value,&val);
								bval = (binding[5].toInt())?TRUE:FALSE;
								beryl_setting_value_set_bell(value,&bval);
								bval = (binding[6].toInt())?TRUE:FALSE;
								beryl_setting_value_set_button_enabled(value,&bval);
								bval = (binding[7].toInt())?TRUE:FALSE;
								beryl_setting_value_set_key_enabled(value,&bval);
							}
						}
						break;
					default:
						break;
				}
			}
			break;
		case BERYL_SETTING_TYPE_BINDING:
			{
				setting->is_default=TRUE;
				if (cfg->hasKey(key + "_key"))
				{
					setting->is_default=FALSE;
					Bool bval = (cfg->readBoolEntry(key + "_key_enabled"))?TRUE:FALSE;
					char * sval = strdup(cfg->readEntry(key + "_key").ascii());

					int keysym = 0;
					int keymods = 0;
					gchar * spos;
					keymods = beryl_settings_get_mods_and_endptr(sval,&spos);
					if (spos && *spos)
					{
						keysym=XStringToKeysym(spos);
					}
					beryl_setting_value_set_keysym(&setting->value,&keysym);
					beryl_setting_value_set_keymods(&setting->value,&keymods);
					beryl_setting_value_set_key_enabled(&setting->value,&bval);
					delete sval;
				}
				if (cfg->hasKey(key + "_button"))
				{
					setting->is_default=FALSE;
					Bool bval = (cfg->readBoolEntry(key + "_button_enabled"))?TRUE:FALSE;
					char * sval = strdup(cfg->readEntry(key + "_button").ascii());

					int button = 0;
					int buttonmods = 0;
					char * spos;
					buttonmods = beryl_settings_get_mods_and_endptr(sval,&spos);
					if (spos && *spos)
					{
						spos=strcasestr(spos,"Button");
						if (spos && *spos)
						{
							spos+=strlen("Button");
							button=atoi(spos);
						}
					}
					beryl_setting_value_set_button(&setting->value,&button);
					beryl_setting_value_set_buttonmods(&setting->value,&buttonmods);
					beryl_setting_value_set_button_enabled(&setting->value,&bval);
					delete sval;
				}
				if (cfg->hasKey(key + "_edge"))
				{
					setting->is_default=FALSE;
					QString sval = cfg->readEntry(key + "_edge");
					int int_val = 0;

					for (int i = 0; i < 8; i++)
					{
						if (sval == edgeName[i])
						{
							int_val=1<<i;
						}
					}
						beryl_setting_value_set_edgemask(&setting->value,&int_val);
				}
				if (cfg->hasKey(key + "_bell"))
				{
					setting->is_default=FALSE;
					Bool bval = (cfg->readBoolEntry(key + "_bell"))?TRUE:FALSE;
					beryl_setting_value_set_bell(&setting->value,&bval);
				}
			}
			break;
		default:
			kdDebug () << "Not supported setting type : " << setting->type << endl;
			setting->is_default=TRUE;
			break;
	}
}

static void copy_bool(BerylSettingValue * value, QValueList<int> *list)
{
	list->append((value->value.as_bool)?1:0);
}

static void copy_int(BerylSettingValue * value, QValueList<int> *list)
{
	list->append(value->value.as_int);
}

static void copy_float(BerylSettingValue * value, QStringList *list)
{
	list->append(QString::number(value->value.as_float));
}

static void copy_binding(BerylSettingValue * value, QStringList *list)
{
	QString binding;
	binding += QString::number(value->value.as_binding.button_mod_mask);
	binding += ",";
	binding += QString::number(value->value.as_binding.button);
	binding += ",";
	binding += QString::number(value->value.as_binding.key_mod_mask);
	binding += ",";
	binding += QString::number(value->value.as_binding.keysym);
	binding += ",";
	binding += QString::number(value->value.as_binding.edge_mask);
	binding += ",";
	binding += QString::number((value->value.as_binding.on_bell)?1:0);
	binding += ",";
	binding += QString::number((value->value.as_binding.enabled.value.button)?1:0);
	binding += ",";
	binding += QString::number((value->value.as_binding.enabled.value.key)?1:0);
	list->append(binding);
}

static void copy_string(BerylSettingValue * value, QStringList *list)
{
	list->append(value->value.as_string);
}

static void copy_color(BerylSettingValue * value, QStringList *list)
{
	QString color;
	color += QString::number(value->value.as_color.color.red);
	color += ",";
	color += QString::number(value->value.as_color.color.green);
	color += ",";
	color += QString::number(value->value.as_color.color.blue);
	color += ",";
	color += QString::number(value->value.as_color.color.alpha);
	list->append(color);
}

static void BerylIntToKde (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	int val = setting->value.value.as_int;

	if (cfg->readNumEntry(specialOptions[num].kdeName) != val)
	{
		cFiles->modified = true;
		cfg->writeEntry(specialOptions[num].kdeName,val);
	}
}

static void BerylBoolToKde (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	bool val = setting->value.value.as_bool;

	if (cfg->readBoolEntry(specialOptions[num].kdeName) != val)
	{
		cFiles->modified = true;
		cfg->writeEntry(specialOptions[num].kdeName,val);
	}
}

static void BerylKeyToKde (ConfigFiles *cFiles, BerylSetting * setting, int num)
{
	KConfig *cfg = (specialOptions[num].global)?cFiles->global:cFiles->kwin;

	Bool enabled = FALSE;
	int keysym = 0;
	int keymod = 0;

	beryl_setting_value_get_key_enabled(&setting->value,&enabled);
	beryl_setting_value_get_keymods(&setting->value,&keymod);
	// do not store only modifiers
	if (!beryl_setting_value_get_keysym(&setting->value,&keysym))
		enabled = FALSE;

	int kde_keymod = 0;

	if (keymod & ShiftMask)
		kde_keymod |= KKey::SHIFT;
	if (keymod & ControlMask)
		kde_keymod |= KKey::CTRL;
	if (keymod & CompAltMask)
		kde_keymod |= KKey::ALT;
	if (keymod & CompSuperMask)
		kde_keymod |= KKey::WIN;

	KKey key(keysym, kde_keymod);
	KKey akey(cfg->readEntry(specialOptions[num].kdeName));

	if (enabled && akey != key)
	{
		cFiles->modified = true;
		cfg->writeEntry(specialOptions[num].kdeName,key.toString());
		if (cFiles->beryl->hasKey(specialOptions[num].berylName + "_key (Integrated)"))
			cFiles->beryl->deleteEntry(specialOptions[num].berylName + "_key (Integrated)");
	}
	if (!enabled)
	{
		if (cfg->readEntry(specialOptions[num].kdeName) != "none")
		{
			cFiles->modified = true;
			cfg->writeEntry(specialOptions[num].kdeName,"none");
		}
		gint ire;
		QString binding;
		QString keysym;

		if (beryl_setting_value_get_keymods(&setting->value,&ire))
			binding += beryl_settings_mods_to_string(ire);
		if (beryl_setting_value_get_keysym(&setting->value,&ire))
			keysym = XKeysymToString(ire);
		if (keysym.isEmpty()) keysym = "None";

		binding += keysym;
		cFiles->beryl->writeEntry(specialOptions[num].berylName + "_key (Integrated)",binding);
	}
}


static void writeIntegratedOption(ConfigFiles *cFiles, BerylSetting * setting)
{
	int option = 0;
	for (unsigned int i = 0; i < N_SOPTIONS; i++)
	{
		if (setting->name == specialOptions[i].berylName &&
			QString(setting->parent->name) == specialOptions[i].berylPlugin)
		{
			option = i;
			break;
		}
	}
	setting->is_default=FALSE;
	switch (specialOptions[option].type)
	{
		case OptionInt:
			BerylIntToKde(cFiles, setting, option);
			break;
		case OptionBool:
			BerylBoolToKde(cFiles, setting, option);
			break;
		case OptionKey:
			BerylKeyToKde(cFiles, setting, option);
			break;
		case OptionSpecial:
			if (specialOptions[option].berylName == "command_screenshot"
				|| specialOptions[option].berylName == "command_window_screenshot"
				|| specialOptions[option].berylName == "command11"
				||specialOptions[option].berylName == "unmaximize_window"
				|| specialOptions[option].berylName == "maximize_window"
				|| specialOptions[option].berylName == "maximize_window_horizontally"
				|| specialOptions[option].berylName == "maximize_window_vertically"
			    || specialOptions[option].berylName == "map_on_shutdown")
				break;

			if (specialOptions[option].berylName == "click_to_focus")
			{
				QString mode = cFiles->kwin->readEntry("FocusPolicy");
				QString val = "ClickToFocus";
				if (!setting->value.value.as_bool)
				{
					val = "FocusFollowsMouse";
				}
				if (mode != val)
				{
					cFiles->modified = true;
					cFiles->kwin->writeEntry("FocusPolicy",val);
				}
			}
			if (specialOptions[option].berylName == "focus_stealing_prevention_level")
			{
				int level = cFiles->kwin->readNumEntry("FocusStealingPreventionLevel");
				level = MAX(0,MIN(4,level));
				int val = 0;
				for (int i = 0; i < 5; i++)
					if (strcmp(fspLevel[i],setting->value.value.as_string) == 0)
						val = i;
				if (val != level)
				{
					cFiles->modified = true;
					cFiles->kwin->writeEntry("FocusStealingPreventionLevel",val);
				}
			}
			if (specialOptions[option].berylName == "number_of_desktops")
			{
				cFiles->kwin->setGroup("Desktops");
				BerylIntToKde(cFiles, setting, option);
				cFiles->kwin->setGroup("Windows");
			}
			if (specialOptions[option].berylName == "resize_mode")
			{
				QString mode = cFiles->kwin->readEntry("ResizeMode");
				QString val = "Opaque";
				if (strcmp(setting->value.value.as_string,"Outline") == 0
					|| strcmp(setting->value.value.as_string,"Filled Outline") == 0)
				{
					val = "Transparent";
				}
				if (mode != val)
				{
					cFiles->modified = true;
					cFiles->kwin->writeEntry("ResizeMode",val);
				}
				cFiles->beryl->writeEntry(specialOptions[option].berylName + " (Integrated)",setting->value.value.as_string);
			}
			break;
		default:
			break;
	}
}

void write_setting(BerylSettingsContext * c, BerylSetting * setting)
{
	ConfigFiles *cFiles = (ConfigFiles *)c->backend_private_ptr;
	KSimpleConfig *cfg = cFiles->beryl;

	QString key(setting->name);
	QString group(setting->parent->name);
	if (setting->is_screen)
		group += "_screen";
	cfg->setGroup(group);

	if (beryl_settings_context_get_de_integration_enabled(c) && isIntegratedOption(setting))
	{
		writeIntegratedOption(cFiles, setting);
		return;
	}

	if (setting->is_default)
	{
		if (cfg->hasKey(key))
			cfg->deleteEntry(key);
		return;
	}

	switch (setting->type)
	{
		case BERYL_SETTING_TYPE_STRING:
			cfg->writeEntry(key,setting->value.value.as_string);
			break;
		case BERYL_SETTING_TYPE_FLOAT:
			cfg->writeEntry(key,setting->value.value.as_float);
			break;
		case BERYL_SETTING_TYPE_INT:
			cfg->writeEntry(key,setting->value.value.as_int);
			break;
		case BERYL_SETTING_TYPE_BOOL:
			cfg->writeEntry(key,bool(setting->value.value.as_bool));
			break;
		case BERYL_SETTING_TYPE_COLOR:
			{
				QValueList<int> list;
				list.append(setting->value.value.as_color.color.red);
				list.append(setting->value.value.as_color.color.green);
				list.append(setting->value.value.as_color.color.blue);
				list.append(setting->value.value.as_color.color.alpha);
				cfg->writeEntry(key,list);
			}
			break;
		case BERYL_SETTING_TYPE_LIST:
			{
				switch(setting->info.for_list.list_of_type)
				{
					case BERYL_SETTING_TYPE_BOOL:
						{
							QValueList<int> list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_bool,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					case BERYL_SETTING_TYPE_INT:
						{
							QValueList<int> list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_int,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					case BERYL_SETTING_TYPE_STRING:
						{
							QStringList list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_string,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					case BERYL_SETTING_TYPE_FLOAT:
						{
							QStringList list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_float,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					case BERYL_SETTING_TYPE_COLOR:
						{
							QStringList list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_color,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					case BERYL_SETTING_TYPE_BINDING:
						{
							QStringList list;
							g_slist_foreach(setting->value.value.as_list,(GFunc)copy_binding,&list);
							cfg->writeEntry(key,list,';');
						}
						break;
					default:
						break;
				}
			}
			break;
		case BERYL_SETTING_TYPE_BINDING:
			{
				gboolean bool_val;
				if (beryl_setting_get_can_set_key(setting,&bool_val)
						&&bool_val)
				{
					if (beryl_setting_value_get_key_enabled(
								&setting->value,&bool_val)&&bool_val)
						cfg->writeEntry(key + "_key_enabled",true);
					else
						cfg->writeEntry(key + "_key_enabled",false);

					gint ire;
					QString binding;
					QString keysym;

					if (beryl_setting_value_get_keymods(&setting->value,&ire))
						binding += beryl_settings_mods_to_string(ire);

					if (beryl_setting_value_get_keysym(&setting->value,&ire))
						keysym = XKeysymToString(ire);
					if (keysym.isEmpty()) keysym = "None";

					binding += keysym;
					cfg->writeEntry(key + "_key",binding);
				}
				if (beryl_setting_get_can_set_button(setting,&bool_val)
						&&bool_val)
				{
					if (beryl_setting_value_get_button_enabled(
								&setting->value,&bool_val)&&bool_val)
						cfg->writeEntry(key + "_button_enabled",true);
					else
						cfg->writeEntry(key + "_button_enabled",false);


					gint ire;
					QString binding;
					QString button = "Any";

					if (beryl_setting_value_get_buttonmods(&setting->value,&ire))
						binding += beryl_settings_mods_to_string(ire);

					if (beryl_setting_value_get_button(&setting->value,&ire))
						button = "Button" + QString::number(ire);

					binding += button;
					cfg->writeEntry(key + "_button",binding);
				}
				if (beryl_setting_get_can_set_edgemask(setting,&bool_val)
						&&bool_val)
				{
					int ema;
					QString edge = "None";

					if (beryl_setting_value_get_edgemask(&setting->value,&ema))
					{
						int i;
						for (i=0;i<8;i++)
							if (ema & 1<<i)
								edge = edgeName[i];
					}
					cfg->writeEntry(key + "_edge",edge);
				}
				if (beryl_setting_get_can_set_bell(setting,&bool_val)
						&&bool_val)
				{
					if (beryl_setting_value_get_bell(&setting->value,&bool_val))
						cfg->writeEntry(key + "_bell",bool(bool_val));
				}
			}
			break;
		default:
			kdDebug () << "Not supported setting type : " << setting->type << endl;
			break;
	}
}

gboolean read_init(BerylSettingsContext * c)
{
	if (!instance)
		instance = new KInstance("beryl-kconfig");

	ConfigFiles *cFiles = new ConfigFiles;
	QString configName("berylrc");
	if (beryl_settings_context_get_profile(c))
	{
		configName += ".";
		configName += beryl_settings_context_get_profile(c);
	}
	cFiles->beryl = new KSimpleConfig(configName);
	cFiles->kwin = new KConfig("kwinrc",true);
	cFiles->global = new KConfig("kdeglobals",true);
	cFiles->kwin->setGroup("Windows");
	cFiles->global->setGroup("Global Shortcuts");

	c->backend_private_ptr=(void *)cFiles;
	return TRUE;
}

void read_done(BerylSettingsContext * c)
{
	ConfigFiles *cFiles = (ConfigFiles *)c->backend_private_ptr;
	delete cFiles->beryl;
	delete cFiles->kwin;
	delete cFiles->global;
	delete cFiles;
	c->backend_private_ptr=NULL;
}

gboolean write_init(BerylSettingsContext * c)
{
	if (!instance)
		instance = new KInstance("beryl-kconfig");

	ConfigFiles *cFiles = new ConfigFiles;
	QString configName("berylrc");
	if (beryl_settings_context_get_profile(c))
	{
		configName += ".";
		configName += beryl_settings_context_get_profile(c);
	}
	cFiles->beryl = new KSimpleConfig(configName);
	cFiles->kwin = new KConfig("kwinrc");
	cFiles->global = new KConfig("kdeglobals");
	cFiles->kwin->setGroup("Windows");
	cFiles->global->setGroup("Global Shortcuts");
	cFiles->modified = false;

	c->backend_private_ptr=(void *)cFiles;
	return TRUE;
}

void write_done(BerylSettingsContext * c)
{
	ConfigFiles *cFiles = (ConfigFiles *)c->backend_private_ptr;
	if (cFiles->modified)
	{
		printf("sending command\n");
		KIPC::sendMessageAll( KIPC::SettingsChanged, KApplication::SETTINGS_SHORTCUTS );
		DCOPClient *client = kapp->dcopClient();
		if (!client->isAttached())
        	client->attach();
    	client->send("kwin", "KWinInterface", "reconfigure()", "");
	}
	delete cFiles->beryl;
	delete cFiles->kwin;
	delete cFiles->global;
	delete cFiles;
	c->backend_private_ptr=NULL;
}

gboolean backend_init (BerylSettingsContext *)
{
	if (!instance)
		instance = new KInstance("beryl-kconfig");
	return TRUE;
}

gboolean backend_fini (BerylSettingsContext *)
{
    return TRUE;
}

const gchar * get_short_desc(void)
{
	return "KDE Configuration Backend";
}

gboolean get_supports_integration(void)
{
	return TRUE;
}

gboolean delete_profile(gchar * profile)
{
	QString file(KGlobal::dirs()->saveLocation("config", QString::null, false));
	file += "berylrc";
	if (profile && strlen(profile))
	{
		file += ".";
		file += profile;
	}

	if (QFile::exists(file))
		return QFile::remove(file);

	return FALSE;
}
