package Cache::Apt::Config;
use Data::Dumper;
use Exporter;
use Carp;
use strict;
use warnings;
use vars qw(@ISA @EXPORT @depends $arch $verbose $suite $mysources
$sourcesname );
@ISA=qw(Exporter);
@EXPORT=qw( set_download_mode get_suite set_suite use_mysources check_mysources
get_cache_architecture check_cache_arch use_hostsources get_sourceslist
get_cache_apt_version );

=pod

=head1 Name

Cache::Apt::Config - Configuration handler for apt-cross and related tools

=head1 Description

Extensively modified portion of NorthernCross using only the
Config and Lookup modules and exposing the package details using a dedicated struct.

=head1 Copyright and Licence

=over

=item *

 Copyright (C) 2006, 2007  Neil Williams <codehelp@debian.org>

=item *

 Copyright (c) 2006, 2007  Alexander Shishkin <alexander.shishkin@siemens.com>

=back

 This package is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

=cut

# provide configuration variables to various parts of Northern Cross

our %commands = (
	native_pkgtool => '/usr/bin/dpkg',
	cross_pkgtool  => '/usr/bin/dpkg-cross',
	# allow aptitude
	aptget_tool    => '/usr/bin/apt-get',
	zcat           => '/bin/zcat',
	gzip           => '/bin/gzip',
);

our %main = (
	# required settings
	suite      => '',
	arch       => '',
	maindir    => '',
	#calculated value
	config_str => '',
	# other values.
	extra_conf => '',  # any apt config can be added here.
	tmp_dir   => '/tmp',
);
my @obligatory_main = qw/arch suite maindir/;

sub new
{
	my $class = shift;
	my %opt = @_;
	for (@obligatory_main) {
		croak "Bug: obligatory configuration value '$_' not given." unless defined $opt{$_};
	}
	map {
		$main{$_} = $opt{$_};
	} keys %opt;
	$main{config_str} = '';
	# build the config string that replaces the apt.conf files.
	# ALL options to pass to apt live here!
	$main{config_str} .= " -o Apt::Architecture=" . $main{arch}
		if ($main{arch} ne '');
	$main{config_str} .= " -o APT::Get::List-Cleanup=off";
	# make Install-Recommends configurable.
	$main{config_str} .= " -o Apt::Install-Recommends=false";
	$main{config_str} .= " -o Dir=" . $main{maindir};
	$main{config_str} .= " -o Dir::Etc=" . $main{maindir};
	$sourcesname = "sources." . $main{suite};
	$sourcesname = $mysources if (defined($mysources));
	$main{config_str} .= " -o Dir::Etc::SourceList=$sourcesname";
	$main{config_str} .= " -o Dir::State=" . $main{suite};
	$main{config_str} .= " -o Dir::State::Status=" . $main{maindir} . '/status';
	$main{config_str} .= " -o Dir::Cache=" . $main{suite};
	$main{config_str} .= " " . $main{extra_conf} if ($main{extraconf});
	my $self = {
		main => \%main,
		cmd  => \%commands,
	};
	bless $self, $class;

	return $self;
}

=head1 set_download_mode

Add Apt configuration options to the current
configuration to set Apt::Get::Assume-Yes,
Apt::Get::Print-URI, Apt::Get::Download-only
ready to identify the location of files needing
to be downloaded from apt sources.

=cut

sub set_download_mode
{
	$verbose = Cache::Apt::Lookup::get_verbose();
	my $config = shift;
	my $base = $config->{'main'}->{'config_str'};
	# this output is not displayed so keep quiet.
	my $str = " -q=2 ";
	$str .= " -o APT::Get::Assume-Yes=yes ";
	$str .= " -o APT::Get::Print-URIs=yes ";
	$str .= " -o APT::Get::Download-Only=yes ";
	$config->{'main'}->{'config_str'} = "$str $base";
	print Dumper($config) if ($verbose >= 4);
	return $config->{'main'}->{'config_str'};
}

=head1 get_cache_architecture

Reimplemented from dpkg-cross to remove an extra dependency.

=cut

sub get_cache_architecture {
	return $ENV{'DPKGCROSSARCH'} || $ENV{'ARCH'} || $arch;
}

=head1 check_cache_arch($arch)

Reimplemented from dpkg-cross to remove an extra dependency.

Checks that the supplied $arch is (or can be converted to)
a DEB_HOST_GNU_TYPE that is known to dpkg-architecture. Sets the
module $arch variable as a side-effect.

returns the DPKG_HOST_GNU_TYPE or undef

=cut

sub check_cache_arch {
	my $check = shift;
	my $deb_host_gnu_type;
	chomp($deb_host_gnu_type = `CC="" dpkg-architecture -f -a$check -qDEB_HOST_GNU_TYPE 2> /dev/null`);
	$arch = $check if (defined($deb_host_gnu_type));
	return $deb_host_gnu_type || undef;
}

=head1 set_suite($suite)

Override the &get_suite default to set a specific repository codename
that must be supported by all sources in your sources.list.

=cut

sub set_suite {
	$suite = shift;
}

=head1 get_sourceslist ()

Returns the name of the current sources list file.

=cut

sub get_sourceslist {
	return $sourcesname;
}

sub check_mysources {
	return $mysources or undef;
}

=head1 use_mysources ($sources_file)

Specify that the cache can only use the specified sources.list file
instead of collating a complete set of sources from the system apt
configuration.

=cut

sub use_mysources
{
	$mysources = shift;
	my $cachedir = &Cache::Apt::Lookup::get_cachedir;
	if (not -f "$cachedir/$mysources")
	{
		carp ("Unable to find $cachedir/$mysources, ignoring.");
		undef $mysources;
		return;
	}
	$sourcesname = $mysources;
	&Cache::Apt::Lookup::update_sources;
}

=head1 use_hostsources

Remove the override and return to the default system sources.

=cut

sub use_hostsources
{
	undef $mysources if (defined($mysources));
}

=head1 get_suite

Logic to identify the current suite from apt-cache policy.

This comes from emdebian-tools - migrate to NorthernCross in due course.

=cut

sub get_suite
{
	return $suite if (defined($suite));
	my ($cur_pri, $max_pri, $max_pri_suite, $suite);
	# list of suites supported by Emdebian toolchain repository.
	my @suites = qw/unstable testing stable sid lenny etch/;
	# Improved to detect priority. Ben Hutchings <ben@decadent.org.uk>
	open POLICY, "apt-cache policy |" or carp "Unable to read apt policy: $!";
	return "unstable" unless <POLICY>;
	for (<POLICY>) {
		if (/^\s*(\d+)\s(\w+):.*\s+(\w+)\/\w+\/?\w*\sPackages$/) {
			$cur_pri = $1;
			# Improved to detect cdrom only installs. Anthony Sharobaiko <anthony@fiord.com>
			if ((!defined($max_pri) || $cur_pri > $max_pri) && $2 eq "cdrom") {
				$max_pri=$cur_pri;
				$max_pri_suite=$3;
			}
		} elsif (defined($cur_pri)
			&& (!defined($max_pri) || $cur_pri > $max_pri)
			&& /^\s+release .*o=Debian,a=(\w+),l=Debian,c=main\s*$/) {
		$max_pri = $cur_pri;
		$max_pri_suite = $1;
		}
	}
	close POLICY;
	if (defined($max_pri_suite)) {
		$suite = $max_pri_suite;
	} else {
		carp ("Unable to determine apt-cache policy for Debian main! - using unstable.");
		return "unstable";
	}
	foreach my $s (@suites)
	{
		# check the matched value is sensible.
		return $s if ($suite eq $s);
	}
	carp ("Unable to use current suite: $suite is unsupported in Emdebian! - assuming unstable.");
	return "unstable";
}

=head1 get_cache_apt_version

Uses dpkg-query to retrieve the version string for the currently
installed libcache-apt-perl package. Falls back to a default version
if dpkg-query fails.

=cut

sub get_cache_apt_version
{
	my $fallback = "0.13.0";
	my $dpkg=`dpkg-query -W -f='\${Version}' libcache-apt-perl 2>/dev/null`;
	return $dpkg if ($dpkg !~ /^\s*$/);
	return $fallback;
}

1
