
//#define LOCAL_DEBUG
#include "debug.h"

#include "job.h"
#include <cstdio>
#include <stdexcept>
#include <limits>
using namespace MYSTD;

#include "conn.h"
#include "acfg.h"
#include "fileitem.h"
#include "dlcon.h"
#include "sockio.h"
#include "fileio.h" // for ::stat
#include <dirent.h>
#include <algorithm>
#include "maintenance.h"

#ifdef HAVE_LINUX_SENDFILE
#include <sys/sendfile.h>
#endif

#include <errno.h>

#define REPLEVEL (m_type==rechecks::FILE_INDEX ? 4 : (m_type==rechecks::FILE_PKG ? 5 : SPAMLEVEL))


#define SPECIAL_FD -42
inline bool IsValidFD(int fd) { return fd>=0 || SPECIAL_FD == fd; }


/*
 * Unlike the regular store-and-forward file item handler, this ones does not store anything to
 * harddisk. Instead, it uses the download buffer and lets job object send the data straight from
 * it to the client.
 */
class tPassThroughFitem : public fileitem
{
protected:

	const char *m_pData;
	size_t m_nConsumable, m_nConsumed;

public:
	tPassThroughFitem(MYSTD::string s) : fileitem(s),
	m_pData(NULL), m_nConsumable(0), m_nConsumed(0)
	{
		m_bAllowStoreData=false;
		m_nSizeChecked = m_nSizeSeen = 0;
	};
	virtual FiStatus Setup(bool)
	{
		m_nSizeChecked = m_nSizeSeen = 0;
		return status = FIST_INITED;
	}
	virtual void Unreg() {
		setLockGuard;
		if(FIST_DLRECEIVING == status)
		{
			// no shit, there is some input waiting? bring the sender down ASAP
			// this is not nice to the sender because it triggers a reconnection, but
			// it's a local source so it shouldn't hurt much
			status = FIST_ERRNOUSER;
			notifyAll();
		}
	};
	virtual int GetFileFd() { return SPECIAL_FD; }; // something, don't care for now
	virtual bool DownloadStartedStoreHeader(const header & h, const char *)
	{
		setLockGuard;
		m_head=h;
		status=FIST_DLGOTHEAD;
		return true;
	}
	virtual bool StoreFileData(const char *data, unsigned int size)
	{
		setLockGuard;

		LOGSTART2("tPassThroughFitem::StoreFileData", "status: " << status);

		// something might care, most likely... also about BOUNCE action
		notifyAll();

		m_nIncommingCount += size;

		dbgline;
		if (status >= FIST_ERROR || status < FIST_DLGOTHEAD)
			return false;

		if (size == 0)
			status = FIST_COMPLETE;
		else
		{
			dbgline;
			status = FIST_DLRECEIVING;
			m_nSizeChecked += size;
			m_pData = data;
			m_nConsumable=size;
			m_nConsumed=0;
			while(0 == m_nConsumed && status<FIST_ERROR)
				wait();

			dbgline;
			// let the downloader abort?
			if(status >= FIST_ERROR)
				return false;

			dbgline;
			m_nConsumable=0;
			m_pData=NULL;
			return m_nConsumed;
		}
		return true;
	}
	ssize_t SendData(int out_fd, int, off_t &nSendPos, size_t nMax2SendNow)
	{
		setLockGuard;

		while(0 == m_nConsumable && status<FIST_ERROR
				&& ! (m_nSizeChecked==0 && status==FIST_COMPLETE))
		{
			wait();
		}
		if (status >= FIST_ERROR || !m_pData)
			return -1;

		if(!m_nSizeChecked)
			return 0;

		ssize_t r = write(out_fd, m_pData, min(nMax2SendNow, m_nConsumable));
		if (r < 0 && (errno == EAGAIN || errno == EINTR)) // harmless
			r = 0;
		if(r<0)
		{
			status=FIST_ERROR;
		}
		else if(r>0)
		{
			m_nConsumable-=r;
			m_pData+=r;
			m_nConsumed+=r;
			nSendPos+=r;
		}
		notify();
		return r;
	}
};

// fileitem replacement with custom storage for local data
class tGeneratedFitemBase : public fileitem
{
public:
	virtual void Unreg() {}; // NOOP, since this item is not shared
	virtual int GetFileFd() { return SPECIAL_FD; }; // something, don't care for now
	tSS m_data;

	tGeneratedFitemBase(const string &sFitemId, const char *szFrontLineMsg) : fileitem(sFitemId), m_data(256)
	{
		status=FIST_COMPLETE;
		m_head.type = header::ANSWER;
		m_head.frontLine = "HTTP/1.1 ";
		m_head.frontLine += (szFrontLineMsg ? szFrontLineMsg : "500 Internal Failure");
		m_head.set(header::CONTENT_TYPE, _SZ2PS("text/html") );
	}
	ssize_t SendData(int out_fd, int, off_t &nSendPos, size_t nMax2SendNow)
	{
		if (status >= FIST_ERROR || out_fd<0)
			return -1;
		int r = m_data.syswrite(out_fd, nMax2SendNow);
		if(r>0) nSendPos+=r;
		return r;
	}
	inline void seal()
	{
		// seal the item
		m_nSizeChecked = m_data.size();
		m_head.set(header::CONTENT_LENGTH, m_nSizeChecked);
	}
};

ssize_t sendfile_generic(int out_fd, int in_fd, off_t *offset, size_t count)
{
	LOGSTART("sendfile_generic (fallback)");
	char buf[4096];
	ssize_t totalcnt=0;
	
	if(!offset)
	{
		errno=EFAULT;
		return -1;
	}
	if(lseek(in_fd, *offset, SEEK_SET)== (off_t)-1)
		return -1;
	while(count>0)
	{
		ssize_t readcount=read(in_fd, buf, count>sizeof(buf)?sizeof(buf):count);
		if(readcount<=0)
		{
			if(errno==EINTR || errno==EAGAIN)
				continue;
			else
				return readcount;
		}
		
		*offset+=readcount;
		totalcnt+=readcount;
		count-=readcount;
		
		ssize_t nPos(0);
		while(nPos<readcount)
		{
			ssize_t r=write(out_fd, buf+nPos, readcount-nPos);
			if(r==0) continue; // not nice but needs to deliver it
			if(r<0)
			{
				if(errno==EAGAIN || errno==EINTR)
					continue;
				else
					return r;
			}
			nPos+=r;
		}
	}
	return totalcnt;
}

ssize_t fileitem::SendData(int out_fd, int in_fd, off_t &nSendPos, size_t count)
{
#ifndef HAVE_LINUX_SENDFILE
	return sendfile_generic(out_fd, in_fd, &nSendPos, count);
#else
	ssize_t r=sendfile(out_fd, in_fd, &nSendPos, count);

	if(r<0 && (errno==ENOSYS || errno==EINVAL))
		return sendfile_generic(out_fd, in_fd, &nSendPos, count);
	else
		return r;
#endif
}


job::job(header *h, con *pParent) :
	m_filefd(-1),
	m_pParentCon(pParent),
	m_bChunkMode(false),
	//m_bCloseCon(false),
	m_bIsHttp11(false),
	m_state(STATE_SEND_MAIN_HEAD),
	m_backstate(STATE_TODISCON),
	m_pReqHead(h),
	m_nSendPos(0),
	m_nRangeLast(OFFT_MAX-1), // range size should never overflow
	m_nAllDataCount(0),
	m_nChunkRemainingBytes(0),
	m_type(rechecks::FILE_INVALID)
{
	LOGSTART2("job::job", "job creating, " << m_pReqHead->frontLine << " and this: " << uintptr_t(this));
}

string miscError("(HTTP error page)");

job::~job()
{
	LOGSTART("job::~job");

	bool bErr=m_sFileLoc.empty();
	m_pParentCon->LogDataCounts(
			( bErr ? (m_pItem ? m_pItem->GetHttpMsg() : miscError ) : m_sFileLoc ),
			m_pReqHead->h[header::XFORWARDEDFOR],
			(m_pItem ? m_pItem->GetTransferCount() : 0),
			m_nAllDataCount, bErr);
	
	checkforceclose(m_filefd);

	fileitem::UnregOneASAP(m_pItem);

	delete m_pReqHead;
}

const string & GetMimeType(const string &path)
{
	tStrPos dpos = path.find_last_of('.');
	if (dpos != stmiss)
	{
		NoCaseStringMap::const_iterator it = acfg::mimemap.find(path.substr(
				dpos + 1));
		if (it != acfg::mimemap.end())
			return it->second;
	}
	static string nix("");
	return nix;
}

void replaceChars(string &s, const char *szBadChars, char goodChar)
{
	for(string::iterator p=s.begin();p!=s.end();p++)
		for(const char *b=szBadChars;*b;b++)
			if(*b==*p)
			{
				*p=goodChar;
				break;
			}
}

inline void job::HandleLocalDownload(const string &visPath,
		const string &fsBase, const string &fsSubpath)
{
	struct stat stbuf;
	string absPath = fsBase+SZPATHSEP+fsSubpath;
	if (0!=::stat(absPath.c_str(), &stbuf))
	{
		switch(errno)
		{
		case EACCES:
			SetErrorResponse("403 Permission denied");
			break;
		case EBADF:
		case EFAULT:
		case ENOMEM:
		case EOVERFLOW:
		default:
			//aclog::err("Internal error");
			SetErrorResponse("500 Internal server error");
			break;
		case ELOOP:
			SetErrorResponse("500 Infinite link recursion");
			break;
		case ENAMETOOLONG:
			SetErrorResponse("500 File name too long");
			break;
		case ENOENT:
		case ENOTDIR:
			SetErrorResponse("404 File or directory not found");
			break;
		}
		return;
	}
/*
	// simplified version, just converts a string into a page
	class bufferitem : public tPassThroughFitem, private string
	{
	public:
		bufferitem(const string &sId, const string &sData)
		: tPassThroughFitem(sId), string(sData)
		{
			status = FIST_COMPLETE;
			m_pData=c_str();
			m_nConsumable=length();
			m_nSizeChecked=m_nConsumable;
			m_head.frontLine.assign(_SZ2PS("HTTP/1.1 200 OK"));
			m_head.set(header::CONTENT_LENGTH, length());
			m_head.set(header::CONTENT_TYPE, _SZ2PS("text/html") );
			m_head.type=header::ANSWER;
		}
	};
*/

	if(S_ISDIR(stbuf.st_mode))
	{
		// unconfuse the browser
		if(!endsWithSzAr(visPath, SZPATHSEPUNIX))
		{
			class dirredirect : public tGeneratedFitemBase
			{
			public:	dirredirect(const string &visPath)
			: tGeneratedFitemBase(visPath, "301 Moved Permanently")
				{
					m_head.set(header::LOCATION, visPath+"/");
					m_data << "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">"
					"<html><head><title>301 Moved Permanently</title></head><body><h1>Moved Permanently</h1>"
					"<p>The document has moved <a href=\""+visPath+"/\">here</a>.</p></body></html>";

					seal();
				}
			};
			m_pItem.reset(static_cast<fileitem*>(new dirredirect(visPath)));
			return;
		}

		class listing: public tGeneratedFitemBase
		{
		public:
			listing(const string &visPath) :
				tGeneratedFitemBase(visPath, "200 OK")
			{
				seal(); // for now...
			}
		};
		listing *p=new listing(visPath);
		m_pItem.reset(static_cast<fileitem*>(p)); // assign to smart pointer ASAP, operations might throw
		tSS & page = p->m_data;

		page << "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">"
		"<html><head><title>Index of " << visPath << "</title></head>"
		"<body><h1>Index of " << visPath << "</h1>"
		"<table><tr><th>&nbsp;</th><th>Name</th><th>Last modified</th><th>Size</th></tr>"
		"<tr><th colspan=\"4\"><hr></th></tr>";

		DIR *dir = opendir(absPath.c_str());
		if (!dir) // weird, whatever... ignore...
			page<<"ERROR READING DIRECTORY";
		else
		{
			// quick hack, good enough
			MYSTD::map<string, tSS> sortMap;
			for(struct dirent dp, *pEndTest;
					0==readdir_r(dir, &dp, &pEndTest) && pEndTest; )
			{
				if (0!=::stat((absPath+SZPATHSEP+dp.d_name).c_str(), &stbuf))
					continue;

				bool bDir=S_ISDIR(stbuf.st_mode);

				char datestr[32]={0};
				struct tm tmtimebuf;
				strftime(datestr, sizeof(datestr)-1,
						"%d-%b-%Y %H:%M", localtime_r(&stbuf.st_mtime, &tmtimebuf));

				tSS &line = sortMap[string(bDir?"a":"b")+dp.d_name];
				if(bDir)
					line << "[DIR]";
				else if(startsWithSz(GetMimeType(dp.d_name), "image/"))
					line << "[IMG]";
				else
					line << "[&nbsp;&nbsp;&nbsp;]";
				line <<  "</td><td><a href=\"" << dp.d_name <<	+(bDir? "/\">" : "\">" ) << dp.d_name
						<< "</a></td><td>" << datestr << "</td><td align=\"right\">";
				if(bDir)
					line << "-";
				else
					line << stbuf.st_size;
			}
			closedir(dir);
			for(MYSTD::map<string, tSS>::const_iterator it=sortMap.begin(); it!=sortMap.end(); it++)
				page << "<tr><td valign=\"top\">" << it->second << "</td></tr>\r\n";
		}
		page << "<tr><td colspan=\"4\">";
		_AddFooter(page);
		page << "</td></tr></table></body></html>";
		p->seal();
		return;
	}
	if(!S_ISREG(stbuf.st_mode))
	{
		SetErrorResponse("403 Unsupported data type");
		return;
	}
	/*
	 * This variant of file item handler sends a local file. The
	 * header data is generated as needed.
	 */
	class tLocalGetFitem : public fileitem
	{
	public:
		tLocalGetFitem(string s, string sLocalPath, struct stat &stdata) : fileitem(s)
		{
			m_bAllowStoreData=false;
			m_sPath=sLocalPath;
			status=FIST_COMPLETE;
			m_nSizeChecked=m_nSizeSeen=stdata.st_size;
			m_bCheckFreshness=false;
			m_head.type=header::ANSWER;
			m_head.frontLine="HTTP/1.1 200 OK";
			m_head.set(header::CONTENT_LENGTH, stdata.st_size);
			const string &sMimeType=GetMimeType(sLocalPath);
			if(!sMimeType.empty())
				m_head.set(header::CONTENT_TYPE, sMimeType);
		};
		virtual void Unreg() {}; // NOOP, since this item is not shared
	};
	m_pItem.reset(static_cast<fileitem*>(new tLocalGetFitem(visPath, absPath, stbuf)));
}

void job::PrepareDownload() {

    LOGSTART("job::PrepareDownload");
    
#ifdef DEBUG
    acfg::localdirs["stuff"]="/tmp/stuff";
    aclog::err(m_pReqHead->ToString());
#endif

    string sRawUriPath, sPathResdiual;
    tHttpUrl tUrl; // parsed URL
        
    const acfg::tRepoData * pBackends(NULL); // appropriate backends
    const string * psVname(NULL); // virtual name for storage pool, if available
    
    FiStatus fistate(FIST_FRESH);
    bool bPtMode(false);
    bool bForceFreshnessChecks(false); // force update of the file, i.e. on dynamic index files?
	tStrVec tokens;
    
    if(m_pReqHead->type!=header::GET && m_pReqHead->type!=header::HEAD)
    	goto report_invpath;
    
	if (3 != Tokenize(m_pReqHead->frontLine, SPACECHARS, tokens))
		goto report_invpath;

	sRawUriPath = tokens[1];
	USRDBG(6, "Raw request URI: " << sRawUriPath);

	m_bIsHttp11=(tokens[2] == "HTTP/1.1");
	// a sane default? close for http 1.0, keep for 1.1.
	// But if set by the client, comply!
	m_bClientWants2Close =!m_bIsHttp11;
	if(m_pReqHead && m_pReqHead->h[header::CONNECTION])
		m_bClientWants2Close = 0==strncasecmp(m_pReqHead->h[header::CONNECTION], "close", 5);

    // "clever" file system browsing attempt?
	if(stmiss != sRawUriPath.find("..")
			|| stmiss != sRawUriPath.find("/_actmp")
			|| startsWithSz(sRawUriPath, "/_"))
		goto report_notallowed;

    MYTRY
	{

		if (0==sRawUriPath.compare(0, 12, "apt-cacher?/"))
		sRawUriPath.erase(0, 12);
		if (0==sRawUriPath.compare(0, 11, "apt-cacher/"))
		sRawUriPath.erase(11);
		
		if(!tUrl.SetHttpUrl(sRawUriPath))
		{
			m_sMaintCmd="/";
			return;
		}
		LOG("refined path: " << tUrl.sPath);

		if(!tUrl.sPort.empty() && tUrl.sPort!="80")
		{
			//USRDBG(3, "illegal port " << tUrl.sPort << "?" << tUrl.sHost << tUrl.sPath);
			if(acfg::port == tUrl.sPort)
				goto report_doubleproxy;

			goto report_invport;
		}

		// make a shortcut
		string & sPath=tUrl.sPath;

		// kill multiple slashes
		for(tStrPos pos; stmiss != (pos = sPath.find("//")); )
			sPath.erase(pos, 1);

		bPtMode=rechecks::MatchUncacheableRequest(tUrl.ToURI());

		LOG("input uri: "<<tUrl.ToURI()<<" , dontcache-flag? " << bPtMode);

		tStrPos nRepNameLen=acfg::reportpage.size();
		if(nRepNameLen>0)
		{
			if(0==tUrl.sHost.compare(0, nRepNameLen, acfg::reportpage))
			{
				m_sMaintCmd=tUrl.sHost;
				return;
			}
			if (tUrl.sHost == "style.css")
			{
				LOG("support CSS style file");
				m_sMaintCmd = "/style.css";
				return;
			}
		}

		for(tStrMap::const_iterator it=acfg::localdirs.find(tUrl.sHost);
				it != acfg::localdirs.end() ; )
		{
			string realSubPath;
			if(! UrlDecode(tUrl.sPath, realSubPath) || realSubPath.find("..") != stmiss)
				goto report_invpath; // what the...
			HandleLocalDownload(sRawUriPath, it->second, realSubPath);
			return;
		}

		// entered directory but not defined as local? Then 404 it with hints
		if(!sPath.empty() && endsWithSzAr(sPath, "/"))
		{
			LOG("generic user information page");
			m_sMaintCmd="/";
			return;
		}

		m_type = rechecks::GetFiletype(sPath);

		if ( m_type == rechecks::FILE_INVALID ) goto report_notallowed;
		
		// got something valid, has type now, trace it
		USRDBG(REPLEVEL, "Processing new job, "<<m_pReqHead->frontLine);

		// resolve to an internal location
		psVname = acfg::GetRepNameAndPathResidual(tUrl, sPathResdiual);
		
		if(psVname)
			m_sFileLoc=*psVname+SZPATHSEP+sPathResdiual;
		else
			m_sFileLoc=tUrl.sHost+tUrl.sPath;
		
		// FIXME: this sucks, belongs into the fileitem
		if(acfg::stupidfs)
		{
			// fix weird chars that we don't like in the filesystem
			replaceChars(tUrl.sPath, ENEMIESOFDOSFS, '_');
			replaceChars(tUrl.sHost, ENEMIESOFDOSFS, '_');
#ifdef WIN32
			replaceChars(tUrl.sPath, "/", '\\');
#endif
		}
	}
	MYCATCH(MYSTD::out_of_range&) // better safe...
	{
    	goto report_invpath;
    }
    
    bForceFreshnessChecks = ( ! acfg::offlinemode && m_type==rechecks::FILE_INDEX);

#ifdef QUATSCH
    // some users want to not be most up-to-date
	if(bForceFreshnessChecks && acfg::updinterval > 0)
	{
		struct stat stbuf;
		bForceFreshnessChecks=(0 == ::stat(m_sFileLoc.c_str(), &stbuf)
				&& ( time(0) - (UINT) stbuf.st_ctime) < acfg::updinterval);
	}
#endif

	m_pItem=fileitem::GetFileItem(m_sFileLoc);
    
    if(!m_pItem)
    {
    	if(acfg::debug)
    		aclog::err(string("Error creating file item for ")+m_sFileLoc);
    	goto report_overload;
    }
    
    fistate = m_pItem->Setup(bForceFreshnessChecks);
	LOG("Got initial file status: " << fistate);

	if (bPtMode && fistate != FIST_COMPLETE)
		fistate = _SwitchToPtItem(m_sFileLoc);

	if(fistate==FIST_COMPLETE)
		return; // perfect, done here

    if(acfg::offlinemode) { // make sure there will be no problems later in SendData or prepare a user message
    	// error or needs download but freshness check was disabled, so it's really not complete.
    	goto report_offlineconf;
    }
    dbgline;
    if( fistate < FIST_DLGOTHEAD) // needs a downloader 
    {
    	dbgline;
    	if(!m_pParentCon->SetupDownloader(m_pReqHead->h[header::XFORWARDEDFOR]))
    	{
    		if(acfg::debug)
    			aclog::err(string("Error creating download handler for ")+m_sFileLoc);
    		goto report_overload;
    	}
    	
    	dbgline;
    	MYTRY
		{
			if(psVname && NULL != (pBackends=acfg::GetBackendVec(psVname)))
			{
				LOG("Backends found, using them with " << sPathResdiual
						<< ", first backend: " <<pBackends->m_backends.front().ToURI());

				if(! bPtMode && rechecks::
						MatchUncacheableTarget(pBackends->m_backends.front().ToURI()+sPathResdiual))
				{
					fistate=_SwitchToPtItem(m_sFileLoc);
				}

				m_pParentCon->m_pDlClient->AddJob(m_pItem, pBackends, sPathResdiual);
			}
			else
			{
			    if(acfg::forcemanaged)
			    	goto report_notallowed;
			    LOG("Passing new job for " << tUrl.ToURI() << " to " << m_pParentCon->m_dlerthr);

				if(! bPtMode && rechecks::MatchUncacheableTarget(tUrl.ToURI()))
					fistate=_SwitchToPtItem(m_sFileLoc);

			    m_pParentCon->m_pDlClient->AddJob(m_pItem, tUrl);
			}
			ldbg("Download job enqueued for " << m_sFileLoc);
		}
		MYCATCH(MYSTD::bad_alloc&) // OOM, may this ever happen here?
		{
			if(acfg::debug)
				aclog::err("Out of memory");			    		
			goto report_overload;
		};
	}
    
	return;
    
report_overload:
	SetErrorResponse("503 Server overload, try later");
    return ;

report_notallowed:
	SetErrorResponse("403 Forbidden file type or location");
    USRDBG(5, sRawUriPath + " -- ACCESS FORBIDDEN");
    return ;

report_offlineconf:
	SetErrorResponse("503 Unable to download in offline mode");
	return;

report_invpath:
	SetErrorResponse("403 Invalid path specification");
    return ;

report_invport:
	SetErrorResponse("403 Prohibited port (configure Remap-...)");
    return ;

report_doubleproxy:
	SetErrorResponse("403 URL seems to be made for proxy but contains apt-cacher-ng port. "
    		"Inconsistent apt configuration?");
    return ;

}

#define THROW_ERROR(x) { if(m_nAllDataCount) return R_DISCON; SetErrorResponse(x); return R_AGAIN; }
job::eJobResult job::SendData(int confd, int nAllJobCount)
{
	LOGSTART("job::SendData");

	if(!m_sMaintCmd.empty())
	{
		DispatchAndRunMaintTask(m_sMaintCmd, confd, m_pReqHead->h[header::AUTHORIZATION]);
			return R_DISCON; // just stop and close connection
	}
	
	off_t nGooddataSize(0);
	FiStatus fistate(FIST_ERROR);
	header respHead; // template of the response header, e.g. as found in cache

	if (confd<0)
		return R_DISCON; // shouldn't be here

	if (m_pItem)
	{
		lockguard g(*m_pItem);
		
		while (true)
		{
			fistate=m_pItem->GetStatusUnlocked(nGooddataSize);
			
			dbgline;
			if (fistate>=FIST_ERROR)
			{
				const header &h = m_pItem->GetHeaderUnlocked();
				g.unLock(); // item lock must be released in order to replace it!
				if(m_nAllDataCount)
					return R_DISCON;
				if(h.h[header::XORIG])
					m_sOrigUrl=h.h[header::XORIG];
				// must be something like "HTTP/1.1 403 Forbidden"
				if(h.frontLine.length()>9 && h.getStatus()!=200)
					SetErrorResponse(h.frontLine.c_str()+9);
				else // good ungood? confused somewhere?!
					SetErrorResponse("500 Unknown error");
				return R_AGAIN;
			}
			
			dbgline;
			if(fistate==FIST_COMPLETE || (m_nSendPos < nGooddataSize && fistate>=FIST_DLGOTHEAD))
				break;
			
			dbgline;
			m_pItem->wait();
			
			dbgline;
		}
		
		respHead = m_pItem->GetHeaderUnlocked();

		if(respHead.h[header::XORIG])
			m_sOrigUrl=respHead.h[header::XORIG];

	}
	else if(m_state != STATE_SEND_BUFFER)
	{
		ASSERT(!"no FileItem assigned and no sensible way to continue");
		return R_DISCON;
	}

	USRDBG(10, "set/restoring state, fileitem " << size_t(m_pItem.get())
			<< " with fistate " << fistate << " and procstate " << m_state );

	while (true) // left by returning
	{
		MYTRY // for bad_alloc in members
		{
			switch(m_state)
			{
				case(STATE_SEND_MAIN_HEAD):
				{
					ldbg("STATE_FRESH");
					if(fistate<FIST_DLGOTHEAD) // be sure about that
						return R_AGAIN;
					m_state=STATE_SEND_BUFFER;
					m_backstate=STATE_HEADER_SENT; // could be changed while creating header
					const char *szErr = BuildAndEnqueHeader(fistate, nGooddataSize, respHead);
					if(szErr) THROW_ERROR(szErr);
					USRDBG(REPLEVEL, "Prepared response header for user: \n" << m_sendbuf );
					return R_AGAIN;
				}
				case(STATE_HEADER_SENT):
				{
					ldbg("STATE_HEADER_SENT");
					
					if( fistate < FIST_DLGOTHEAD)
					{
						ldbg("ERROR condition detected: starts activity while downloader not ready")
						return R_AGAIN;
					}

					m_filefd=m_pItem->GetFileFd();
					if(!IsValidFD(m_filefd)) THROW_ERROR("503 IO error");

					m_state=m_bChunkMode ? STATE_SEND_CHUNK_HEADER : STATE_SEND_PLAIN_DATA;
					ldbg("next state will be: " << m_state);
					continue;
				}
				case(STATE_SEND_PLAIN_DATA):
				{
					ldbg("STATE_SEND_PLAIN_DATA, max. " << nGooddataSize);

					// eof?
#define GOTOENDE { m_state=STATE_FINISHJOB ; continue; }
					if(m_nSendPos>=nGooddataSize)
					{
						if(fistate>=FIST_COMPLETE)
							GOTOENDE;
						LOG("Cannot send more, not enough fresh data yet");
						return R_AGAIN;
					}

					size_t nMax2SendNow=min(nGooddataSize-m_nSendPos, m_nRangeLast+1-m_nSendPos);
					ldbg("~sendfile: on "<< m_nSendPos << " up to : " << nMax2SendNow);
					int n = m_pItem->SendData(confd, m_filefd, m_nSendPos, nMax2SendNow);
					ldbg("~sendfile: " << n << " new m_nSendPos: " << m_nSendPos);

					if(n>0)
						m_nAllDataCount+=n;

					// shortcuts
					if(m_nSendPos>m_nRangeLast || (fistate==FIST_COMPLETE && m_nSendPos==nGooddataSize))
						GOTOENDE;
					
					if(n<0)
						THROW_ERROR("400 Client error");
					
					
					return R_AGAIN;
				}
				case(STATE_SEND_CHUNK_HEADER):
				{
					m_nChunkRemainingBytes=nGooddataSize-m_nSendPos;
					ldbg("STATE_SEND_CHUNK_HEADER for " << m_nChunkRemainingBytes);
					m_sendbuf << tSS::hex << m_nChunkRemainingBytes << tSS::dec
							<< (m_nChunkRemainingBytes ? "\r\n" : "\r\n\r\n");

					m_state=STATE_SEND_BUFFER;
					m_backstate=m_nChunkRemainingBytes ? STATE_SEND_CHUNK_DATA : STATE_FINISHJOB;
					continue;
				}
				case(STATE_SEND_CHUNK_DATA):
				{
					ldbg("STATE_SEND_CHUNK_DATA");

					if(m_nChunkRemainingBytes==0)
						GOTOENDE; // done
					int n = m_pItem->SendData(confd, m_filefd, m_nSendPos, m_nChunkRemainingBytes);
					if(n<0)
						THROW_ERROR("400 Client error");
					m_nChunkRemainingBytes-=n;
					m_nAllDataCount+=n;
					if(m_nChunkRemainingBytes<=0)
					{ // append final newline
						m_sendbuf<<"\r\n";
						m_state=STATE_SEND_BUFFER;
						m_backstate=STATE_SEND_CHUNK_HEADER;
						continue;
					}
					return R_AGAIN;
				}
				case(STATE_SEND_BUFFER):
				{
					/* Sends data from the local buffer, used for page or chunk headers.
					* -> DELICATE STATE, should not be interrupted by exception or throw
					* to another state path uncontrolled. */
					MYTRY
					{
						ldbg("prebuf sending: "<< m_sendbuf.c_str());
						int r=send(confd, m_sendbuf.rptr(), m_sendbuf.size(),
								m_backstate == STATE_TODISCON ? 0 : MSG_MORE);
						if (r<0)
						{
							if (errno==EAGAIN || errno==EINTR || errno == ENOBUFS)
								return R_AGAIN;
							return R_DISCON;
						}
						m_nAllDataCount+=r;
						m_sendbuf.drop(r);
						if(m_sendbuf.empty())
						{
							USRDBG(REPLEVEL, "Returning to last state, " << m_backstate);
							m_state=m_backstate;
							continue;
						}
					}
					MYCATCH(...)
					{
						return R_DISCON;
					}
					return R_AGAIN;
				}
				
				case(STATE_ALLDONE):
					LOG("State: STATE_ALLDONE?");
				case (STATE_ERRORCONT):
					LOG("or STATE_ERRORCONT?");
				case(STATE_FINISHJOB):
					LOG("or STATE_FINISHJOB");
					{
						if(m_bClientWants2Close)
							return R_DISCON;
						LOG("Reporting job done")
						return R_DONE;
					}
					break;
				case(STATE_TODISCON):
				default:
					return R_DISCON;
			}
		}
		MYCATCH(bad_alloc&) {
			// TODO: report memory failure?
			return R_DISCON;
		}
		//ASSERT(!"UNREACHED");
	}
}


inline const char * job::BuildAndEnqueHeader(const FiStatus &fistate,
		const off_t &nGooddataSize, header& respHead)
{
	LOGSTART("job::BuildAndEnqueHeader");

	if(respHead.type != header::ANSWER)
		return "500 Rotten Data";

#if 0 // defined DEBUG
	respHead.h[header::CONTENT_LENGTH]=0;
	respHead.set(header::TRANSFER_ENCODING, "chunked");
#endif

	// make sure that header has consistent state and there is data to send which is expected by the client
	bool bHasSendableData = false;
	int httpstatus = respHead.getStatus();
	LOG("State: " << httpstatus);
	if (!BODYFREECODE(httpstatus)) // never for not-unmodified&co.
	{
		bool bChunked = respHead.h[header::TRANSFER_ENCODING] && 0 == strncasecmp( respHead.h[header::TRANSFER_ENCODING], "chunk", 5);
		if (httpstatus == 200)
		{
			if (bChunked)
			{
				// chunked data, usually ok...
				if (respHead.h[header::CONTENT_LENGTH])
				{
					aclog::err("internal error: chunked transfer encoding and content length value are both set");
					respHead.del(header::CONTENT_LENGTH);
				}
				bHasSendableData = true;
			}
			else
				bHasSendableData	= respHead.h[header::CONTENT_LENGTH];
			LOG("has contlen? " << bool(respHead.h[header::CONTENT_LENGTH]));
		}
		else if (httpstatus >= 300)
		{
			bHasSendableData = ( (respHead.h[header::CONTENT_LENGTH]
			                                 && atoi(respHead.h[header::CONTENT_LENGTH])>0)
					|| bChunked);
		}
	}

	tSS &sb=m_sendbuf;
	sb.clear();
	sb << respHead.frontLine <<"\r\n";

	bool bGotLen(false);

	if(bHasSendableData)
	{
		LOG("has sendable content");
		if( ! respHead.h[header::CONTENT_LENGTH])
		{
			// unknown length but must have data, will have to improvise: prepare chunked transfer
			if ( ! m_bIsHttp11 ) // you cannot process this? go away
				return "505 HTTP version not supported for this file";
			m_bChunkMode=true;
			sb<<"Transfer-Encoding: chunked\r\n";
		}
		else if(200==httpstatus) // good data response with known length, can try some optimizations
		{
			LOG("has known content length, optimizing response...");

			// Handle If-Modified-Since and Range headers;
			// we deal with them equally but need to know which to use
			const char *pIfmo = m_pReqHead->h[header::RANGE] ?
					m_pReqHead->h[header::IFRANGE] : m_pReqHead->h[header::IF_MODIFIED_SINCE];

			// consider contents "fresh" for non-volatile data, or when "our" special client is there, or the client simply doesn't care
			bool bFreshnessForced = (m_type != rechecks::FILE_INDEX
				|| m_pReqHead->h[header::ACNGFSMARK] || !pIfmo);

			// this plays well with lighttpd <-> Apt, but may not work with the alternative time format
			// TODO: add format tolerant time comparer
			bool bIfModSeenAndChecked=(pIfmo && respHead.h[header::LAST_MODIFIED] &&
					0==strcmp(pIfmo, respHead.h[header::LAST_MODIFIED]));

			// is it fresh? or is this relevant? or is range mode forced?
			if(  bFreshnessForced || bIfModSeenAndChecked)
			{
				off_t nContLen=atoofft(respHead.h[header::CONTENT_LENGTH]);

				/*
				 * Range: bytes=453291-
				 * ...
				 * Content-Length: 7271829
				 * Content-Range: bytes 453291-7725119/7725120
				 */

				const char *pRange=m_pReqHead->h[header::RANGE];

				// working around a bug in old curl versions
				if(!pRange)
					pRange=m_pReqHead->h[header::CONTENT_RANGE];

				if(pRange)
				{
					off_t nRangeFrom(0), nRangeTo(0);
					int nRangeItems=sscanf(pRange, "bytes=" OFF_T_FMT
                            "-" OFF_T_FMT, &nRangeFrom, &nRangeTo);
                    // working around bad (old curl style) requests
                    if(nRangeItems <= 0)
                    {
                        nRangeItems=sscanf(pRange, "bytes "
                                OFF_T_FMT "-" OFF_T_FMT,
                                &nRangeFrom, &nRangeTo);
                    }
					if(nRangeItems == 1) // open-end? set the limit
						nRangeTo=nContLen-1;

					/*
					 * make sure that our client doesn't just hang here while the download thread is
					 * fetching from 0 to start position for many minutes. If the resumed position
					 * is beyond of what we already have, fall back to 200 (complete download).
					 */
					if(nRangeItems>0
						&&
						(  fistate==FIST_COMPLETE
							// or can start sending within this range
							|| (
									fistate >= FIST_DLGOTHEAD
									&& fistate <= FIST_COMPLETE
									&& nGooddataSize>=nRangeFrom
								)
							// don't care, found special hint from acngfs (kludge...)
							|| m_pReqHead->h[header::ACNGFSMARK]
							)
						)
					{
						USRDBG(20, "request range problem: ")
						// detect errors, out-of-range case
						if(nRangeTo>=nContLen || nRangeFrom>=nContLen || nRangeTo<nRangeFrom)
							return "416 Requested Range Not Satisfiable";

						m_nSendPos = nRangeFrom;
						m_nRangeLast = nRangeTo;
						// replace with partial-response header
						sb.clear();
						sb << "HTTP/1.1 206 Partial Response\r\nContent-Length: "
						 << (m_nRangeLast-m_nSendPos+1) <<
								"\r\nContent-Range: bytes "<< m_nSendPos
								<< "-" << m_nRangeLast << "/" << nContLen << "\r\n";
						bGotLen=true;
					}
				}
				else if(bIfModSeenAndChecked)
				{
					// file is fresh, and user sent if-mod-since -> fine
					return "304 Not Modified";
				}
			}
		}
		// has cont-len available but this header was not set yet in the code above
		if( !bGotLen && !m_bChunkMode)
			sb<<"Content-Length: "<<respHead.h[header::CONTENT_LENGTH]<<"\r\n";

		// OK, has data for user and has set content-length and/or range or chunked transfer mode, now add various meta headers...

		if(respHead.h[header::LAST_MODIFIED])
			sb<<"Last-Modified: "<<respHead.h[header::LAST_MODIFIED]<<"\r\n";

		sb<<"Content-Type: ";
		if(respHead.h[header::CONTENT_TYPE])
			sb<<respHead.h[header::CONTENT_TYPE]<<"\r\n";
		else
			sb<<"application/octet-stream\r\n";
	}
	else
	{
		sb<<"Content-Length: 0\r\n";

		m_backstate=STATE_ALLDONE;
	}

	sb << header::GenInfoHeaders();

	if(respHead.h[header::LOCATION])
		sb<<"Location: "<<respHead.h[header::LOCATION]<<"\r\n";

	if(!m_sOrigUrl.empty())
		sb<<"X-Original-Source: "<< m_sOrigUrl <<"\r\n";

	// whatever the client wants
	sb<<"Connection: "<<(m_bClientWants2Close?"close":"Keep-Alive")<<"\r\n";

	sb<<"\r\n";
	LOG("response prepared:" << sb);

	if(m_pReqHead->type==header::HEAD)
		m_backstate=STATE_ALLDONE; // simulated head is prepared but don't send stuff

	return 0;
}

FiStatus job::_SwitchToPtItem(const MYSTD::string &fileLoc)
{
	// Changing to local pass-through file item
	LOGSTART("job::_SwitchToPtItem");
	// exception-safe sequence
	tFileItemPtr tmp=m_pItem;
	m_pItem.reset();
	tmp->Unreg();
	m_pItem.reset(new tPassThroughFitem(m_sFileLoc));
	return m_pItem->Setup(true);
}


void job::SetErrorResponse(const char * errorLine, int nMinUserLogLevel)
{
	LOGSTART2("job::SetErrorResponse", errorLine);


	class erroritem: public tGeneratedFitemBase
	{
	public:
		erroritem(const string &sId, const char *szError) : tGeneratedFitemBase(sId, szError)
		{
			if(BODYFREECODE(m_head.getStatus()))
				return;
			// otherwise do something meaningful
			m_data << "<!DOCTYPE HTML PUBLIC \"-//IETF//DTD HTML 2.0//EN\">\n"
				"<html><head><title>" << szError << "</title>\n</head>\n<body><h1>"
				<< szError << "</h1></body></html>";
			m_head.set(header::CONTENT_TYPE, "text/html");
			seal();
		}
	};

	tFileItemPtr pNew(static_cast<fileitem*>(new erroritem("noid", errorLine)));

	fileitem::UnregOneASAP(m_pItem);

	m_pItem=pNew;

	//aclog::err(tSS() << "fileitem is now " << uintptr_t(m_pItem.get()));
	m_state=STATE_SEND_MAIN_HEAD;
}
