
#include "conserver.h"
#include "meta.h"
#include "lockable.h"
#include "conn.h"
#include "rfc2553emu.h"
#include "acfg.h"

#include "sockio.h"
#include <sys/un.h>
#include <signal.h>
#include <arpa/inet.h>
#include <errno.h>

#include <netdb.h>

#include <cstdio>
#include <list>
#include <map>
#include <iostream>

#include "debug.h"

using namespace MYSTD;

// for cygwin, incomplete ipv6 support
#ifndef AF_INET6
#define AF_INET6        23              /* IP version 6 */
#endif

namespace conserver
{

int g_sockunix(-1);
vector<int> g_vecSocks;

condition g_ThreadPoolCondition;
list<con*> g_freshConQueue;
int g_nStandbyThreads(0);
int g_nAllConThreadCount(0);
bool bTerminationMode(false);

// safety mechanism, detect when the number of incomming connections 
// is growing A LOT faster than it can be processed 
#define MAX_BACKLOG 200

void * ThreadAction(void *)
{
	list<con*> & Qu = g_freshConQueue;
	//USRDBG(3, "Thread started");

	lockguard g(g_ThreadPoolCondition);
	
	while (true)
	{
		if (bTerminationMode)
			break;

		if(Qu.empty())
		{
			g_nStandbyThreads++;
			while (Qu.empty())
				g_ThreadPoolCondition.wait();
			g_nStandbyThreads--;
		}

		con *c=Qu.front();
		Qu.pop_front();
		g.unLock();
	
		c->WorkLoop();
		delete c;
		
		g.reLock();
		if (g_nStandbyThreads >= acfg::tpstandbymax)
			goto stop_worker;
	}
	
	stop_worker:
	g_nAllConThreadCount--;
	//USRDBG(3, "Terminating thread");
	return NULL;
}

// Thread creation helper, global stuff must be protected while using this
inline bool _SpawnThreadWhenNeeded()
{
	if(g_nStandbyThreads>0)
		return true;

	// check the kill-switch
	if(g_nAllConThreadCount+1>=acfg::tpthreadmax)
		return false;

	pthread_t thr;
	pthread_attr_t attr; // be detached from the beginning
	bool bRet(false);
	
	if(pthread_attr_init(&attr))
		goto thr_deinit_attr;

	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	if(0 == pthread_create(&thr, &attr, ThreadAction, NULL))
	{
		g_nAllConThreadCount++;
		bRet=true;
	}

	thr_deinit_attr:
	pthread_attr_destroy(&attr);
	return bRet;
}

void SetupConAndGo(int fd, const char *szClientName=NULL)
{
	LOGSTART2("SetupConAndGo", fd);

	// thread pool control, and also see Shutdown(), protect from 
	// interference of OS on file descriptor management
	lockguard g(g_ThreadPoolCondition);
	
	if(!szClientName)
		szClientName="";
	
	USRDBG(1, "Client name: " << szClientName);

	con *c(NULL);
	MYTRY
	{
		c=new con(fd, szClientName);
		if(!c)
		{
#ifdef NO_EXCEPTIONS
			USRDBG(4, "Out of memory");
#endif
			goto failure_mode;
		}
	}
	MYCATCH(bad_alloc)
	{
		USRDBG(4, "Out of memory");
		goto failure_mode;
	}
	
	// DOS prevention
	if (g_freshConQueue.size() > MAX_BACKLOG)
	{
		USRDBG(4, "Worker queue overrun");
		goto failure_mode;
	}

	if (!_SpawnThreadWhenNeeded())
	    goto failure_mode;
	
	g_freshConQueue.push_back(c);
	g_ThreadPoolCondition.notifyAll();

	return;

	failure_mode: 
	if (c)
		delete c;
	USRDBG(3, "Connection setup error");
	forceShutdownClose(fd);
}

void CreateUnixSocket() {
	string & sPath=acfg::fifopath;
	struct sockaddr_un addr_unx;
	memset(&addr_unx, 0, sizeof(addr_unx));
	
	int jo=1;
	size_t size = sPath.length()+1+offsetof(struct sockaddr_un, sun_path);
	
	if(sPath.length()>sizeof(addr_unx.sun_path))
	{
		errno=ENAMETOOLONG;
		goto unx_err;
	}
	
	addr_unx.sun_family = AF_UNIX;
	strncpy(addr_unx.sun_path, sPath.c_str(), sPath.length());
	
	mkbasedir(sPath);
	unlink(sPath.c_str());
	
	g_sockunix = socket(PF_UNIX, SOCK_STREAM, 0);
	setsockopt(g_sockunix, SOL_SOCKET, SO_REUSEADDR, &jo, sizeof(jo));

	if (g_sockunix<0)
		goto unx_err;
	
	if (bind(g_sockunix, (struct sockaddr *)&addr_unx, size) < 0)
		goto unx_err;
	
	if (0==listen(g_sockunix, SO_MAXCONN))
	{
		g_vecSocks.push_back(g_sockunix);
		return;
	}
	
	unx_err:
	cerr << "Error creating Unix Domain Socket, ";
	cerr.flush();
	perror(acfg::fifopath.c_str());
	cerr << "Check socket file and directory permissions" <<endl;
	exit(EXIT_FAILURE);

}

void Setup()
{
	using namespace acfg;
	
	if (fifopath.empty() && port.empty())
	{
		cerr << "Neither TCP nor UNIX interface configured, cannot proceed.\n";
		exit(EXIT_FAILURE);
	}
	
	if (atoi(port.c_str())>0)
	{
		struct addrinfo    hints;
		memset(&hints, 0, sizeof hints);
		hints.ai_socktype = SOCK_STREAM;
		hints.ai_flags = AI_PASSIVE;
		hints.ai_family = 0;
		int yes(1);
		
		tStrVec sAdds;
		if(bindaddr.empty())
			sAdds.push_back(""); // one dummy entry to get one NULL later
		else
			Tokenize(bindaddr, SPACECHARS, sAdds);
		for(tStrVec::iterator it=sAdds.begin(); it!=sAdds.end(); it++)
		{
			trimString(*it);
		    struct addrinfo *res, *p;
		    
		    if(0!=getaddrinfo(bindaddr.empty() ? NULL : it->c_str(),
				   port.c_str(), &hints, &res))
			   goto error_getaddrinfo;
		    
		    for(p=res; p; p=p->ai_next)
		    {
		    	int sockip = socket(p->ai_family, p->ai_socktype, p->ai_protocol);
				if (sockip<0)
					goto error_socket;

          // if we have a dual-stack IP implementation (like on Linux) then
          // explicitely disable the shadow v4 listener. Otherwise it might be
          // bound, or maybe not, and then maybe because of the dual-behaviour,
          // or maybe because of real errors; we just cannot know for sure but
          // we have to.
#ifdef IPV6_V6ONLY
		    	if(p->ai_family==AF_INET6)
		    		setsockopt(sockip, SOL_IPV6, IPV6_V6ONLY, &yes, sizeof(yes));
#endif		    	
				setsockopt(sockip, SOL_SOCKET, SO_REUSEADDR, &yes, sizeof(yes));
				
				
		    	if (bind(sockip, p->ai_addr, p->ai_addrlen))
		    		goto error_bind;
		    	if (listen(sockip, SO_MAXCONN))
		    		goto error_listen;
		    	
		    	g_vecSocks.push_back(sockip);
		    	
		    	continue;

				error_socket:
				
				if(EAFNOSUPPORT != errno &&
						EPFNOSUPPORT != errno &&
						ESOCKTNOSUPPORT != errno &&
						EPROTONOSUPPORT != errno)
				{
					perror("Error creating socket");
				}
				goto close_socket;

				error_listen:
				perror("Couldn't listen on socket");
				goto close_socket;

				error_bind:
				perror("Couldn't bind socket");
				cerr.flush();
				if(EADDRINUSE == errno)
					cerr << "Port " << port << " is busy, see the manual (Troubleshooting chapter) for details." <<endl;
				cerr.flush();
				goto close_socket;

				close_socket:
				forceclose(sockip);
		    }
		    freeaddrinfo(res);
		    continue;
		    
		    error_getaddrinfo:
		    perror("Error resolving address for binding");
		}
		if(g_vecSocks.empty())
		{
			cerr << "No socket(s) could be create/prepared. "
			"Check the network, check or unset the BindAddress directive.\n";
			exit(EXIT_FAILURE);
		}
	}
	else
		aclog::err("Not creating TCP listening socket, no valid port specified!");

	if ( !acfg::fifopath.empty() )
		CreateUnixSocket();
	else
		aclog::err("Not creating Unix Domain Socket, fifo_path not specified");
}

int Run()
{
	LOGSTART("Run");
	fd_set rfds, wfds;
	int maxfd(0);
	for(vector<int>::iterator it=g_vecSocks.begin(); it!=g_vecSocks.end(); it++)
		if(*it > maxfd)
			maxfd=*it;
	maxfd++;
	USRDBG(1, "Listening to incoming connections...");

	while (1)
	{ // main accept() loop

		FD_ZERO(&rfds);
		FD_ZERO(&wfds);
		for(vector<int>::iterator it=g_vecSocks.begin(); it!=g_vecSocks.end(); it++)
			FD_SET(*it, &rfds);
		
		//cerr << "Polling..." <<endl;
		int nReady=select(maxfd, &rfds, &wfds, NULL, NULL);
		if (nReady<0)
		{
			if(errno == EINTR)
				continue;
			
			aclog::err("select", "failure");
			perror("Select died");
			exit(EXIT_FAILURE);
		}
		
		for(vector<int>::iterator it=g_vecSocks.begin(); it!=g_vecSocks.end(); it++)
		{
			if(!FD_ISSET(*it, &rfds)) 	continue;

			if(g_sockunix == *it)
			{
				int fd = accept(g_sockunix, NULL, NULL);
				if (fd>=0)
				{
					set_nb(fd);
					USRDBG(1, "Detected incoming connection from the UNIX socket");
					SetupConAndGo(fd);
				}
				else if(errno == EMFILE || errno == ENOMEM || ENOBUFS == errno)
				{
					// play nicely, give it a break
					sleep(1);
				}
			}
			else
			{
				struct sockaddr_storage addr;

				socklen_t addrlen = sizeof(addr);
				int fd=accept(*it,(struct sockaddr *)&addr, &addrlen);
//fd_accepted:
				if (fd>=0)
				{
					set_nb(fd);
					char hbuf[NI_MAXHOST];
					USRDBG(1, "Detected incoming connection from the TCP socket");
					if (getnameinfo((struct sockaddr*) &addr, addrlen, hbuf, sizeof(hbuf),
									NULL, 0, NI_NUMERICHOST))
					{
						aclog::err("ERROR: could not resolve hostname for incoming TCP host");
						forceShutdownClose(fd);
						sleep(1);
						continue;
					}
					SetupConAndGo(fd, hbuf);
				}
				else if(errno == EMFILE || errno == ENOMEM || ENOBUFS == errno)
				{
					// play nicely, give it a break
					sleep(1);
				}
			}
		}
	}
	return 0;
}

void Shutdown()
{
	lockguard g(g_ThreadPoolCondition);
	
	if(bTerminationMode)
		return; // double SIGWHATEVER? Prevent it.
	
	//for (map<con*,int>::iterator it=mConStatus.begin(); it !=mConStatus.end(); it++)
	//	it->first->SignalStop();
	// TODO: maybe call shutdown on all?
	//printf("Signaled stop to all cons\n");
	
	bTerminationMode=true;
	printf("Notifying waiting threads\n");
	g_ThreadPoolCondition.notifyAll();
	
	printf("Closing sockets\n");
	for(vector<int>::iterator it=g_vecSocks.begin(); it!=g_vecSocks.end(); it++)
		forceShutdownClose(*it);
}

}
