# $Id: ZOO.pm,v 1.10 2003/03/31 06:06:35 bengen Exp $

#
# Module for extracting ZOO files
#

package AMAVIS::Extract::ZOO;
use strict;
use vars qw($VERSION);
$VERSION='0.1';

use AMAVIS;
use AMAVIS::Logging;

use vars qw(
	    $cfg_zoo_binary
	    $cfg_unzoo_binary
	   );

sub init {
  my $self = shift;
  my $args = shift;
  my $types = shift;
  $cfg_zoo_binary=$AMAVIS::cfg->val('external', 'zoo');
  if (defined $cfg_zoo_binary) {
    if (! -x  $cfg_zoo_binary) {
      writelog($args,LOG_CRIT, __PACKAGE__.
	       ": ZOO binary ($cfg_zoo_binary) not executable");
      return 0;
    }
    writelog($args,LOG_DEBUG, __PACKAGE__.": Using zoo ($cfg_zoo_binary)");
    $cfg_unzoo_binary=undef;
  }
  else {
    $cfg_unzoo_binary=$AMAVIS::cfg->val('external', 'unzoo');
    if (! -x  $cfg_unzoo_binary) {
      writelog($args,LOG_CRIT, __PACKAGE__.
	       ": UNZOO binary ($cfg_unzoo_binary) not executable");
      return 0;
    }
    writelog($args,LOG_DEBUG, __PACKAGE__.": Using unzoo ($cfg_unzoo_binary)");
  }
  $$types{'application/x-zoo'}=$self;
  writelog($args,LOG_DEBUG,__PACKAGE__." initialized.");
  return 1;
}

sub extract ( $ ) {
  my $self = shift;
  my $args = shift;
  my $filename = shift;
  my $unpacked_size = 0;

  my @command;

  my $buffer;
  my $len;

  writelog($args,LOG_DEBUG, "Attempting to unpack $filename as ZOO file");

  # Zoo needs extension of .zoo!
  symlink("$$args{'directory'}/parts/$filename", 
	  "$$args{'directory'}/parts/$filename.zoo");

  # This is safe. No filenames specified from outside
  my @list;
  my $in_handle;

  if (defined $cfg_zoo_binary) {
    @command = ($cfg_zoo_binary, 'lf1q',
		"$$args{'directory'}/parts/$filename.zoo");
    writelog($args,LOG_DEBUG,__PACKAGE__.": Running ".join(' ',@command));
    $in_handle=cmd_pipe($args, @command);

    while (<$in_handle>) {
      chomp;
      push @list, $_;
    }
  } elsif (defined $cfg_unzoo_binary) {
    @command = ($cfg_unzoo_binary, '-l',
		"$$args{'directory'}/parts/$filename.zoo");
    writelog($args,LOG_DEBUG,__PACKAGE__.": Running ".join(' ',@command));
    $in_handle=cmd_pipe($args, @command);
    while (<$in_handle>) {
      last if (/^-+/);
    }
    while (<$in_handle>) {
      last if (/^-+/);
      my $filename;
      chomp;
      (undef,undef,undef,undef,undef,undef,undef,undef,$filename) = split /\s+/;
      push @list, $filename;
    }
    $in_handle->close();
  } else {
    return 0;
  }

  foreach (@list) {
    # Ignore directories
    unless (/.*\/$/) {
      if ($$args{'unpacked_files'}++ > $cfg_maxfiles) {
	writelog($args,LOG_ERR, __PACKAGE__.": Unpacking uses too many files");
	return 0;
      }
      my $in_handle;

      if (defined $cfg_zoo_binary) {
	@command = ($cfg_zoo_binary,
		    'xpqqq:',
		    "$$args{'directory'}/parts/$filename.zoo",
		    $_);
	writelog($args,LOG_DEBUG,__PACKAGE__.": Running ".join(' ',@command));
	$in_handle=cmd_pipe($args, @command);
      } elsif (defined $cfg_unzoo_binary) {
	@command = ($cfg_unzoo_binary,
		    '-x','-p',
		    "$$args{'directory'}/parts/$filename.zoo",
		    $_);
	writelog($args,LOG_DEBUG,__PACKAGE__.": Running ".join(' ',@command));
	$in_handle=cmd_pipe($args, @command);
	# Skip over the first three lines (which contain the file name)
	<$in_handle>;
	<$in_handle>;
	<$in_handle>;
      } else {
	return 0;
      }
      my $securename=get_secure_filename($args);
      my $out_handle=IO::File->new(">$$args{'directory'}/parts/"
				   .$securename);

      while ($len=$in_handle->read($buffer, 4096)) {
	$out_handle->write($buffer, $len);
	$unpacked_size += $len;
	if ($$args{'unpacked_size'} + $unpacked_size >= $cfg_maxspace) {
	  $in_handle->close();
	  $out_handle->close();
	  writelog($args,LOG_ERR, __PACKAGE__.": Unpacking takes too much space");
	  return 0;
	}
      }

      $ {$$args{'contents'}}{$securename} = {'original_filename' => $_};
      $ {$ {$$args{'contents'}}{$securename}}{insecure_type}='';
    }
  }
  unlink("$$args{'directory'}/parts/$filename.zoo");
  $$args{'unpacked_size'} += $unpacked_size;
  return 1;
}

1;
